## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  out.width = "47%",
  fig.show = "hold",
  fig.ext = "svg",
  dev = "svg"
)

# Load problematic correlogram for example
pcorr <- structure(c(0.500528302565943, -0.754836705515775, -1.19039679450543, 
  -0.178590255566502, -0.60652246495491, 1.82273156499271, -0.172010338128128, 
  -2.0163653383661, -0.3745369917274, 0.456123617810818, 1.67415100748476, 
  0.937900901537193, -1.29338484030548, -1.28069618875814, -1.82633767163309, 
  -0.614386402460464, 0.584995100586833, -0.504037627819764, 0.0226448302823769, 
  -1.20080447279195, -0.0617993438737834, 0.528551859910452, 1.96448337848478, 
  0.072350278901269, 0.225152117571247, -1.25610158595963, -0.845732815433606, 
  0.471680065854876, 2.4943925938456, 1.80772326821158, 0.204809764053051, 
  -0.150961614287815, -0.0197795008143637, -1.05994768012253, -1.29810950765603, 
  -1.17853411156364, -0.643232293492797, -2.20411574357046, 0.216829082430902, 
  -0.941814658830531, -0.781300200015838, 0.273580690612867, 1.09702483374736, 
  -1.36990396610849, -2.07605582194368, 0.451160815513836, 0.86688665896418, 
  0.51840092005946, 1.28921143353651, 0.875645430906388, 0.591064637882466, 
  -0.55966314736262, -0.367191668064688, -2.76744332433181, -1.2443630247938, 
  -0.154933017518784, 0.0667021828181572, 0.197007379842123, 0.490808179023661, 
  2.2336743342413, 1.89448119625069, 1.47138565779839, -0.666213196115124, 
  0.843870629377204, -0.203203819755488, -2.20410902950622, 0.827755498838514, 
  1.78219490528679, -0.111762758688384, -2.22360645019641, -1.0056276510069, 
  -0.329364603192913, 1.33056799025663, 0.149121700442515, 0.267676241642407, 
  -0.679653050876458, 0.0901160031947412, -0.563650739275465, 0.666301686016796, 
  1.70298732225847, 0.673722209645992, 1.32892165992321, -1.29016755347508, 
  -2.11926333043274, -0.434170672177059, -1.11952602901843, -0.282918307090735, 
  -0.296513315892018, -0.215455257897354, -0.229591553459229, 1.90335315883752, 
  -0.640041683494477, 0.555037751515986, 0.498140844212503, -0.295915459399637, 
  0.587207819554741, -0.52622399410093, 1.32416109800959, -0.825699074434715, 
  -2.51977385384327, -0.681811843738689, 1.0515603999965, 1.75547334518565, 
  1.46352961234024, 0.0972088740341473, -1.41469490750948, -2.39336234402329, 
  0.596660462796983, 2.29853427605131, 1.65898229920668, -0.0276102807086275, 
  -1.64374943302099, 0.602395878758738, 0.229442193055737, 2.1687970121186, 
  0.643910340438762, -0.290718077669911, -1.02619172899549, -0.433490300518998, 
  -0.978870909554381, -0.0098964640725574, -1.96478096657371, -0.0969197029820769, 
  -2.08344229475593, -0.293555313118738, -1.29271002623495, 0.768226868279432, 
  0.714533099091565, -0.739381839162954, 0.486811684636613, -0.0462968073811469, 
  0.119532030148799, 0.699661964254544, -0.121299551979732, -0.952343329737353, 
  0.47622151198097, -0.235031006098588, -0.274831514573784, -0.0760339102360359, 
  1.25475007283353, 1.46495001193988, 0.167876131248247, 0.191014261173205, 
  1.20954335642528, 1.84131238261517, 1.11873900227391, -0.948077984555249, 
  -1.06745345491371, -1.4245470888048, -0.538478752695991, -0.277037416300269, 
  -0.887735440660801, -0.960970972045782, -1.17008385676073, 0.291022149761318, 
  1.57923122715123, 2.55155997330415, 0.225931504246328, 0.00220891571648253, 
  -0.416842778537851, -0.607033513606461, -0.40792204210633, 0.0685179557733119, 
  1.14003218398751, 1.82522025910671, 1.35481822913805, 1.52147329773245, 
  0.328519767736851, -0.0225431382950235, -0.0632294307946625, 
  -0.34692001495444, 0.609522181403036, 1.18523376254118, -0.2922169564903, 
  0.486710723126883, 0.0171559573700382, -0.841704196640039, -0.170564928787282, 
  1.09088790395037, 1.01711383301683, -0.57576763235922, -0.813788055144079, 
  0.845138966533092, -0.506617097735509, -0.180391805588545, -0.0731710033636772, 
  0.650969458632428, -0.544567330307273, -0.697447901212486, -0.843571490826714, 
  -1.04624243603828, 0.862168240020156, 0.393550845622028, -1.48053698592407, 
  -1.750632331433, 1.47944447592151, 1.1384887083871, 1.19239984208863, 
  1.05781365301425, 0.231828681437716), .Tsp = c(1, 200, 1), class = "ts")

## ----setup--------------------------------------------------------------------
library(blocklength)
set.seed(32)

## ----Inspect Parameters-------------------------------------------------------
# Generate an AR(1) simulation
sim <- stats::arima.sim(list(order = c(1, 0, 0), ar = 0.5),
                        n = 500, innov = rnorm(500))

# Run the PWSD method and suppress output of the correlogram
b <- pwsd(sim, correlogram = FALSE)

# Inspect interim parameters
b$parameters

## ----Inpect b_max-------------------------------------------------------------
# Print block-length estimates
b$BlockLength

# Test if optimal block-length is censored by b_max
b$parameters[, "b_max"] == b$BlockLength

# Print the upper-bound
b$parameters[, "b_max"]

## ----Correlogram--------------------------------------------------------------
# Print rho_k_critical
b$parameters[, "rho_k_critical"]

# Plot correlogram from previous selection
plot(b, main = "c = qnorm(0.975)", ylim = c(-0.2, 1))

# Expand confidence level and plot again
b1 <- pwsd(sim, c = 4, correlogram = FALSE)
plot(b1, main = "c = 4", ylim = c(-0.2, 1))

## ----Convex MSE---------------------------------------------------------------
# Select a block-length using HHJ method
b2 <- hhj(sim, sub_sample = 10, k = "bias/variance", plots = FALSE)
plot(b2)

## ----Concave MSE--------------------------------------------------------------
# Generate a AR(2) simulation
sim2 <- stats::arima.sim(list(order = c(2, 0, 0), ar = c(0.5, 0.3)),
                        n = 500, innov = rnorm(500))

# Select a block-length using HHJ method
b3 <- hhj(sim2, plots = FALSE)
plot(b3)

## ----Problematic Correlogram, fig.align = "center", out.width = "80%"---------
# Select a block-length using PWSD method
b4 <- pwsd(pcorr, K_N = 5, correlogram = FALSE)
plot(b4, main = "Problematic Correlogram")

# Output m_hat to console
b4$parameters[, "m_hat"]

## ----Problematic Correlogram 2, fig.align = "center", out.width = "80%"-------
# Select block-length using PWSD method
b5 <- pwsd(pcorr, K_N = 6, correlogram = FALSE)
plot(b5, main = "Problematic Correlogram 2")

# Output m_hat to console
b5$parameters[, "m_hat"]

## ----Problematic Correlogram 3, fig.align = "center", out.width = "80%"-------
# Select a block-length using PWSD method
b6 <- pwsd(pcorr, c = qnorm(0.995), correlogram = FALSE)
plot(b6, main = "Problematic Correlogram 3")

# Output m_hat to console
b6$parameters[, "m_hat"]

