## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 8,
  fig.height = 5,
  fig.align = "center",
  out.width = "85%",
  dpi = 150
)

## ----eval=FALSE---------------------------------------------------------------
#  remotes::install_github("marceelrf/tidyspec")

## ----warning=FALSE,message=FALSE----------------------------------------------
library(tidyspec)
library(tidyverse)

## -----------------------------------------------------------------------------
CoHAspec

## -----------------------------------------------------------------------------
set_spec_wn("Wavenumber")
check_wn_col()

## ----fig.cap="Static spectral plot showing all samples"-----------------------
spec_smartplot(CoHAspec)

## ----ig.cap="Interactive spectral plot with line geometry"--------------------
spec_smartplotly(CoHAspec,geom = "line")

## ----fig.cap="Custom ggplot2 visualization of spectral data", fig.height=4----
CoHAspec %>% 
  tidyr::pivot_longer(cols = -Wavenumber,
                      names_to = "spectrums",
                      values_to = "absorbance") %>% 
  ggplot(aes(x = Wavenumber, y = absorbance, col = spectrums)) +
  geom_line() #<------ Customize your data from here

## -----------------------------------------------------------------------------
CoHAspec %>% 
    spec_abs2trans() %>%
    spec_smartplot()

## -----------------------------------------------------------------------------
CoHAspec %>% 
    spec_abs2trans() %>%
    spec_trans2abs() %>%
    spec_smartplot()

## -----------------------------------------------------------------------------
CoHAspec %>% 
    dplyr::select(Wavenumber,CoHA100) %>%
    tidyspec::spec_smartplot()

## -----------------------------------------------------------------------------
CoHAspec %>% 
    tidyspec::spec_select(CoHA100) %>%
    tidyspec::spec_smartplot()

## -----------------------------------------------------------------------------
CoHAspec_filt <- 
    CoHAspec %>% 
    tidyspec::spec_select(CoHA100) %>%
    dplyr::filter(Wavenumber > 1000,
                Wavenumber < 1950)

## ----out.width = "100%",fig.alt="With `spec_select`", fig.dpi=600-------------
CoHAspec_filt <- 
    CoHAspec %>% 
    spec_select(CoHA100) %>%
    spec_filter(wn_min = 1000,
                wn_max = 1950)

spec_smartplot(CoHAspec_filt, geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_filt %>% 
  spec_smooth_avg() %>%
  spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_filt %>% 
  spec_smooth_sga() %>%
  spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_filt %>% 
    tidyspec::spec_smooth_sga() %>%
    spec_diff(degree = 2) %>%
    tidyspec::spec_smartplot(geom = "line")

## ----out.width="100%", fig.dpi=600--------------------------------------------
CoHAspec_filt %>% 
    spec_smooth_sga() %>%
    spec_blc_rollingBall(wn_min = 1030,
                         wn_max = 1285,
                         ws = 10,
                         wm = 50) %>%
    tidyspec::spec_smartplot(geom = "line")

## ----out.width="100%", fig.dpi=600--------------------------------------------
CoHAspec_filt %>% 
    spec_smooth_sga() %>%
    spec_bl_rollingBall(wn_col = "Wavenumber",
                        wn_min = 1030,
                        wn_max = 1285,
                        ws = 10,
                        wm = 50) %>% 
  spec_smartplot()

## ----out.width="100%", fig.dpi=600--------------------------------------------
bl <- CoHAspec_filt %>% 
    spec_smooth_sga() %>%
    spec_bl_rollingBall(wn_col = "Wavenumber",
                         wn_min = 1030,
                         wn_max = 1285,
                         ws = 10, wm = 50)

CoHAspec_filt %>%
  spec_smooth_sga() %>%
  spec_filter(wn_min = 1030, wn_max = 1285) %>% 
  left_join(bl) %>% 
  spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_region <- CoHAspec  %>%
    dplyr::filter(Wavenumber > 1300,
                Wavenumber < 1950) 
CoHAspec_region %>% 
    spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_region %>%
    spec_norm_01() %>% 
    spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_region %>%
    spec_norm_minmax(min = 1, max = 2) %>% 
    spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
CoHAspec_region %>%
    spec_norm_var() %>% 
    spec_smartplot(geom = "line")

## -----------------------------------------------------------------------------
dados_1030_1285 <- CoHAspec_filt %>% 
    tidyspec::spec_smooth_sga() %>%
    dplyr::filter(Wavenumber <= 1285, Wavenumber >= 1030)


## -----------------------------------------------------------------------------
params <- tidyr::crossing(ws_val = c(2,4,6,8,10,12),
                          wm_val = c(10, 25, 40, 50))
params 

## -----------------------------------------------------------------------------
df <- params %>%
    dplyr::mutate(spectra = list(dados_1030_1285)) %>%
    dplyr::mutate(spectra_blc = purrr::pmap(list(spectra, ws_val, wm_val),
                             \(x, y, z ) spec_blc_rollingBall(.data = x, ws = y, wm = z))) %>%
    dplyr::mutate(title = paste0("ws = ", ws_val," , wm = ", wm_val )) %>%
    dplyr::mutate(plot = purrr::map2(spectra_blc, title, ~ spec_smartplot(
            .data = .x,
            wn_col = "Wavenumber",
            geom = "line") + labs(title = .y)
            ))

## -----------------------------------------------------------------------------
library(gridExtra)

grid.arrange(grobs = df$plot[1:4], nrow = 2, ncol = 2)
grid.arrange(grobs = df$plot[5:8], nrow = 2, ncol = 2)
grid.arrange(grobs = df$plot[9:12], nrow = 2, ncol = 2)
grid.arrange(grobs = df$plot[13:16], nrow = 2, ncol = 2)
grid.arrange(grobs = df$plot[17:20], nrow = 2, ncol = 2)
grid.arrange(grobs = df$plot[21:24], nrow = 2, ncol = 2)

