# -*- coding: utf-8 -*-
"""
Created on Wed Oct  8 11:06:10 2014
Copyright François Durand 2014, 2015
fradurand@gmail.com

This file is part of SVVAMP.

    SVVAMP is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    SVVAMP is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with SVVAMP.  If not, see <http://www.gnu.org/licenses/>.
"""

import numpy as np
import networkx as nx

from svvamp.VotingSystems.ElectionResult import ElectionResult
from svvamp.Preferences.Population import Population


class RankedPairsResult(ElectionResult):
    """Results of an election using Tideman's Ranked Pairs.

    In the matrix of duels, victories (and ties) are sorted by decreasing
    amplitude. If two duels have the same score, we take first the one where
    the winner has the smallest index; if there is still a choice to make,
    we take first the duel where the loser has the highest index.

    Starting with the largest victory, we build a directed graph whose
    nodes are the candidates and edges are victories. But if a victory
    creates a cycle in the graph, it is not validated and the edge is not
    added.

    At the end, we have a transitive connected directed graph, whose adjacency
    relation is included in the relation of victories (with ties broken),
    matrix_victories_vtb_ctb. The maximal node of this graph (by topological
    order) is declared the winner.
    """
    
    _options_parameters = ElectionResult._options_parameters.copy()

    def __init__(self, population, **kwargs):
        super().__init__(population, **kwargs)
        self._log_identity = "RANKED_PAIRS_RESULT"
        self._one_v_might_be_pivotal = None

    def _forget_results_subclass(self):
        self._one_v_might_be_pivotal = None

    #%% Counting the ballots

    def _count_ballots(self):
        self._mylog("Count ballots", 1)
        matrix_duels_vtb_temp = np.copy(self.pop.matrix_duels_vtb)
        G = nx.DiGraph()
        G.add_nodes_from(range(self.pop.C))
        while True:
            best_duel_result_c = np.max(matrix_duels_vtb_temp, 1)
            c = np.argmax(best_duel_result_c)
            best_duel_result = best_duel_result_c[c]
            d = np.where(
                matrix_duels_vtb_temp[c, :] == best_duel_result
            )[0][-1]
            if best_duel_result == 0:
                break
            matrix_duels_vtb_temp[c, d] = 0
            matrix_duels_vtb_temp[d, c] = 0
            if not nx.has_path(G, d, c):
                G.add_edge(c, d, weight=self.pop.matrix_duels_vtb[c, d])
        self._candidates_by_scores_best_to_worst = nx.topological_sort(G)   
        self._w = self._candidates_by_scores_best_to_worst[0]
        self._scores = nx.to_numpy_matrix(G)
            
    @property
    def w(self):
        """Integer (winning candidate).
        """
        if self._w is None:
            self._count_ballots()
        return self._w

    @property
    def candidates_by_scores_best_to_worst(self):
        """1d array of integers. candidates_by_scores_best_to_worst[k] is the
        k-th candidate by topological order on the graph generated by Ranked
        Pairs.
        
        By definition, candidates_by_scores_best_to_worst[0] = w.
        """
        if self._candidates_by_scores_best_to_worst is None:
            self._count_ballots()
        return self._candidates_by_scores_best_to_worst     

    @property
    def scores(self):
        """2d array of integers. scores[c, d] is equal to matrix_duels_vtb[
        c, d] iff this duel was validated in Ranked Pairs, 0 otherwise.

        N.B. Unlike for most other voting systems, scores matrix must be
        read in rows: c's score vector is scores[c, :].
        """
        if self._scores is None:
            self._count_ballots()
        return self._scores
        
    @property
    def score_w(self):
        """score_w is w's score vector. I.e. score_w = scores[w, :].
        """
        if self._score_w is None:
            self._mylog("Compute winner's score", 1)
            self._score_w = self.scores[self.w, :]
        return self._score_w

    @property 
    def scores_best_to_worst(self):
        """2d array. scores_best_to_worst is the scores of the candidates, 
        from the winner to the last candidate of the election.
        
        scores_best_to_worst[k, j] is the score of the k-th best
        candidate of the election against the j-th. I.e. it is 
        the result in matrix_duels_vtb iff this duels was validated by Ranked
        Pairs, 0 otherwise.
        """
        if self._scores_best_to_worst is None:
            self._mylog("Compute scores_best_to_worst", 1)
            self._scores_best_to_worst = self.scores[ 
                self.candidates_by_scores_best_to_worst, :][
                    :, self.candidates_by_scores_best_to_worst]
        return self._scores_best_to_worst

    @property
    def _v_might_IM_for_c(self):
        # This is a quite lazy version, we could be more precise (possible
        # future work).
        # For a voter to be pivotal, it is necessary that she makes a duel
        # examined before another one in Ranked Pairs algorithm (this includes
        # the case of making a duel (c, d) examined before (d, c), i.e. 
        # changing a defeat into a victory)
        if self._one_v_might_be_pivotal is None:
            scores_duels = np.sort(np.concatenate((
                self.pop.matrix_duels_vtb[np.triu_indices(self.pop.C, 1)],
                self.pop.matrix_duels_vtb[np.tril_indices(self.pop.C, -1)]
            )))
            self._one_v_might_be_pivotal = np.any(scores_duels[:-1] + 2 >= 
                                                  scores_duels[1:])
        return np.full((self.pop.V, self.pop.C), self._one_v_might_be_pivotal)


if __name__ == '__main__':
    # A quick demo
    preferences_utilities = np.random.randint(-5, 5, (10, 5))
    pop = Population(preferences_utilities)
    election = RankedPairsResult(pop)
    election.demo(log_depth=3)