#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Module to control interaction with an ANSYS shell instance.
Built using ansys documentation from
https://www.sharcnet.ca/Software/Ansys/

This module makes no claim to own any rights to ANSYS.  It's merely an interface to
software owned by ANSYS.

"""
import re
import os
import tempfile
import appdirs
import warnings
import logging
import pexpect
import time
import pyansys

# settings directory
settings_dir = appdirs.user_data_dir('pyansys')
if not os.path.isdir(settings_dir):
    try:
        os.mkdir(settings_dir)
    except:
        warnings.warn('Unable to create settings directory.\n' +
                      'Will be unable to cache ANSYS executable location')

config_file = os.path.join(settings_dir, 'config.txt')

ready_items = [b'BEGIN:',
               b'PREP7:',
               b'SOLU_LS[0-9]+:',
               b'POST1:',
               b'POST26:',
               b'RUNSTAT:',
               b'AUX2:',
               b'AUX3:',
               b'AUX12:',
               b'AUX15:',
               # continue
               b'YES,NO OR CONTINUOUS\)\=',
               b'executed\?',
               # errors
               b'SHOULD INPUT PROCESSING BE SUSPENDED\?',
               # prompts
               b'ENTER FORMAT for',
]

continue_idx = ready_items.index(b'YES,NO OR CONTINUOUS\)\=')
warning_idx = ready_items.index(b'executed\?')
error_idx = ready_items.index(b'SHOULD INPUT PROCESSING BE SUSPENDED\?')
prompt_idx = ready_items.index(b'ENTER FORMAT for')

nitems = len(ready_items)
expect_list = []
for item in ready_items:
    expect_list.append(re.compile(item))

# idenfity ignored commands
# ignored = re.compile('\s+'.join(['WARNING', 'command', 'ignored']))
ignored = re.compile('[\s\S]+'.join(['WARNING', 'command', 'ignored']))
# re.DOTALL = True
# ignored = re.compile('(?:.|\n)+'.join(['WARNING', 'ignored']))
# ignored = re.compile('[\s\S]+'.join(['WARNING', 'ignored']))
# print(ignored.search(response))


def SetupLogger(loglevel='INFO'):
    """ Setup logger """

    # return existing log if this function has already been called
    if hasattr(SetupLogger, 'log'):
        SetupLogger.log.setLevel(loglevel)
        ch = SetupLogger.log.handlers[0]
        ch.setLevel(loglevel)
        return SetupLogger.log

    # create logger
    log = logging.getLogger(__name__)
    log.setLevel(loglevel)

    # create console handler and set level to debug
    ch = logging.StreamHandler()
    ch.setLevel(loglevel)

    # create formatter
    formatstr = '%(asctime)s [%(levelname)s] %(name)s: %(message)s'
    formatter = logging.Formatter(formatstr)

    # add formatter to ch
    ch.setFormatter(formatter)

    # add ch to logger
    log.addHandler(ch)

    # make persistent
    SetupLogger.log = log

    return log


def GetANSYSPath():
    """ Acquires ANSYS Path from a cached file or user input """
    if os.path.isfile(config_file):
        with open(config_file) as f:
            exe_loc = f.read()

        # verify
        if not os.path.isfile(exe_loc):
            print('Cached ANSYS executable %s not found' % exe_loc)
            exe_loc = SaveANSYSPath()

    else:  # create configuration file
        exe_loc = SaveANSYSPath()

    return exe_loc


def SaveANSYSPath():
    # try:
    # check to see if the ansys path is stored as an env_var
    # for env_var in os.environ:
    #     if 'ANSYS' in env_var:
    #         if os.environ[env_var][-5:] == 'ANSYS':
    #             exe_loc = os.environ[env_var] + env_var[-4:]
    #             print('Found ANSYS at %s' % exe_loc)
    #             resp = input('Use this location?  [Y/n]')
    #             if resp != 'n':
    #                 return exe_loc

    # otherwise, query user for the location
    with open(config_file, 'w') as f:
        try:
            exe_loc = raw_input('Enter location of ANSYS executable: ')
        except NameError:
            exe_loc = input('Enter location of ANSYS executable: ')
        if not os.path.isfile(exe_loc):
            raise Exception('ANSYS executable not found at this location:\n%s' % exe_loc)

        f.write(exe_loc)
        # config = configparser.ConfigParser()
        # config['DEFAULT'] = {'ansys_app': exe_loc}
        # config.write(f)

    return exe_loc


class ANSYS(object):
    """
    This class opens ANSYS in the background and allows commands to be
    passed to a persistent session.

    Parameters
    ----------
    exec_file : str, optional
        The location of the ANSYS executable.  Will use the cached location when
        left at the default None.

    run_location : str, optional
        ANSYS working directory.  Defaults to a temporary working directory.

    jobname : str, optional
        ANSYS jobname.  Defaults to 'file'.

    nproc : int, optional
        Number of processors.  Defaults to None, which for ANSYS this is usually 2.

    override : bool, optional
        Attempts to delete the *.lock file at the run_location.  Useful when a prior
        ANSYS session has exited prematurely and the lock file has not been deleted.

    wait : bool, optional
        When True, waits until ANSYS has been initialized before initializing the
        python ansys object.  Set this to False for debugging.

    loglevel : str, optional
        Sets which messages are printed to the console.  Default 'INFO' prints out
        all ANSYS messages, 'WARNING` prints only messages containing ANSYS warnings,
        and 'ERROR' prints only error messages.

    Examples
    --------
    >>> import pyansys
    >>> ansys = pyansys.ANSYS()

    """
    # default settings
    allow_ignore = False
    auto_continue = True
    block_override = None

    def __init__(self, exec_file=None, run_location=None, jobname='file', nproc=None,
                 override=False, wait=True, loglevel='INFO'):
        """ Initialize connection with ANSYS program """

        self.log = SetupLogger(loglevel.upper())

        if exec_file is None:
            # Load cached path
            try:
                exec_file = GetANSYSPath()
            except:
                raise Exception('Invalid or path or cannot load cached ansys path' +
                                'Enter one manually using pyansys.ANSYS(exec_file=...)')
        else:  # verify ansys exists at this location
            if not os.path.isfile(exec_file):
                raise Exception('Invalid ANSYS executable at %s' % exec_file +
                                'Enter one manually using pyansys.ANSYS(exec_file="")')

        # spawn temporary directory
        if run_location is None:
            temp_dir = tempfile.gettempdir()
            run_location = os.path.join(temp_dir, 'ansys')
            if not os.path.isdir(run_location):
                try:
                    os.mkdir(run_location)
                except:
                    raise Exception('Unable to create temporary working '
                                    'directory %s\n' % run_location +
                                    'Please specify run_location')
        else:
            if not os.path.isdir(run_location):
                raise Exception('%s is not a valid folder' % run_location)

        # initialize ANSYS process
        self.lockfile = os.path.join(run_location, jobname + '.lock')
        if os.path.isfile(self.lockfile):
            if not override:
                raise Exception('Lock file exists for jobname %s \n' % jobname +
                                'Set override=True to delete lock and start ANSYS')
            else:
                os.remove(self.lockfile)

        command = '%s -j %s ' % (exec_file, jobname)
        if nproc:
            command += '-np %d ' % nproc

        # add run location to command
        # command += '-dir "%s" ' % run_location
        self.log.debug('Spawning shell process with: "%s"' % command)
        self.log.debug('At "%s"' % run_location)
        if os.name == 'nt':
            from pexpect import popen_spawn
            self.process = popen_spawn.PopenSpawn(command, cwd=run_location)
        else:
            self.process = pexpect.spawn(command, cwd=run_location)
        self.process.delaybeforesend = None
        self.log.debug('Waiting')
        if wait:
            self.process.expect('CONTINUE')
            self.process.sendline('')  # enter to continue
            self.process.expect('BEGIN:')
            self.log.debug('Initialized ANSYS')
            self.log.debug(self.process.before.decode('utf-8'))

        # store for later reference
        self.path = run_location
        self.exec_file = exec_file
        self.jobname = jobname

    def SetLogLevel(self, loglevel):
        """ Sets log level """
        SetupLogger(loglevel=loglevel.upper())

    def __enter__(self):
        return self

    def RunCommand(self, command, return_response=False, block=True,
                   continue_on_error=False, ignore_prompt=False, timeout=None):
        """ Sends command and returns ANSYS's response """
        if not self.process.isalive():
            raise Exception('ANSYS process closed')

        if self.block_override is not None:
            block = self.block_override

        # send the command
        self.log.debug('Sending command %s' % command)
        if ignore_prompt:
            self.log.debug('... with ignore_prompt=True')
        self.process.sendline(command)

        if block:
            self.log.debug('Waiting: TIMEOUT %s' % str(timeout))
            while True:
                i = self.process.expect_list(expect_list, timeout=timeout)
                response = self.process.before.decode('utf-8')
                if i >= continue_idx and i < warning_idx:  # continue
                    self.log.debug('Continue: Response index %i.  Matched %s'
                                   % (i, ready_items[i].decode('utf-8')))
                    self.log.info(response + ready_items[i].decode('utf-8'))
                    if self.auto_continue:
                        user_input = 'y'
                    else:
                        user_input = input('Response: ')
                    self.process.sendline(user_input)

                elif i >= warning_idx and i < error_idx:  # warning
                    self.log.debug('Prompt: Response index %i.  Matched %s'
                                   % (i, ready_items[i].decode('utf-8')))
                    self.log.warning(response + ready_items[i].decode('utf-8'))
                    if self.auto_continue:
                        user_input = 'y'
                    else:
                        user_input = input('Response: ')
                    self.process.sendline(user_input)

                elif i >= error_idx and i < prompt_idx:  # error
                    self.log.debug('Error index %i.  Matched %s'
                                   % (i, ready_items[i].decode('utf-8')))
                    self.log.error(response)
                    response += ready_items[i].decode('utf-8')
                    if continue_on_error:
                        self.process.sendline(user_input)
                    else:
                        raise Exception(response)

                elif i >= prompt_idx:  # prompt
                    self.log.debug('Prompt index %i.  Matched %s'
                                   % (i, ready_items[i].decode('utf-8')))
                    self.log.info(response + ready_items[i].decode('utf-8'))
                    if ignore_prompt:
                        self.log.debug('Ignoring prompt')
                        # time.sleep(0.1)
                        break
                    else:
                        user_input = input('Response: ')
                        self.process.sendline(user_input)

                else:  # continue item
                    self.log.debug('continue index %i.  Matched %s'
                                   % (i, ready_items[i].decode('utf-8')))
                    break

            # handle response
            if '*** ERROR ***' in response:  # flag error
                self.log.error(response)
                if not continue_on_error:
                    raise Exception(response)
            elif ignored.search(response):  # flag ignored command
                if not self.allow_ignore:
                    self.log.error(response)
                    raise Exception(response)
                else:
                    self.log.warning(response)
            else:  # all else
                self.log.info(response)

            if return_response:
                response = self.process.before.decode('utf-8')
                return response

    def last_message(self):
        """ Returns the last output from ANSYS """
        return self.process.before.decode('utf-8')

    def __del__(self):
        self.process.sendline('FINISH')
        self.process.sendline('EXIT')

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.Exit()

    def Exit(self, twait=10):
        """
        Gracefully ANSYS process

        Returns True when process is still active.

        """
        self.process.sendline('FINISH')
        self.process.sendline('EXIT')
        t = time.time()
        wait_til = t + twait
        while self.process.isalive() and time.time() < wait_til:
            time.sleep(0.01)

        return self.process.isalive()

    def Kill(self):
        """ Forces ANSYS process to end and removes lock file """
        if self.process.terminate(True):
            if os.path.isfile:
                try:
                    os.remove(self.lockfile)
                except:
                    self.log.warning('Unable to remove lock file %s ' % self.lockfile)
        else:
            self.log.warning('Unable to terminate')
            return True

        return False

    @property
    def results(self):
        """ Returns a binary interface to the result file """
        resultfile = os.path.join(self.path, '%s.rst' % self.jobname)
        if not os.path.isfile(resultfile):
            raise Exception('No results found at %s' % resultfile)

        return pyansys.ResultReader(resultfile)

    def __call__(self, command, **kwargs):
        self.RunCommand(command, **kwargs)

    def Vdot(self, labr="", labx1="", laby1="", labz1="", labx2="", laby2="",
             labz2="", **kwargs):
        """
        APDL Command: VDOT

        Forms an element table item from the dot product of two vectors.

        Parameters
        ----------
        labr
            Label assigned to dot product result.

        labx1, laby1, labz1
            X, Y, and Z-component of first vector label.

        labx2, laby2, labz2
            X, Y, and Z-component of second vector label.

        Notes
        -----
        Forms labeled result items for the selected element from the dot
        product of two vectors:

        LabR = {LabX1, LabY1, LabZ1} : :  {LabX2, LabY2, LabZ2}

        Data must be in a consistent coordinate system.  Labels are those
        associated with the ETABLE command.

        """
        command = "VDOT,%s,%s,%s,%s,%s,%s,%s" % (str(labr), str(labx1), str(laby1), str(labz1), str(labx2), str(laby2), str(labz2))
        self.RunCommand(command, **kwargs)

    def Cwd(self, dirpath="", **kwargs):
        """
        APDL Command: /CWD

        Changes the current working directory.

        Parameters
        ----------
        dirpath
            The full path name of the new working directory.

        Notes
        -----
         After issuing the /CWD command, all new files opened with no default
        directory specified (via the FILE, /COPY, or RESUME commands, for
        example) default to the new DIRPATH directory.

        """
        command = "/CWD,%s" % (str(dirpath))
        self.RunCommand(command, **kwargs)

    def Mpdres(self, labf="", matf="", labt="", matt="", **kwargs):
        """
        APDL Command: MPDRES

        Reassembles existing material data with the temperature table.

        Parameters
        ----------
        labf
            Material property label associated with MATF.

        matf
            Material reference number of property to restore from virtual
            space.

        labt
            Material property label associated with MATT (defaults to label
            associated with MATF).

        matt
            Material reference number assigned to generated property (defaults
            to MATF).

        Notes
        -----
        Restores into the database (from virtual space) a data table previously
        defined [MP] for a particular property, assembles data with current
        database temperature table, and stores back in virtual space as a new
        property.

        This command is also valid in SOLUTION.

        """
        command = "MPDRES,%s,%s,%s,%s" % (str(labf), str(matf), str(labt), str(matt))
        self.RunCommand(command, **kwargs)

    def Mlist(self, node1="", node2="", ninc="", **kwargs):
        """
        APDL Command: MLIST

        Lists the MDOF of freedom.

        Parameters
        ----------
        node1, node2, ninc
            List master degrees of freedom from NODE1 to NODE2 (defaults
            toNODE1) in steps of NINC (defaults to 1).  If NODE1 = ALL
            (default), NODE2 and NINC are ignored and masters for all selected
            nodes [NSEL] are listed.  If NODE1 = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be substituted for NODE1
            (NODE2 and NINC are ignored).

        Notes
        -----
        Lists the master degrees of freedom.

        """
        command = "MLIST,%s,%s,%s" % (str(node1), str(node2), str(ninc))
        self.RunCommand(command, **kwargs)

    def R(self, nset="", r1="", r2="", r3="", r4="", r5="", r6="", **kwargs):
        """
        APDL Command: R

        Defines the element real constants.

        Parameters
        ----------
        nset
            Real constant set identification number (arbitrary).  If same as a
            previous set number, set is redefined. Set number relates to that
            defined with the element [REAL]. Note that the GUI automatically
            assigns this value.

        r1, r2, r3, . . . , r6
            Real constant values (interpreted as area, moment of inertia,
            thickness, etc., as required for the particular element type using
            this set), or table names for tabular input of boundary conditions.
            Use RMORE command if more than six real constants per set are to be
            input.

        Notes
        -----
        Defines the element real constants.  The real constants required for an
        element are shown in the Input Summary of each element description in
        the Element Reference.  Constants must be input in the same order as
        shown in that table.  If more than the required number of element real
        constants are specified in a set, only those required are used.  If
        fewer than the required number are specified, zero values are assumed
        for the unspecified constants.

        If using table inputs (SURF151, SURF152, FLUID116, CONTA171, CONTA172,
        CONTA173, CONTA174, and CONTA175 only), enclose the table name in %
        signs (e.g., %tabname%).

        Specify NSET = GCN to define real constants for real constant sets that
        were previously assigned by the GCDEF command (that is, real constants
        used in general contact interactions).

        When copying real constants to new sets, ANSYS, Inc. recommends that
        you use the command input. If you do use the GUI, restrict the real
        constant copy to only the first six real constants (real constants
        seven and greater will be incorrect for both the master and copy set).

        This command is also valid in SOLUTION.

        """
        command = "R,%s,%s,%s,%s,%s,%s,%s" % (str(nset), str(r1), str(r2), str(r3), str(r4), str(r5), str(r6))
        self.RunCommand(command, **kwargs)

    def Nlgeom(self, key="", **kwargs):
        """
        APDL Command: NLGEOM

        Includes large-deflection effects in a static or full transient
        analysis.

        Parameters
        ----------
        key
            Large-deflection key:

            OFF - Ignores large-deflection effects (that is, a small-deflection analysis is
                  specified). This option is the default.

            ON - Includes large-deflection (large rotation) effects or large strain effects,
                 according to the element type.

        Notes
        -----
        Large-deflection effects are categorized as either large deflection (or
        large rotation) or large strain, depending on the element type. These
        are listed (if available) under Special Features in the input data
        table for each element in the Element Reference. When large deflection
        effects are included (NLGEOM,ON), stress stiffening effects are also
        included automatically.

        If used during the solution (/SOLU), this command is valid only within
        the first load step.

        In a large-deflection analysis, pressure loads behave differently than
        other load types. For more information, see Load Direction in a Large-
        Deflection Analysis.

        The gyroscopic matrix (that occurs due to rotational angular velocity)
        does not support large-deflection effects. The theoretical formulations
        for the gyroscopic matrix support small deflection (linear formulation)
        only.

        When large-deflection effects are included in a substructure or CMS
        transient analysis use pass, the OUTRES command ignores DSUBres = ALL.

        This command is also valid in PREP7.

        In ANSYS Professional NLT, large deflection effects should not be
        turned on if 2-D solid (PLANEn) or 3-D solid (SOLIDn) elements are
        defined. ANSYS Professional NLS supports NLGEOM,ON for plane and solid
        elements.

        """
        command = "NLGEOM,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Read(self, fname="", nskip="", format="", xfield="", yfield="",
             zfield="", prfield="", pifield="", **kwargs):
        """
        APDL Command: READ

        Reads coordinate and pressure data from a file.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        nskip
            Number of lines at the beginning of the file that will be skipped
            while it is read. Default = 0. NSKIP is ignored for FileType =
            CFXTBR or CFDPOST on the FTYPE command.

        format
            For FileType = FORMATTED on the FTYPE command, Format is the read
            format in the FORTRAN FORMAT convention enclosed in parentheses;
            for example: (3e10.0,10x,e10.0,70x,e10.0)

        xfield, yfield, zfield, prfield, pifield
            For FileType = CSV on the FTYPE command, these are field numbers
            locating the coordinates and real and imaginary (if present)
            pressures. The field value may not exceed 20.

        Notes
        -----
        Reads coordinate and pressure data from the specified file. The file
        type must have been previously specified on the FTYPE command.

        Upon reading the file, nodes are created for the source points. For
        FileType = CFXTBR or CFDPOST on the FTYPE command, if face data is
        available, SURF154 elements are also created. A nodal component named
        SOURCENODES and an element component named SOURCEELEMS are created
        automatically.

        """
        command = "READ,%s,%s,%s,%s,%s,%s,%s,%s" % (str(fname), str(nskip), str(format), str(xfield), str(yfield), str(zfield), str(prfield), str(pifield))
        self.RunCommand(command, **kwargs)

    def Cycle(self, **kwargs):
        """
        APDL Command: *CYCLE

        Bypasses commands within a do-loop.

        Notes
        -----
        Bypasses all commands between this command and the  *ENDDO command
        within a do-loop.  The next loop (if applicable) is initiated.  The
        cycle option may also be conditionally executed [Use the *IF].  The
        *CYCLE command must appear on the same file as the *DO command.

        This command is valid in any processor.

        """
        command = "*CYCLE," % ()
        self.RunCommand(command, **kwargs)

    def Nopr(self, **kwargs):
        """
        APDL Command: /NOPR

        Suppresses the expanded interpreted input data listing.

        Notes
        -----
        Suppresses printout of interpreted input data, including information
        labeled as "Notes."  When this printout is not suppressed, the data
        input to the analysis is echoed to the output file in an expanded
        format.  Printout is suppressed until a /GOPR or /GO command is read.

        Use of /NOPR is not recommended when the graphical user interface (GUI)
        is active.  The GUI sometimes issues "hidden" /NOPR and /GOPR command
        sequences, which will countermand user-issued /NOPR commands, thus
        making the use of /NOPR in the GUI environment unpredictable.

        This command is valid in any processor.

        """
        command = "/NOPR," % ()
        self.RunCommand(command, **kwargs)

    def Rthick(self, par="", iloc="", jloc="", kloc="", lloc="", **kwargs):
        """
        APDL Command: RTHICK

        Defines variable thickness at nodes for shell elements.

        Parameters
        ----------
        par
             Array parameter (indexed by node number) that expresses the
            function to be mapped.  For example, func (17) should be the
            desired shell thickness at node 17.

        iloc
            Position in real constant set for thickness at node I of the
            element (default 1).

        jloc
            Position in real constant set for thickness at node J of the
            element (default 2).

        kloc
            Position in real constant set for thickness at node K of the
            element (default 3).

        lloc
            Position in real constant set for thickness at node L of the
            element (default 4).

        Notes
        -----
        After RTHICK, the REAL number will match the ELEM number of each
        selected element. For example, R (ILOC) = func (I NODE), R (JLOC) =
        func (J NODE), etc.

        R(ILOC), R(JLOC), R(KLOC), and R(LLOC) on a previously defined real
        constant will be overwritten.  Any other real constants on a previously
        defined real constant set will remain unchanged. This command cannot be
        used for beam elements.

        """
        command = "RTHICK,%s,%s,%s,%s,%s" % (str(par), str(iloc), str(jloc), str(kloc), str(lloc))
        self.RunCommand(command, **kwargs)

    def Vcross(self, labxr="", labyr="", labzr="", labx1="", laby1="",
               labz1="", labx2="", laby2="", labz2="", **kwargs):
        """
        APDL Command: VCROSS

        Forms element table items from the cross product of two vectors.

        Parameters
        ----------
        labxr, labyr, labzr
            Label assigned to X, Y, and Z-component of resultant vector.

        labx1, laby1, labz1
            X, Y, and Z-component of first vector label.

        labx2, laby2, labz2
            X, Y, and Z-component of second vector label.

        Notes
        -----
        Forms labeled result items for the selected element from the cross
        product of two vectors:

        {LabXR, LabYR, LabZR} = {LabX1, LabY1, LabZ1} X {LabX2, LabY2, LabZ2}

        Data must be in a consistent coordinate system.  Labels are those
        associated with the ETABLE command.

        """
        command = "VCROSS,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(labxr), str(labyr), str(labzr), str(labx1), str(laby1), str(labz1), str(labx2), str(laby2), str(labz2))
        self.RunCommand(command, **kwargs)

    def Elem(self, **kwargs):
        """
        APDL Command: ELEM

        Specifies "Elements" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command. Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu>: List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "ELEM," % ()
        self.RunCommand(command, **kwargs)

    def Prod(self, ir="", ia="", ib="", ic="", name="", facta="", factb="",
             factc="", **kwargs):
        """
        APDL Command: PROD

        Multiplies variables.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]).  If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.

        ia, ib, ic
            Reference numbers of the three variables to be operated on.  If
            only two leave IC blank.  If only one, leave IB blank also.

        name
            Thirty-two character name identifying the variable on printouts and
            displays.  Embedded blanks are compressed for output.

        --, --
            Unused fields

        facta, factb, factc
            Scaling factors (positive or negative) applied to the corresponding
            variables (default to 1.0).

        Notes
        -----
        Multiplies variables (up to three at once) according to the operation:

        IR = (FACTA x IA) x (FACTB x IB) x (FACTC x IC)

        """
        command = "PROD,%s,%s,%s,%s,%s,%s,%s,%s" % (str(ir), str(ia), str(ib), str(ic), str(name), str(facta), str(factb), str(factc))
        self.RunCommand(command, **kwargs)

    def Paresu(self, lab="", fname="", ext="", **kwargs):
        """
        APDL Command: PARESU

        Restores previously saved paths from a file.

        Parameters
        ----------
        lab
            Read operation:

            ALL - Read all paths from the selected file (default).

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        This command removes all paths from virtual memory and then reads path
        data from a file written with the PASAVE command.  All paths on the
        file will be restored.  All paths currently in memory will be deleted.

        """
        command = "PARESU,%s,%s,%s" % (str(lab), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Lgen(self, itime="", nl1="", nl2="", ninc="", dx="", dy="", dz="",
             kinc="", noelem="", imove="", **kwargs):
        """
        APDL Command: LGEN

        Generates additional lines from a pattern of lines.

        Parameters
        ----------
        itime
            Do this generation operation a total of ITIMEs, incrementing all
            keypoints in the given pattern automatically (or by KINC) each time
            after the first.  ITIME must be > 1 for generation to occur.

        nl1, nl2, ninc
            Generate lines from pattern beginning with NL1 to NL2 (defaults to
            NL1) in steps of NINC (defaults to 1).  If NL1 = ALL, NL2 and NINC
            are ignored and pattern is all selected lines [LSEL].  If NL1 = P,
            graphical picking is enabled and all remaining command fields are
            ignored (valid only in the GUI).  A component name may also be
            substituted for NL1 (NL2 and NINC are ignored).

        dx, dy, dz
            Keypoint location increments in the active coordinate system (--,
            Dθ, DZ for cylindrical, --, Dθ, -- for spherical).

        kinc
            Keypoint increment between generated sets.  If zero, the lowest
            available keypoint numbers are assigned [NUMSTR].

        noelem
            Specifies if elements and nodes are also to be generated:

            0 - Generate nodes and elements associated with the original lines, if they exist.

            1 - Do not generate nodes and elements.

        imove
            Specifies whether to redefine the existing lines:

            0 - Generate additional lines as requested with the ITIME argument.

            1 - Move original lines to new position retaining the same keypoint numbers (ITIME,
                KINC, and NOELM are ignored).  Valid only if the old lines are
                no longer needed at their original positions.  Corresponding
                meshed items are also moved if not needed at their original
                position.

        Notes
        -----
        Generates additional lines (and their corresponding keypoints and mesh)
        from a given line pattern.  The MAT, TYPE, REAL, and ESYS attributes
        are based upon the lines in the pattern and not upon the current
        settings.  End slopes of the generated lines remain the same (in the
        active coordinate system) as those of the given pattern.  For example,
        radial slopes remain radial, etc.  Generations which produce lines of a
        size or shape different from the pattern (i.e., radial generations in
        cylindrical systems, radial and phi generations in spherical systems,
        and theta generations in elliptical systems) are not allowed.  Note
        that solid modeling in a toroidal coordinate system is not recommended.
        New line numbers are automatically assigned (beginning with the lowest
        available values [NUMSTR]).

        """
        command = "LGEN,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(itime), str(nl1), str(nl2), str(ninc), str(dx), str(dy), str(dz), str(kinc), str(noelem), str(imove))
        self.RunCommand(command, **kwargs)

    def Mkdir(self, dir="", **kwargs):
        """
        APDL Command: /MKDIR

        Creates a directory.

        Parameters
        ----------
        dir
            The directory to create (248 characters maximum on Linux; 233 on
            Windows). If no path is provided, it will be created in the current
            working directory. Must be a valid name (and path) for the system
            you are working on.

        Notes
        -----
        Creates a directory on the computer ANSYS is currently running on.

        """
        command = "/MKDIR,%s" % (str(dir))
        self.RunCommand(command, **kwargs)

    def Torqsum(self, cnam1="", cnam2="", cnam3="", cnam4="", cnam5="",
                cnam6="", cnam7="", cnam8="", cnam9="", **kwargs):
        """
        APDL Command: TORQSUM

        Summarizes electromagnetic torque calculations on element components.

        Parameters
        ----------
        cnam1, cnam2, cnam3, . . . , cnam9
            Names of existing element components for which Maxwell or virtual
            work boundary conditions were applied in the preprocessor.  Must be
            enclosed in single quotes (e.g., 'CNAM1') when the command typed in
            the command input box.

        Notes
        -----
        TORQSUM invokes an ANSYS macro that summarizes the Maxwell and virtual
        work torque values.  The element components must have had appropriate
        Maxwell or virtual work boundary conditions established in the
        preprocessor prior to solution in order to retrieve torques (see the
        FMAGBC command).  The torque values are stored on a per-element basis
        for the adjacent air layer elements surrounding the components and are
        retrieved and summed by the macro.  For a harmonic analysis, the
        calculated torque represents a time-average value.

        TORQSUM is valid only for 2-D planar analysis.

        """
        command = "TORQSUM,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(cnam1), str(cnam2), str(cnam3), str(cnam4), str(cnam5), str(cnam6), str(cnam7), str(cnam8), str(cnam9))
        self.RunCommand(command, **kwargs)

    def Cmsel(self, type="", name="", entity="", **kwargs):
        """
        APDL Command: CMSEL

        Selects a subset of components and assemblies.

        Parameters
        ----------
        type
            Label identifying the type of select:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

            ALL - Also select all components.

            NONE - Unselect all components.

        name
            Name of component or assembly whose items are to be selected (valid
            only if Type = S, R, A, or U).

        entity
            If Name is blank, then the following entity types can be specified:

            VOLU - Select the volume components only.

            AREA - Select the area components only.

            LINE - Select the line components only.

            KP - Select the keypoint components only.

            ELEM - Select the element components only.

            NODE - Select the node components only.

        Notes
        -----
        Selecting by component is a convenient adjunct to individual item
        selection (e.g., VSEL, ESEL, etc.). CMSEL, ALL allows you to select
        components in addition to other items you have already selected.

        If Type = R for an assembly selection [CMSEL,R,<assembly-name>], the
        reselect operation is performed on each component in the assembly in
        the order in which the components make up the assembly.  Thus, if one
        reselect operation results in an empty set, subsequent operations will
        also result in empty sets.  For example, if the first reselect
        operation tries to reselect node 1 from the selected set of nodes 3, 4,
        and 5, the operation results in an empty set (that is, no nodes are
        selected).  Since the current set is now an empty set, if the second
        reselect operation tries to reselect any nodes, the second operation
        also results in an empty set, and so on.  This is equivalent to
        repeating the command CMSEL,R,<component-name> once for each component
        making up the assembly.

        This command is valid in any processor.

        """
        command = "CMSEL,%s,%s,%s" % (str(type), str(name), str(entity))
        self.RunCommand(command, **kwargs)

    def Post1(self, **kwargs):
        """
        APDL Command: /POST1

        Enters the database results postprocessor.

        Notes
        -----
        Enters the general database results postprocessor (POST1).  All load
        symbols (/PBC, /PSF, or /PBF) are automatically turned off with this
        command.

        This command is valid only at the Begin Level.

        """
        command = "/POST1," % ()
        self.RunCommand(command, **kwargs)

    def Cdopt(self, option="", **kwargs):
        """
        APDL Command: CDOPT

        Specifies format to be used for archiving geometry.

        Parameters
        ----------
        option
            IGES

            IGES - Write solid model geometry information using IGES format (default).

            ANF - Write solid model geometry information using ANSYS Neutral File format.

            STAT - Print out the current format setting.

        Notes
        -----
        This command controls your solid model geometry format for CDWRITE
        operations. The ANF option affects only the COMB and SOLID options of
        the CDWRITE command. All other options remain unaffected.

        This option setting is saved in the database.

        """
        command = "CDOPT,%s" % (str(option))
        self.RunCommand(command, **kwargs)

    def Presol(self, item="", comp="", **kwargs):
        """
        APDL Command: PRESOL

        Prints the solution results for elements.

        Parameters
        ----------
        item
            Label identifying the item.  Valid item labels are shown in Table
            224: PRESOL - Valid Item and Component Labels for Element Results
            below.  Some items also require a component label.

        comp
            Component of the item (if required).  Valid component labels are
            shown in Table 224: PRESOL - Valid Item and Component Labels for
            Element Results below.

        Notes
        -----
        Prints the solution results for the selected elements in the sorted
        sequence.  For example, PRESOL,S prints the stress items SX, SY, SZ,
        SXY, SYZ, and SXZ for the node locations of the element.  Component
        results are in the global Cartesian coordinate directions unless
        transformed (RSYS).

        Shell elements print values at the top, then bottom of the element (or
        layer). If KEYOPT(8) = 2 (for SHELL181, SHELL208, SHELL209, SHELL281,
        or ELBOW290), the results are printed in the order TOP, BOT and then
        MID of each element, (or layer). The MID value will be the actual value
        as written to the results file.

        Items are listed as columns of a table versus element number.  An
        exception occurs for item ELEM which uses an element format (all
        applicable line element results are listed per element) instead of a
        tabular format.

        The FORCE command can be used to define which component of the nodal
        load is to be used (static, damping, inertia, or total).  See the
        ETABLE and PRETAB commands for printing items not available through
        this command (such as line element results).

        For PowerGraphics [/GRAPHICS,POWER], results are listed only for the
        element surface.  The items marked with [1] are not supported  by
        PowerGraphics.

        Table: 224:: : PRESOL - Valid Item and Component Labels for Element
        Results

        Failure criteria for virgin material [1][2].

        Default components: Maximum of all failure criteria defined at current
        location (MAX), maximum strain (EMAX), maximum stress (SMAX), Tsai-Wu
        Strength Index (TWSI), inverse of Tsai-Wu Strength Ratio Index (TWSR).

        Other available components: Hashin Fiber Failure (HFIB), Hashin Matrix
        Failure (HMAT), Puck Fiber Failure (PFIB), Puck Matrix Failure (PMAT),
        LaRc03 Fiber Failure (L3FB),  LaRc03 Matrix Failure (L3MT), LaRc04
        Fiber Failure (L4FB), LaRc04 Matrix Failure (L4MT), and any user-
        defined failure criteria (USR1 through USR9) [4].

        Issue the FCTYP command to activate or remove failure criteria.

        Failure criteria based on the effective stresses in the damaged
        material.

        Components: Maximum of all failure criteria defined at current location
        (MAX), fiber tensile failure (FT), fiber compressive failure (FC),
        matrix tensile failure (MT), and matrix compressive (MC).

        Progressive damage parameters.

        Components: Damage status (STAT, 0 - undamaged, 1 - damaged, 2 -
        complete damage), fiber tensile damage variable (FT), fiber compressive
        damage variable (FC), matrix tensile damage variable (MT), matrix
        compressive damage variable (MC), shear damage variable (S),  energy
        dissipated per unit volume (SED), energy per unit volume due to viscous
        damping (SEDV).

        Not supported by PowerGraphics

        Works only if failure criteria information is provided. (For more
        information, see the documentation for the FC and TB commands.)

        Some element- and material-type limitations apply. (For more
        information, see the documentation for the PRERR command.)

        Works only if user failure criteria routine is provided.

        Do not use the PRESOL command to obtain contact forces for contact
        elements. The force values reported by this command may not be accurate
        for these elements. Instead, use the ETABLE command to obtain contact
        force values.

        """
        command = "PRESOL,%s,%s" % (str(item), str(comp))
        self.RunCommand(command, **kwargs)

    def Pdclr(self, type="", **kwargs):
        """
        APDL Command: PDCLR

        Clears the probabilistic design database.

        Parameters
        ----------
        type
            Specifies the part of the probabilistic database to be cleared.

            ALL - Clear the entire probabilistic database. Both the preprocessing and
                  postprocessing parts are cleared.

            POST - Clear only the postprocessing part of the probabilistic database. This is
                   necessary if you want to modify the deterministic model (the
                   analysis file) or probabilistic model (random variables,
                   correlations between random variables, or the random output
                   parameter) after a probabilistic analysis has already been
                   performed.

        Notes
        -----
        Clears the probabilistic design system (PDS) database. The settings are
        reset to their default values and the memory is cleared. Remember that
        the result files containing the results of the probabilistic loops are
        never deleted unless you intentionally delete them. We recommend that
        you use this command before switching to a new probabilistic analysis
        using a different probabilistic model (a different analysis loop file
        or deterministic model), or changing random input variables, their
        correlations, or random output parameters. Clearing the probabilistic
        database is not necessary if the probabilistic model remains the same
        and will be analyzed with a different probabilistic method or different
        method options. See the PDEXE command for restrictions. Before issuing
        the PDCLR command, you should save the probabilistic database using the
        PDSAVE command.

        """
        command = "PDCLR,%s" % (str(type))
        self.RunCommand(command, **kwargs)

    def Sspd(self, d11="", d21="", d31="", d22="", d32="", d33="", t="",
             **kwargs):
        """
        APDL Command: SSPD

        Specifies a preintegrated bending stiffness for shell sections.

        Parameters
        ----------
        d11, d21, d31, d22, d32, d33
            Bending stiffness component (symmetric lower part of submatrix
            [D]).

        t
            Temperature.

        Notes
        -----
        The behavior of shell elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        The SSPD command, one of several preintegrated shell section commands,
        specifies the bending stiffness quantity (submatrix [D] data) for a
        preintegrated shell section. The section data defined is associated
        with the section most recently defined (via the SECTYPE command).

        Unspecified commands default to zero.

        Related commands are SSPA, SSPB, SSPE, SSMT, SSBT, and SSPM.

        If you are using the SHELL181 or SHELL281 element's Membrane option
        (KEYOPT(1) = 1), it is not necessary to issue this command.

        For complete information, see Using Preintegrated General Shell
        Sections.

        """
        command = "SSPD,%s,%s,%s,%s,%s,%s,%s" % (str(d11), str(d21), str(d31), str(d22), str(d32), str(d33), str(t))
        self.RunCommand(command, **kwargs)

    def Sdelete(self, sfirst="", sl_ast="", sinc="", knoclean="", lchk="",
                **kwargs):
        """
        APDL Command: SDELETE

        Deletes sections from the database.

        Parameters
        ----------
        sfirst
            First section ID to be deleted; defaults to first available section
            in the database.

        slast
            Last section ID to be deleted; defaults to last available section
            in the database.

        sinc
            Increment of the section ID; defaults to 1.

        knoclean
            Pretension element cleanup key (pretension sections only).

            0 - Perform cleanup of pretension elements (delete pretension elements and
                reconnect elements split during PSMESH).

            1 - Do not perform cleanup.

        lchk
            Specifies the level of element-associativity checking:

            NOCHECK - No element-associativity check occurs. This option is the default.

            WARN - When a section, material, or real constant is associated with an element, ANSYS
                   issues a message warning that the necessary entity has been
                   deleted.

            CHECK - The command terminates, and no section, material, or real constant is deleted
                    if it is associated with an element.

        Notes
        -----
        Deletes one or more specified sections and their associated data from
        the ANSYS database.

        """
        command = "SDELETE,%s,%s,%s,%s,%s" % (str(sfirst), str(sl_ast), str(sinc), str(knoclean), str(lchk))
        self.RunCommand(command, **kwargs)

    def Lcwrite(self, lcno="", fname="", ext="", **kwargs):
        """
        APDL Command: LCWRITE

        Creates a load case by writing results to a load case file.

        Parameters
        ----------
        lcno
            Arbitrary pointer number (1-99) to be assigned to this load case.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        Creates a load case by writing the results data in the database to a
        load case file.  The database remains unchanged by this operation.  A
        pointer is also established to the written set of results on the load
        case file.  This pointer (LCNO) can then be used on the LCASE or LCOPER
        commands to read the load case data into the database.  By default,
        only summable results data (such as displacements, stresses, elastic
        strains) and constant results data (such as volume) are written to the
        load case file unless requested (LCSUM command).  Non-summable results
        data (such as plastic strains, strain energy), boundary conditions, and
        nodal loads are not written to the load case file.  The load case file
        may be named by default or by a user name.  Rewriting to the same file
        overwrites the previous data.  See the LCDEF command for status and
        erase operations.

        """
        command = "LCWRITE,%s,%s,%s" % (str(lcno), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Slashstatus(self, lab="", **kwargs):
        """
        APDL Command: /STATUS

        Lists the status of items for the run.

        Parameters
        ----------
        lab
            Items to list status for:

            ALL - List all below (default).

            TITLE - List only titles, Jobname, and revision number.

            UNITS - List only units.

            MEM - List only memory data statistics.

            DB - List only database statistics

            CONFIG - List only configuration parameters.

            GLOBAL - Provides a global status summary.

            SOLU - Provides a solution status summary.

            PROD - Provides a product summary.

        Notes
        -----
        Displays various items active for the run (such as the ANSYS revision
        number, Jobname, titles, units, configuration parameters, database
        statistics, etc.).

        This command is valid in any processor.

        """
        command = "/STATUS,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Sfcum(self, lab="", oper="", fact="", fact2="", **kwargs):
        """
        APDL Command: SFCUM

        Specifies that surface loads are to be accumulated.

        Parameters
        ----------
        lab
            Valid surface load label.  If ALL, use all appropriate labels.

        oper
            Accumulation key:

            REPL - Subsequent values replace the previous values (default).

            ADD - Subsequent values are added to the previous values.

            IGNO - Subsequent values are ignored.

        fact
            Scale factor for the first surface load value. A (blank) or '0'
            entry defaults to 1.0.

        fact2
            Scale factor for the second surface load value. A (blank) or '0'
            entry defaults to 1.0.

        Notes
        -----
        Allows repeated surface loads (pressure, convection, etc.) to be
        replaced, added, or ignored.  Surface loads are applied with the SF,
        SFE, and SFBEAM commands.  Issue the SFELIST command to list the
        surface loads.  The operations occur when the next surface load
        specifications are defined.  For example, issuing the SF command with a
        pressure value of 25 after a previous SF command with a pressure value
        of 20 causes the current value of that pressure to be 45 with the add
        operation, 25 with the replace operation, or 20 with the ignore
        operation.  All new pressures applied with SF after the ignore
        operation will be ignored, even if no current pressure exists on that
        surface.

        Scale factors are also available to multiply the next value before the
        add or replace operation.  A scale factor of 2.0 with the previous
        "add" example results in a pressure of 70.  Scale factors are applied
        even if no previous values exist.  Issue SFCUM,STAT to show the current
        label, operation, and scale factors.  Solid model boundary conditions
        are not affected by this command, but boundary conditions on the FE
        model are affected.

        Note:: : The FE boundary conditions may still be overwritten by
        existing solid model boundary conditions if a subsequent boundary
        condition transfer occurs.

        SFCUM does not work for tabular boundary conditions.

        This command is also valid in PREP7.

        """
        command = "SFCUM,%s,%s,%s,%s" % (str(lab), str(oper), str(fact), str(fact2))
        self.RunCommand(command, **kwargs)

    def Wmid(self, key="", **kwargs):
        """
        APDL Command: WMID

        Specifies reordering options for the WAVES command.

        Parameters
        ----------
        key
             Determines whether midside nodes are considered when reordering.

            NO - Do not consider midside nodes when reordering (default).

            YES - Consider midside nodes when reordering. This option is useful for models where
                  line elements are only attached to midside nodes of solid
                  elements.

        Notes
        -----
        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "WMID,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Bfelist(self, elem="", lab="", **kwargs):
        """
        APDL Command: BFELIST

        Lists the element body force loads.

        Parameters
        ----------
        elem
            Element at which body load is to be listed.  If ALL (or blank),
            list for all selected elements [ESEL].  If ELEM = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).  A component name may also be substituted
            for ELEM.

        lab
            Valid body load label. If ALL (or blank), use all appropriate
            labels. See BFE command for labels.

        Notes
        -----
        Lists the element body force loads for the specified element and label.
        Element body loads may be defined with the BFE command.

        This command is valid in any processor.

        """
        command = "BFELIST,%s,%s" % (str(elem), str(lab))
        self.RunCommand(command, **kwargs)

    def Tble(self, **kwargs):
        """
        APDL Command: TBLE

        Specifies "Data table properties" as the subsequent status topic.

        Notes
        -----
        This is a status (STAT) topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "TBLE," % ()
        self.RunCommand(command, **kwargs)

    def Path(self, name="", npts="", nsets="", ndiv="", **kwargs):
        """
        APDL Command: PATH

        Defines a path name and establishes parameters for the path.

        Parameters
        ----------
        name
            Name for this path (eight characters maximum. If nPts is blank, set
            the current path to the path with this name. If nPts is greater
            than zero, create a path of this name. If a path with this name
            already exists, replace it with a new path. If the NAME value is
            STATUS, display the status for path settings.

        npts
            The number of points used to define this path.  The minimum number
            is two, and the maximum is 1000. Default is 2.

        nsets
            The number of sets of data which you can map to this path.  You
            must specify at least four:  X, Y, Z, and S.  Default is 30.

        ndiv
            The number of divisions between adjacent points.  Default is 20.
            There is no maximum number of divisions.

        Notes
        -----
        The PATH command is used to define parameters for establishing a path.
        The path geometry is created by the PPATH command.  Multiple paths may
        be defined and named; however, only one path may be active for data
        interpolation [PDEF] and data operations [PCALC, etc.].  Path geometry
        points and data are stored in memory while in POST1.  If you leave
        POST1, the path information is erased.  Path geometry and data may be
        saved in a file by archiving the data using the PASAVE command.  Path
        information may be restored by retrieving the data using the PARESU
        command.

        For overlapping nodes, the lowest numbered node is assigned to the
        path.

        The number of divisions defined using nDiv does NOT affect the number
        of divisions used by PLSECT and PRSECT.

        For information on displaying paths you have defined, see  the Basic
        Analysis Guide.

        """
        command = "PATH,%s,%s,%s,%s" % (str(name), str(npts), str(nsets), str(ndiv))
        self.RunCommand(command, **kwargs)

    def Fitem(self, nfield="", item="", itemy="", itemz="", **kwargs):
        """
        APDL Command: FITEM

        Identifies items chosen by a picking operation (GUI).

        Parameters
        ----------
        nfield
            Field number on the command which uses the picking data.  (Count
            the command name as a field, so that a 2 indicates the first
            command argument, 3 the second command argument, etc.)  The
            corresponding field on the command will have a P51X label.

        item
            Entity number of the entity picked.  Negative entity numbers are
            used to indicate a range of entities.  If the item picked is a
            coordinate location, then this field represents the X-coordinate.
            See also the FLST command.

        itemy, itemz
            Y and Z coordinates of a picked coordinate location.  ITEM
            represents the X coordinate.  See also the FLST  command.

        Notes
        -----
        This is a command generated by the GUI and will appear in the log file
        (Jobname.LOG) if graphical picking is used.  This command is not
        intended to be typed in directly in an ANSYS session (although it can
        be included in an input file for batch input or for use with the /INPUT
        command).

        On the log file, a set of FITEM commands is preceded by one FLST
        command which defines the picking specifications for that pick
        operation.  The data listed in the FITEM commands are used by the first
        subsequent command containing a P51X label in one of its fields.

        Caution:: : For a given entity type, a list containing an ITEM that is
        larger than the maximum defined entity, could deplete the system memory
        and produce unpredictable results.

        This command is valid in any processor.

        """
        command = "FITEM,%s,%s,%s,%s" % (str(nfield), str(item), str(itemy), str(itemz))
        self.RunCommand(command, **kwargs)

    def Ldrag(self, nk1="", nk2="", nk3="", nk4="", nk5="", nk6="", nl1="",
              nl2="", nl3="", nl4="", nl5="", nl6="", **kwargs):
        """
        APDL Command: LDRAG

        Generates lines by sweeping a keypoint pattern along  path.

        Parameters
        ----------
        nk1, nk2, nk3, . . . , nk6
            List of keypoints in the pattern to be dragged (6 maximum if using
            keyboard entry).  If NK1 = P, graphical picking is enabled and all
            remaining command fields are ignored (valid only in the GUI).  If
            NK1 = ALL, all selected keypoints (except those that define the
            drag path) will be swept along the path.  A component name may also
            be substituted for NK1.

        nl1, nl2, nl3, . . . , nl6
            List of lines defining the path along which the pattern is to be
            dragged (6 maximum if using keyboard entry).  Must be a continuous
            set of lines.

        Notes
        -----
        Generates lines (and their corresponding keypoints) by sweeping a given
        keypoint pattern along a characteristic drag path.  If the drag path
        consists of multiple lines, the drag direction is determined by the
        sequence in which the path lines are input (NL1, NL2, etc.).  If the
        drag path is a single line (NL1), the drag direction is from the
        keypoint on the drag line that is closest to the first keypoint of the
        given pattern to the other end of the drag line.

        The magnitude of the vector between the keypoints of the given pattern
        and the first path keypoint remains constant for all generated keypoint
        patterns and the path keypoints.  The direction of the vector relative
        to the path slope also remains constant so that patterns may be swept
        around curves.  Keypoint and line numbers are automatically assigned
        (beginning with the lowest available values [NUMSTR]).  For best
        results, the entities to be dragged should be orthogonal to the start
        of the drag path.  Drag operations that produce an error message may
        create some of the desired entities prior to terminating.

        """
        command = "LDRAG,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nk1), str(nk2), str(nk3), str(nk4), str(nk5), str(nk6), str(nl1), str(nl2), str(nl3), str(nl4), str(nl5), str(nl6))
        self.RunCommand(command, **kwargs)

    def Edwrite(self, option="", fname="", ext="", **kwargs):
        """
        APDL Command: EDWRITE

        Writes explicit dynamics input to an LS-DYNA input file.

        Parameters
        ----------
        option
            Sets a flag in the LS-DYNA input file (Fname.Ext) to produce
            desired output.

            ANSYS - Set a flag to write results files for the ANSYS postprocessors (default).  The
                    files that will be written are Jobname.RST and Jobname.HIS
                    (see Notes below).

            LSDYNA - Set a flag to write results files for the LS-DYNA postprocessor (LS-POST).  The
                     files that will be written are D3PLOT, and files specified
                     by EDOUT and EDHIST (see Notes below).

            BOTH - Set a flag to write results files for both ANSYS and LS-DYNA postprocessors.

        fname
            File name and directory path (80 characters maximum, including
            directory; this limit is due to an LS-DYNA program limitation). If
            you do not specify a directory path, it will default to your
            working directory. The file name defaults to Jobname. Previous data
            on this file, if any, are overwritten.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        This command writes an LS-DYNA input file for the LS-DYNA solver.
        EDWRITE is only valid if explicit dynamic elements have been specified.
        This command is not necessary if the LS-DYNA solver is invoked from
        within ANSYS, in which case Jobname.K (or Jobname.R) is written
        automatically when the solution is initiated. (If LS-DYNA is invoked
        from within ANSYS, use EDOPT to specify desired output.)

        If the analysis is a small restart (EDSTART,2), the file that is
        written will have the name Jobname.R (by default) and will only contain
        changes from the original analysis.

        If the analysis is a full restart (EDSTART,3), the file that is written
        will have the name Jobname_nn.K (by default) and will contain all the
        information from the database. In a full restart, the jobname is
        changed to Jobname_nn (nn = 01 initially, and is incremented for each
        subsequent full restart.)

        A command is included in the LS-DYNA input file to instruct the LS-DYNA
        solver to write the results files indicated by Option.  By default, LS-
        DYNA will write the ANSYS results file Jobname.RST (see the EDRST
        command).  If Jobname.HIS is desired, you must also issue EDHIST.

        Option = LSDYNA or BOTH will cause LS-DYNA to write results files for
        the LS-POST postprocessor. The D3PLOT file is always written for these
        two options. If other LS-POST files are desired, you must issue the
        appropriate EDHIST and EDOUT commands.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDWRITE,%s,%s,%s" % (str(option), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Selist(self, sename="", kopt="", kint="", **kwargs):
        """
        APDL Command: SELIST

        Lists the contents of a superelement matrix file.

        Parameters
        ----------
        sename
            The name (case-sensitive) of the superelement matrix file created
            by the substructure generation pass (Sename.SUB).   Defaults to the
            current Jobname.  If a number, it is the element number of the
            superelement as used in the use pass.

        kopt
            List key:

            0 - List summary data only.

            1 - List contents, except load vectors and matrices.

            2 - List contents, except matrices.

            3 - List full contents.  Be aware that the listing may be extensive.

        kint
            Integer printout format key:

            OFF - Default.

            ON - Long format for large integers.

        Notes
        -----
        This command is valid in any processor.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "SELIST,%s,%s,%s" % (str(sename), str(kopt), str(kint))
        self.RunCommand(command, **kwargs)

    def Fecons(self, **kwargs):
        """
        APDL Command: FECONS

        Specifies "Constraints on nodes" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "FECONS," % ()
        self.RunCommand(command, **kwargs)

    def Ksymm(self, ncomp="", np1="", np2="", ninc="", kinc="", noelem="",
              imove="", **kwargs):
        """
        APDL Command: KSYMM

        Generates a reflected set of keypoints.

        Parameters
        ----------
        ncomp
            Symmetry key:

            X - X (or R) symmetry (default).

            Y - Y (or θ) symmetry.

            Z - Z (or Φ) symmetry.

        np1, np2, ninc
            Reflect keypoints from pattern beginning with NP1 to NP2 (defaults
            to NP1) in steps of NINC (defaults to 1).  If NP1 = ALL, NP2 and
            NINC are ignored and pattern is all selected keypoints [KSEL].  If
            Ncomp = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).  A component name may
            also be substituted for NP1 (NP2 and NINC are ignored).

        kinc
            Keypoint increment between sets.  If zero, the lowest available
            keypoint numbers are assigned [NUMSTR].

        noelem
            Specifies whether nodes and elements are also to be generated:

            0 - Generate nodes and point elements associated with the original keypoints, if
                they exist.

            1 - Do not generate nodes and elements.

        imove
            Specifies whether keypoints will be moved or newly defined:

            0 - Generate additional keypoints.

            1 - Move original keypoints to new position retaining the same keypoint numbers
                (KINC and NOELEM are ignored).  Valid only if the old keypoints
                are no longer needed at their original positions.
                Corresponding meshed items are also moved if not needed at
                their original position.

        Notes
        -----
        Generates a reflected set of keypoints (and corresponding mesh) from a
        given keypoint pattern by a symmetry reflection (see analogous node
        symmetry command, NSYM).  The MAT, TYPE, REAL, and ESYS attributes are
        based upon the keypoints in the pattern and not upon the current
        settings.  Reflection is done in the active coordinate system by
        changing a particular coordinate sign.  Keypoints in the pattern may
        have been generated in any coordinate system.  However, solid modeling
        in a toroidal coordinate system is not recommended.

        """
        command = "KSYMM,%s,%s,%s,%s,%s,%s,%s" % (str(ncomp), str(np1), str(np2), str(ninc), str(kinc), str(noelem), str(imove))
        self.RunCommand(command, **kwargs)

    def Celist(self, neqn1="", neqn2="", ninc="", option="", **kwargs):
        """
        APDL Command: CELIST

        Lists the constraint equations.

        Parameters
        ----------
        neqn1, neqn2, ninc
            List constraint equations from NEQN1 to NEQN2 (defaults to NEQN1)
            in steps of NINC (defaults to 1).  If NEQN1 = ALL (default), NEQN2
            and NINC are ignored and all constraint equations are listed.

        option
            Options for listing constraint equations:

            ANY - List equation set if any of the selected nodes are in the set (default). Only
                  externally-generated constraint equations are listed.

            ALL - List equation set only if all of the selected nodes are in the set. Only
                  externally-generated constraint equations are listed.

            INTE - List internally-generated constraint equations that are associated with MPC-
                   based contact. Constraint equations are listed only if all
                   the nodes in the set are selected.

            CONV - Convert internal constraint equations to external constraint equations.
                   Internal constraint equations are converted only if all of
                   the nodes in the set are selected.

        Notes
        -----
        This command is valid in any processor. However, the INTE and CONV
        options are only valid in the Solution processor after a SOLVE command
        has been issued.

        """
        command = "CELIST,%s,%s,%s,%s" % (str(neqn1), str(neqn2), str(ninc), str(option))
        self.RunCommand(command, **kwargs)

    def Focus(self, wn="", xf="", yf="", zf="", ktrans="", **kwargs):
        """
        APDL Command: /FOCUS

        Specifies the focus point (center of the window).

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        xf, yf, zf
            Location of the object to be at the focus point (center of the
            window) in the global Cartesian coordinate system.  If XF = AUTO,
            allow automatic location calculation.  If XF = USER, use focus
            location of last display (useful when last display had auto focus).

        ktrans
            Translate key:

            0 - Interpret numerical XF, YF, ZF values as described above.

            1 - Interpret XF, YF, ZF values as multiples of half-screens to translate from the
                current position in the screen coordinate system.  Example: XF
                of 2.4 translates the display approximately 2.4 half-screens to
                the left in the screen X (horizontal) direction.

            2 - Interpret XF, YF, ZF values as multiples of half-screens to translate from the
                current position in the global Cartesian coordinate system.
                Example: XF of 1.5 translates the display approximately 1.5
                half-screens in the global Cartesian X direction of the model.

        Notes
        -----
        Specifies the location on (or off) the model which is to be located at
        the focus point (center of the window).  For section and capped
        displays, the cutting plane is also assumed to pass through this
        location (unless the working plane is used via /CPLANE).  See also
        /AUTO and /USER commands.

        This command is valid in any processor.

        """
        command = "/FOCUS,%s,%s,%s,%s,%s" % (str(wn), str(xf), str(yf), str(zf), str(ktrans))
        self.RunCommand(command, **kwargs)

    def Secmodif(self, secid="", kywrd="", **kwargs):
        """
        APDL Command: SECMODIF

        Modifies a pretension section

        Parameters
        ----------
        secid
            Unique section number. This number must already be assigned to a
            section.

        norm
            Keyword specifying that the command will modify the pretension
            section normal direction.

        nx, ny, nz
            Specifies the individual normal components to modify.

        kcn
            Coordinate system number. This can be either 0 (Global Cartesian),
            1 (Global Cylindrical) 2 (Global Spherical), 4 (Working Plane), 5
            (Global Y Axis Cylindrical) or an arbitrary reference number
            assigned to a coordinate system.

        Notes
        -----
        The SECMODIF command either modifies the normal for a specified
        pretension section, or changes the name of the specified pretension
        surface.

        """
        command = "SECMODIF,%s,%s" % (str(secid), str(kywrd))
        self.RunCommand(command, **kwargs)

    def Anmres(self, delay="", min="", max="", inc="", autocntrky="", freq="",
               ext="", **kwargs):
        """
        APDL Command: ANMRES

        Performs animation of results over multiple results files in an
        explicit dynamic structural analysis or fluid flow analysis with
        remeshing.

        Parameters
        ----------
        delay
            Time delay during animation (default = 0.5 seconds).

        min
            Minimum results file number to animate. Default = 1 (for
            Jobname.RS01).

        max
            Maximum results file number to animate. Defaults to the highest
            numbered results file, Jobname.RSnn.

        inc
            Increment between results file numbers. Default = 1.

        autocntrky
            Automatic contour scaling option.

            0 - No auto-scaling (default).

            1 - Auto-scaling on.

        freq
             Results frequency key.

            0 or 1 - Animate every results set in each Jobname.EXT file (default).

            2 - Animate every other results set in each Jobname.EXT file.

            n - Animate every nth results set in each Jobname.EXT file.

        ext
            Extension of result files

            'rfl' - Animate Jobname.rflnn

            'rs' - Animate Jobname.rsnn. Default = 'rs'.

        Notes
        -----
        ANMRES invokes an ANSYS macro that performs animation across multiple
        results files (Jobname.EXT, Jobname.EXT, etc.) produced by an explicit
        dynamic structural analysis or fluid flow analysis with remeshing.
        Multiple results files typically occur when adaptive meshing is used in
        an explicit dynamic structural analysis or fluid flow analysis with
        remeshing. Each results file must have more than one set of results.
        ANMRES cannot be used for multiple results files that are caused by
        file splitting.

        ANMRES animates results from files having the currently specified
        jobname (Jobname.EXT - Jobname.EXT).  To change the current jobname,
        use  the /FILNAME command. The animation is based on the last plot
        command (e.g., PLDISP).

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "ANMRES,%s,%s,%s,%s,%s,%s,%s" % (str(delay), str(min), str(max), str(inc), str(autocntrky), str(freq), str(ext))
        self.RunCommand(command, **kwargs)

    def Plnsol(self, item="", comp="", kund="", fact="", fileid ="", **kwargs):
        """
        APDL Command: PLNSOL

        Displays results as continuous contours.

        Parameters
        ----------
        item
            Label identifying the item.  Valid item labels are shown in Table
            220: PLNSOL - Valid Item and Component Labels below.  Some items
            also require a component label.

        comp
            Component of the item (if required).  Valid component labels are
            shown in  Table 220: PLNSOL - Valid Item and Component Labels
            below.

        kund
            Undisplaced shape key:

            0 - Do not overlay undeformed structure display

            1 - Overlay displaced contour plot with undeformed display (appearance is system-
                dependent)

            2 - Overlay displaced contour plot with undeformed edge display (appearance is
                system-dependent)

        fact
            Scale factor for 2-D display for contact items.  Default value is
            1.  A negative scaling factor may be used to invert the display.

        fileid
            The file index number (obtained via the NLDIAG,NRRE,ON command).
            Valid only for Item = NRRE.

        Notes
        -----
        Displays the solution results as continuous contours across element
        boundaries for the selected nodes and elements. For example, PLNSOL,S,X
        displays the X component of stress S (that is,  the SX stress
        component). Various element results depend upon the recalculation
        method and the selected results location [AVPRIN, RSYS, LAYER, SHELL,
        and NSEL].  Contours are determined by linear interpolation within each
        element from the nodal values, which are averaged at a node whenever
        two or more elements connect to the same node (except for FMAG, which
        is summed at the node).

        For PowerGraphics displays [/GRAPHICS,POWER], results are plotted only
        for the model exterior surface.  The items marked with [2] are not
        supported by PowerGraphics. To plot midside nodes, you must first issue
        /EFACET,2.

        Table: 220:: : PLNSOL - Valid Item and Component Labels

        For SHELL131 and SHELL132 elements with KEYOPT(3) = 0 or 1, use the
        labels TBOT, TE2, TE3, . . ., TTOP instead of TEMP to view the
        individual temperature degree of freedom. When other thermal elements
        are included in the model, they should be unselected to avoid plotting
        undefined information. To view all temperatures in the same plot, set
        /ESHAPE,1 and /GRAPHICS,POWER and issue PLNSOL,TEMP.

        Not supported by PowerGraphics

        For the CONT items for elements CONTA171 through CONTA177, the reported
        data is averaged across the element. To obtain a more meaningful STAT
        value, use the PLESOL command.

        For MPC-based contact definitions, the value of STAT can be negative.
        This indicates that one or more contact constraints were intentionally
        removed to prevent overconstraint. STAT = -3 is used for MPC bonded
        contact; STAT = -2 is used for MPC no-separation contact.

        When plotting  Newton-Raphson residual items (Item = NRRE) from a file
        on the deformed geometry, the displacements are based on the current
        set of results in the database. These displacements may not correspond
        to the loadstep and substep in the .nrxxxxx file. (For more information
        about .nrxxxxx files and nonlinear diagnostics postprocessing, see the
        description of the NLDPOST command and Performing Nonlinear
        Diagnostics.)

        Works only if failure criteria information is provided. (For more
        information, see the documentation for the FC and TB commands.)

        Works only if user failure criteria routine is provided.

        Must be added via the FCTYP command first.

        Failure criteria are based on the effective stresses in the damaged
        material.

        """
        command = "PLNSOL,%s,%s,%s,%s,%s" % (str(item), str(comp), str(kund), str(fact), str(fileid ))
        self.RunCommand(command, **kwargs)

    def Reorder(self, **kwargs):
        """
        APDL Command: REORDER

        Specifies "Model reordering" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "REORDER," % ()
        self.RunCommand(command, **kwargs)

    def Voper(self, parr="", par1="", oper="", par2="", con1="", con2="",
              **kwargs):
        """
        APDL Command: *VOPER

        Operates on two array parameters.

        Parameters
        ----------
        parr
            The name of the resulting array parameter vector.  See *SET  for
            name restrictions.

        par1
            First array parameter vector in the operation.  May also be a
            scalar parameter or a literal constant.

        oper
            Operations:

            Addition: Par1+Par2. - Subtraction: Par1-Par2.

            Multiplication: Par1*Par2. - Division: Par1/Par2  (a divide by zero results in a value of zero).

            Minimum: minimum of Par1 and Par2. - Maximum: maximum of Par1 and Par2.

            Less than comparison: Par1<Par2 gives 1.0 if true, 0.0 if false. - Less than or equal comparison: Par1 Par2 gives 1.0 if true, 0.0 if false.

            Equal comparison: Par1 = Par2 gives 1.0 if true, 0.0 if false. - Not equal comparison: Par1 ≠ Par2 gives 1.0 if true, 0.0 if false.

            Greater than or equal comparison: Par1 Par2 gives 1.0 if true, 0.0 if false. - Greater than comparison: Par1>Par2 gives 1.0 if true, 0.0 if false.

            First derivative: d(Par1)/d(Par2).  The derivative at a point is determined  over points half way between the previous and next points (by linear  interpolation). Par1 must be a function (a unique Par1 value for each Par2 value) and Par2 must be in ascending order. - Second derivative: d2(Par1)/d(Par2)2.  See also DER1.

            Single integral:    Par1 d(Par2), where CON1 is the integration constant.  The integral at a point is determined by using the single integration procedure described in the Mechanical APDL Theory Reference. - Double integral:       Par1 d(Par2), where CON1 is the integration constant of
                              the first integral and CON2 is the integration
                              constant of the second integral.  If Par1
                              contains acceleration data, CON1 is the initial
                              velocity and CON2 is the initial displacement.
                              See also INT1.

            Dot product: Par1  . Par2.  Par1 and Par2 must each have three consecutive columns of data, with the columns containing the i, j, and k vector components, respectively.  Only the starting row index and the column index for the i components are specified for Par1 and Par2, such as A(1,1).  The j and k components of the vector are assumed to begin in the corresponding next columns, such as A(1,2) and A(1,3). - Cross product: Par1 x Par2.  Par1, Par2, and ParR must each have 3 components,
                              respectively.  Only the starting row index and
                              the column index for the i components are
                              specified for Par1, Par2, and ParR, such as
                              A(1,1).  The j and k components of the vector are
                              assumed to begin in the corresponding next
                              columns, such as A(1,2) and A(1,3).

            Gather:  For a vector of position numbers, Par2, copy the value of Par1 at each position number to ParR.  Example:  for Par1 = 10,20,30,40 and Par2 = 2,4,1;  ParR = 20,40,10. - Scatter:  Opposite of GATH operation.  For a vector of position numbers, Par2,
                              copy the value of Par1 to that position number in
                              ParR.  Example: for Par1 = 10,20,30,40,50 and
                              Par2 = 2,1,0,5,3; ParR = 20,10,50,0,40.

            Arctangent: arctangent of Par1/Par2 with the sign of each component considered. - Transform the data in Par1 from the global Cartesian coordinate system to the
                              local coordinate system given in CON1. Par1 must
                              be an N x 3 (i.e., vector) or an N x 6 (i.e.,
                              stress or strain tensor) array. If the local
                              coordinate system is a cylindrical, spherical, or
                              toroidal system, then you must provide the global
                              Cartesian coordinates in Par2 as an N x 3 array.
                              Set CON2 = 1 if the data is strain data.

        par2
            Second array parameter vector in the operation.  May also be a
            scalar parameter or a literal constant.

        con1
            First constant (used only with the INT1 and INT2 operations).

        con2
            Second constant (used only with the INT2 operation).

        Notes
        -----
        Operates on two input array parameter vectors and produces one output
        array parameter vector according to:

        ParR = Par1 o Par2

        where the operations (o) are described below.  ParR may be the same as
        Par1 or Par2.  Absolute values and scale factors may be applied to all
        parameters [*VABS, *VFACT].  Results may be cumulative [*VCUM].
        Starting array element numbers must be defined for each array parameter
        vector if it does not start at the first location, such as
        *VOPER,A,B(5),ADD,C(3) which adds the third element of C to the fifth
        element of B and stores the result in the first element of A.
        Operations continue on successive array elements [*VLEN, *VMASK] with
        the default being all successive elements.  Skipping array elements via
        *VMASK or *VLEN for the DER_ and INT_ functions skips only the writing
        of the results (skipped array element data are used in all
        calculations).

        Parameter functions and operations are available to operate on a scalar
        parameter or a single element of an array parameter, such as SQRT(B) or
        SQRT(A(4)).  See the *SET command for details.  Operations on a
        sequence of array elements can be done by repeating the desired
        function or operation in a do-loop [*DO].  The vector operations within
        the ANSYS program (*VXX commands) are internally programmed do-loops
        that conveniently perform the indicated operation over a sequence of
        array elements.  If the array is multidimensional, only the first
        subscript is incremented in the do-loop, that is, the operation repeats
        in column vector fashion "down" the array.  For example, for A(1,5),
        A(2,5), A(3,5), etc.  The starting location of the row index must be
        defined for each parameter read and for the result written.

        The default number of loops is from the starting result location to the
        last result location and can be altered with the *VLEN command.  A
        logical mask vector may be defined to control at which locations the
        operations are to be skipped [*VMASK].  The default is to skip no
        locations.  Repeat operations automatically terminate at the last array
        element of the result array column if the number of loops is undefined
        or if it exceeds the last result array element.  Zeroes are used in
        operations for values read beyond the last array element of an input
        array column.  Existing values in the rows and columns of the results
        matrix remain unchanged where not changed by the requested operation
        values.  The result array column may be the same as the input array
        column since results in progress are stored in a temporary array until
        being moved to the results array at the end of the operation.  Results
        may be overwritten or accumulated with the existing results [*VCUM].
        The default is to overwrite results.  The absolute value may be used
        for each parameter read or written [*VABS].  A scale factor (defaulting
        to 1.0) is also applied to each parameter read and written [*VFACT].

        This command is valid in any processor.

        """
        command = "*VOPER,%s,%s,%s,%s,%s,%s" % (str(parr), str(par1), str(oper), str(par2), str(con1), str(con2))
        self.RunCommand(command, **kwargs)

    def Cycopt(self, option="", value1="", value2="", value3="", value4="",
               value5="", value6="", value7="", **kwargs):
        """
        APDL Command: CYCOPT

        Specifies solution options for a cyclic symmetry analysis.

        Parameters
        ----------
        option
            One of the following options:

            BCMULT - Controls whether cyclic sector array parameter names are reused or created new
                     for multiple entities.

            Value1 - The flag value.

            0 (OFF or NO) - Create new array parameter names (default)

            1(ON or YES) - Reuse array parameter names

            COMBINE - For linear static cyclic symmetry analysis with non-cyclically symmetric
                      loading only, expands and combines all harmonic index
                      solutions and writes them to the results file during the
                      solution phase of the analysis.

            Value1 - The flag value.

            0 (OFF or NO) - Disable combining of harmonic index solutions (default)

            1 (ON or YES) - Enable combining of harmonic index solutions

            DEFAULT - Set the default cyclic solution settings.

            DOF - The degrees of freedom to couple from the nodes on the low sector boundary to
                  nodes on the high boundary:

            Value1 - The component pair ID number.

            Value2, Value3, Value4,  . . . ,  Value7 - The constraint-equation/-coupling degree of freedom (DOF) for this pair. Repeat
                              the command to add other DOFs. The default is
                              constraint-equation/-coupling all applicable
                              DOFs.

            FACETOL - Tolerance for inclusion of surface nodes into your basic sector. Autodetect
                      defaults to 15°, accommodating most sections. Specify a
                      new Value1 only when extreme cut angles or complex model
                      geometry cause surface nodes to be excluded. See Notes
                      (below) for more information.

            ANSYS, Inc. recommends that successful auto-detection depends more on the value of ANGTOL than the value of FACETOL. Please refer to CYCOPT Auto Detection Tolerance Adjustments for Difficult Cases for more information about auto-detection and the CYCOPT command. - Value1

            The face tolerance applies only to auto detection from node/element models (already meshed and no solid model), and it defaults to 15°.   - HINDEX

            The harmonic index solution ranges for modal or buckling cyclic symmetry analyses. The SOLVE command initiates a cyclic symmetry solution sequence at the harmonic indices specified. (By default, the SOLVE command solves for all available harmonic indices.) Static and harmonic cyclic symmetry solutions always use all harmonic indices required for the applied loads. - EVEN / ODD

            For low-frequency electromagnetic analysis only, EVEN specifies a symmetric solution and ODD specifies an antisymmetric solution.  - The value you specify is based on the harmonic index: EVEN (default) indicates
                              harmonic index = 0, and ODD indicates harmonic
                              index = N / 2 (where N is an integer representing
                              the number of sectors in 360°). A value of ODD
                              applies only when N is an even number.

            The CYCOPT command with this HINDEX option is cumulative. To remove an option (for example, EVEN), issue this command: CYCOPT,HINDEX,EVEN,,,-1 - ALL

            Solve all applicable harmonic indices. - Note:  Value2 must be blank.

            Value1, Value2, Value3 - Solve harmonic indices in range Value1 through Value2 in steps of Value3.
                              Repeat the command to add other ranges. The
                              default solves all applicable harmonic indices.

            Value4 - The only valid value is -1. If specified, it removes Value1 through Value2 in
                     steps of Value3 from the set to solve. By default, if
                     Value4 = -1 then Value1 = 0, Value2 = 0, and Value3 = 1.

            Value5 - For static and harmonic analyses, the tolerance for determining if a Fourier
                     contribution of a load contributes to the response
                     (default = 1.0E-5).

            If Value5=STATIC, it forces the program to solve only the specified harmonic indices (even if a load may have a Fourier contribution in an index not specified). - LDSECT

            Restricts subsequently defined force loads and surface loads to a specified sector. The restriction remains in effect until you change or reset it. This option is not available for harmonic analyses based on mode-superposition (CYCOPT,MSUP,1) - Value1

            The sector number. A value other than 0 (default) is valid for a cyclic symmetry analysis with non-cyclically symmetric loading only. A value of 0 (or ALL) resets the default behavior for cyclic loading (where the loads are identical on all sectors). - MOVE

            Specifies if the program should move high- or low-edge component nodes paired within the specified tolerance (TOLER) to create precisely matching pairs.  - Value1

            The flag value. - 0

            Do not move edge component nodes (default) - 1 or HIGH

            Move the high-edge component nodes to precisely match the low-edge component nodes - -1 or LOW

            Move the low-edge component nodes to precisely match the high-edge component nodes - MSUP

            For modal cyclic symmetry analysis only, this flag is used to limit the results written to the Jobname.MODE and Jobname.RST files in preparation for a subsequent mode-superposition-based analysis. In a linear perturbation modal analysis, this option must be specified in the first load step of the preceding base analysis.  - Value1

            The flag value. - 0 (OFF or NO)

            Write results for the basic and duplicate sectors to the Jobname.MODE and Jobname.RST files (default) - 1 (ON or YES)

            Write only the basic sector results to the Jobname.MODE and Jobname.RST files for use in a subsequent mode-superposition-based analysis - STATUS

            List the solution option settings active for the cyclic model. - TOLER

            The tolerance used to determine whether a node on the low edge is paired with a node on the high edge. - Value1

            The tolerance value. - Greater than 0

            The absolute distance tolerance for automatic sector-boundary detection and low-/high-edge component node pairing - Less than 0

            The relative tolerance for automatic sector-boundary detection and low-/high-edge component node pairing. In this case, the tolerance is Value1 * Length, where Length is the length of the diagonal of an imaginary box enclosing the model - 0

            Tolerance is set to -1.0 x 10-4 (default) - Value2

            ANGTOL = Maximum allowable angle tolerance. (default = 0.01°) - The valid range for ANGTOL is model dependent.

            If you input both the number of sectors and a sector angle, the angle must match 360/(number of sectors) within ANGTOL.  - If you input only a sector angle, it must divide evenly into 360° within
                              ANGTOL.

            If you input a sector angle, the final cyclic sector must span that angle within ANGTOL.   - For auto detected sector angle, the final cyclic sector must span 360/(number
                              of sectors) within ANGTOL, everywhere along the
                              LOW/HIGH boundaries.

            If ANGTOL is too small, your CAD or FEA model may not be accurate enough to allow auto detection or verification. - If ANGTOL is too large, you may get an unexpected or incorrect boundary
                              definition, or in other cases fail to detect the
                              boundaries.

            For some difficult cases from FEA models (not solid models), you  may need to change the value of FACETOL to achieve auto detection. Please refer to CYCOPT Auto Detection Tolerance Adjustments for Difficult Cases for more information about auto-detection and the CYCOPT command. - USRROT

            Flag specifying whether ANSYS should override automatic nodal rotations to edge components and allow you to apply nodal rotations manually.  - Value1

            The flag value. - 0 (OFF or NO)

            Allow automatic node rotation (default) - 1 (ON or YES)

            Suppress automatic node rotation. If you select this option, you must apply appropriate nodal rotations to all edge component nodes; otherwise, your analysis will yield incorrect solution results. - LOW

            Suppresses automatic rotation of low-edge component nodes only, allowing you to apply them manually. Automatic rotation of high-edge component nodes occurs to produce the matching edge nodes required for a valid cyclic solution. - HIGH

            Suppresses automatic rotation of high-edge component nodes only, allowing you to apply them manually. Automatic rotation of low-edge component nodes occurs to produce the matching edge nodes required for a valid cyclic solution. - VTSOL

            Employ Variational Technology to accelerate your cyclic symmetry analysis.  - Value1

            The flag value. - 0 (OFF or NO)

            Do not use Variational Technology (VT) (default) - 1 (ON or YES)

        Notes
        -----
        The program solves a cyclically symmetric model (set up via the CYCLIC
        command during preprocessing) at the harmonic indices specified via the
        CYCOPT command.

        The CYCOPT,COMBINE option is an alternative to the /CYCEXPAND command
        and is especially useful for testing purposes. However, ANSYS, Inc.
        recommends specifying COMBINE only when the number of sectors is
        relatively small. (The option expands nodes and elements into the full
        360° and can slow postprocessing significantly.

        If you issue a CYCOPT,TOLER command to set a tolerance for edge-
        component pairing before issuing the CYCLIC command, the CYCLIC command
        uses the specified tolerance when performing automatic edge-component
        detection.

        In cases involving non-cyclically symmetric loading (that is, when
        LDSECT > 0), the underlying command operations create or modify the
        required SECTOR tabular boundary condition (BC) data to apply on the
        appropriate sector. Therefore, it is not necessary to manipulate tables
        for situations where the applied BC is not a function of other tabular
        BC variables such as TIME, X, Y, Z, and so on.

        To delete a previously applied load on a specified sector, issue an
        FDELE command.

        Because edge nodes are rotated into the cyclic coordinate system during
        solution, any applied displacements or forces on sector edges will be
        in the cyclic coordinate system.

        The CYCOPT command is valid in the preprocessing and solution stages of
        an analysis.

        To learn more about analyzing a cyclically symmetric structure, see the
        Cyclic Symmetry Analysis Guide.

        Distributed ANSYS Restriction: The COMBINE option is not supported in
        Distributed ANSYS.

        """
        command = "CYCOPT,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(value1), str(value2), str(value3), str(value4), str(value5), str(value6), str(value7))
        self.RunCommand(command, **kwargs)

    def Bfalist(self, area="", lab="", **kwargs):
        """
        APDL Command: BFALIST

        Lists the body force loads on an area.

        Parameters
        ----------
        area
            Area at which body load is to be listed.  If ALL (or blank), list
            for all selected areas [ASEL].  If AREA = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be substituted for AREA.

        lab
            Valid body load label. If ALL, use all appropriate labels. Load
            labels are listed under "Body Loads" in the input table for each
            element type in the Element Reference. See the BFA command for
            labels.

        Notes
        -----
        Lists the body force loads for the specified area and label.  Body
        loads may be defined on an area with the BFA command.

        This command is valid in any processor.

        """
        command = "BFALIST,%s,%s" % (str(area), str(lab))
        self.RunCommand(command, **kwargs)

    def Lsbw(self, nl="", sepo="", keep="", **kwargs):
        """
        APDL Command: LSBW

        Subtracts the intersection of the working plane from lines (divides
        lines).

        Parameters
        ----------
        nl
            Line (or lines, if picking is used) to be subtracted from.  If NL =
            ALL, use all selected lines.  If NL = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be input for NL.

        sepo
            Behavior of the created boundary.

            (blank) - The resulting lines will share keypoint(s) where they touch.

            SEPO - The resulting lines will have separate, but coincident keypoint(s).

        keep
            Specifies whether NL lines are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete NL lines after LSBW operation (override BOPTN command settings).

            KEEP - Keep NL lines after LSBW operation (override BOPTN command settings).

        Notes
        -----
        Generates new lines by subtracting the intersection of the working
        plane from the NL lines.  The intersection will be a keypoint(s).  The
        working plane must not be in the same plane as the NL line(s).  If SEPO
        is blank, the NL line is divided and the resulting lines will be
        connected, sharing a common keypoint where they touch.  If SEPO is set
        to SEPO, NL is divided into two unconnected lines with separate
        keypoints.  See the Modeling and Meshing Guide for an illustration.
        See the BOPTN command for an explanation of the options available to
        Boolean operations.  Element attributes and solid model boundary
        conditions assigned to the original entities will not be transferred to
        the new entities generated.  Areas that completely contain the input
        lines will be updated if the lines are divided by this operation.

        """
        command = "LSBW,%s,%s,%s" % (str(nl), str(sepo), str(keep))
        self.RunCommand(command, **kwargs)

    def Dalist(self, area="", **kwargs):
        """
        APDL Command: DALIST

        Lists the DOF constraints on an area.

        Parameters
        ----------
        area
            List constraints for this area.  If ALL (default), list for all
            selected areas [ASEL].  If P1 = P, graphical picking is enabled and
            all remaining command fields are ignored (valid only in the GUI).
            A component name may also be substituted for AREA.

        Notes
        -----
        Lists the degree of freedom constraints on an area previously specified
        with the DA command.

        This command is valid in any processor.

        """
        command = "DALIST,%s" % (str(area))
        self.RunCommand(command, **kwargs)

    def Addam(self, af="", aa="", ab="", ac="", ad="", amin="", **kwargs):
        """
        APDL Command: ADDAM

        Specifies the acceleration spectrum computation constants for the
        analysis of shock resistance of shipboard structures.

        Parameters
        ----------
        af
            Direction-dependent acceleration coefficient for elastic or
            elastic-plastic analysis option (default = 0).

        aa, ab, ac, ad
            Coefficients for the DDAM acceleration spectrum equations. Default
            for these coefficients is zero.

        amin
            The minimum acceleration value in inch/sec2.  It defaults to 2316
            inch/sec2 which equals 6g, where g is acceleration due to gravity
            (g = 386 inch/sec2).

        Notes
        -----
        This command specifies acceleration coefficients to analyze shock
        resistance of shipboard equipment.  These coefficients are used to
        compute mode coefficients according to the equations given in Dynamic
        Design Analysis Method in the Mechanical APDL Theory Reference.  The
        form of these equations is based on the Naval NRL Dynamic Design
        Analysis Method.  This command, along with the VDDAM and SED commands,
        is used with the spectrum (ANTYPE,SPECTR) analysis as a special purpose
        alternative to the SV, FREQ, and SVTYP commands.  The mass and length
        units of the model must be in pounds and inches, respectively.

        DDASPEC may alternatively be used to calculate spectrum coefficients.

        This command is also valid in PREP7.

        """
        command = "ADDAM,%s,%s,%s,%s,%s,%s" % (str(af), str(aa), str(ab), str(ac), str(ad), str(amin))
        self.RunCommand(command, **kwargs)

    def Srss(self, signif="", label="", abssumkey="", forcetype="", **kwargs):
        """
        APDL Command: SRSS

        Specifies the square root of sum of squares mode combination method.

        Parameters
        ----------
        signif
            Combine only those modes whose significance level exceeds the
            SIGNIF threshold.  For single point, multipoint, or DDAM response
            (SPOPT,SPRS, MPRS or DDAM), the significance level of a mode is
            defined as the mode coefficient of the mode, divided by the maximum
            mode coefficient of all modes.  Any mode whose significance level
            is less than SIGNIF is considered insignificant and is not
            contributed to the mode combinations.  The higher the SIGNIF
            threshold, the fewer the number of modes combined. SIGNIF defaults
            to 0.001.  If SIGNIF is specified as 0.0, it is taken as 0.0.
            (This mode combination method is not valid for SPOPT,PSD.)

        label
            Label identifying the combined mode solution output.

            DISP - Displacement solution (default).  Displacements, stresses, forces, etc., are
                   available.

            VELO - Velocity solution.  Velocities, "stress velocities," "force velocities," etc.,
                   are available.

            ACEL - Acceleration solution.  Accelerations, "stress accelerations," "force
                   accelerations," etc., are available.

        abssumkey
            Absolute Sum combination key (for SPOPT,MPRS only):

            NO - Do not use the Absolute Sum method (default).

            YES - Combine the modes per excitation direction using the Absolute Sum method, then
                  combine the resulting quantities using the square root of sum
                  of squares method.

        forcetype
            Label identifying the forces to be combined:

            STATIC - Combine the modal static forces (default).

            TOTAL - Combine the modal static plus inertial forces.

        Notes
        -----
        This command is also valid for PREP7.

        """
        command = "SRSS,%s,%s,%s,%s" % (str(signif), str(label), str(abssumkey), str(forcetype))
        self.RunCommand(command, **kwargs)

    def Wsort(self, lab="", kord="", wopt="", oldmax="", oldrms="", **kwargs):
        """
        APDL Command: WSORT

        Initiates element reordering based upon a geometric sort.

        Parameters
        ----------
        lab
            Coordinate (in the active system) along which element centroid
            locations are sorted.  Valid labels are:  X, Y, Z, ALL.  If ALL
            (default), all three directions will be used, and the order
            corresponding to the lowest MAX or RMS wavefront value will be
            retained.

        kord
            Sort order:

            0 - Sort according to ascending coordinate values.

            1 - Sort according to descending coordinate values.

        --
            Unused field.

        wopt
            Option for comparison:

            MAX - Use maximum wavefront value for comparison (default).

            RMS - Use RMS wavefront value.

        oldmax, oldrms
            MAX and RMS wavefront values of model to be used in place of the
            old values.  OLDRMS defaults to OLDMAX (and vice versa).  If
            neither is  specified, each defaults to its calculated old value.

        Notes
        -----
        Initiates element reordering based upon a geometric sort of the element
        centroid locations.  Wave lists, if any [WSTART], are ignored.
        Reordering affects only the element order for the solution phase and
        not the element numbers (input referring to element numbers, such as
        element pressures, is unaffected by reordering).

        Note:: : The new order is retained only if new the new maximum or RMS
        wavefront values are lower than the old values, as described below.
        See the WAVES command for another reordering procedure and for more
        details on reordering.  The resulting element ordering can be shown by
        listing the wavefront history [WFRONT,1] or by displaying elements with
        their element location numbers [/PNUM].

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "WSORT,%s,%s,%s,%s,%s" % (str(lab), str(kord), str(wopt), str(oldmax), str(oldrms))
        self.RunCommand(command, **kwargs)

    def Edrst(self, nstep="", dt="", **kwargs):
        """
        APDL Command: EDRST

        Specifies the output interval for an explicit dynamic analysis.

        Parameters
        ----------
        nstep
            Number of steps at which output is written to the results file
            (Jobname.RST). Defaults to 100. When you specify NSTEP, NSTEP+2
            results are written to the Jobname.RST file. The time interval
            between output is TIME / NSTEP, where TIME is the analysis end-time
            specified on the TIME command. Do not specify a value of NSTEP = 0.

        dt
            Time interval at which output is written to the results file
            (Jobname.RST). If NSTEP is input, DT is ignored.

        Notes
        -----
        You can use NSTEP or DT to specify the output interval to be used for
        Jobname.RST. You should not specify both quantities; if both are input,
        NSTEP will be used.

        In an explicit dynamic small restart (EDSTART,2) or full restart
        analysis (EDSTART,3), the EDRST setting will default to the NSTEP or DT
        value used in the original analysis. You can issue EDRST in the restart
        to change this setting.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDRST,%s,%s" % (str(nstep), str(dt))
        self.RunCommand(command, **kwargs)

    def Trlcy(self, lab="", tlevel="", n1="", n2="", ninc="", **kwargs):
        """
        APDL Command: /TRLCY

        Specifies the level of translucency.

        Parameters
        ----------
        lab
            Apply translucency level to the items specified by the following
            labels:

            ELEM - Elements.  Use N1, N2, NINC fields for element numbers.

            AREA - Solid model areas.  Use N1, N2, NINC fields for area numbers.

            VOLU - Solid model volumes.  Use N1, N2, NINC fields for volume numbers.

            ISURF - Isosurfaces (surfaces of constant stress, etc., value).  Translucency varies
                    with result value, to a maximum of the specified
                    translucency level.

            CM - Component group.  Use N1 for component name, ignore N2 and NINC.

            CURVE - Filled areas under curves of line graphs.  Use N1, N2, NINC fields for curve
                    numbers.

            ZCAP - If /TYPE,WN,ZCAP is the current display type, then /TRLCY,ZCAP,TLEVEL will
                   display the model in window WN with the portion of the model
                   in front of the section plane displayed at the translucency
                   level TLEVEL.

            ON, OFF - Sets the specified translucency display on or off. All other fields are
                      ignored.

        tlevel
            Translucency level:  0.0 (opaque) to 1.0 (transparent).

        n1, n2, ninc
            Used only with labels as noted above.  Apply translucency level to
            Lab items numbered N1 to N2 (defaults to N1) in steps of NINC
            (defaults to 1).  If N1 is blank or ALL, apply specified
            translucency level to entire selected range.  If Lab is CM, use
            component name for N1 and ignore N2 and NINC. A value of N1 = P
            allows you to graphically pick elements, areas, and volumes. You
            can then assign translucency levels to the entities via the picker.
            The Lab and TLEVEL fields are ignored when translucency is applied
            by picking.

        Notes
        -----
        Specifies the level of translucency for various items.  Issue
        /TRLCY,DEFA to reset the default (0) translucency levels.  This command
        is valid only on selected 2-D and 3-D graphics devices; see  in the
        Basic Analysis Guide for more information on applying translucency.

        For 2-D devices, ANSYS displays only the visible faces of the items
        being displayed.  The information behind the facing planes is not
        displayed.  Issuing the /SHRINK command will force the hardware to
        display information behind the translucent items.

        This command is valid in any processor.

        """
        command = "/TRLCY,%s,%s,%s,%s,%s" % (str(lab), str(tlevel), str(n1), str(n2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Sabs(self, key="", **kwargs):
        """
        APDL Command: SABS

        Specifies absolute values for element table operations.

        Parameters
        ----------
        key
            Absolute value key:

            0 - Use algebraic values in operations.

            1 - Use absolute values in operations.

        Notes
        -----
        Causes absolute values to be used in the SADD, SMULT, SMAX, SMIN, and
        SSUM operations.

        """
        command = "SABS,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Axpy(self, vr="", vi="", m1="", wr="", wi="", m2="", **kwargs):
        """
        APDL Command: *AXPY

        Performs the matrix operation M2= v*M1 + w*M2.

        Parameters
        ----------
        vr, vi
            The real and imaginary parts of the scalar v. Default value is 0.

        m1
            Name of matrix M1. If not specified, the operation M2 = w*M2 will
            be performed.

        wr, wi
            The real and imaginary parts of the scalar w. Default value is 0.

        m2
            Name of matrix M2. Must be specified.

        Notes
        -----
        The matrices M1 and M2 must have the same dimensions and same type
        (dense or sparse). If M2 is real, vi and wi are ignored.

        """
        command = "*AXPY,%s,%s,%s,%s,%s,%s" % (str(vr), str(vi), str(m1), str(wr), str(wi), str(m2))
        self.RunCommand(command, **kwargs)

    def Keep(self, key="", **kwargs):
        """
        APDL Command: KEEP

        Stores POST26 definitions and data during active session.

        Parameters
        ----------
        key
            State or value

            On or 1  - Allows you to exit and reenter /POST26 without losing your current time history
                       variable information. Keeps a cache of the /POST26
                       variable information including the active file name
                       (FILE),  variable definitions (NSOL, ESOL, GAPF, RFORCE,
                       SOLU, and EDREAD) and stored variable data in memory for
                       the current ANSYS session.

            Off or 0  - /POST26 variable information is deleted when you exit /POST26.

        Notes
        -----
        Your variable information is saved in memory only for the current
        active ANSYS session. It is deleted when you exit ANSYS. This
        information is also deleted when you issue /CLEAR, RESUME, SOLVE, or
        RESET.

        When you reenter /POST26 all time history variable data is available
        for use. When you issue STORE,NEW, variable definitions created by math
        operations such as ADD or PROD will not be restored. However, variables
        defined with NSOL, ESOL, GAPF, RFORCE, SOLU, and EDREAD will be
        restored. Only the last active results file name is kept in memory
        (FILE).

        Commands such as LAYERP26, SHELL, and FORCE that specify the location
        or a component of data to be stored will retain the setting at the time
        of exiting /POST26 .

        """
        command = "KEEP,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Mfimport(self, fnumb="", option="", fname="", ext="", **kwargs):
        """
        APDL Command: MFIMPORT

        Imports a new field into a current ANSYS Multi-field solver analysis.

        Parameters
        ----------
        fnumb
            Field number specified by the MFELEM command.

        option
            Selects data to read.

            DB - Reads a CDB file. The CDB file name and extension are specified by Fname and
                 Ext.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        Notes
        -----
        The field to be imported should be written to a CDB file (CDWRITE
        command). This file is read into the database, offsetting all existing
        element type numbers, node numbers, etc. in order to accommodate the
        imported field. (See the NUMOFF command for information on offset
        capabilities.) It then updates all of the previously issued MFxx
        commands to the new element type numbers. A new field is created using
        the specified field number, which must not currently exist. If there
        are no ANSYS Multi-field solver command files written for the existing
        fields in the database, one will be written for each field with the
        default name (see the MFCMMAND command). A MFCMMAND will be issued for
        the imported field as well.

        Repeat the MFIMPORT command to import additional fields.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFIMPORT,%s,%s,%s,%s" % (str(fnumb), str(option), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Cfwrite(self, command="", **kwargs):
        """
        APDL Command: *CFWRITE

        Writes an ANSYS command (or similar string) to a "command" file.

        Parameters
        ----------
        command
            Command or string to be written.  The standard command form of a
            label followed by arguments separated by commas is assumed.
            Command may be a parameter assignment (e.g.,  *CFWRITE, A = 5).

        Notes
        -----
        Writes an ANSYS command (or similar string) to the file opened with
        *CFOPEN.  The Command string is not executed (except that numeric and
        character parameter substitution and operations (with imbedded *, /, >,
        etc. characters) are performed before writing).  When used with *GET
        results and parameter substitution, an ANSYS command can be created
        from results and then read back into the ANSYS program (or used
        elsewhere).  For example, if the command *CFWRITE,BF,NNUM,TEMP,TVAL is
        used in a do-loop, where TVAL is a parameter value returned from the
        *GET operation and NNUM is a specified or returned parameter value, a
        series of BF  commands, with numerical values substituted for the two
        parameters, will be written.  To create a file without parameter
        substitution, use *CREATE.

        This command is valid in any processor.

        """
        command = "*CFWRITE,%s" % (str(command))
        self.RunCommand(command, **kwargs)

    def Rstmac(self, file1="", lstep1="", sbstep1="", file2="", lstep2="",
               sbstep2="", tolern="", maclim="", cname="", keyprint="",
               unvscale="", keym_ass="", **kwargs):
        """
        APDL Command: RSTMAC

        Calculates modal assurance criterion (MAC) and matches nodal solutions
        from two results files or from one results file and one universal
        format file.

        Parameters
        ----------
        file1
            File name (32 characters maximum) corresponding to the first
            results file (.rst or .rstp file). If the file name does not
            contain the extension, it defaults to .rst.

        lstep1
            Load step number of the results to be read in File1.

            N  - Reads load step N. Defaults to 1.

        sbstep1
            Substep number of the results to be read in File1.

            N  - Reads substep N.

            All  - Reads all substeps. This value is the default.

        file2
            File name (32 characters maximum) corresponding to the second file
            (.rst, .rstp, or .unv file). If the file name does not contain the
            extension, it defaults to .rst.

        lstep2
            Load step number of the results to be read in File2.

            N  - Reads load step N. Defaults to 1.

        sbstep2
            Substep number of the results to be read in File2.

            N  - Reads substep N.

            All  - Reads all substeps. This value is the default.

        tolern
            Tolerance for nodes matching. If TolerN = -1, the nodes on File2
            are mapped into the selected elements of File1. If TolerN is
            positive, the selected nodes of File1 are matched to the nodes of
            File2. The default value is 0.01.

        maclim
            Smallest acceptable MAC value. Must be  0 and  1. The default value
            is 0.90.

        cname
            Name of the component from the first file (File1). The component
            must be based on nodes. If unspecified, all nodes are matched and
            used for MAC calculations. If a component name is specified, only
            nodes included in the specified component are used. Not applicable
            to node mapping (TolerN=-1).

        keyprint
            Printout options:

            0  - Printout matched solutions table. This value is the default.

            1  - Printout matched solutions table and full MAC table.

            2  - Printout matched solutions table, full MAC table and matched nodes table.

        unvscale
            Scaling of the nodal coordinates when File2 is a universal format
            file (.unv).

            Blank or 0  - No scaling (default).

            Non-zero value  - Scaling factor applied to the nodal coordinates.

        keymass
            Key to include the mass matrix diagonal in the calculations.

            ON  - Include the mass matrix diagonal. This is the default.

            OFF  - Do not include the mass matrix diagonal.

        Notes
        -----
        The RSTMAC command allows the comparison of the solutions from either:

        Two different results files

        One result file and one universal format file

        The modal assurance criterion (MAC) is used.

        The meshes read on File1 and File2 may be different. If TolerN>0, the
        nodes are matched. This is the default. If TolerN = -1, the nodes are
        mapped and the solutions are interpolated from File1.

        Units and coordinate systems must be the same for both models. When a
        universal format file is used, the nodal coordinates can be scaled
        using UNVscale.

        The corresponding database file (.db) for File1 must be resumed before
        running the command only if a component (Cname) is used or if the nodes
        are mapped (TolerN = -1).

        Results may be real or complex; however, if results from File1 have a
        different type from results in File2, only the real parts of the
        solutions are taken into account in MAC calculations. The analysis type
        can be arbitrary.

        Only structural degrees of freedom are considered. Degrees of freedom
        can vary between File1 and File2, but at least one common degree of
        freedom must exist.

        When node mapping and solution interpolation is performed (TolerN=-1),
        File1 must correspond to a model meshed in solid and/or shell elements.
        Other types of elements can be present but the node mapping is not
        performed for those elements. Interpolation is performed on UX, UY, and
        UZ degrees of freedom.

        The solutions read on the results files are not all written to the
        database, therefore, subsequent plotting or printing of solutions is
        not possible.  A SET command must be issued after the RSTMAC command to
        post-process each solution.

        RSTMAC comparison on cyclic symmetry analysis works only if the number
        of sectors on File1 and File2 are the same. Also comparison cannot be
        made between cyclic symmetry results and full 360 degree model results
        (File1 – cyclic solution, File2 – full 360 degree model solution).
        Comparing cyclic symmetry solutions written on selected set of node
        (OUTRES) is not supported.

        The modal assurance criterion values can be retrieved as parameters
        using the *GET command (Entity = RSTMAC).

        For more information and an example, see Comparing Nodal Solutions From
        Two Models (RSTMAC) in the Basic Analysis Guide.

        """
        command = "RSTMAC,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(file1), str(lstep1), str(sbstep1), str(file2), str(lstep2), str(sbstep2), str(tolern), str(maclim), str(cname), str(keyprint), str(unvscale), str(keym_ass))
        self.RunCommand(command, **kwargs)

    def Psearch(self, pname="", **kwargs):
        """
        APDL Command: /PSEARCH

        Specifies a directory to be searched for "unknown command" macro files.

        Parameters
        ----------
        pname
            Path name (64 characters maximum, and must include the final
            delimiter) of the middle directory to be searched.  Defaults to the
            user home directory.  If Pname = OFF, search only the ANSYS and
            current working directories.  If Pname = STAT, list the current
            middle directory and show the ANSYS_MACROLIB setting.

        Notes
        -----
        Specifies the pathname of a directory for file searches when reading
        "unknown command" macro files.  The search for the files is typically
        from the ANSYS directory, then from the user home directory, and then
        from the current working directory.  This command allows the middle
        directory searched to be other than the user home directory.

        This command is valid only at the Begin Level.

        """
        command = "/PSEARCH,%s" % (str(pname))
        self.RunCommand(command, **kwargs)

    def Nwpave(self, n1="", n2="", n3="", n4="", n5="", n6="", n7="", n8="",
               n9="", **kwargs):
        """
        APDL Command: NWPAVE

        Moves the working plane origin to the average location of nodes.

        Parameters
        ----------
        n1, n2, n3, . . . , n9
            Nodes used in calculation of the average.  At least one must be
            defined.  If N1 = P, graphical picking is enabled and all remaining
            command fields are ignored (valid only in the GUI).

        Notes
        -----
        Averaging is based on the active coordinate system.

        This command is valid in any processor.

        """
        command = "NWPAVE,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(n1), str(n2), str(n3), str(n4), str(n5), str(n6), str(n7), str(n8), str(n9))
        self.RunCommand(command, **kwargs)

    def Dj(self, elem="", label="", value="", **kwargs):
        """
        APDL Command: DJ

        Specifies boundary conditions on the components of relative motion of a
        joint element.

        Parameters
        ----------
        elem
            Element number or ALL to be specified.

        label
            Valid labels are:

            UX - Displacement in local x direction.

            UY - Displacement in local y direction.

            UZ - Displacement in local z direction.

            ROTX - Rotation about local x axis.

            ROTY - Rotation about local y axis.

            ROTZ - Rotation about local y axis.

            VELX - Linear velocity in local x direction.

            VELY - Linear velocity in local y direction.

            VELZ - Linear velocity in local z direction.

            OMGX - Angular velocity in local x direction.

            OMGY - Angular velocity in local y direction.

            OMGZ - Angular velocity in local z direction.

            ACCX - Linear acceleration in local x direction.

            ACCY - Linear acceleration in local y direction.

            ACCZ - Linear acceleration in local z direction.

            DMGX - Angular acceleration in local x direction.

            DMGY - Angular acceleration in local y direction.

            DMGZ - Angular acceleration in local z direction.

        value
            Value of the label.

        Notes
        -----
        This command is valid for MPC184 joint elements. See DJDELE for
        information on deleting boundary conditions applied with the DJ
        command.

        You can apply only one displacement, velocity, or acceleration load at
        any relative degree of freedom. If multiple loads are specified, the
        last applied load overrides the previous ones. For example, the
        following commands apply loads to element 100:

        In this case, the velocity load (VELX) applied in the last command will
        override the displacement load (UX).

        Tabular boundary conditions (VALUE = %tabname%) can be used.

        %_FIX% is an ANSYS reserved table name. When VALUE is set to %_FIX%,
        ANSYS will prescribe the degree of freedom to the “current” relative
        displacement value. This option is only valid for the following labels:
        UX, UY, UZ, ROTX, ROTY, ROTZ. In most cases, %_FIX% usage is efficient
        and recommended for all structural degrees of freedom.

        """
        command = "DJ,%s,%s,%s" % (str(elem), str(label), str(value))
        self.RunCommand(command, **kwargs)

    def Ugin(self, name="", extension="", path="", entity="", layer="", fmt="",
             **kwargs):
        """
        APDL Command: ~UGIN

        Transfers an NX part into the ANSYS program.

        Parameters
        ----------
        name
            The file name of the NX part to be imported, which cannot exceed 64
            characters in length. The path name must begin with an alphanumeric
            character. Special characters such as &, -,  and * are not
            permitted in the part name.

        extension
            The NX part file extension. The default is .prt.

        path
            The full path name to the directory containing the part, enclosed
            in single quotes; for example, '/ug_parts'. The default is the
            current working directory.

        entity
            Entity to be imported.

            0 or Solid - Solids only, imported as ANSYS volumes (the default).

            1 or Surface - Surfaces only, imported as ANSYS areas.

            2 or Wireframe - Wireframe only, imported as ANSYS lines.

            3 or All - All entities. Use this option when the part contains entities that may not be
                       attached to each other, such as a solid in one location
                       and a surface in another.

        layer
            The number(s) assigned to the layer(s) to be imported. You can
            import one layer or a range of layers (designated by hyphens).
            Defaults to 1-256 (all layers).

        fmt
            The format in which ANSYS will store the model.

            0 - Neutral format (default). Defeaturing after import is restricted.

            1 - Solid format; this allows defeaturing after import.

        Notes
        -----
        More information on importing NX parts is available in UG/NX in the
        Connection User's Guide.

        """
        command = "~UGIN,%s,%s,%s,%s,%s,%s" % (str(name), str(extension), str(path), str(entity), str(layer), str(fmt))
        self.RunCommand(command, **kwargs)

    def Cnvtol(self, lab="", value="", toler="", norm="", minref="", **kwargs):
        """
        APDL Command: CNVTOL

        Sets convergence values for nonlinear analyses.

        Parameters
        ----------
        lab
            Valid convergence labels. If STAT, list the status of the currently
            specified criteria.

        value
            Typical reference value for the specified convergence label (Lab).

        toler
            Tolerance; defaults to 0.005 (0.5%) for force and moment, 1.0E-4
            (0.01%) for DVOL, 0.05 (5%) for displacement when rotational DOFs
            are not present, and 0.05 (5%) for HDSP.

        norm
            Specifies norm selection:

            2 - L2 norm (check SRSS value). Default, except for Lab = U.

            1 - L1 norm (check absolute value sum).

            0 - Infinite norm (check each DOF separately). Default for Lab = U.

        minref
            The minimum value allowed for the program calculated reference
            value. If negative, no minimum is enforced. Used only if VALUE is
            blank. Defaults to 0.01 for force, moment, and volume convergence,
            1.0E-6 for heat flow, 1.0E-12 for VLTG and CHRG, 1.0E-6 for HDSP,
            and 0.0 otherwise.

        Notes
        -----
        This command is usually not needed because the default convergence
        criteria are sufficient for most nonlinear analyses. In rare cases, you
        may need to use this command to diagnose convergence difficulties.

        Values may be set for the degrees of freedom (DOF) and/or the out-of-
        balance load for the corresponding forcing quantities.

        Issuing CNVTOL to set a convergence criterion for a specific
        convergence label (Lab) does not affect the convergence criterion for
        any other label. All other convergence criteria will remain at their
        default setting or at the value set by a previous CNVTOL command.

         When the GUI is on, if a "Delete" operation in a Nonlinear Convergence
        Criteria dialog box writes this command to a log file (Jobname.LOG or
        Jobname.LGW), you will observe that Lab is blank, VALUE = -1, and TOLER
        is an integer number.  In this case, the GUI has assigned a value of
        TOLER that corresponds to the location of a chosen convergence label in
        the dialog box's list.  It is not intended that you type in such a
        location value for TOLER in an ANSYS session.  However, a file that
        contains a GUI-generated CNVTOL command of this form can be used for
        batch input or with the /INPUT command.

        Convergence norms specified with CNVTOL may be graphically tracked
        while the solution is in process using the ANSYS program's Graphical
        Solution Tracking (GST) feature.  Use the /GST command to turn GST on
        or off.  By default, GST is ON for interactive sessions and OFF for
        batch runs.

        This command is also valid in PREP7.

        """
        command = "CNVTOL,%s,%s,%s,%s,%s" % (str(lab), str(value), str(toler), str(norm), str(minref))
        self.RunCommand(command, **kwargs)

    def Cone(self, rbot="", rtop="", z1="", z2="", theta1="", theta2="",
             **kwargs):
        """
        APDL Command: CONE

        Creates a conical volume centered about the working plane origin.

        Parameters
        ----------
        rbot, rtop
            Radii of the bottom and top faces of the cone.  A value of zero or
            blank for either RBOT or RTOP defines a degenerate face at the
            center axis (i.e., the vertex of the cone).  The same value for
            both RBOT and RTOP defines a cylinder instead of a cone.

        z1, z2
            Working plane Z coordinates of the cone.  The smaller value is
            always associated with the bottom face.

        theta1, theta2
            Starting and ending angles (either order) of the cone.  Used for
            creating a conical sector.  The sector begins at the algebraically
            smaller angle, extends in a positive angular direction, and ends at
            the larger angle.  The  starting angle defaults to 0° and the
            ending angle defaults to 360°.  See the Modeling and Meshing Guide
            for an illustration.

        Notes
        -----
        Defines a solid conical volume centered about the working plane origin.
        The non-degenerate face (top or bottom) is parallel to the working
        plane but not necessarily coplanar with (i.e., "on") the working plane.
        The cone must have a spatial volume greater than zero. (i.e., this
        volume primitive command cannot be used to create a degenerate volume
        as a means of creating an area.)  For a cone of 360°, top and bottom
        faces will be circular (each area defined with four lines), and they
        will be connected with two areas (each spanning 180°).  See the CON4
        command for an alternate way to create cones.

        """
        command = "CONE,%s,%s,%s,%s,%s,%s" % (str(rbot), str(rtop), str(z1), str(z2), str(theta1), str(theta2))
        self.RunCommand(command, **kwargs)

    def Numstr(self, label="", value="", **kwargs):
        """
        APDL Command: NUMSTR

        Establishes starting numbers for automatically numbered items.

        Parameters
        ----------
        label
            Apply starting number to one of the following sets of items:

            NODE - Node numbers.  Value defaults (and is continually reset) to 1 + maximum node
                   number in model.  Cannot be reset lower.

            ELEM - Element numbers.  Value defaults (and is continually reset) to 1 + maximum
                   element number in model.  Cannot be reset lower.

            KP - Keypoint numbers.  Value defaults to 1.  Only undefined numbers are used.
                 Existing keypoints are not overwritten.

            LINE - Line numbers.  Value defaults to 1.  Only undefined numbers are used.  Existing
                   lines are not overwritten.

            AREA - Area numbers.  Value defaults to 1.  Only undefined numbers are used.  Existing
                   areas are not overwritten.

            VOLU - Volume numbers.  Value defaults to 1.  Only undefined numbers are used.
                   Existing volumes are not overwritten.

            DEFA - Default.  Returns all starting numbers to their default values.

        value
            Starting number value.

        Notes
        -----
        Establishes starting numbers for various items that may have numbers
        automatically assigned (such as element numbers with the EGEN command,
        and node and solid model entity numbers with the mesh [AMESH, VMESH,
        etc.] commands).  Use NUMSTR,STAT to display settings.  Use NUMSTR,DEFA
        to reset all specifications back to defaults.  Defaults may be lowered
        by deleting and compressing items (i.e., NDELE and NUMCMP,NODE for
        nodes, etc.).

        Note:: : A mesh clear operation (VCLEAR, ACLEAR, LCLEAR, and KCLEAR)
        automatically sets starting node and element numbers to the highest
        unused numbers.  If a specific starting node or element number is
        desired, issue NUMSTR after the clear operation.

        """
        command = "NUMSTR,%s,%s" % (str(label), str(value))
        self.RunCommand(command, **kwargs)

    def Blc5(self, xcenter="", ycenter="", width="", height="", depth="",
             **kwargs):
        """
        APDL Command: BLC5

        Creates a rectangular area or block volume by center and corner points.

        Parameters
        ----------
        xcenter, ycenter
            Working plane X and Y coordinates of the center of the rectangle or
            block face.

        width
            The total distance on or parallel to the working plane X-axis
            defining the width of the rectangle or block face.

        height
            The total distance on or parallel to the working plane Y-axis
            defining the height of the rectangle or block face.

        depth
            The perpendicular distance (either positive or negative based on
            the working plane Z direction) from the working plane representing
            the depth of the block.  If DEPTH = 0 (default), a rectangular area
            is created on the working plane.

        Notes
        -----
        Defines a rectangular area anywhere on the working plane or a
        hexahedral volume with one face anywhere on the working plane by
        specifying the center and corner points.  A rectangle will be defined
        with four keypoints and four lines.  A volume will be defined with
        eight keypoints, twelve lines, and six areas, with the top and bottom
        faces parallel to the working plane.  See the BLC4, RECTNG, and BLOCK
        commands for alternate ways to create rectangles and blocks.

        """
        command = "BLC5,%s,%s,%s,%s,%s" % (str(xcenter), str(ycenter), str(width), str(height), str(depth))
        self.RunCommand(command, **kwargs)

    def Plpath(self, lab1="", lab2="", lab3="", lab4="", lab5="", lab6="",
               **kwargs):
        """
        APDL Command: PLPATH

        Displays path items on a graph.

        Parameters
        ----------
        lab1, lab2, lab3, . . . , lab6
            Labels identifying the path items to be displayed.  Up to six items
            may be drawn per frame.  Predefined path geometry items XG, YG, ZG,
            and S [PDEF] may also be displayed.

        Notes
        -----
        The path must have been defined by the PATH and PPATH commands.  Path
        items and their labels must have been defined with the PDEF, PVECT,
        PCALC, PDOT, PCROSS, or PLNEAR commands.  Path items may also be
        printed with the PRPATH command.  Graph scaling may be controlled with
        the /XRANGE, /YRANGE, and PRANGE commands. You need to type all six
        characters to issue this command.

        """
        command = "PLPATH,%s,%s,%s,%s,%s,%s" % (str(lab1), str(lab2), str(lab3), str(lab4), str(lab5), str(lab6))
        self.RunCommand(command, **kwargs)

    def Kmesh(self, np1="", np2="", ninc="", **kwargs):
        """
        APDL Command: KMESH

        Generates nodes and point elements at keypoints.

        Parameters
        ----------
        np1, np2, ninc
            Mesh keypoints from NP1 to NP2 (defaults to NP1) in steps of NINC
            (defaults to 1).  If NP1 = ALL, NP2 and NINC are ignored and all
            selected keypoints [KSEL] are meshed.  If NP1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).  A component name may also be substituted
            for NP1.

        Notes
        -----
        Missing nodes required for the generated elements are created and
        assigned the lowest available numbers.

        """
        command = "KMESH,%s,%s,%s" % (str(np1), str(np2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Spmwrite(self, method="", nmode="", inputs="", inputlabels="",
                 outputs="", outputlabels="", nic="", velacckey="",
                 fileformat="", **kwargs):
        """
        APDL Command: SPMWRITE

        Calculates the state-space matrices and writes them to the SPM file.

        Parameters
        ----------
        method
            Reduction method for the calculation of the state-space matrices.

            MODAL - Method based on modal analysis results from LANB, LANPCG, SNODE, or SUBSP
                    eigensolver (default).

        nmode
            Number of modes to be used. Defaults to all modes.

        inputs
            Definition of the inputs. Defaults to all load vectors on the MODE
            file.

        inputlabels
            Definition of the input labels. Defaults to the load vector numbers
            or input definition (node and degree of freedom array parameter),
            depending on the Inputs specification.

        outputs
            Definition of the outputs. Defaults to the inputs.

        outputlabels
            Definition of the output labels. Defaults to the output definition
            (node and degree of freedom) if used, else defaults to the
            InputLabels.

        nic
            Load vector on the MODE file used for the calculation of the
            initial conditions. Defaults to no initial condition.

        velacckey
            Output velocities and accelerations key.

            OFF - Output displacements only (default).

            ON - Output displacements, velocities and accelerations.

        fileformat
            The format of the SPM file.

            0 - Dense format.

            1 - Matrix Market Exchange format (non-zero terms only).

            2 - Simplorer SML format without reference (default).

            3 - Simplorer SML format with common reference.

            4 - Simplorer SML format with independent references.

        Notes
        -----
        The SPMWRITE generates the file Jobname.SPM containing the state-space
        matrices and other information.

        The following applies to the SML formats (FileFormat = 2, 3, and 4):

        For conservative systems where the outputs are equal to the inputs
        (Outputs is left blank):

        The labels for the inputs (InputLabels) are required.

        The Inputs must use the array parameter option so that the input
        degrees of freedom (DOFs) are known.

        For non-conservative systems where the outputs are not equal to the
        inputs:

        The labels for the outputs (OutputLabels) are required.

        The file formats with references (FileFormat = 3 and 4) do not apply.

        Velocity and acceleration results are not included in the state-space
        matrices calculation (VelAccKey = OFF)

        File format with common reference (FileFormat = 3) does not apply if
        the inputs are based on DOFs of a different nature.  All input DOFs
        must be either all rotational or all translational and not a mix of the
        two.

        A graphics file (Jobname_SPM.PNG) is generated. It contains an element
        plot of the model.

        For more details about the reduction method and the generation of the
        state-space matrices, see Reduced-Order Modeling for State-Space
        Matrices Export in the Mechanical APDL Theory Reference.

        For examples of the command usage, see State-Space Matrices Export.

        """
        command = "SPMWRITE,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(method), str(nmode), str(inputs), str(inputlabels), str(outputs), str(outputlabels), str(nic), str(velacckey), str(fileformat))
        self.RunCommand(command, **kwargs)

    def Bfecum(self, lab="", oper="", fact="", tb_ase="", **kwargs):
        """
        APDL Command: BFECUM

        Specifies whether to ignore subsequent element body force loads.

        Parameters
        ----------
        lab
            Valid body load label.  If ALL, use all appropriate labels.

        oper
            Replace or ignore key:

            REPL - Subsequent values replace the previous values (default).

            IGNO - Subsequent values are ignored.

        fact
            Scale factor for the element body load values.  Zero (or blank)
            defaults to 1.0.  Use a small number for a zero scale factor.  The
            scale factor is not applied to body load phase angles.

        tbase
            Used (only with  Lab = TEMP) to calculate the temperature used in
            the add or replace operation (see Oper) as:

        Notes
        -----
        Allows repeated element body force loads to be replaced or ignored.
        Element body loads are applied with the BFE command.  Issue the BFELIST
        command to list the element body loads.  The operations occur when the
        next body loads are defined.  For example, issuing the BFE command with
        a temperature value of 25 after a previous BFE command with a
        temperature value of 20 causes the new value of that temperature to be
        25 with the replace operation, or 20 with the ignore operation.  A
        scale factor is also available to multiply the next value before the
        replace operation.  A scale factor of 2.0 with the previous "replace"
        example results in a temperature of 50.  The scale factor is applied
        even if no previous values exist.  Issue BFECUM,STAT to show the
        current label, operation, and scale factors.

        BFECUM does not work for tabular boundary conditions.

        This command is also valid in PREP7.

        """
        command = "BFECUM,%s,%s,%s,%s" % (str(lab), str(oper), str(fact), str(tb_ase))
        self.RunCommand(command, **kwargs)

    def Rsys(self, kcn="", **kwargs):
        """
        APDL Command: RSYS

        Activates a coordinate system for printout or display of element and
        nodal results.

        Parameters
        ----------
        kcn
            The coordinate system to use for results output:

            0 - Global Cartesian coordinate system (default, except for spectrum analyses).

            1 - Global cylindrical coordinate system.

            2 - Global spherical coordinate system.

            > 10 - Any existing local coordinate system.

            SOLU - Solution coordinate systems. For element quantities, these are the element
                   coordinate system for each element.  For nodal quantities,
                   these are the nodal coordinate systems. If an element or
                   nodal coordinate system is not defined, ANSYS uses the
                   global Cartesian coordinate system. If you issue a LAYER,N
                   command (where N refers to a layer number), the results
                   appear in the layer coordinate system. (SOLU is the default
                   for spectrum analyses.)

            LSYS - Layer coordinate system. For layered shell and solid elements, the results
                   appear in their respective layer coordinate systems.  For a
                   specific layer of interest, issue a LAYER,N command (where N
                   refers to a layer number). If a model has both nonlayered
                   and layered elements, you can use RSYS,SOLU and RSYS,LSYS
                   simultaneously (with RSYS,SOLU applicable to nonlayered
                   elements and RSYS,LSYS applicable to layered elements).  To
                   reverse effects of the LSYS option, issue an RSYS,0 command.
                   LSYS is the default for spectrum analysis.

        Notes
        -----
        The RSYS command activates a coordinate system for printing or
        displaying element results data such as stresses and heat fluxes, and
        nodal results data such as degrees of freedom and reactions. ANSYS
        rotates the results data to the specified coordinate system during
        printout, display, or element table operations (such as PRNSOL, PRESOL,
        PLNSOL, and ETABLE). You can define coordinate systems with various
        ANSYS commands such as LOCAL, CS, CLOCAL, and CSKP.

        If you issue RSYS with KCN > 10 (indicating a local coordinate system),
        and the specified system is subsequently redefined, you must reissue
        RSYS for results to be rotated into the redefined system.

        Note:: : The default coordinate system for certain elements, notably
        shells, is not global Cartesian and is frequently not aligned at
        adjacent elements.

        The use of RSYS,SOLU with these elements can make nodal averaging of
        component element results, such as SX, SY, SZ, SXY, SYZ, and SXZ,
        invalid and is not recommended.

        The RSYS command has no effect on beam or pipe stresses, which ANSYS
        displays (via /ESHAPE,1 and PowerGraphics) in the element coordinate
        system.

        Element results such as stresses and heat fluxes are in the element
        coordinate systems when KCN = SOLU. Nodal requests for element results
        (for example, PRNSOL,S,COMP) average the element values at the common
        node; that is, the orientation of the node is not a factor in the
        output of element quantities.  For nearly all solid elements, the
        default element coordinate systems are parallel to the global Cartesian
        coordinate system. For shell elements and the remaining solid elements,
        the default element coordinate system can differ from element to
        element. For layered shell and layered solid elements, ANSYS initially
        selects the element coordinate system when KCN = SOLU; you can then
        select the layer coordinate system via the LAYER command.

        Nodal results such as degrees of freedom and reactions can be properly
        rotated only if the resulting component set is consistent with the
        degree-of-freedom set at the node. (The degree-of-freedom set at a node
        is determined by the elements attached to the node.) For example, if a
        node does not have a UZ degree of freedom during solution, then any Z
        component resulting from a rotation does not print or display in POST1.
        Therefore, results at nodes with a single degree-of-freedom (UY only,
        for example) should not be rotated; that is, they should be viewed only
        in the nodal coordinate system or a system parallel to the nodal
        system. (The global Cartesian system--the RSYS command default--may not
        be parallel to the nodal system.) Results at nodes with a 2-D degree-
        of-freedom set (UX and UY, for example) should not be rotated out of
        the 2-D plane.

        PowerGraphics

        For PowerGraphics, ANSYS plots PLVECT vector arrow displays (such
        temperature, velocity, and force) in the global Cartesian coordinate
        system (RSYS = 0). Subsequent operations revert to your original
        coordinate system.

        PGR File

        When you generate a .PGR file in SOLUTION, you can use the Results
        Viewer to display your stresses only in the coordinate system in which
        you write your .PGR file. To view stresses in other coordinate systems,
        load your results file into the Results Viewer and regenerate the data.

        Large Deflections

        If large deflection is active (NLGEOM,ON), ANSYS rotates the element
        component result directions by the amount of rigid body rotation.

        ANSYS displays the element component results in the initial global
        coordinate system for the following elements: SHELL181, SHELL281,
        ELBOW290, PLANE182, PLANE183, SOLID185, SOLID186, SOLID187, SOLID272,
        SOLID273, SOLID285, SOLSH190, SHELL208, and SHELL209. All other element
        result transformations are, therefore, also relative to the initial
        global system. Nodal degree-of-freedom results are based on the initial
        (and not the updated) geometry. For all other element types, component
        results displayed in the co-rotated coordinate system include the
        element rigid body rotation from the initial global coordinate system,
        and all other element result transformations are relative to the
        rotated global system.

        LS-DYNA

        You can use the RSYS command to rotate stress data for all explicit
        (ANSYS LS-DYNA) elements except BEAM161, COMBI165, and composite
        SHELL163 (KEYOPT(3) = 1). In models that contain these element types
        combined with other explicit elements, you must unselect the
        unsupported elements before issuing the RSYS command. The command does
        not support strain data for any explicit element types. If you request
        strain results for explicit elements when RSYS is not set to the global
        Cartesian coordinate system (KCN = 0), ANSYS ignores the printing or
        plotting command. (ANSYS always rotates displacements into the results
        coordinate system, independent of the explicit element type.)

        """
        command = "RSYS,%s" % (str(kcn))
        self.RunCommand(command, **kwargs)

    def Sallow(self, strs1="", strs2="", strs3="", strs4="", strs5="",
               strs6="", **kwargs):
        """
        APDL Command: SALLOW

        Defines the allowable stress table for safety factor calculations.

        Parameters
        ----------
        strs1, strs2, strs3, . . . , strs6
            Input up to six allowable stresses corresponding to the temperature
            points [TALLOW].

        Notes
        -----
        Defines the allowable stress table for safety factor calculations
        [SFACT,SFCALC].  Use the STAT command to list current allowable stress
        table.  Repeat SALLOW to zero table and redefine points (6 maximum).

        Safety factor calculations are not supported by PowerGraphics. Both the
        SALLOW and TALLOW commands must be used with the Full Model Graphics
        display method active.

        """
        command = "SALLOW,%s,%s,%s,%s,%s,%s" % (str(strs1), str(strs2), str(strs3), str(strs4), str(strs5), str(strs6))
        self.RunCommand(command, **kwargs)

    def Autots(self, key="", **kwargs):
        """
        APDL Command: AUTOTS

        Specifies whether to use automatic time stepping or load stepping.

        Parameters
        ----------
        key
            Automatic time stepping key:

            OFF - Do not use automatic time stepping.

            ON - Use automatic time stepping (default).

            AUTO - The program determines whether to use automatic time stepping (used by
                   Workbench).

        Notes
        -----
        Specifies whether to use automatic time stepping (or load stepping)
        over this load step. If Key = ON, both time step prediction and time
        step bisection will be used.

        You cannot use automatic time stepping [AUTOTS], line search [LNSRCH],
        or the DOF solution predictor [PRED] with the arc-length method
        [ARCLEN, ARCTRM]. If you activate the arc-length method after you set
        AUTOTS, LNSRCH, or PRED, a warning message appears. If you choose to
        proceed with the arc-length method, the program disables your automatic
        time stepping, line search, and DOF predictor settings, and the time
        step size is controlled by the arc-length method internally.

        This command is also valid in PREP7.

        """
        command = "AUTOTS,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Tshap(self, shape="", **kwargs):
        """
        APDL Command: TSHAP

        Defines simple 2-D and 3-D geometric surfaces for target segment
        elements.

        Parameters
        ----------
        shape
            Specifies the geometric shapes for target segment elements TARGE169
            and TARGE170.

            LINE - Straight line (2-D, 3-D) (Default for 2-D)

            PARA - Parabola (2-D, 3-D)

            ARC - Clockwise arc (2-D)

            CARC - Counterclockwise arc (2-D)

            CIRC - Complete circle (2-D)

            TRIA - Three-node triangle (3-D) (Default for 3-D)

            TRI6 - Six-node triangle (3-D)

            QUAD - Four-node quadrilateral (3-D)

            QUA8 - Eight-node quadrilateral (3-D)

            CYLI - Cylinder (3-D)

            CONE - Cone (3-D)

            SPHE - Sphere (3-D)

            PILO - Pilot node (2-D, 3-D)

            POINT - Point (rigid surface node) (2-D, 3-D)

        Notes
        -----
        Use this command to specify the target segment shapes for the rigid
        target surface associated with surface-to-surface contact (TARGE169,
        CONTA171, CONTA172 (2-D) and TARGE170, CONTA173, CONTA174 (3-D)), 3-D
        beam-to-beam contact (TARGE170 and CONTA176), and 3-D line-to-surface
        contact (TARGE170 and CONTA177).  Once you issue TSHAP, all subsequent
        target elements generated via the direct element generation technique
        will have the same shape, until you issue TSHAP again with a different
        Shape value.

        """
        command = "TSHAP,%s" % (str(shape))
        self.RunCommand(command, **kwargs)

    def Prsect(self, rho="", kbr="", **kwargs):
        """
        APDL Command: PRSECT

        Calculates and prints linearized stresses along a section path.

        Parameters
        ----------
        rho
            In-plane (X-Y) average radius of curvature of the inside and
            outside surfaces of an axisymmetric section.  If zero (or blank), a
            plane or 3-D structure is assumed.  If nonzero, an axisymmetric
            structure is assumed.  Use any large number (or -1) for an
            axisymmetric straight section.

        kbr
            Through-thickness bending stresses key for an axisymmetric analysis
            (RHO  ≠ 0):

            0 - Include the thickness-direction bending stresses.

            1 - Ignore the thickness-direction bending stresses.

            2 - Include the thickness-direction bending stress using the same formula as the Y
                (axial direction ) bending stress. Also use the same formula
                for the shear stress.

        Notes
        -----
        You may choose to linearize the stresses through a section and separate
        them into categories for various code calculations.  PRSECT calculates
        and reports linearized stresses along a section path.  The linearized
        stresses are also separated into membrane, bending, membrane plus
        bending, peak, and total stress categories.

        First, define your section path using the PATH and PPATH (with the NODE
        option) commands.  Your path must lie entirely within the selected set
        of elements (that is, there must be no element gaps along the path).
        PATH and PPATH are used only to retrieve the two end nodes.  The path
        data is not retained.  The section path is defined by the two end
        nodes, and by 47 intermediate points that are automatically determined
        by linear interpolation in the active display coordinate system [DSYS].
        The number and location of the intermediate points are not affected by
        the number of divisions set by PATH,,,,nDiv.

        Your  linearized component stress values are obtained by interpolating
        each element’s  average corner nodal values along the section path
        points within each path element.  PRSECT reports the linearized
        component and principal stresses for each stress category at the
        beginning, mid-length, and end of the section path.  PRPATH can be used
        to report the total stresses at the intermediate points.

        Section paths may be through any set of solid (2-D plane, 2-D
        axisymmetric or 3-D) elements.  However, section paths are usually
        defined to be through the thickness of the structure and normal to the
        inner and outer structure surfaces.  Section paths (in-plane only) may
        also be defined for shell element structures.  See the Mechanical APDL
        Theory Reference for details.

        If the RHO option is set to indicate the axisymmetric option (non-
        zero), PRSECT reports the linearized stresses in the section
        coordinates (SX – along the path, SY – normal to the path, and SZ –
        hoop direction).  If the RHO option is set to indicate the 2-D planar
        or 3-D option (zero or blank), PRSECT reports the linearized stresses
        in the active results coordinate system [RSYS].  If the RHO option is
        zero or blank and either RSYS, SOLU or RSYS, -1 are active, the
        linearized stresses are calculated and reported in the global Cartesian
        coordinate system.  It is recommended that linearized stress
        calculations be performed in a rectangular coordinate system.
        Principal stresses are recalculated from the component stresses and are
        invariant with the coordinate system as long as SX is in the same
        direction at all points along the defined path.  The PLSECT command
        displays the linearized stresses in the same coordinate system as
        reported by PRSECT.

        Stress components through the section are linearized by a line integral
        method and separated into constant membrane stresses, bending stresses
        varying linearly between end points, and peak stresses (defined as the
        difference between the actual (total) stress and the membrane plus
        bending combination).

        For nonaxisymmetric structures,  the bending stresses are calculated
        such that the neutral axis is at the midpoint of the path.
        Axisymmetric results include the effects of both the radius of
        revolution (automatically determined from the node locations) and the
        in-plane average radius of curvature of the section surfaces (user
        input).

        For axisymmetric cases, Mechanical APDL calculates the linearized
        bending stress in the through-thickness direction as the difference
        between the total outer fiber stress and the membrane stress if KBR =
        1. The calculation method may be conservative for locations with a
        highly nonlinear variation of stress in the through-thickness
        direction.  Alternatively, you can specify KBR = 2 to calculate the
        bending stress using the same method and formula as the Y (axial
        direction) bending stress. For more information, see the discussion of
        axisymmetric cases (specifically Equation: 17–40) in the Mechanical
        APDL Theory Reference.

        Portions of this command are not supported by PowerGraphics
        [/GRAPHICS,POWER].

        """
        command = "PRSECT,%s,%s" % (str(rho), str(kbr))
        self.RunCommand(command, **kwargs)

    def Cpngen(self, nset="", lab="", node1="", node2="", ninc="", **kwargs):
        """
        APDL Command: CPNGEN

        Defines, modifies, or adds to a set of coupled degrees of freedom.

        Parameters
        ----------
        nset
            Set reference number [CP].

        lab
            Degree of freedom label [CP].

        node1, node2, ninc
            Include in coupled set nodes NODE1 to NODE2 in steps of NINC
            (defaults to 1).  If NODE1 = P, graphical picking is enabled and
            all remaining command fields are ignored (valid only in the GUI).
            If -NODE1, delete range of nodes from set instead of including.  A
            component name may also be substituted for NODE1 (NODE2 and NINC
            are ignored).

        Notes
        -----
        Defines, modifies, or adds to a set of coupled degrees of freedom.  May
        be used in combination with (or in place of) the CP command.  Repeat
        CPNGEN command for additional nodes.

        """
        command = "CPNGEN,%s,%s,%s,%s,%s" % (str(nset), str(lab), str(node1), str(node2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Sstate(self, action="", cm_name="", val1="", val2="", val3="", val4="",
               val5="", val6="", val7="", val8="", val9="", **kwargs):
        """
        APDL Command: SSTATE

        Defines a steady-state analysis.

        Parameters
        ----------
        action
            Action to perform for defining or manipulating steady-state
            analysis data:

            DEFINE - Define steady-state analysis data

            LIST - List current steady-state analysis data

            DELETE - Delete steady-state analysis data

        cm_name
            Element component name

        val1, ..., val9
            Input values (based on the Action type)

        Notes
        -----
        The SSTATE command specifies steady-state analysis parameters for the
        given element component. The program runs the steady-state analysis if
        the corresponding element key option is enabled for that element
        component.

        The command supports the following elements:

        SOLID185

        3-D 8-Node Structural Solid

        SOLID186

        3-D 20-Node Structural Solid

        SOLID187

        3-D 10-Node Tetrahedral Structural Solid

        SOLSH190

        3-D 8-Node Structural Solid Shell

        Degenerated shape (prism) option not supported.

        SOLID285

        3-D 4-Node Tetrahedral Structural Solid with Nodal Pressures

        For information about steady-state rolling for rebar and solid
        elements, see Steady State Rolling in the Mechanical APDL Theory
        Reference.

        The following data types can be defined:

        SPIN -- Steady-state spinning motion

        TRANSLATE -- Rigid body motion (velocity) that the spinning component
        is undergoing

        Define the steady-state spinning motion:

        SSTATE, DEFINE, CM_Name, SPIN, OMEGA, Method, Val4, Val5, Val6, Val7,
        Val8, Val9

        Spin velocity

        Method to use for defining the spin axis:

        Define the spin axis using two points:

        Val4, Val5, Val6 -- Coordinates of the first point

        Val7, Val8, Val9 -- Coordinates of the second point

        This definition method is currently the only option.

        This command defines a steady state spinning motion of 120 rad/s around
        the spin axis:

        In this case, two points with coordinates (0,0,0) and (0,1,0) define
        the spin axis in the global Y direction.

        Define the rigid body motion (velocity):

        SSTATE, DEFINE, CM_Name, TRANSLATE, Val2, Val3, Val4

        SSTATE, LIST, CM_Name

        Lists all steady-state analysis data defined on the specified element
        component. All data is listed if no component (CM_Name) is specified.

        SSTATE, DELETE, CM_Name

        Deletes all steady-state analysis data defined on the specified element
        component. All data is deleted if no component (CM_Name) is specified.

        """
        command = "SSTATE,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(action), str(cm_name), str(val1), str(val2), str(val3), str(val4), str(val5), str(val6), str(val7), str(val8), str(val9))
        self.RunCommand(command, **kwargs)

    def Paget(self, parray="", popt="", **kwargs):
        """
        APDL Command: PAGET

        Writes current path information into an array variable.

        Parameters
        ----------
        parray
            The name of the array parameter that the ANSYS program creates to
            store the path information.  If the array parameter already exists,
            it will be replaced with the current path information.

        popt
            Determines how data will be stored in the parameter specified with
            PARRAY:

            POINTS - Store the path points, the nodes (if any), and coordinate system.  (For
                     information on defining paths and path points, see the
                     descriptions of the PATH and PPATH commands.)

            TABLE - Store the path data items.  (See the PDEF command description for path data
                    items.)

            LABEL - Stores path data labels.

        Notes
        -----
        Use the PAGET command together with the PAPUT command to store and
        retrieve path data in array variables for archiving purposes.  When
        retrieving path information, restore the path points (POINTS option)
        first, then the path data (TABLE option), and then the path labels
        (LABEL option).

        """
        command = "PAGET,%s,%s" % (str(parray), str(popt))
        self.RunCommand(command, **kwargs)

    def Alist(self, na1="", na2="", ninc="", lab="", **kwargs):
        """
        APDL Command: ALIST

        Lists the defined areas.

        Parameters
        ----------
        na1, na2, ninc
            List areas from NA1 to NA2 (defaults to NA1) in steps of NINC
            (defaults to 1).  If NA1 = ALL (default), NA2 and NINC are ignored
            and all selected areas [ASEL] are listed.  If NA1 = P, graphical
            picking is enabled and all remaining arguments are ignored (valid
            only in the GUI).  A component name may also be substituted for NA1
            (NA2 and NINC are ignored).

        lab
            Determines what type of listing is used (one of the following):

            (blank) - Prints information about all areas in the specified range.

            HPT - Prints information about only those areas that contain hard points.

        Notes
        -----
        An attribute (TYPE, MAT, REAL, or ESYS) listed as a zero is unassigned;
        one listed as a positive value indicates that the attribute was
        assigned with the AATT command (and will not be reset to zero if the
        mesh is cleared); one listed as a negative value indicates that the
        attribute was assigned using the attribute pointer [TYPE, MAT, REAL, or
        ESYS] that was active during meshing (and will be reset to zero if the
        mesh is cleared).  A "-1" in the "nodes" column indicates that the area
        has been meshed but there are no interior nodes.  The area size is
        listed only if an ASUM command has been performed on the area.

        """
        command = "ALIST,%s,%s,%s,%s" % (str(na1), str(na2), str(ninc), str(lab))
        self.RunCommand(command, **kwargs)

    def Nummrg(self, label="", toler="", gtoler="", action="", switch="",
               **kwargs):
        """
        APDL Command: NUMMRG

        Merges coincident or equivalently defined items.

        Parameters
        ----------
        label
            Items to be merged:

            NODE - Nodes

            ELEM - Elements

            KP - Keypoints (will also merge lines, areas, and volumes)

            MAT - Materials

            TYPE - Element types

            REAL - Real constants

            CP - Coupled sets

            CE - Constraint equations

            ALL - All items

        toler
            Range of coincidence.  For Label = NODE and KP, defaults to 1.0E-4
            (based on maximum Cartesian coordinate difference between nodes or
            keypoints).  For Label = MAT, REAL, and CE, defaults to 1.0E-7
            (based on difference of the values normalized by the values).  Only
            items within range are merged.  (For keypoints attached to lines,
            further restrictions apply.  See the GTOLER field and Merging Solid
            Model Entities below.)

        gtoler
            Global solid model tolerance -- used only when merging keypoints
            attached to lines.  If specified, GTOLER will override the internal
            relative solid model tolerance.  See Merging Solid Model Entities
            below.

        action
            Specifies whether to merge or select coincident items.

            SELE - Select coincident items but do not merge. Action = SELE is only valid for Label
                   = NODE.

            (Blank) - Merge the coincident items (default).

        switch
            Specifies whether the lowest or highest numbered coincident item is
            retained after the merging operation.  This option does not apply
            to keypoints; i.e., for Label = KP, the lowest numbered keypoint is
            retained regardless of the Switch setting.

            LOW - Retain the lowest numbered coincident item after the merging operation
                  (default).

            HIGH - Retain the highest numbered coincident item after the merging operation.

        Notes
        -----
        After issuing the command, the area and volume sizes (ASUM and VSUM)
        may give slightly different results. In order to obtain the same
        results as before, use /FACET, /NORMAL, and ASUM / VSUM.

        The merge operation is useful for tying separate, but coincident, parts
        of a model together. If not all items are to be checked for merging,
        use the select commands (NSEL, ESEL, etc.) to select items.  Only
        selected items are included in the merge operation for nodes,
        keypoints, and elements.

        By default, the merge operation retains the lowest numbered coincident
        item.  Higher numbered coincident items are deleted.  Set Switch to
        HIGH to retain the highest numbered coincident item after the merging
        operation.  Applicable related items are also checked for deleted item
        numbers and if found, are replaced with the retained item number.  For
        example, if nodes are merged, element connectivities (except
        superelements), mesh item range associativity, coupled degrees of
        freedom, constraint equations, master degrees of freedom, gap
        conditions, degree of freedom constraints, nodal force loads, nodal
        surface loads, and nodal body force loads are checked.  Merging
        material numbers [NUMMRG,ALL or NUMMRG,MAT] does not update the
        material number referenced:

        By temperature-dependent film coefficients as part of convection load
        or a temperature-dependent emissivity as part of a surface-to-surface
        radiation load [SF, SFE, SFL, SFA]

        By real constants for multi-material elements (such as SOLID65)

        If a unique load is defined among merged nodes, the value is kept and
        applied to the retained node.  If loads are not unique (not
        recommended), only the value on the lowest node (or highest if Switch =
        HIGH) will be kept, except for "force" loads for which the values will
        be summed if they are not defined using tabular boundary conditions.

        Note:: : The unused nodes (not recommended) in elements, couplings,
        constraint equations, etc. may become active after the merge operation.

        The Action field provides the option of visualizing the coincident
        items before the merging operation.

        Caution:: : When merging entities in a model that has already been
        meshed, the order in which you issue multiple NUMMRG commands is
        significant.  If you want to merge two adjacent meshed regions that
        have coincident nodes and keypoints, always merge nodes [NUMMRG,NODE]
        before merging keypoints [NUMMRG,KP].  Merging keypoints before nodes
        can result in some of the nodes becoming "orphaned"; that is, the nodes
        lose their association with the solid model.  Orphaned nodes can cause
        certain operations (such as boundary condition transfers, surface load
        transfers, and so on) to fail. However, using NUMMRG should be avoided
        if at all possible, as the procedure outlined above may even cause
        meshing failure, especially after multiple merging and meshing
        operations.

        After a NUMMRG,NODE, is issued, some nodes may be attached to more than
        one solid entity. As a result, subsequent attempts to transfer solid
        model loads to the elements may not be successful. Issue NUMMRG,KP to
        correct this problem. Do NOT issue VCLEAR before issuing NUMMRG,KP.

        For NUMMRG,ELEM, elements must be identical in all aspects, including
        the direction of the element coordinate system.

        For certain solid and shell elements (181, 185, 190, etc) ANSYS will
        interpret coincident faces as internal and eliminate them. To prevent
        this from occurring, shrink the entities by a very small factor to
        delineate coincident items (/SHRINK, 0.0001) and no internal nodes,
        lines, areas or elements will be eliminated.

        When working with solid models, you may have better success with the
        gluing operations (AGLUE, LGLUE, VGLUE). Please read the following
        information when attempting to merge solid model entities.

        Gluing Operations vs. Merging Operations

        Adjacent, touching regions can be joined by gluing them (AGLUE, LGLUE,
        VGLUE) or by merging coincident keypoints (NUMMRG,KP, which also causes
        merging of identical lines, areas, and volumes).  In many situations,
        either approach will work just fine. Some factors, however, may lead to
        a preference for one method over the other.

        Geometric Configuration

        Gluing is possible regardless of the initial alignment or offset of the
        input entities. Keypoint merging is  possible only if each keypoint on
        one side of the face to be joined is matched by a coincident keypoint
        on the other side. This is commonly the case after a symmetry
        reflection (ARSYM or VSYMM) or a copy (AGEN or VGEN),  especially for a
        model built entirely in ANSYS rather than imported from a CAD system.
        When the geometry is  extremely precise, and the configuration is
        correct for  keypoint merging, NUMMRG is more efficient and robust than
        AGLUE or VGLUE.

        Model Accuracy

        Gluing, like all ANSYS boolean operations, requires that the input
        entities meet the current boolean tolerance  (BTOL). Otherwise, AGLUE
        or VGLUE may fail. In such cases,  relaxing the tolerance may allow the
        glue to complete. An advantage of gluing is that it is unlikely to
        degrade the accuracy  of a geometric model. Keypoint merging can
        operate on almost any combination of entities (although you may have to
        override the default tolerances on NUMMRG).  However, it can also
        introduce or increase accuracy flaws, making later boolean operations
        less likely to succeed. If the input tolerances are too large, NUMMRG
        can collapse out small lines, areas, or volumes you intended to keep,
        possibly rendering the model unusable.

        Mesh Status

        Gluing, like all ANSYS boolean operations, requires that the input
        entities be unmeshed. Keypoint merging is effective for meshed models
        under the right conditions. More information on keypoint merging
        follows.

        Merging Solid Model Entities:

        When merging solid model entities (Label = KP or ALL), keypoint
        locations are used as the basis for merging.  Once keypoints are
        merged, any higher order solid model entities (lines, areas, and
        volumes), regardless of their select status or attachment to the merged
        keypoints, are considered for merging.

        Keypoints that are attached to lines will be merged only if:

        ΔX, : ΔY, and: ΔZ are each less than TOLER

        where,

        and

         is less than 1E-5 times the length of the longest line attached to
        those keypoints (internal relative solid model tolerance), or  is less
        than GTOLER (global solid model tolerance) if specified.

        The TOLER field is a consideration tolerance.  If a keypoint is within
        TOLER of another keypoint, then those two keypoints are candidates to
        be merged.  If, when "moving" the higher numbered keypoint, the
        distance exceeds the internal relative solid model tolerance, or the
        global solid model tolerance (GTOLER) if specified, the keypoints will
        not be merged.  Lines, areas, and volumes are considered for merging in
        a similar manner.

        The internal relative solid model tolerance should be overridden by the
        global solid model tolerance (GTOLER) only when absolutely necessary.
        GTOLER is an absolute tolerance; if specified, relative lengths of
        lines in the model will no longer be considered in the merge operation.
        If GTOLER is too large, you can "merge-out" portions of your model
        accidently, effectively defeaturing the model. If GTOLER is used, it is
        recommended that NUMMRG be preceded by saving the database (since
        undesired merges of solid model entities could occur).

        Note:: : Use of the NUMMRG command does not cause changes to a model's
        geometry; only the topology is affected.

        """
        command = "NUMMRG,%s,%s,%s,%s,%s" % (str(label), str(toler), str(gtoler), str(action), str(switch))
        self.RunCommand(command, **kwargs)

    def M(self, node="", lab1="", nend="", ninc="", lab2="", lab3="", lab4="",
          lab5="", lab6="", **kwargs):
        """
        APDL Command: M

        Defines master degrees of freedom for superelement generation analyses.

        Parameters
        ----------
        node
            Node number at which master degree of freedom is defined.  If ALL,
            define master degrees of freedom at all selected nodes (NSEL).  If
            NODE = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).  A component name may
            also be substituted for NODE.

        lab1
            Valid degree of freedom label. If ALL, use all appropriate labels.
            Structural labels:  UX, UY, or UZ (displacements); ROTX, ROTY, or
            ROTZ (rotations). Thermal labels: TEMP, TBOT, TE2, TE3, . . ., TTOP
            (temperature).  Electric labels:  VOLT (voltage).

        nend, ninc
            Define all nodes from NODE to NEND (defaults to NODE) in steps of
            NINC (defaults to 1) as master degrees of freedom in the specified
            direction.

        lab2, lab3, lab4, . . . , lab6
            Additional master degree of freedom labels.  The nodes defined are
            associated with each label specified.

        Notes
        -----
        Defines master degrees of freedom (MDOF) for superelement generation.
        If defined for other analyses, MDOF are ignored.  If used in SOLUTION,
        this command is valid only within the first load step.

        Repeat M command for additional master degrees of freedom.  The limit
        for the number of master nodes used is determined by the maximum system
        memory available.

        The substructure (ANTYPE,SUBSTR) analysis utilizes the matrix
        condensation technique to reduce the structure matrices to those
        characterized by a set of master degrees of freedom.

        Master degrees of freedom are identified by a list of nodes and their
        nodal directions.  The actual degree of freedom directions available
        for a particular node depends upon the degrees of freedom associated
        with element types (ET) at that node.  There must be some mass (or
        stress stiffening in the case of the buckling analysis) associated with
        each master degree of freedom (except for the VOLT label).  The mass
        may be due either to the distributed mass of the element or due to
        discrete lumped masses at the node.  If a master degree of freedom is
        specified at a constrained point, it is ignored.  If a master degree of
        freedom is specified at a coupled node, it should be specified at the
        prime node of the coupled set.

        Substructure analysis connection points must be defined as master
        degrees of freedom.

        This command is also valid in PREP7.

        """
        command = "M,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(node), str(lab1), str(nend), str(ninc), str(lab2), str(lab3), str(lab4), str(lab5), str(lab6))
        self.RunCommand(command, **kwargs)

    def Cgloc(self, xloc="", yloc="", zloc="", **kwargs):
        """
        APDL Command: CGLOC

        Specifies the origin location of the acceleration coordinate system.

        Parameters
        ----------
        xloc, yloc, zloc
            Global Cartesian X, Y, and Z coordinates of the acceleration
            coordinate system origin.

        Notes
        -----
        Specifies the origin location of the acceleration coordinate system
        with respect to the global Cartesian system.  The axes of this
        acceleration coordinate system are parallel to the global Cartesian
        axes.

        A structure may be rotating about the global Cartesian origin [OMEGA,
        DOMEGA], which may in turn be rotating about another point (the origin
        of the acceleration coordinate system), introducing Coriolis effects.
        The location of this point (relative to the global Cartesian origin) is
        specified with this CGLOC command.  For example, if Y is vertical and
        the global system origin is at the surface of the earth while the
        acceleration system origin is at the center of the earth, YLOC should
        be -4000 miles (or equivalent) if the rotational effects of the earth
        are to be included.  The rotational velocity of the global Cartesian
        system about this point is specified with the CGOMGA command, and the
        rotational acceleration is specified with the DCGOMG command.

        The rotational velocities and accelerations are mainly intended to
        include mass effects in a static (ANTYPE,STATIC) analysis.  If used in
        dynamic analyses, no coupling exists between the user input terms and
        the time history response of the structure.  See Acceleration Effect in
        the Mechanical APDL Theory Reference for details.  Related commands are
        ACEL, CGOMGA, DCGOMG, DOMEGA, and OMEGA.

        See Analysis Tools in the Mechanical APDL Theory Reference for more
        information.

        This command is also valid in PREP7.

        """
        command = "CGLOC,%s,%s,%s" % (str(xloc), str(yloc), str(zloc))
        self.RunCommand(command, **kwargs)

    def Sf(self, nlist="", lab="", value="", value2="", **kwargs):
        """
        APDL Command: SF

        Specifies surface loads on nodes.

        Parameters
        ----------
        nlist
            Nodes defining the surface upon which the load is to be applied.
            Use the label ALL or P, or a component name.  If ALL, all selected
            nodes [NSEL] are used (default).  If P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).

        lab
            Valid surface load label. Load labels are listed under "Surface
            Loads" in the input table for each element type in the Element
            Reference.

        value
            Surface load value or table name reference for specifying tabular
            boundary conditions.

        value2
            Second surface load value (if any).

        Notes
        -----
        Individual nodes may not be entered for this command.  The node list is
        to identify a surface and the Nlist field must contain a sufficient
        number of nodes to define an element surface.  The loads are internally
        stored on element faces defined by the specified nodes.  All nodes on
        an element face (including midside nodes, if any) must be specified for
        the face to be used, and the element must be selected.

        If all nodes defining a face are shared by an adjacent face of another
        selected element, the face is not free and will not have a load
        applied.  If more than one element can share the same nodes (for
        example, a surface element attached to a solid element), select the
        desired element type before issuing the SF command. The SF command
        applies only to area and volume elements.

        For shell elements, if the specified nodes include face one (which is
        usually the bottom face) along with other faces (such as edges), only
        face one is used.  Where faces cannot be uniquely determined from the
        nodes, or where the face does not fully describe the load application,
        use the SFE command.  A load key of 1 (which is typically the first
        loading condition on the first face) is used if the face determination
        is not unique.  A uniform load value is applied over the element face.

        See the SFBEAM command for applying surface loads to beam elements.
        See the SFGRAD command for an alternate tapered load capability.  See
        the SFFUN command for applying loads from a node vs. value function.
        Also see the SFE command for applying tapered loads on individual
        element faces.  Use the SFDELE command to delete loads applied with
        this command.  Use the SFCUM command to accumulate (add) surface loads
        applied with SF.

        Tabular boundary conditions (VALUE = %tabname% and/or VALUE2 =
        %tabname%) are available for the following surface load labels (Lab)
        only: : PRES (real and/or imaginary components), CONV (film coefficient
        and/or bulk temperature) or HFLUX, DFLUX (diffusion flux), RAD (surface
        emissivity and ambient temperature), IMPD (resistance and reactance),
        SHLD (normal velocity and phase or acceleration), and ATTN (attenuation
        coefficient). Use the *DIM command to define a table.

        This command is also valid in PREP7 and in the /MAP processor.

        """
        command = "SF,%s,%s,%s,%s" % (str(nlist), str(lab), str(value), str(value2))
        self.RunCommand(command, **kwargs)

    def Edcsc(self, key="", **kwargs):
        """
        APDL Command: EDCSC

        Specifies whether to use subcycling in an explicit dynamics analysis.

        Parameters
        ----------
        key
            Subcycling key:

            OFF - Do not use subcycling (default).

            ON - Use subcycling.

        Notes
        -----
        Subcycling can be used to speed up an analysis when element sizes
        within a model vary significantly. Relatively small elements will
        result in a small time step size. When subcycling is on, the minimum
        time step size is increased for the smallest elements.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDCSC,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Plgeom(self, item="", nodekey="", **kwargs):
        """
        APDL Command: PLGEOM

        Plots target and source geometries.

        Parameters
        ----------
        item
            Items to plot:

            BOTH - Plot both target and source geometries (default).

            TARGET - Plot only the target geometry.

            SOURCE - Plot only the source geometry.

        nodekey
            If the source data contains faces (that is, surface elements were
            created upon the READ command), set NODEkey = 1 to plot only the
            source nodes rather than both the nodes and the elements.

        Notes
        -----
        Target faces are displayed in gray and source points in yellow. If the
        source data contains faces (that is, surface elements were created upon
        the READ command), the source faces are also displayed in blue (unless
        NODEkey = 1), and both surfaces are made translucent.

        """
        command = "PLGEOM,%s,%s" % (str(item), str(nodekey))
        self.RunCommand(command, **kwargs)

    def Fill(self, node1="", node2="", nfill="", nstrt="", ninc="", itime="",
             inc="", space="", **kwargs):
        """
        APDL Command: FILL

        Generates a line of nodes between two existing nodes.

        Parameters
        ----------
        node1, node2
            Beginning and ending nodes for fill-in.  NODE1 defaults to next to
            last node specified, NODE2 defaults to last node specified.  If
            NODE1 = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).

        nfill
            Fill NFILL nodes between NODE1 and NODE2 (defaults to
            |NODE2-NODE1|-1).  NFILL must be positive.

        nstrt
            Node number assigned to first filled-in node (defaults to NODE1 +
            NINC).

        ninc
            Add this increment to each of the remaining filled-in node numbers
            (may be positive or negative).  Defaults to the integer result of
            (NODE2-NODE1)/(NFILL + 1), i.e., linear interpolation.  If the
            default evaluates to zero, or if zero is input, NINC is set to 1.

        itime, inc
            Do fill-in operation a total of ITIMEs, incrementing NODE1, NODE2
            and NSTRT by INC each time after the first.  ITIME and INC both
            default to 1.

        space
            Spacing ratio.  Ratio of last division size to first division size.
            If > 1.0, divisions increase.  If < 1.0, divisions decrease.  Ratio
            defaults to 1.0 (uniform spacing).

        Notes
        -----
        Generates a line of nodes (in the active coordinate system) between two
        existing nodes.  The two nodes may have been defined in any coordinate
        system.  Nodal locations and rotation angles are determined by
        interpolation.  Any number of nodes may be filled-in and any node
        number sequence may be assigned.  See the CSCIR command when filling
        across the 180° singularity line in a non-Cartesian system.

        """
        command = "FILL,%s,%s,%s,%s,%s,%s,%s,%s" % (str(node1), str(node2), str(nfill), str(nstrt), str(ninc), str(itime), str(inc), str(space))
        self.RunCommand(command, **kwargs)

    def Psdval(self, tblno="", sv1="", sv2="", sv3="", sv4="", sv5="", sv6="",
               sv7="", **kwargs):
        """
        APDL Command: PSDVAL

        Defines PSD values.

        Parameters
        ----------
        tblno
            Input table number being defined.

        sv1, sv2, sv3, . . . , sv7
            Spectral values corresponding to the frequency points [PSDFRQ].
            Values are interpreted as defined with the PSDUNIT command.

        Notes
        -----
        Defines PSD values to be associated with the previously defined
        frequency points.

        Repeat PSDVAL command for additional values, up to the number of
        frequency points [PSDFRQ].  Values are added after the last nonzero
        value.

        This command is also valid in PREP7.

        """
        command = "PSDVAL,%s,%s,%s,%s,%s,%s,%s,%s" % (str(tblno), str(sv1), str(sv2), str(sv3), str(sv4), str(sv5), str(sv6), str(sv7))
        self.RunCommand(command, **kwargs)

    def Sfgrad(self, lab="", slkcn="", sldir="", slzer="", slope="", **kwargs):
        """
        APDL Command: SFGRAD

        Specifies a gradient (slope) for surface loads.

        Parameters
        ----------
        lab
            Valid surface load label.  Load labels are listed under "Surface
            Loads" in the input table for each element type in the Element
            Reference.

        slkcn
            Reference number of slope coordinate system (used with Sldir and
            SLZER to determine COORD).  Defaults to 0 (the global Cartesian
            coordinate system).

        sldir
            Slope direction in coordinate system SLKCN:

            X - Slope is along X direction (default).  Interpreted as R direction for non-
                Cartesian coordinate systems.

            Y - Slope is along Y direction.  Interpreted as  θ direction for non-Cartesian
                coordinate systems.

            Z - Slope is along Z direction.  Interpreted as Φ direction for spherical or
                toroidal coordinate systems.

        slzer
            Coordinate location (degrees for angular input) where slope
            contribution is zero (CVALUE = VALUE).  Allows the slope
            contribution to be shifted along the slope direction.  For angular
            input, SLZER should be between ±180° if the singularity [CSCIR]  is
            at 180° and should be between 0° and 360° if the singularity is at
            0°.

        slope
            Slope value (load per unit length or per degree).

        Notes
        -----
        Specifies a gradient (slope) for surface loads.  All surface loads
        issued with the SF, SFE, SFL, or SFA commands while this specification
        is active will have this gradient applied (for complex pressures, only
        the real component will be affected; for convections, only the bulk
        temperature will be affected).  The load value, CVALUE, calculated at
        each node is:

        CVALUE = VALUE + (SLOPE X (COORD-SLZER))

        where VALUE is the load value specified on the subsequent SF, SFE, SFL,
        or SFA commands and COORD is the coordinate value (in the Sldir
        direction of coordinate system SLKCN) of the node.  Only one SFGRAD
        specification may be active at a time (repeated use of this command
        replaces the previous specification with the new specification).  Issue
        SFGRAD (with blank fields) to remove the specification.  Issue
        SFGRAD,STAT to show the current command status.  The SFGRAD
        specification (if active) is removed when the LSREAD (if any) command
        is issued.

        SFGRAD does not work for tabular boundary conditions.

        This command is also valid in PREP7.

        """
        command = "SFGRAD,%s,%s,%s,%s,%s" % (str(lab), str(slkcn), str(sldir), str(slzer), str(slope))
        self.RunCommand(command, **kwargs)

    def Nerr(self, nmerr="", nmabt="", ifkey="", num="", **kwargs):
        """
        APDL Command: /NERR

        Limits the number of warning and error messages displayed.

        Parameters
        ----------
        nmerr
            Maximum number of warning and error messages displayed per command.
            Defaults to 5 for interactive runs with the GUI turned on, 20 for
            interactive runs with the GUI turned off, 200 for batch runs.  If
            NMERR is negative, the absolute value of NMERR is used as the
            maximum number of warning and error messages written to the error
            file (file.ERR) per command, as well as the maximum number of
            messages displayed per command.

        nmabt
            Maximum number of warning and error messages allowed per command
            before run aborts (must be greater than zero).  Maximum value is
            99,999,999. Defaults to 10,000.

        --
            Unused field.

        ifkey
            Specifies whether or not to abort if an error occurs during a
            /INPUT operation:

            0 or OFF - Do not abort. This option is the default.

            1 or ON - Abort.

        num
            The number of invalid command warnings before a stop warning will
            be issued:

            0 - Disables the stop warning/error function.

            n - An integer value representing the number of warnings that will be encountered
                before prompting the user to stop (default = 5). The first
                error encountered will ALWAYS result in a prompt.

        Notes
        -----
        Limits the number of warning and error messages displayed for any one
        command in an interactive run.

        Warning and error messages continue to be written to Jobname.ERR
        regardless of these limits (unless NMERR is negative).

        Issue this command with NUM = n to specify the number of "invalid
        command" warnings to be encountered before the user is prompted to
        stop. You can then continue or abort the run. If you choose to abort
        the run, the log file can be saved so that any of the processing up to
        that point can be appended to an input that rectifies the condition. A
        batch run always aborts on the first error.  Issue /NERR,STAT to list
        current settings.

        Issue /NERR,DEFA to reset values to initial defaults.

        An IFKEY value of 1 or ON causes the ANSYS program to abort immediately
        upon encountering an error during a file /INPUT operation. However, use
        of this option may cause the following conditions to occur:

        The /INPUT command may abort if issued for a log file (jobname.log).

        Some macros may abort.

        A CAD connection may fail after reading only a small portion of a CAD
        model.

        The command is valid in any processor.

        """
        command = "/NERR,%s,%s,%s,%s" % (str(nmerr), str(nmabt), str(ifkey), str(num))
        self.RunCommand(command, **kwargs)

    def Stat(self, **kwargs):
        """
        APDL Command: STAT

        Displays the status of database settings.

        Notes
        -----
        In the DISPLAY program, STAT will show the current status of the
        program settings.

        In the ANSYS program, STAT is a command generated by the GUI and will
        appear in the log file (Jobname.LOG) if status is requested for some
        items under Utility Menu> List> Status.  Generally, STAT will be
        preceded by one of the commands listed below, which specifies the
        particular topic that status was requested for.

        If entered directly into the program, the STAT command should be
        immediately preceded by the desired topic command listed below.  In
        processors other than those listed below (e.g., AUX12), no topic
        command should proceed STAT.

        This command is valid in any processor.

        PREP7 topic commands (and their corresponding topics) are:

        SOLUTION topic commands (and their corresponding topics) are:

        POST1 topic commands (and their corresponding topics) are:

        POST26 topic commands (and their corresponding topics) are:

        """
        command = "STAT," % ()
        self.RunCommand(command, **kwargs)

    def Rmrplot(self, refname="", type="", mode1="", mode2="", **kwargs):
        """
        APDL Command: RMRPLOT

        Plots response surface of ROM function or its derivatives with respect
        to the dominant mode(s).

        Parameters
        ----------
        refname
            Reference name of ROM function. Valid reference names are "SENE"
            for the strain energy of the mechanical domain and any capacitance
            definition, previously defined by means of the RMCAP command, for
            the electrostatic domain.

        type
            Type of data to be plotted. Valid types are:

            FUNC - Response surface (default)

        mode1
            First mode number (used for Type = "FIRST" and Type = "SECOND"
            only).

        mode2
            Second mode number (used for Type = "SECOND" only).

        Notes
        -----
        The objective of response surface fit is to compute an analytical
        expression for the strain energy and the capacitance as functions of
        modal amplitudes. This command assumes that the coefficient files
        jobnam_ijk.pcs are available [RMRGENERATE]. Visualization of the
        response surface will help to evaluate the validity of the function
        fit.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "RMRPLOT,%s,%s,%s,%s" % (str(refname), str(type), str(mode1), str(mode2))
        self.RunCommand(command, **kwargs)

    def Ptxy(self, x1="", y1="", x2="", y2="", x3="", y3="", x4="", y4="",
             **kwargs):
        """
        APDL Command: PTXY

        Defines coordinate pairs for use in polygons and prisms.

        Parameters
        ----------
        x1, y1, x2, y2, x3, y3, x4, y4
            X and Y coordinate pairs on the working plane.

        Notes
        -----
        Defines coordinate pairs for use in polygons and prisms [POLY, RPRISM].
        The coordinates must be in the Cartesian coordinate system.  The
        coordinate pairs must be input in a continuous order.  PTXY  may be
        repeated (up to 100 pairs) until the required pairs have been defined.
        The pairs will be saved until either the POLY or PRISM command is
        entered.  Use PTXY,STAT to list the saved coordinate pairs.  Use
        PTXY,DELE to delete all the saved coordinate pairs.  See the RPOLY,
        RPRISM, and RPR4 commands for other ways to create polygons and prisms.

        """
        command = "PTXY,%s,%s,%s,%s,%s,%s,%s,%s" % (str(x1), str(y1), str(x2), str(y2), str(x3), str(y3), str(x4), str(y4))
        self.RunCommand(command, **kwargs)

    def Golist(self, **kwargs):
        """
        APDL Command: /GOLIST

        Reactivates the suppressed data input listing.

        Notes
        -----
        Reactivates printout of the data input listing suppressed with /NOLIST.

        This command is valid in any processor, but only within a batch run
        [/BATCH].

        """
        command = "/GOLIST," % ()
        self.RunCommand(command, **kwargs)

    def Cmacel(self, cm_name="", cmacel_x="", cmacel_y="", cmacel_z="",
               **kwargs):
        """
        APDL Command: CMACEL

        Specifies the translational acceleration of an element component

        Parameters
        ----------
        cm_name
            The name of the element component.

        cmacel_x, cmacel_y, cmacel_z
            Acceleration of the element component CM_NAME in the global
            Cartesian X, Y, and Z axis directions, respectively.

        Notes
        -----
        The CMACEL command specifies the translational acceleration of the
        element component in each of the global Cartesian (X, Y, and Z) axis
        directions.

        Components for which you want to specify acceleration loading must
        consist of elements only. The elements you use cannot be part of more
        than one component, and elements that share nodes cannot exist in
        different element components. You cannot apply the loading to an
        assembly of element components.

        To simulate gravity (by using inertial effects), accelerate the
        structure in the direction opposite to gravity. For example, apply a
        positive CMACELY to simulate gravity acting in the negative Y
        direction. Units are length/time2.

        You can define the acceleration for the following analyses types:

        Static (ANTYPE,STATIC)

        Harmonic (ANTYPE,HARMIC), full or mode-superposition method

        Transient (ANTYPE,TRANS), full or mode-superposition method

        Substructure (ANTYPE,SUBSTR)

        Accelerations are combined with the element mass matrices to form a
        body force load vector term. Units of acceleration and mass must be
        consistent to give a product of force units.

        In a modal harmonic or transient analysis, you must apply the load in
        the modal portion of the analysis. Mechanical APDL calculates a load
        vector and writes it to the mode shape file, which you can apply via
        the LVSCALE command.

        The CMACEL command supports tabular boundary conditions (%TABNAME_X%,
        %TABNAME_Y%, and %TABNAME_Z%) for CMACEL_X, CMACEL_Y, and CMACEL_Z
        input values (*DIM) as a function of both time and frequency for full
        transient and harmonic analyses.

        Related commands for inertia loads are ACEL, CGLOC, CGOMGA, DCGOMG,
        DOMEGA, OMEGA, CMOMEGA, and CMDOMEGA.

        See Analysis Tools in the Mechanical APDL Theory Reference for more
        information.

        This command is also valid in /PREP7.

        """
        command = "CMACEL,%s,%s,%s,%s" % (str(cm_name), str(cmacel_x), str(cmacel_y), str(cmacel_z))
        self.RunCommand(command, **kwargs)

    def Enersol(self, nvar="", item="", name="", **kwargs):
        """
        APDL Command: ENERSOL

        Specifies the total energies to be stored.

        Parameters
        ----------
        nvar
            Arbitrary reference number assigned to this variable (2 to NV).

        """
        command = "ENERSOL,%s,%s,%s" % (str(nvar), str(item), str(name))
        self.RunCommand(command, **kwargs)

    def Ltran(self, kcnto="", nl1="", nl2="", ninc="", kinc="", noelem="",
              imove="", **kwargs):
        """
        APDL Command: LTRAN

        Transfers a pattern of lines to another coordinate system.

        Parameters
        ----------
        kcnto
            Reference number of coordinate system where the pattern is to be
            transferred.  Transfer occurs from the active coordinate system.
            The coordinate system type and parameters of KCNTO must be the same
            as the active system.

        nl1, nl2, ninc
            Transfer lines from pattern beginning with NL1 to NL2 (defaults to
            NL1) in steps of NINC (defaults to 1).  If NL1 = ALL, NL2 and NINC
            are ignored and pattern is all selected lines [LSEL].  If NL1 = P,
            graphical picking is enabled and all remaining command fields are
            ignored (valid only in the GUI).  A component name may also be
            substituted for NL1 (NL2 and NINC are ignored).

        kinc
            Keypoint increment between sets.  If zero, the lowest available
            keypoint numbers are assigned [NUMSTR].

        noelem
            Specifies whether nodes and elements are also to be generated:

            0 - Generate nodes and elements associated with the original lines, if they exist.

            1 - Do not generate nodes and elements.

        imove
            Specifies whether lines will be moved or newly defined:

            0 - Generate additional lines.

            1 - Move original lines to new position retaining the same keypoint numbers (KINC
                and NOELM are ignored).  Valid only if the old lines are no
                longer needed at their original positions.  Corresponding
                meshed items are also moved if not needed at their original
                position.

        Notes
        -----
        Transfers a pattern of lines (and their corresponding keypoints and
        mesh) from one coordinate system to another (see analogous node
        transfer command, TRANSFER).  The MAT, TYPE, REAL, and ESYS attributes
        are based upon the lines in the pattern and not upon the current
        settings.  Coordinate systems may be translated and rotated relative to
        each other.  Initial pattern may be generated in any coordinate system.
        However, solid modeling in a toroidal coordinate system is not
        recommended.  Coordinate and slope values are interpreted in the active
        coordinate system and are transferred directly.  Lines are generated as
        described in the LGEN command.

        """
        command = "LTRAN,%s,%s,%s,%s,%s,%s,%s" % (str(kcnto), str(nl1), str(nl2), str(ninc), str(kinc), str(noelem), str(imove))
        self.RunCommand(command, **kwargs)

    def Cmomega(self, cm_name="", omegax="", omegay="", omegaz="", x1="",
                y1="", z1="", x2="", y2="", z2="", **kwargs):
        """
        APDL Command: CMOMEGA

        Specifies the rotational velocity of an element component about a user-
        defined rotational axis.

        Parameters
        ----------
        cm_name
            The name of the element component.

        omegax, omegay, omegaz
            If the X2, Y2, Z2 fields are not defined, OMEGAX, OMEGAY, and
            OMEGAZ specify the components of the rotational velocity vector in
            the global Cartesian X, Y, Z directions.

        x1, y1, z1
            If the X2, Y2, Z2 fields are defined,X1, Y1, and Z1 define the
            coordinates of the beginning point of the rotational axis vector.
            Otherwise, X1, Y1, and Z1 are the coordinates of a point through
            which the rotational axis passes.

        x2, y2, z2
            The coordinates of the end point of the rotational axis vector.

        Notes
        -----
        Specifies the rotational velocity components OMEGAX, OMEGAY, and OMEGAZ
        of an element component CM_NAME about a user-defined rotational axis.
        The rotational axis can be defined either as a vector passing through a
        single point or a vector connecting two points.

        You can define rotational velocity and rotational axis for these
        analysis types:

        Static (ANTYPE,STATIC)

        Harmonic (ANTYPE,HARMIC) -- Full or modal superposition

        Transient (ANTYPE,TRANS)  -- Full or modal superposition

        Substructuring (ANTYPE,SUBSTR)

        Modal (ANTYPE,MODAL)

        Rotational velocities are combined with the element mass matrices to
        form a body force load vector term. Units are radians/time. Related
        commands are ACEL, CGLOC, CGLOC,  CGOMGA, CMDOMEGA, DCGOMG, DOMEGA.

        See Analysis Tools in the Mechanical APDL Theory Reference for more
        information.

        You can use the CMOMEGA command in conjunction with either one of the
        following two groups of commands, but not with both groups
        simultaneously:

        Components for which you want to specify rotational loading must
        consist of elements only. The elements you use cannot be part of more
        than one component, and elements that share nodes cannot exist in
        different element components. You cannot apply the loading to an
        assembly of element components.

        If you have applied the Coriolis effect (CORIOLIS) using a stationary
        reference frame, the CMOMEGA command takes the gyroscopic damping
        matrix into account for the elements listed under “Stationary Reference
        Frame” in the notes section of the CORIOLIS command. ANSYS verifies
        that the rotation vector axis is parallel to the axis of the element;
        if not, the gyroscopic effect is not applied. If you issue a CMOMEGA
        command when the Coriolis or gyroscopic effect is present, a
        subsequently issued OMEGA command has no effect.

        The CMOMEGA command supports tabular boundary conditions (%TABNAME_X%,
        %TABNAME_Y%, and %TABNAME_Z%) for CMOMEGA_X, CMOMEGA_Y, and CMOMEGA_Z
        input values (*DIM) for full transient and harmonic analyses.

        In a mode-superposition harmonic or transient analysis, you must apply
        the load in the modal portion of the analysis. Mechanical APDL
        calculates a load vector and writes it to the MODE file, which you can
        apply via the LVSCALE command.

        """
        command = "CMOMEGA,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(cm_name), str(omegax), str(omegay), str(omegaz), str(x1), str(y1), str(z1), str(x2), str(y2), str(z2))
        self.RunCommand(command, **kwargs)

    def Plcrack(self, loc="", num="", **kwargs):
        """
        APDL Command: PLCRACK

        Displays cracking and crushing locations in SOLID65 elements.

        Parameters
        ----------
        loc
            Location at which symbols are to be displayed:

            0 - Plot symbols at integration points (default).

            1 - Plot symbol at element centroids (averaged).

        num
            Crack to be displayed:

            0 - Plot all cracks (default).

            1 - Plot only the first crack.

            2 - Plot only the second crack.

            3 - Plot only the third crack.

        Notes
        -----
        PLCRACK displays circles at locations of cracking or crushing in
        concrete elements.  Cracking is shown with a circle outline in the
        plane of the crack, and crushing is shown with an octahedron outline.
        If the crack has opened and then closed, the circle outline will have
        an X through it.  Each integration point can crack in up to three
        different planes.  The first crack at an integration point is shown
        with a red circle outline, the second crack with a green outline, and
        the third crack with a blue outline.

        Symbols shown at the element centroid (LOC = 1) are based on the status
        of all of the element's integration points.  If any integration point
        in the element has crushed, the crushed (octahedron) symbol is shown at
        the centroid.  If any integration point has cracked or cracked and
        closed, the cracked symbol is shown at the element centroid.  If at
        least five integration points have cracked and closed, the cracked and
        closed symbol is shown at the element centroid.  Finally, if more than
        one integration point has cracked, the circle outline at the element
        centroid shows the average orientation of all cracked planes for that
        element.

        Portions of this command are not supported by PowerGraphics
        [/GRAPHICS,POWER].

        """
        command = "PLCRACK,%s,%s" % (str(loc), str(num))
        self.RunCommand(command, **kwargs)

    def Psymb(self, label="", key="", **kwargs):
        """
        APDL Command: /PSYMB

        Shows various symbols on displays.

        Parameters
        ----------
        label
            Show symbols as selected from the following labels:

            CS - Local coordinate systems.

            NDIR - Nodal coordinate systems (on rotated nodes only).

            ESYS - Element coordinate systems (element displays only).

            LDIR - Line directions (line displays only).

            LDIV - Controls the display of element divisions on lines.

            ADIR - Area direction symbol (for keypoint, line, area and volume plots).

            LAYR - Layer orientations (relative to the projected element x-axis) or fiber
                   orientations in smeared reinforcing elements. Used only
                   within an element display. Use KEY to specify the layer
                   number.

            ECON - Element mesh symbols on keypoints and lines.

            DOT - Larger symbols displayed for node and keypoint locations.  When Label = DOT,
                  KEY = 1 by default.

            XNOD - Extra node of surface or circuit elements.

            FBCS - Force boundary condition scaling. Subsequent KEY value determines whether or
                   not to scale the applied and derived forces/moments to the
                   same maximum value.

            DEFA - Resets the symbol keys so that ANSYS displays none of the symbols controlled by
                   the /PSYMB command.  The value of the KEY field is ignored.

            STAT - Prints the status of the settings of the symbol keys controlled by the /PSYMB
                   command.  The KEY field is ignored.

        key
            Symbol key:

            -1  - Effective only if Label = LAYR and solid shape element display (/ESHAPE) is
                  active. Orientation of all layers appears with the solid
                  shape element display.

            0  - No symbol (default). If Label = LDIV, then KEY= 0 indicates that the displayed
                 element divisions will correspond to the existing mesh (the
                 word MESHED or EXISTING can also be substituted). Also, for
                 Label = LDIV, if you execute any meshing command (such as
                 AMESH or VMESH), KEY is set to 0 (MESHED) automatically. If
                 Label = FBCS, then KEY= 0 indicates that boundary condition
                 scaling will not be common. The applied and derived
                 forces/moments will be scaled to their respective maximum
                 values.

            1  - Include symbol. If Label = LDIV, then KEY = 1 indicates that the displayed line
                 divisions will correspond to the value assigned by LESIZE (the
                 word LESIZE can also be substituted). Also, for Label = LDIV,
                 if you execute the LESIZE command, KEY is set to 1 (LESIZE)
                 automatically. If Label = FBCS, then KEY= 1 indicates that
                 boundary condition scaling will be common. The applied and
                 derived forces/moments will be scaled to the same maximum
                 value.

            N  - If Label = LAYR, then N is equal to the layer number. If Label = DOT, then N
                 can be equal to 0,1,.....15, indicating the dot size. If Label
                 = LDIV, then KEY = -1, indicates that no element divisions
                 will be displayed (the word OFF can also be substituted).

        Notes
        -----
        Includes various symbols on the display.  Triads are right-handed with
        x displayed as the longest leg.  Where color is displayed, x is white,
        y is green, and z is blue.  For beams, x is always along the length of
        the element.   For lines, an arrow represents the direction of a line,
        from the beginning keypoint to the end keypoint.  See /PLOPTS command
        for additional display options.  Use /PSTATUS or /PSYMB,STAT to display
        settings.  Use /PSYMB,DEFA to reset all specifications back to their
        defaults.  The command /PSYMB,ECON,1 causes the symbol "M" to be
        displayed on keypoints and lines associated with meshed entities.  When
        you issue the command /PSYMB,DOT,1, a larger symbol is displayed for
        each node and keypoint location.

        PowerGraphics (/GRAPHICS,POWER) does not support /PSYMB,ESYS and
        /PSYMB,LAYR.

        If KEY = N and PowerGraphics is off, the centroid of the surface
        elements is connected to the extra node using a gray line.  However, if
        PowerGraphics is on, the color of the line connecting the centroid to
        the extra node is the same as that for the elements themselves (as
        determined by /PNUM).

        When Label = LAYR, the layer systems can be visualized with all
        current-technology layered elements and the smeared reinforcing element
        REINF265. To use /PSYMB,LAYR with REINF265, first set the vector-mode
        graphics option (/DEVICE,VECTOR,1).

        This command is valid in any processor.

        """
        command = "/PSYMB,%s,%s" % (str(label), str(key))
        self.RunCommand(command, **kwargs)

    def Pdropt(self, rvar="", corr="", stat="", shis="", hist="", cdf="",
               sens="", cmat="", conf="", **kwargs):
        """
        APDL Command: PDROPT

        Specifies the options for an HTML report.

        Parameters
        ----------
        rvar
            Specifies in which form to show the definitions of random variables
            in the report.

            0 - Using tables (including name and distribution parameter) and figures (including
                a probability density function plot and a cumulative
                distribution plot) (default).

            1 - Using tables only.

            2 - Using figures only.

            3 - None.

        corr
            Specifies if a table describing the correlation between random
            variables should be included in the report.

            0 - Yes, include this table (default).

            1 - No, do not include this table.

        stat
            Specifies which statistics to include in the report. In general,
            statistics are provided in a tabular form.

            0 - Statistics of the random output parameters only (default).

            1 - Statistics of the random input variables only.

            2 - Statistics of both the random input variables and the random output parameters.

            3 - None.

        shis
            Specifies which sample history plots to include in the report. This
            option applies to the random output parameters only.

            0 - None (default).

            1 - Mean value and standard deviation as a sample plot.

            2 - Mean value, standard deviation and sample values as a sample plot.

            3 - All types of sample plots - mean value, standard deviation, minimum value,
                maximum values, and the sample values.

        hist
            Specifies which histogram plots to include in the report.

            0 - Histogram of the random output parameters only (default).

            1 - Histogram of the random input variables only.

            2 - Histogram of both the random input variables and the random output parameters.

            3 - None.

        cdf
            Specifies which cumulative distribution function (CDF) plots to
            include in the report.

            0 - CDF of the random output parameters only (default).

            1 - CDF of the random input variables only.

            2 - CDF of both the random input variables and the random output parameters.

            3 - None.

        sens
            Specifies which sensitivity plots to include in the report.

            0 - Plots the sensitivities for all random output parameters based on Spearman-
                rank-order correlation coefficient (default).

            1 - Plots the sensitivities for all random output parameters based on linear
                (Pearson) correlation coefficient.

            2 - Plots the sensitivities according to option SENS=1 and SENS=2.

            3 - None.

        cmat
            Specifies which correlation matrices to include in the report.

            0 - Correlation matrix between random output parameters and random output
                parameters only (default).

            1 - Correlation matrix between random input variables and random output parameters
                only.

            2 - Correlation matrix between random input variables and random input variables
                only.

            3 - Correlation matrices according to option CMAT=0 and CMAT=1.

            4 - Correlation matrices according to option CMAT=0 and CMAT=2.

            5 - Correlation matrices according to option CMAT=1 and CMAT=2.

            6 - Correlation matrices according to option CMAT=0, CMAT=1, and CMAT=2.

            7 - None.

        conf
            Confidence level. The confidence level is used to plot confidence
            bounds for the history value. The value for the confidence level
            must be between 0.0 and 1.0 and it defaults to 0.95 (95%).
            Confidence bound(s) plotting is suppressed for CONF  0.5. This
            option is ignored if the report does not include plots for which
            confidence bounds are applicable.

        Notes
        -----
        Specifies the options for an HTML report. An HTML report includes a
        description of the deterministic model, the probabilistic model, the
        probabilistic methods used for the analyses, and the results obtained
        from the analyses.  The deterministic model is documented in the report
        by including a link to the analysis file (see PDANL command). In
        addition, an element plot of the component is shown, if available,
        based on the current view settings. The command ALLSEL is issued
        automatically prior to the respective plot command.

        """
        command = "PDROPT,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(rvar), str(corr), str(stat), str(shis), str(hist), str(cdf), str(sens), str(cmat), str(conf))
        self.RunCommand(command, **kwargs)

    def Mfconv(self, lab="", toler="", minref="", **kwargs):
        """
        APDL Command: MFCONV

        Sets convergence values for an ANSYS Multi-field solver analysis.

        Parameters
        ----------
        lab
            Valid labels:

        toler
            Convergence tolerance about program calculated reference value (the
            L2 norm of the new load in a multi-field analysis). Defaults to
            0.01 (1%) for all labels. Must be less than 1.0.

        --
            Unused field.

        minref
            The minimum value allowed for the program calculated reference
            value. If negative, no minimum is enforced. Defaults to 1.0e-6 for
            all labels. Not available in the GUI. MINREF corresponds to
            ||ϕnew|| as defined in Set up Stagger Solution in the Coupled-Field
            Analysis Guide.

        Notes
        -----
        MFCONV sets convergence values for variables at the ANSYS Multi-field
        solver interface.

        This command is also valid in PREP7.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFCONV,%s,%s,%s" % (str(lab), str(toler), str(minref))
        self.RunCommand(command, **kwargs)

    def Octype(self, datatype="", name="", **kwargs):
        """
        APDL Command: OCTYPE

        Specifies the type of ocean load data to follow.

        Parameters
        ----------
        datatype
            The type of ocean data to be input following this command:

            BASIC - The basic ocean load, required for any ocean loading.

            CURR - An optional drift current.

            WAVE - An optional ocean wave state.

        name
            An eight-character name for the ocean load. An ocean name can
            consist of letters and numbers, but cannot contain punctuation,
            special characters, or spaces.

        Notes
        -----
        The OCTYPE command specifies the type of ocean load data to follow
        (basic, current, or wave). Issue this command before defining your
        ocean load data (OCDATA and OCTABLE).

        Ocean loading applies only to current-technology pipe (PIPE288 and
        PIPE289), surface (SURF154), link (LINK180) and beam (BEAM188 and
        BEAM189) elements.

        An ocean current or wave is accessible repeatedly. For example, it is
        not necessary to input an identical current table again just because
        the drag coefficients of the basic input table have changed.

        The following example shows how you can use the basic (DataType =
        BASIC), current (DataType = CURR), and wave (DataType = WAVE) ocean
        data types within the context of a simple input file fragment:

        """
        command = "OCTYPE,%s,%s" % (str(datatype), str(name))
        self.RunCommand(command, **kwargs)

    def Tbdele(self, lab="", mat1="", mat2="", inc="", **kwargs):
        """
        APDL Command: TBDELE

        Deletes previously defined material data tables.

        Parameters
        ----------
        lab
            Data table label. (See the TB command for valid labels.)  If ALL,
            delete all data tables.

        mat1, mat2, inc
            Delete tables for materials MAT1 to (MAT2 defaults to MAT1) in
            steps of INC (defaults to 1).  If MAT1= ALL, ignore MAT2 and INC
            and delete data tables for all materials.

        Notes
        -----
        This command is also valid in SOLUTION.

        """
        command = "TBDELE,%s,%s,%s,%s" % (str(lab), str(mat1), str(mat2), str(inc))
        self.RunCommand(command, **kwargs)

    def Dspoption(self, reord_option="", memory_option="", memory_size="",
                  solve_info="", **kwargs):
        """
        APDL Command: DSPOPTION

         Sets memory option for the distributed sparse solver.

        Parameters
        ----------
        reord_option
            Reordering option:

            DEFAULT - Use the default reordering scheme.

            SEQORDER - Use a sequential equation reordering scheme within the distributed sparse
                       solver. Relative to PARORDER, this option typically
                       results in longer equation ordering times and therefore
                       longer overall solver times. Occasionally, however, this
                       option will produce better quality orderings which
                       decrease the matrix factorization times and improve
                       overall solver performance.

            PARORDER - Use a parallel equation reordering scheme within the distributed sparse solver.
                       Relative to SEQORDER, this option typically results in
                       shorter equation ordering times and therefore shorter
                       overall solver times. Occasionally, however, this option
                       will produce lower quality orderings which increase the
                       matrix factorization times and degrade overall solver
                       performance.

        memory_option
            Memory allocation option:

            DEFAULT - Use the default memory allocation strategy for the distributed sparse solver.
                      The default strategy attempts to run in the INCORE memory
                      mode. If there is not enough physical memory available
                      when the solver starts to run in the INCORE memory mode,
                      the solver will then attempt to run in the OUTOFCORE
                      memory mode.

            INCORE -  Use a memory allocation strategy in the distributed sparse solver that will
                     attempt to obtain enough memory to run with the entire
                     factorized matrix in memory. This option uses the most
                     amount of memory and should avoid doing any I/O. By
                     avoiding I/O, this option achieves optimal solver
                     performance. However, a significant amount of memory is
                     required to run in this mode, and it is only recommended
                     on machines with a large amount of memory. If the
                     allocation for in-core memory fails, the solver will
                     automatically revert to out-of-core memory mode.

            OUTOFCORE - Use a memory allocation strategy in the distributed sparse solver that will
                        attempt to allocate only enough work space to factor
                        each individual frontal matrix in memory, but will
                        share the entire factorized matrix on disk. Typically,
                        this memory mode results in poor performance due to the
                        potential bottleneck caused by the I/O to the various
                        files written by the solver.

            FORCE - This option, when used in conjunction with the Memory_Size option, allows you
                    to force the distributed sparse solver to run with a
                    specific amount of memory. This option is only recommended
                    for the advanced user who understands distributed sparse
                    solver memory requirements for the problem being solved,
                    understands the physical memory on the system, and wants to
                    control the distributed sparse solver memory usage.

        memory_size
            Initial memory size allocation for the sparse solver in MB. The
            Memory_Size setting should always be well within the physical
            memory available, but not so small as to cause the distributed
            sparse solver to run out of memory. Warnings and/or errors from the
            distributed sparse solver will appear if this value is set too low.
            If the FORCE memory option is used, this value is the amount of
            memory allocated for the entire duration of the distributed sparse
            solver solution.

        --, --
            Unused fields

        solve_info
            Solver output option:

            OFF - Turns off additional output printing from the distributed sparse solver
                  (default).

            PERFORMANCE - Turns on additional output printing from the distributed sparse solver,
                          including a performance summary and a summary of file
                          I/O for the distributed sparse solver. Information on
                          memory usage during assembly of the global matrix
                          (that is, creation of the Jobname.FULL file) is also
                          printed with this option.

        Notes
        -----
        This command controls options related to the distributed sparse solver
        in all analysis types where the distributed sparse solver can be used.

        The amount of memory required for the distributed sparse solver is
        unknown until the matrix structure is preprocessed, including equation
        reordering. The amount of memory allocated for the distributed sparse
        solver is then dynamically adjusted to supply the solver what it needs
        to compute the solution.

        If you have a large memory system, you may want to try selecting the
        INCORE memory mode for larger jobs to improve performance. Also, when
        running the distributed sparse solver with many processors on the same
        machine or on a machine with very slow I/O performance (e.g., slow hard
        drive speed), you may want to try using the INCORE memory mode to
        achieve better performance. However, doing so may require much more
        memory compared to running in the OUTOFCORE memory mode.

        Running with the INCORE memory mode is best for jobs which comfortably
        fit within the limits of the physical memory on a given system. If the
        distributed sparse solver workspace exceeds physical memory size, the
        system will be forced to use virtual memory (or the system page/swap
        file). In this case, it is typically more efficient to run with the
        OUTOFCORE memory mode.

        """
        command = "DSPOPTION,%s,%s,%s,%s" % (str(reord_option), str(memory_option), str(memory_size), str(solve_info))
        self.RunCommand(command, **kwargs)

    def Nsym(self, ncomp="", inc="", node1="", node2="", ninc="", **kwargs):
        """
        APDL Command: NSYM

        Generates a reflected set of nodes.

        Parameters
        ----------
        ncomp
            Symmetry key:

            X - X (or R) symmetry (default).

            Y - Y (or θ) symmetry.

            Z - Z (or Φ) symmetry.

        inc
            Increment all nodes in the given pattern by INC to form the
            reflected node pattern.

        node1, node2, ninc
            Reflect nodes from pattern beginning with NODE1 to NODE2 (defaults
            to NODE1) in steps of NINC (defaults to 1).  If  NODE1 = ALL, NODE2
            and NINC are ignored and pattern is all selected nodes [NSEL].  If
            NODE1 = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).  A component name may
            also be substituted for NODE1 (NODE2 and NINC are ignored).

        Notes
        -----
        Generates nodes from a given node pattern by a symmetry reflection.
        Reflection is done in the active coordinate system by changing a
        particular coordinate sign.  Nodes in the pattern may have been
        generated in any coordinate system.   Nodal rotation angles are not
        reflected.

        Symmetry reflection may be used with any node pattern, in any
        coordinate system, as many times as desired.  Reflection is
        accomplished by a coordinate sign change (in the active coordinate
        system).  For example, an X-reflection in a Cartesian coordinate system
        generates additional nodes from a given pattern, with a node increment
        added to each node number, and an X coordinate sign change.  An
        R-reflection in a cylindrical coordinate system gives a reflected
        "radial" location by changing the "equivalent" Cartesian (i.e., the
        Cartesian system with the same origin as the active cylindrical system)
        X and Y coordinate signs.  An R-reflection in a spherical coordinate
        system gives a reflected "radial" location by changing the equivalent
        Cartesian X, Y, and Z coordinate location signs.  Nodal coordinate
        system rotation angles are not reflected.

        """
        command = "NSYM,%s,%s,%s,%s,%s" % (str(ncomp), str(inc), str(node1), str(node2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Xflist(self, enrichmentid="", **kwargs):
        """
        APDL Command: XFLIST

        Lists enrichment details and associated crack information

        Parameters
        ----------
        enrichmentid or (blank)
            Name of the enrichment specified via the associated XFENRICH
            command. Specifying EnrichmentID is optional.

        Notes
        -----
        This command is valid in PREP7 (/PREP7) and SOLUTION (/SOLU).

        """
        command = "XFLIST,%s" % (str(enrichmentid))
        self.RunCommand(command, **kwargs)

    def Lsdump(self, enginename="", filename="", **kwargs):
        """
        APDL Command: *LSDUMP

        Dumps a linear solver engine to a binary File.

        Parameters
        ----------
        enginename
            Name used to identify this engine. Must have been previously
            created using *LSENGINE and factorized using *LSFACTOR.

        filename
            Name of the file to create.

        Notes
        -----
        Dumps a previously factorized linear solver system to a binary file.
        Only LAPACK and BCS linear solvers can be used with this feature. The
        Linear Solver can later be restored with the *LSRESTORE command.

        A BCS Sparse Solver can be dumped only if uses the INCORE memory option
        (see BCSOPTION).

        """
        command = "*LSDUMP,%s,%s" % (str(enginename), str(filename))
        self.RunCommand(command, **kwargs)

    def Ftype(self, filetype="", prestype="", **kwargs):
        """
        APDL Command: FTYPE

        Specifies the file type and pressure type for the subsequent import of
        source points and pressures.

        Parameters
        ----------
        filetype
            Type of file from which the pressure data will be retrieved (no
            default):

            CFXTBR - File from a CFX Transient Blade Row (TBR) analysis export.

            CFDPOST - File from a CFD-Post BC Profile export.

            FORMATTED - Formatted file.

            CSV - Comma-Separated Values file.

        prestype
            Type of pressure data contained in the file:

            0 - Only real-valued pressures are on the file.

            1 - Real-valued and imaginary-valued pressures are on the file (default).

        Notes
        -----
        CFX Transient Blade Row files (FileType = CFXTBR) are obtained from the
        Export Results Tab in CFX-Pre, with [Export Surface Name]: Option set
        to Harmonic Forced Response.

        CFD-Post files (FileType = CFDPOST) are obtained from the Export action
        in CFD-Post with Type set to BC Profile.

        Formatted files (FileType = FORMATTED) contain the coordinates and
        pressure data in fixed-format columns in the order  x, y, z, pressure.
        You may have other columns of data in the file which can be skipped
        over in the Format specifier on the READ command, but the data must be
        in that order.

        Comma-separated values files (FileType = CSV) contain the coordinates
        and pressure data in comma-separated fields. The data can be in any
        order, and other fields of data may also be present.

        """
        command = "FTYPE,%s,%s" % (str(filetype), str(prestype))
        self.RunCommand(command, **kwargs)

    def Edndtsd(self, vect1="", vect2="", datap="", fitpt="", vect3="",
                vect4="", disp="", **kwargs):
        """
        APDL Command: EDNDTSD

        Allows smoothing of noisy data for explicit dynamics analyses and
        provides a graphical representation of the data.

        Parameters
        ----------
        vect1
            Name of the first vector that contains the noisy data set (i.e.,
            independent variable). You must create and fill this vector before
            issuing EDNDTSD.

        vect2
            Name of the second vector that contains the dependent set of data.
            Must be the same length as the first vector. You must create and
            fill this vector before issuing EDNDTSD.

        datap
            Number of data points to be fitted, starting from the beginning of
            the vector. If left blank, the entire vector will be fitted. The
            maximum number of data points is 100,000 (or greater, depending on
            the memory of the computer).

        fitpt
            Curve fitting order to be used as a smooth representation of the
            data. This number should be less than or equal to the number of
            data points. However, because high order polynomial curve fitting
            can cause numerical difficulties, a polynomial order less than 7 is
            suggested. The default (blank) is one-half the number of data
            points or 7, which ever is less. The following values are
            available:

            1 - Curve is the absolute average of all of the data points.

            2 - Curve is the least square average of all of the data points.

            3 or more - Curve is a polynomial of the order (n-1), where n is the number of data fitting
                        order points.

        vect3
            Name of the vector that contains the smoothed data of the
            independent variable. This vector should have a length equal to or
            greater than the number of smoothed data points. In batch (command)
            mode, you must create this vector before issuing the EDNDTSD
            command. In interactive mode, the GUI automatically creates this
            vector (if it does not exist). If you do not specify a vector name,
            the GUI will name the vector smth_ind.

        vect4
            Name of the vector that contains the smoothed data of the dependent
            variable.  This vector must be the same length as Vect3.  In batch
            (command) mode, you must create this vector before issuing the
            EDNDTSD command. In interactive mode, the GUI automatically creates
            this vector (if it does not exist). If you do not specify a vector
            name, the GUI will name the vector smth_dep.

        disp
            Specifies how you want to display data. No default; you must
            specify an option.

            1 - Unsmoothed data only

            2 - Smoothed data only

            3 - Both smoothed and unsmoothed data

        Notes
        -----
        You can control the attributes of the graph using standard ANSYS
        controls (/GRID, /GTHK, /COLOR, etc.). If working interactively, these
        controls appear in this dialog box for convenience, as well as in their
        standard dialog boxes. You must always create Vect1 and Vect2 (using
        *DIM) and fill these vectors before smoothing the data. If you're
        working interactively, ANSYS automatically creates Vect3 and Vect4, but
        if you're working in batch (command) mode, you must create Vect3 and
        Vect4 (using *DIM) before issuing EDNDTSD.  Vect3 and Vect4 are then
        filled automatically by ANSYS.  In addition, ANSYS creates an
        additional TABLE type array that contains the smoothed array and the
        unsmoothed data to allow for plotting later with *VPLOT.  Column 1 in
        this table corresponds to Vect1, column 2 to Vect2, and column 3 to
        Vect4.  This array is named Vect3_SMOOTH, up to a limit of 32
        characters. For example, if the array name is X1, the table name is
        X1_SMOOTH.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDNDTSD,%s,%s,%s,%s,%s,%s,%s" % (str(vect1), str(vect2), str(datap), str(fitpt), str(vect3), str(vect4), str(disp))
        self.RunCommand(command, **kwargs)

    def Ansol(self, nvar="", node="", item="", comp="", name="", mat="",
              real="", ename="", **kwargs):
        """
        APDL Command: ANSOL

        Specifies averaged nodal data to be stored from the results file in the
        solution coordinate system.

        Parameters
        ----------
        nvar
            Arbitrary reference number assigned to this variable (2 to NV
            [NUMVAR]). Overwrites any existing results for this variable.

        node
            Node number for which data are to be stored.

        item
            Label identifying the item. General item labels are shown in Table
            126: ANSOL - General Item and Component Labels below. Some items
            also require a component label.

        comp
            Component of the item (if required). General component labels are
            shown in Table 126: ANSOL - General Item and Component Labels
            below.

        name
            Thirty-two character name for identifying the item on the printout
            and displays. Defaults to an eight character label formed by
            concatenating the first four characters of the Item and Comp
            labels.

        mat
            The material number. Average will be computed based on the subset
            of elements with the specified material number. DEFAULT: Use all
            elements in the active set unless Real and/or Ename is specified.

        real
            The real number. Average will be computed based on the subset of
            elements with the specified real number. DEFAULT: Use all elements
            in the active set unless Mat and/or Ename is specified.

        ename
            The element type name. Average will be computed based on the subset
            of elements with the specified element type name. DEFAULT: Use all
            elements in the active set unless Mat and/or Real is specified.

        Notes
        -----
        Valid item and component labels for averaged nodal results are listed
        in Table: 126:: ANSOL - General Item and Component Labels, below.

        All element nodal quantities are obtained in RSYS, Solu and then
        averaged.

        The ANSOL command defines averaged nodal results data to be stored from
        a results file [FILE]. Not all items are valid for all nodes. See the
        input and output summary tables of the Element Reference of each
        element that is attached to the node for the available items.

        COORDINATE SYSTEMS: All element nodal results used by ANSOL for
        averaging are in the element coordinate system, except for layered
        elements. Layered element results are in the layer coordinate system.
        You can further specify the element nodal results, for some elements,
        with the SHELL, LAYERP26, and FORCE commands.

        ANSOL does not transform results from RSYS, SOLU to other coordinate
        systems. Verify that all elements attached to the subject node have the
        same coordinate system before using ANSOL.

        SHELL ELEMENTS: The default shell element coordinate system is based on
        node ordering. For shell elements the adjacent elements could have a
        different RSYS,SOLU, making the resultant averaged data inconsistent. A
        note to this effect is issued when ANSOL is used in models containing
        shell elements. Ensure that consistent coordinate systems are active
        for all associated elements used by the ANSOL command.

        DERIVED QUANTITIES: Some of the result items supported by ANSOL (see
        Table: 126:: ANSOL - General Item and Component Labels) are derived
        from the component quantities. Use AVPRIN to specify the principal and
        vector sum quantity averaging methods.

        DEFAULT: If Mat, Real , and Ename are not specified, all of the
        elements attached to the node will be considered. When a material ID,
        real constant ID, or element type discontinuity is detected at a node,
        a note is issued. For example, in a FSI analysis, a FLUID30 element at
        the structure interface would be considered. But since it contains no
        SX result, it will not be used during STORE operations.

        Table: 126:: : ANSOL - General Item and Component Labels

        For more information on the meaning of contact status and its possible
        values, see Reviewing Results in POST1 in the Contact Technology Guide.

        """
        command = "ANSOL,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nvar), str(node), str(item), str(comp), str(name), str(mat), str(real), str(ename))
        self.RunCommand(command, **kwargs)

    def Edpvel(self, option="", pid="", vx="", vy="", vz="", omegax="",
               omegay="", omegaz="", xc="", yc="", zc="", angx="", angy="",
               angz="", **kwargs):
        """
        APDL Command: EDPVEL

        Applies initial velocities to parts or part assemblies in an explicit
        dynamic analysis.

        Parameters
        ----------
        option
            Label identifying the option to be performed.

            VGEN - Define initial velocities for the part or part assembly based on translational
                   velocities (relative to global Cartesian) and the rotational
                   velocity about an arbitrary axis. For this option, use the
                   fields VX, VY, VZ to specify the translational velocities,
                   and use OMEGAX, XC, YC, ZC, ANGX, ANGY, ANGZ to specify the
                   rotational velocity and the axis of rotation.

            VELO - Define initial velocity for the part or part assembly based on translational
                   velocities and nodal rotational velocities input relative to
                   the global Cartesian axes. For this option, use the
                   following fields to define the initial velocity: VX, VY, VZ,
                   OMEGAX, OMEGAY, OMEGAZ.

            LIST - List initial velocity for the part or part assembly specified by PID. If PID is
                   blank, all initial velocities defined on parts and part
                   assemblies are listed. Remaining fields are ignored for this
                   option.

            DELE - Delete initial velocity defined for the part or part assembly specified by PID.
                   If PID is blank, all initial velocities defined on parts and
                   part assemblies are deleted. Remaining fields are ignored
                   for this option.

        pid
            Part ID or part assembly ID to which the initial velocity is to be
            applied. The part or assembly ID must be defined (EDPART or EDASMP)
            before issuing this command.

        vx
            Initial velocity in X direction. Defaults to 0.

        vy
            Initial velocity in Y direction. Defaults to 0.

        vz
            Initial velocity in Z direction. Defaults to 0.

        omegax
            For Option = VGEN, OMEGAX is the initial rotational velocity of the
            part or part assembly about the specified rotational axis. For
            Option = VELO, OMEGAX is the initial nodal rotational velocity
            about the X-axis. OMEGAX defaults to 0.

        omegay
            Initial nodal rotational velocity about the Y-axis (used only if
            Option = VELO). Defaults to 0.

        omegaz
            Initial nodal rotational velocity about the Z-axis (used only if
            Option = VELO). Defaults to 0.

        Notes
        -----
        You cannot mix the two methods of initial velocity input (Option = VELO
        and Option = VGEN) in the same analysis. You must use only one method
        for all initial velocity definitions.

        The VGEN and VELO methods differ in how the rotational velocity is
        defined. Use Option = VGEN to input the initial velocities of a
        rotating part or part assembly. Use Option = VELO to apply the
        rotations directly to the nodes' rotation degrees of freedom. Since
        only shell and beam elements have rotation degrees of freedom, the
        rotations input with Option = VELO are only applicable to SHELL163 and
        BEAM161 elements. The rotational velocities input with Option = VELO
        are ignored for nodes not having rotational degrees of freedom (such as
        nodes attached to a SOLID164 or SOLID168 element).

        It is normally acceptable to mix nodes belonging to deformable bodies
        and rigid bodies in the part assembly used in an initial velocity
        definition. However, when defining initial velocities in an implicit-
        to-explicit sequential solution, this is not an acceptable practice. In
        order for the initial velocities to be defined correctly in this type
        of analysis, you must define the initial velocities on the deformable
        body nodes separately from the initial velocities on the rigid body
        nodes.

        Issuing the EDPVEL command again for the same part or part assembly
        (PID) will overwrite previous initial velocities defined for that part
        or part assembly.

        To set the initial velocities to zero, issue the EDPVEL command with
        only the Option (use VELO or VGEN) and PID fields specified.

        In a small restart analysis (EDSTART,2), you can only use the Option =
        VELO method to change initial velocities. When used in a small restart,
        the command EDPVEL,VELO changes the velocity of the specified part or
        part assembly. If you don't change the velocity of the parts and
        assemblies, their velocity at the beginning of the restart will be the
        same as the velocity at the end of the previous analysis.

        Except for the LIST option, the EDPVEL command is not supported in a
        full restart analysis (EDSTART,3). You can list initial velocities
        defined in the previous analysis with the command EDPVEL,LIST. However,
        you cannot change initial velocities for parts that existed in the
        previous analysis; their velocity at the beginning of the analysis will
        be the same as the velocity at the end of the previous analysis. In
        addition, you cannot define initial velocities for any parts that are
        added in the full restart; the velocity of new parts will be zero.

        To apply initial velocities to node components or nodes, use the EDVEL
        command.

        You can use EDPVEL and EDVEL in the same analysis. If a node or node
        component input on the EDVEL command shares common nodes with a part or
        part assembly input on the EDPVEL command, the initial velocities
        defined on the common nodes will be determined by the last command
        input.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDPVEL,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(pid), str(vx), str(vy), str(vz), str(omegax), str(omegay), str(omegaz), str(xc), str(yc), str(zc), str(angx), str(angy), str(angz))
        self.RunCommand(command, **kwargs)

    def Latt(self, mat="", real="", type="", kb="", ke="", secnum="",
             **kwargs):
        """
        APDL Command: LATT

        Associates element attributes with the selected, unmeshed lines.

        Parameters
        ----------
        mat, real, type
            Material number, real constant set number, and type number to be
            associated with selected, unmeshed lines.

        --
            Unused field.

        kb, ke
            Beginning and ending orientation keypoints to be associated with
            selected, unmeshed lines.  ANSYS uses the location of these
            keypoints to determine how to orient beam cross sections during
            beam meshing.  Beam elements may be created along a line with a
            constant orientation by specifying only one orientation keypoint
            (KB), or a pre-twisted beam may be created by selecting different
            orientation keypoints at each end of the line (KB and KE).  (For a
            line bounded by two keypoints (KP1 and KP2), the orientation vector
            at the beginning of the line extends from KP1 to KB, and the
            orientation vector at the end of the line extends from KP2 to KE.
            The orientation vectors are used to compute the orientation nodes
            of the elements.)

        secnum
            Section identifier to be associated with selected, unmeshed lines.
            For details, see the description of the SECTYPE and SECNUM
            commands.

        Notes
        -----
        The element attributes specified by the LATT command will be used when
        the lines are meshed.

        Lines subsequently generated from the lines will also have the
        attributes specified by MAT, REAL, TYPE, and SECNUM.  If a line does
        not have these attributes associated with it (by this command) at the
        time it is meshed, the attributes are obtained from the then current
        MAT, REAL, TYPE, and SECNUM command settings.

        In contrast, the values specified by KB and KE apply only to the
        selected lines; that is, lines subsequently generated from these lines
        will not share these attributes.  Similarly, if a line does not have KB
        and KE attributes associated with it via the LATT command at the time
        it is meshed, ANSYS cannot obtain the attributes from elsewhere.  See
        the discussion on beam meshing in Meshing Your Solid Model in the
        Modeling and Meshing Guide for more information.

        Reissue the LATT command (before lines are meshed) to change the
        attributes.  A zero (or blank) argument removes the corresponding
        association. If any of the arguments are defined as -1, then that value
        will be left unchanged in the selected set.

        In some cases, ANSYS can proceed with a line meshing operation even
        when no logical element type has been assigned via LATT,,,TYPE or TYPE.
        See Meshing Your Solid Model in the Modeling and Meshing Guide for more
        information about setting element attributes.

        """
        command = "LATT,%s,%s,%s,%s,%s,%s" % (str(mat), str(real), str(type), str(kb), str(ke), str(secnum))
        self.RunCommand(command, **kwargs)

    def Gfile(self, size="", **kwargs):
        """
        APDL Command: /GFILE

        Specifies the pixel resolution on Z-buffered graphics files.

        Parameters
        ----------
        size
            Pixel resolution.  Defaults to a pixel resolution of 800.  Valid
            values are from 256 to 2400.

        Notes
        -----
        Defines the pixel resolution on subsequently written graphics files
        (Jobname.GRPH) for software Z-buffered displays [/TYPE].  Lowering the
        pixel resolution produces a "fuzzier" image; increasing the resolution
        produces a "sharper" image but takes a little longer.

        This command is valid in any processor.

        """
        command = "/GFILE,%s" % (str(size))
        self.RunCommand(command, **kwargs)

    def Numcmp(self, label="", **kwargs):
        """
        APDL Command: NUMCMP

        Compresses the numbering of defined items.

        Parameters
        ----------
        label
            Items to be compressed:

            NODE - Node numbers

            ELEM - Element numbers

            KP - Keypoint numbers

            LINE - Line numbers

            AREA - Area numbers

            VOLU - Volume numbers

            MAT - Material numbers

            TYPE - Element type numbers

            REAL - Real constant numbers

            CP - Coupled set numbers

            SECN - Section numbers

            CE - Constraint equation numbers

            ALL - All item numbers

        Notes
        -----
        The NUMCMP command effectively compresses out unused item numbers by
        renumbering all the items, beginning with one and continuing throughout
        the model.  The renumbering order follows the initial item numbering
        order (that is, compression lowers the maximum number by "sliding"
        numbers down to take advantage of unused or skipped numbers).  All
        defined items are renumbered, regardless of whether or not they are
        actually used or selected.  Applicable related items are also checked
        for renumbering as described for the merge operation (NUMMRG).

        Compressing material numbers (NUMCMP,ALL or NUMCMP,MAT) does not update
        the material number referenced by either of the following:

        A temperature-dependent convection or surface-to-surface radiation load
        (SF, SFE, SFL, SFA)

        Real constants for multi-material elements (such as SOLID65)

         Compression is usually not required unless memory space is limited and
        there are large gaps in the numbering sequence.

        """
        command = "NUMCMP,%s" % (str(label))
        self.RunCommand(command, **kwargs)

    def Tbin(self, oper="", par1="", par2="", par3="", par4="", **kwargs):
        """
        APDL Command: TBIN

        Sets parameters used for interpolation of the material data tables.

        Parameters
        ----------
        oper
            Operation to perform:

            Operation to perform: - SCALE

        par1
            Independent variable, which can be any field variable specified via
            the TBFIELD command.

        par2
            Index of any material parameter specified via the TBDATA command.

        par3
            Scale to be used for the independent variable. Valid options are
            LINEAR (linear) or LOG (logarithmic).

        par4
            Scale to be used for the dependent variable (the material parameter
            specified via Par2). Valid options are LINEAR (linear) or LOG
            (logarithmic).

        Notes
        -----
        For a list of the supported material data tables (TB), see Logarithmic
        Interpolation and Scaling in the Material Reference.

        """
        command = "TBIN,%s,%s,%s,%s,%s" % (str(oper), str(par1), str(par2), str(par3), str(par4))
        self.RunCommand(command, **kwargs)

    def Gssol(self, nvar="", item="", comp="", name="", **kwargs):
        """
        APDL Command: GSSOL

        Specifies which results to store from the results file when using
        generalized plane strain.

        Parameters
        ----------
        nvar
            Arbitrary reference number or name assigned to this variable.
            Variable numbers can be 2 to NV (NUMVAR) while the name can be an
            eight byte character string. Overwrites any existing results for
            this variable.

        item
            Label identifying item to be stored.

            LENGTH - Change of fiber length at the ending point.

            ROT - Rotation of the ending plane during deformation.

            F - Reaction force at the ending point in the fiber direction.

            M - Reaction moment applied on the ending plane.

        comp
            Component of the item, if Item = ROT or M.

            X - The rotation angle or reaction moment of the ending plane about X.

            Y - The rotation angle or reaction moment of the ending plane about Y.

        name
            Thirty-two character name identifying the item on the printout and
            display. Defaults to the label formed by concatenating the first
            four characters of the Item and Comp labels.

        Notes
        -----
        This command stores the results (new position of the ending plane after
        deformation) for generalized plane strain. All outputs are in the
        global Cartesian coordinate system. For more information about the
        generalized plane strain feature, see Generalized Plane Strain Option
        of Current-Technology Solid Elements in the Element Reference.

        """
        command = "GSSOL,%s,%s,%s,%s" % (str(nvar), str(item), str(comp), str(name))
        self.RunCommand(command, **kwargs)

    def Lsrestore(self, enginename="", filename="", **kwargs):
        """
        APDL Command: *LSRESTORE

        Restores a linear solver engine from a binary file.

        Parameters
        ----------
        enginename
            Name used to identify this engine.

        filename
            Name of the file to read from.

        Notes
        -----
        Restores a previously dumped Linear Solver (see the *LSDUMP command).
        This Linear Solver can be used to solve a linear system using the
        *LSBAC command.

        """
        command = "*LSRESTORE,%s,%s" % (str(enginename), str(filename))
        self.RunCommand(command, **kwargs)

    def Smrtsize(self, sizlvl="", fac="", expnd="", trans="", angl="", angh="",
                 gratio="", smhlc="", smanc="", mxitr="", sprx="", **kwargs):
        """
        APDL Command: SMRTSIZE

        Specifies meshing parameters for automatic (smart) element sizing.

        Parameters
        ----------
        sizlvl
            Overall element size level for meshing.  The level value controls
            the fineness of the mesh.  (Any input in this field causes
            remaining arguments to be ignored.)   Valid inputs are:

            n - Activate SmartSizing and set the size level to n.  Must be an integer value
                from 1 (fine mesh) to 10 (coarse mesh).  Remaining arguments
                are ignored, and argument values are set as shown in Table 229:
                SMRTSIZE - Argument Values for h-elements .

            STAT - List current SMRTSIZE settings.

            DEFA - Set all SMRTSIZE settings to default values (as shown in Table 229: SMRTSIZE -
                   Argument Values for h-elements  for size level 6).

            OFF - Deactivate SmartSizing.  Current settings of DESIZE will be used.  To
                  reactivate SmartSizing, issue SMRTSIZE,n.

        fac
            Scaling factor applied to the computed default mesh sizing.
            Defaults to 1 for h-elements (size level 6), which is medium.
            Values from 0.2 to 5.0 are allowed.

        expnd
            Mesh expansion (or contraction) factor.  (This factor is the same
            as MOPT,EXPND,Value.)  EXPND is used to size internal elements in
            an area based on the size of the elements on the area's boundaries.
            For example, issuing SMRTSIZE,,,2 before meshing an area will allow
            a mesh with elements that are approximately twice as large in the
            interior of an area as they are on the boundary.   If EXPND is less
            than 1, a mesh with smaller elements on the interior of the area
            will be allowed.  EXPND should be greater than 0.5 but less than 4.
            EXPND defaults to 1 for h-elements (size level 6), which does not
            allow expansion or contraction of internal element sizes (except
            when using AESIZE element sizing).  (If EXPND is set to zero, the
            default value of 1 will be used.)  The actual size of the internal
            elements will also depend on the TRANS option or upon AESIZE or
            ESIZE sizing, if used.

        trans
            Mesh transition factor.  (This factor is the same as
            MOPT,TRANS,Value.)  TRANS is used to control how rapidly elements
            are permitted to change in size from the boundary to the interior
            of an area.  TRANS defaults to 2.0 for h-elements (size level 6),
            which permits elements to approximately double in size from one
            element to the next as they approach the interior of the area.  (If
            TRANS is set to zero, the default value will be used.)   TRANS must
            be greater than 1 and, for best results, should be less than 4.
            The actual size of the internal elements will also depend on the
            EXPND option or upon AESIZE or ESIZE sizing, if used.

        angl
            Maximum spanned angle per lower-order element for curved lines.
            Defaults to 22.5 degrees per element (size level 6).  This angle
            limit may be exceeded if the mesher encounters a small feature
            (hole, fillet, etc.).  (This value is not the same as that set by
            DESIZE,,,,ANGL.)

        angh
            Maximum spanned angle per higher-order element for curved lines.
            Defaults to 30 degrees per element (size level 6).  This angle
            limit may be exceeded if the mesher encounters a small feature
            (hole, fillet, etc.).  (This value is NOT the same as that set by
            DESIZE,,,,,ANGH.)

        gratio
            Allowable growth ratio used for proximity checking.  Defaults to
            1.5 for h-elements (size level 6).  Values from 1.2 to 5.0 are
            allowed; however, values from 1.5 to 2.0 are recommended.

        smhlc
            Small hole coarsening key, can be ON (default for size level 6) or
            OFF.  If ON, this feature suppresses curvature refinement that
            would result in very small element edges (i.e., refinement around
            small features).

        smanc
            Small angle coarsening key, can be ON (default for all levels) or
            OFF.  If ON, this feature restricts proximity refinement in areas
            where it is ill-advised (that is, in tight corners on areas,
            especially those that approach 0 degrees).

        mxitr
            Maximum number of sizing iterations (defaults to 4 for all levels).

        sprx
            Surface proximity refinement key, can be off (SPRX = 0, which is
            the default for all levels) or on via two different values (SPRX =
            1 or SPRX = 2).  If SPRX = 1, surface proximity refinement is
            performed and any shell elements that need to be modified are
            modified.  If SPRX=2, surface proximity refinement is performed but
            no shell elements are altered.

        Notes
        -----
        If a valid level number (1 (fine) to 10 (coarse)) is input on SIZLVL,
        inputs for remaining arguments are ignored, and the argument values are
        set as shown in Table: 229:: SMRTSIZE - Argument Values for h-elements
        .

        The settings shown are for h-elements. The first column contains SIZLV
        data, ranging from 10 (coarse) to 1 (fine). The default is 6 (indicated
        by the shaded row).

        Table: 229:: : SMRTSIZE - Argument Values for h-elements

        Where appropriate, SmartSizing will start with AESIZE settings.
        Elsewhere, it will start with any defined ESIZE,SIZE setting. It will
        locally override AESIZE or ESIZE for proximity and curvature.
        SmartSizing ignores any ESIZE,,NDIV setting.

        LESIZE line division and spacing specifications will be honored by
        SmartSizing, unless you give permission for SmartSizing to override
        them (for proximity or curvature) by setting KYNDIV to 1. Lines not
        having an LESIZE specification are meshed as well as they can be.

        This command is also valid for rezoning.

        """
        command = "SMRTSIZE,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(sizlvl), str(fac), str(expnd), str(trans), str(angl), str(angh), str(gratio), str(smhlc), str(smanc), str(mxitr), str(sprx))
        self.RunCommand(command, **kwargs)

    def Anstoaqwa(self, fname="", vertaxis="", gc="", rho="", hwl="",
                  diffkey="", symxkey="", symykey="", **kwargs):
        """
        APDL Command: ANSTOAQWA

        Creates an AQWA-LINE input file from the current ANSYS model.

        Parameters
        ----------
        fname
            AQWA file name. Defaults to Jobname.

        vertaxis
            Axis in the vertical direction:

            Y (or 2)  - Global Y axis.

            Z (or 3)  - Global Z axis (default).

        gc
            Gravitational acceleration. Defaults to 9.81.

        rho
            Density of water. Defaults to 1025.0.

        hwl
            Waterline height in model coordinates. Defaults to 0.0.

        diffkey
            Diffracting model key:

            0 - Create a non-diffracting AQWA model.

            1 - Create a diffracting AQWA model (default).

        symxkey
            Key indicating if model is symmetric about the global XZ plane:

            0 - No symmetry about XZ plane (default).

            1 - Use symmetry about XZ plane. Only include (or select) half the model.

        symykey
            Key indicating if model is symmetric about the global YZ plane:

            0 - No symmetry about YZ plane (default).

            1 - Use symmetry about YZ plane. Only include (or select) half the model.

        Notes
        -----
        This command creates the input file Fname.aqwa for the ANSYS Aqwa
        Multi-Body Hydrodynamics System for diffraction analysis in AQWA-LINE
        from the model currently in the database, based on the currently
        selected set of elements. The selected set must only include the hull
        envelope; no internal structure should be selected.

        There should be a line of nodes defined at the waterline. Only those
        elements that are entirely below the waterline will be specified as
        diffracting. If there are no waterline nodes, there will be no
        diffracting elements at the waterline, which will severely reduce the
        accuracy of the diffraction analysis.

        The translator maps PLANE42, SHELL41, SHELL63, and SHELL181 elements to
        PANELs, and maps PIPE16 and PIPE59 elements to TUBEs. It does not
        recognize any other element types. Any material or geometric properties
        can be used for the shell elements, as AQWA does not need any
        properties at all and the command does not use them. All the shell
        elements below the water must have their normals pointing outward.

        TUBE elements in AQWA have material density, outside diameter, wall
        thickness, added mass, and drag coefficients, so appropriate properties
        should be used in the ANSYS model. PIPE59 elements can have added mass
        and damping coefficients; these will be written to the file. The ANSYS
        program uses the inertia coefficient CM, whereas AQWA uses the added
        mass coefficient CA, where CM = (1 + CA). This correction is made
        automatically.

        In AQWA the vertical axis is always the Z-axis. The command can convert
        a model built with either the Y or Z-axis vertical, but the X-axis must
        be horizontal and should preferably be along the fore/aft axis of the
        vessel.  If the structure is symmetric and you wish to use the symmetry
        options, you must only select one half or one quarter of the model, as
        appropriate. If you model a complete vessel and specify X symmetry, the
        AQWA model will contain two sets of coincident elements.

        If you are working from a model created for a structural analysis, it
        will probably be necessary to remesh the model as the structural mesh
        is most likely finer than needed for a diffraction analysis.

        If you enter this command interactively (with the GUI active) and no
        data is provided for the command options, you will be prompted for
        their values.

        You must verify the completeness and accuracy of the data written.

        """
        command = "ANSTOAQWA,%s,%s,%s,%s,%s,%s,%s,%s" % (str(fname), str(vertaxis), str(gc), str(rho), str(hwl), str(diffkey), str(symxkey), str(symykey))
        self.RunCommand(command, **kwargs)

    def Dset(self, node1="", node2="", node3="", ddev="", **kwargs):
        """
        APDL Command: DSET

        Sets the scale and drawing plane orientation for a digitizing tablet.

        Parameters
        ----------
        node1, node2, node3
            Any three (noncolinear) nodes defining a plane parallel to the
            drawing.  Nodes and actual locations (in any coordinate system)
            must have been previously defined.

        ddev
            Digitizing device type number (device dependent).

        Notes
        -----
        Sets drawing scale size and defines the drawing plane orientation for
        use with a digitizing tablet.  Drawings must be to scale.  Views must
        represent standard orthogonal parallel projections.  The three nodes
        indicated must be digitized [DIG] from the tablet after this command is
        issued.

        """
        command = "DSET,%s,%s,%s,%s" % (str(node1), str(node2), str(node3), str(ddev))
        self.RunCommand(command, **kwargs)

    def Ceintf(self, toler="", dof1="", dof2="", dof3="", dof4="", dof5="",
               dof6="", movetol="", **kwargs):
        """
        APDL Command: CEINTF

        Generates constraint equations at an interface.

        Parameters
        ----------
        toler
            Tolerance about selected elements, based on a fraction of the
            element dimension (defaults to 0.25 (25%)).  Nodes outside the
            element by more than the tolerance are not accepted as being on the
            interface.

        dof1, dof2, dof3, . . . , dof6
            Degrees of freedom for which constraint equations are written.
            Defaults to all applicable DOFs.  DOF1 accepts ALL as a valid
            label, in which case the rest are ignored (all DOFs are applied).

        movetol
            The allowed "motion" of a node (see Note below).  This distance is
            in terms of the element coordinates (-1.0 to 1.0).  A typical value
            is 0.05.  Defaults to 0 (do not move).  MoveTol must be less than
            or equal to TOLER.

        Notes
        -----
        This command can be used to "tie" together two regions with dissimilar
        mesh patterns by generating constraint equations that connect the
        selected nodes of one region to the selected elements of the other
        region.  At the interface between regions, nodes should be selected
        from the more dense mesh region, A, and the elements selected from the
        less dense mesh region, B.  The degrees of freedom of region A nodes
        are interpolated with the corresponding degrees of freedom of the nodes
        on the region B elements, using the shape functions of the region B
        elements.  Constraint equations are then written that relate region A
        and B nodes at the interface.

        The MoveTol field lets the nodes in the previously mentioned region A
        change coordinates when slightly inside or outside the elements of
        region B.  The change in coordinates causes the nodes of region A to
        assume the same surface as the nodes associated with the elements of
        region B.   The constraint equations that relate the nodes at both
        regions of the interface are then written.

        Solid elements with six degrees of freedom should only be interfaced
        with other six degree-of-freedom elements.  The region A nodes should
        be near the region B elements.   A location tolerance based on the
        smallest region B element length may be input.  Stresses across the
        interface are not necessarily continuous.  Nodes in the interface
        region should not have specified constraints.

        Use the CPINTF command to connect nodes by coupling instead of
        constraint equations.  Use the EINTF command to connect nodes by line
        elements.  See also the NSEL and ESEL commands for selecting nodes and
        elements.  See the Mechanical APDL Theory Reference for a description
        of 3-D space used to determine if a node will be considered by this
        command.

        As an alternative to the CEINTF command, you can use contact elements
        and the internal multipoint constraint (MPC) algorithm to tie together
        two regions having dissimilar meshes. See Solid-Solid and Shell-Shell
        Assemblies for more information.

        """
        command = "CEINTF,%s,%s,%s,%s,%s,%s,%s,%s" % (str(toler), str(dof1), str(dof2), str(dof3), str(dof4), str(dof5), str(dof6), str(movetol))
        self.RunCommand(command, **kwargs)

    def View(self, wn="", xv="", yv="", zv="", **kwargs):
        """
        APDL Command: /VIEW

        Defines the viewing direction for the display.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        xv, yv, zv
            The object is viewed along the line from point XV,YV,ZV (in the
            global coordinate system) to the global coordinate system origin.
            For section displays, the cutting plane is assumed to be
            perpendicular to this line.  If XV = WP, modify view to be normal
            to the currently defined working plane.  Defaults to (0,0,1).

        Notes
        -----
        The view line is always normal to the screen.  The view is selected by
        defining a point (in the global Cartesian coordinate system)
        representing a point along the viewing line.  This point, and the
        global Cartesian coordinate system origin, define the line along which
        the object is viewed while looking toward the origin.  Any point along
        the view line may be used, i.e., (1,1,1) and (2,2,2) give the same
        view.  The display orientation may be changed as desired [/ANGLE].  The
        display coordinate system type may be changed (from Cartesian to
        cylindrical, spherical, toroidal, etc.) with the DSYS command.

        This command is valid in any processor.

        """
        command = "/VIEW,%s,%s,%s,%s" % (str(wn), str(xv), str(yv), str(zv))
        self.RunCommand(command, **kwargs)

    def Dsys(self, kcn="", **kwargs):
        """
        APDL Command: DSYS

        Activates a display coordinate system for geometry listings and plots.

        Parameters
        ----------
        kcn
            Coordinate system reference number.  KCN may be 0,1,2 or any
            previously defined local coordinate system number.

        Notes
        -----
          Boundary condition symbols, vector arrows, and element coordinate
        system triads are not transformed to the display coordinate system. The
        display system orientation (for the default view) is X horizontal to
        the right, Y vertical upward, and Z out of the screen (normal).

        Line directions and area directions (/PSYMB,LDIR and /PSYMB,ADIR) are
        not plotted for DSYS >0.

        When you create ANSYS 3-D annotation, the coordinates are stored to the
        database in the DSYS that was active at the time of creation. Changing
        the DSYS does not change the annotation coordinate data in the
        database.

        This command is valid in any processor.

        """
        command = "DSYS,%s" % (str(kcn))
        self.RunCommand(command, **kwargs)

    def Kscon(self, npt="", delr="", kctip="", nthet="", rrat="", **kwargs):
        """
        APDL Command: KSCON

        Specifies a keypoint about which an area mesh will be skewed.

        Parameters
        ----------
        npt
            Keypoint number at concentration.  If NPT = ALL, use all selected
            keypoints.  If remaining fields are blank, remove concentration
            from this keypoint (if unmeshed).  If NPT = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be substituted for NPT.

        delr
            Radius of first row of elements about keypoint.

        kctip
            Crack tip singularity key:

            0 - Do not skew midside nodes, if any, within the element.

            1 - Skew midside nodes of the first row of elements to the 1/4 point for crack tip
                singularity.

        nthet
            Number of elements in circumferential direction (defaults to
            approximately one per 45° (or one per 30°, if KCTIP = 1)).

        rrat
            Ratio of 2nd row element size to DELR (defaults to 0.75, or 0.5 if
            KCTIP = 1).

        Notes
        -----
        Defines a concentration keypoint about which an area mesh will be
        skewed. Useful for modeling stress concentrations and crack tips.
        During meshing, elements are initially generated circumferentially
        about, and radially away, from the keypoint. Lines attached to the
        keypoint are given appropriate divisions and spacing ratios. Only one
        concentration keypoint per unmeshed area is allowed.  Use KSCON,STAT to
        list current status of concentration keypoints. The KSCON command does
        not support 3-D modeling.

        This command is also valid for rezoning.

        """
        command = "KSCON,%s,%s,%s,%s,%s" % (str(npt), str(delr), str(kctip), str(nthet), str(rrat))
        self.RunCommand(command, **kwargs)

    def Mstart(self, label="", key="", **kwargs):
        """
        APDL Command: /MSTART

        Controls the initial GUI components.

        Parameters
        ----------
        label
            Label identifying the GUI component:

            ZOOM - Pan, Zoom, Rotate dialog box, off by default.

            WORK - Offset Working Plane dialog box, off by default.

            WPSET - Working Plane Settings dialog box, off by default.

            ABBR - Edit Toolbar/Abbreviations dialog box, off by default.

            PARM - Scalar Parameters dialog box, off by default.

            SELE - Select Entities dialog box, off by default.

            ANNO - Annotation dialog box, off by default.

            HARD - Hard Copy dialog box, off by default.

            UTIL - Turns on the pre-ANSYS 6.1 (UIDL) GUI, off by default.

        key
            Switch value:

            OFF or 0 - Component does not appear when GUI is initialized.

            ON or 1 - Component appears when GUI is initialized.

        Notes
        -----
        Controls which components appear when the Graphical User Interface
        (GUI) is initially brought up.  This command is valid only before the
        GUI is brought up [/MENU,ON] and is intended to be used in the
        start162.ans file.  It only affects how the GUI is initialized; you can
        always bring up or close any component once you are in the GUI.

        This command is valid only at the Begin Level.

        """
        command = "/MSTART,%s,%s" % (str(label), str(key))
        self.RunCommand(command, **kwargs)

    def Edbx(self, option="", boxid="", xmin="", xmax="", ymin="", ymax="",
             zmin="", zmax="", **kwargs):
        """
        APDL Command: EDBX

        Creates a box shaped volume to be used in a contact definition for
        explicit dynamics.

        Parameters
        ----------
        option
            Label identifying the contact box definition option to be
            performed.

            ADD - Adds a contact box definition (default).

            DELETE - Deletes a contact box definition.

            LIST - Lists each box ID number, and the coordinates that make up each box shaped
                   volume.

        boxid
            User defined list ID number.

        xmin
            Minimum x-coordinate.

        xmax
            Maximum x-coordinate.

        ymin
            Minimum y-coordinate.

        ymax
            Maximum y-coordinate.

        zmin
            Minimum z-coordinate.

        zmax
            Maximum z-coordinate.

        Notes
        -----
        The ANSYS LS-DYNA command EDCGEN allows you to define contact and
        target volumes using box ID numbers BOXID1 and BOXID2, respectively. If
        you use these arguments to define contact volumes, you must first
        define their coordinates using the EDBX command.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDBX,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(boxid), str(xmin), str(xmax), str(ymin), str(ymax), str(zmin), str(zmax))
        self.RunCommand(command, **kwargs)

    def Nlmesh(self, control="", val1="", val2="", val3="", val4="", **kwargs):
        """
        APDL Command: NLMESH

        Controls remeshing in nonlinear adaptivity.

        Parameters
        ----------
        control
            The mesh-quality control to adjust:

            NANG  - Specifies the surface facet dihedral angle threshold. Use this option to retain
                    source mesh geometry features. The dihedral angle is
                    defined by the angle between the normal vectors from two
                    neighboring surface facet sharing an edge. If the dihedral
                    angle is larger than the specified threshold, the edge is
                    treated as soft edge so that the new nodes are forced to
                    the edge.

            VAL1 is the dihedral angle threshold (in degrees) on concave surfaces. VAL2 is the dihedral angle threshold (in degrees) on convex surfaces.  - Default: VAL1 = 15 and VAL2 = 15.

            When NLMESH,EXPL is issued, the VAL1 and VAL2 become the lower bounds of dihedral angles for mesh exploration. Use VAL3 and VAL4 to define the upper bounds of dihedral angles on concave and convex surfaces (respectively) for mesh exploration. - Generally, larger VAL1 and VAL2 values lead to better quality new meshes (and
                              may even repair local tiny facets of poor
                              quality); however, larger values may also smooth
                              out some geometric features, leading to slightly
                              different results and causing possible
                              convergence difficulty in the substeps
                              immediately following remeshing.

            AEDG  - Specifies the edge angle threshold (in degrees). Use this option to split patch
                    segments. The edge angle is the angle between adjacent
                    surface segment edges sharing a node. If the edge angle is
                    larger than the specified threshold (VAL1), the segment
                    splits and the node is automatically treated as a hard node
                    to be retained.

            Default: VAL1 = 10. - Generally, larger VAL1 values improve the quality of the new mesh, but may
                              result in fewer feature nodes. The effect is
                              similar to that of dihedral angles.

            SRAT  - Specifies the global sizing ratio for remeshing.

            Generally, set the lower value (VAL1) to >= 0.7 and the upper value (VAL2) to <= 1.5. Within this range, the model can be refined (< 1.0) or coarsened (> 1.0) up to 3x depending on the number of elements (if performing a remesh of the entire model). - Default: VAL1 = 1.0. The default value results in the new mesh having a similar
                              size as that of the source mesh.

            NLAY - Specifies the number of sculpting layers beginning with detected seed elements.
                   This option helps to detect remeshing regions from whole
                   model.

            Default: VAL1 = 2. - Generally, a larger VAL1 value leads to larger remeshing regions and tends to
                              unite isolated multiple regions. A larger value
                              also tends to result in better remeshing quality
                              (and increases mapping and solution overhead
                              accordingly).

            VAL1 = 0 is not valid, as the remeshing regions would contain only detected seed elements, resulting in many small cavities within remeshing regions (especially if the specified skewness threshold (NLADAPTIVE) is relatively large). - When NLMESH,EXPL is issued, VAL1 becomes the lower bound of mesh exploration.
                              Use VAL2 to define the upper bound for mesh
                              exploration.

            LSRT  - Specifies the local sizing ratio threshold (VAL1). If the length of adjacent
                    segments over that of surface short segments exceeds the
                    specified threshold ratio, the neighboring segments are
                    candidates for local sizing to improve target mesh quality.

            Use local sizing in cases where any of the following conditions exist: - Short edges significantly smaller than average

            Poor surface mesh (triangles) on top edges - Small surface patches composed of few triangles caused by small user-specified
                              dihedral angles.

            Valid values are VAL1 >= 1.0. Default: VAL1 = 1.0. - When NLMESH, EXPL is issued, VAL1 becomes the lower bound of mesh exploration.
                              Use VAL2 to define the upper bound for mesh
                              exploration.

            For more information about this control, see "Notes". - EXPL

            Specifies the nonlinear mesh-exploration behavior. Mesh exploration consists of trying various mesh controls to obtain the best quality mesh during remeshing process. - For more information about this control, see "Notes".

            LIST  - Lists all defined advanced control parameters.

        val1, val2, val3, val4
            Numerical input values that vary according to the specified Control
            option.

        Notes
        -----
        NLMESH is a global control command enabling mesh-quality adjustments
        for remeshing in nonlinear adaptivity. The command can be used when
        components are associated with mesh-quality criteria (NLADAPTIVE with
        Criterion = MESH).

        Issue the NLMESH command only in cases where advanced mesh-quality
        control is desirable for remeshing in nonlinear adaptivity. The
        settings specified by this command apply to all components having mesh-
        quality-based criteria defined.

        Following are LSRT usage examples to help you determine a suitable
        threshold value for the local sizing ratio:

        If the value is only slightly greater than the minimum (and default)
        value of 1.0, local sizing is imposed on all segments. Recommended:
        VAL1 > 1.1.

        If the value is large enough such that no neighboring segments have
        lengths that would cause the threshold ratio to be exceeded, all
        segments are treated as though local sizing is disabled.

        For mesh exploration (NLMESH,EXPL,VAL1):

        VAL1 = 0 -- The exception to the default behavior (no mesh exploration)
        occurs when remeshing fails to create a mesh for the user-specified
        NLMESH input parameters. In this case, mesh exploration is performed as
        though VAL1 = 1, with default NANG upper bounds of 60,60 in order to
        continue the solution, and the lower bounds being user-specified.

        VAL1 = 1 -- The NANG lower and upper bounds must be input; otherwise,
        the command is ignored. The upper bound can be input for NLAY also, but
        the exploration still triggers remeshings with the whole model as seed
        elements.

        VAL1 = 2 -- The NANG lower and upper bounds must be input; otherwise,
        the command is ignored.

        VAL1 = 3 -- An optional upper bound can be specified via LSRT. By
        default, the upper bound is set to be 30 percent more than the (user-
        specified) lower bound.

        Mesh exploration is needed only when it is difficult to obtain a good
        quality mesh via standard remeshing. It is good practice to first try
        less aggressive exploration with VAL1 = 1.

        """
        command = "NLMESH,%s,%s,%s,%s,%s" % (str(control), str(val1), str(val2), str(val3), str(val4))
        self.RunCommand(command, **kwargs)

    def Angle(self, wn="", theta="", axis="", kincr="", **kwargs):
        """
        APDL Command: /ANGLE

        Rotates the display about an axis.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        theta
            Angle (degrees) for changing display orientation (positive,
            counterclockwise about specified axis).

        axis
            Rotation axis:  XS, YS, or ZS (default) for the screen axes;  XM,
            YM, or ZM for the global Cartesian model axes.  ZS is normal to the
            screen; all axes pass through the focus point.

        kincr
            Cumulative rotation key:

            0 - Do not use cumulative successive rotations.

            1 - Use cumulative rotations.  Rotations are relative to the previous rotation.
                View settings (/VIEW) are recalculated.

        Notes
        -----
        Default orientation is YS vertical. When the /XFRM command is set for
        rotation about two points, or for entities, the /ANGLE command is
        functional only for Axis = ZS or ZM and KINCR = 1.

        This command is valid in any processor.

        """
        command = "/ANGLE,%s,%s,%s,%s" % (str(wn), str(theta), str(axis), str(kincr))
        self.RunCommand(command, **kwargs)

    def Areas(self, **kwargs):
        """
        APDL Command: AREAS

        Specifies "Areas" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "AREAS," % ()
        self.RunCommand(command, **kwargs)

    def Sfscale(self, lab="", fact="", fact2="", **kwargs):
        """
        APDL Command: SFSCALE

        Scales surface loads on elements.

        Parameters
        ----------
        lab
            Valid surface load label.  If ALL, use all appropriate labels.

        fact
            Scale factor for the first surface load value.  Zero (or blank)
            defaults  to 1.0.  Use a small number for a zero scale factor.

        fact2
            Scale factor for the second surface load value.  Zero (or blank)
            defaults  to 1.0.  Use a small number for a zero scale factor.

        Notes
        -----
        Scales surface loads (pressure, convection, etc.) in the database on
        the selected elements.  Surface loads are applied with the SF, SFE, or
        SFBEAM commands.  Issue the SFELIST command to list the surface loads.
        Solid model boundary conditions are not scaled by this command, but
        boundary conditions on the FE model are scaled.

        Note:: : Such scaled FE boundary conditions may still be overwritten by
        unscaled solid model boundary conditions if a subsequent boundary
        condition transfer occurs.

        SFSCALE does not work for tabular boundary conditions.

        This command is also valid in PREP7 and in the /MAP processor.

        """
        command = "SFSCALE,%s,%s,%s" % (str(lab), str(fact), str(fact2))
        self.RunCommand(command, **kwargs)

    def Fssect(self, rho="", nev="", nlod="", kbr="", **kwargs):
        """
        APDL Command: FSSECT

        Calculates and stores total linearized stress components.

        Parameters
        ----------
        rho
            In-plane (X-Y) average radius of curvature of the inside and
            outside surfaces of an axisymmetric section.  If zero (or blank), a
            plane or 3-D structure is assumed.  If nonzero, an axisymmetric
            structure is assumed.  Use a suitably large number (see the
            Mechanical APDL Theory Reference) or use -1 for an axisymmetric
            straight section.

        nev
            Event number to be associated with these stresses (defaults to 1).

        nlod
            Loading number to be associated with these stresses (defaults to
            1).

        kbr
            For an axisymmetric analysis (RHO ≠ 0):

            0 - Include the thickness-direction bending stresses

            1 - Ignore the thickness-direction bending stresses

            2 - Include the thickness-direction bending stress using the same formula as the Y
                (axial direction ) bending stress. Also use the same formula
                for the shear stress.

        Notes
        -----
        Calculates and stores the total linearized stress components at the
        ends of a section path [PATH] (as defined by the first two nodes with
        the PPATH command). The path must be entirely within the selected
        elements (that is, there must not be any element gaps along the path).
        Stresses are stored according to the fatigue event number and loading
        number specified.  Locations (one for each node) are associated with
        those previously defined for these nodes [FL] or else they are
        automatically defined.  Stresses are separated into six total
        components (SX through SXZ) and six membrane-plus-bending (SX through
        SXZ) components.  The temperature at each end point and the current
        time are also stored along with the total stress components.
        Calculations are made from the stresses currently in the database (last
        SET or LCASE command).  Stresses are stored as section coordinate
        components if axisymmetric or as global Cartesian coordinate components
        otherwise, regardless of the active results coordinate system [RSYS].
        The FSLIST command may be used to list stresses.  The FS command can be
        used to modify stored stresses.  See also the PRSECT and PLSECT
        commands for similar calculations.

        """
        command = "FSSECT,%s,%s,%s,%s" % (str(rho), str(nev), str(nlod), str(kbr))
        self.RunCommand(command, **kwargs)

    def Mat(self, mat="", **kwargs):
        """
        APDL Command: MAT

        Sets the element material attribute pointer.

        Parameters
        ----------
        mat
            Assign this material number to subsequently defined elements
            (defaults to 1).

        Notes
        -----
        Identifies the material number to be assigned to subsequently defined
        elements.  This number refers to the material number (MAT) defined with
        the material properties [MP].  Material numbers may be displayed
        [/PNUM].

        """
        command = "MAT,%s" % (str(mat))
        self.RunCommand(command, **kwargs)

    def Lang(self, nl1="", p3="", ang="", phit="", locat="", **kwargs):
        """
        APDL Command: LANG

        Generates a straight line at an angle with a line.

        Parameters
        ----------
        nl1
            Number of the line to be hit (touched by the end of the new line).
            If negative, assume P1 (see below) is the second keypoint of the
            line instead of the first.  If NL1 = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).

        p3
            Keypoint at which generated line must end.

        ang
            Angle of intersection of generated line PHIT-P3 with tangent to
            line P1-P2 at PHIT.  If 0 (default), the generated line is tangent
            to NL1 toward end P1; if 90, the generated line is perpendicular to
            NL1.  If 180, the generated line is tangent to NL1 toward end P2.
            ANG can be any value, but is adjusted to the corresponding acute
            angle with respect to LOCAT. See "Notes" for a discussion of
            accuracy.

        phit
            Number to be assigned to keypoint generated at hit location
            (defaults to lowest available keypoint number [NUMSTR]).

        locat
            Approximate location of PHIT in terms of the ratio of the distance
            along the line (NL1) to the length of the line.  LOCAT can range
            from 0 to 1.  If LOCAT is blank, the point will be located with
            less speed and accuracy, and an arbitrary location may result.

        Notes
        -----
        Generates a straight line (PHIT-P3) at an angle (ANG) with a line NL1
        (P1-P2).  The location of PHIT on the line is automatically calculated.
        Line P1-P2 becomes P1-PHIT and new lines PHIT-P2 and PHIT-P3 are
        generated.  Line divisions are set to zero (use LESIZE, etc. to
        modify).

        PHIT is positioned closest to LOCAT for the given angle, ANG.  To
        ensure better performance, it is recommended that LOCAT be input, even
        if it is 0.

        The program uses an iterative procedure to position PHIT.  The
        procedure is not exact, with the result that the actual value of ANG
        will sometimes differ slightly from the specified value.

        """
        command = "LANG,%s,%s,%s,%s,%s" % (str(nl1), str(p3), str(ang), str(phit), str(locat))
        self.RunCommand(command, **kwargs)

    def Madapt(self, errtargt="", nadapt="", nmax="", kplt="", ksmooth="",
               klst="", kcd="", device="", **kwargs):
        """
        APDL Command: MADAPT

        Adaptively meshes and solves an edge-based model.

        Parameters
        ----------
        errtargt
            Target percentage for Zienkiewitz Zhu magnetic flux error (defaults
            to 5).

        nadapt
            Maximum number of adaptive steps (defaults to 5).

        nmax
            Maximum number of elements at which the iterations may continue
            (defaults to 50,000). Limits the number of elements that can be
            chosen for refinement.

        kplt
            Plotting options:

            0 - No plot (default)

            1 - Elements and H

            2 - BERR error estimates

            3 - BDSG, BEPC error estimates

            4 - Adaptive details

        ksmooth
            Smoothing options for refinement

            0 - No postprocessing will be done (default).

            1 - Smoothing will be done.  Node locations may change.

            2 - Smoothing and cleanup will be done.  Existing elements may be deleted, and node
                locations may change.

        klst
            Listing options

            0 - No printing (default)

            1 - Final report

            2 - Report at each iteration step

            3 - Report Zienkiewitz Zhu magnetic errors BERR and BSGD

            4 - Regular details

            5 - Full details

            6 - Refine details

            7 - Track

        kcd
            Allows you to issue a CDWRITE or CDREAD at every iteration.

            0 - Do not issue CDWRITE or CDREAD (default).

            1 - Issue CDWRITE at every iteration (to save every mesh variation). This option
                issues CDWRITE,geom, writing the information to jobnameN.cdb.

            2 - Issue CDREAD at every iteration (to read every mesh variation). Reads from an
                existing jobnameN.cdb.

        device
            Defines the output device for plotting.

            0 - Screen only (default)

            1 - JPEG frames. Each frame is written to a file (jobnameN.jpg by default). See
                /SHOW.

        Notes
        -----
        MADAPT invokes a predefined ANSYS macro for adaptive meshing and
        solution of edge-based magnetic analyses.  The macro causes repeated
        runs of the PREP7, SOLUTION, and POST1 phases of the ANSYS program with
        mesh density refinements based upon the percentage error in energy
        norm.

        The MADAPT command macro requires a second, user-defined macro, which
        must be named madaptld.mac and must reside in the same directory where
        ANSYS is being run. This madaptld macro must contain loads and boundary
        conditions, based on permanent geometry or solid model features (such
        as sides or vertices). Loads specified in the madaptld macro cannot be
        based on node numbers because the node numbers will change throughout
        the refinement process. This secondary macro is required because the
        MADAPT macro process must delete all loads and boundary conditions at
        every refinement step.

        MADAPT refines tetrahedral volume elements based on error. Hexahedra,
        wedges, and pyramids are not refined (see NREFINE).

        This command is also valid at the Begin level.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MADAPT,%s,%s,%s,%s,%s,%s,%s,%s" % (str(errtargt), str(nadapt), str(nmax), str(kplt), str(ksmooth), str(klst), str(kcd), str(device))
        self.RunCommand(command, **kwargs)

    def Sfl(self, line="", lab="", vali="", valj="", val2i="", val2j="",
            **kwargs):
        """
        APDL Command: SFL

        Specifies surface loads on lines of an area.

        Parameters
        ----------
        line
            Line to which surface load applies.  If ALL, apply load to all
            selected lines [LSEL].  If Line = P, graphical picking is enabled
            and all remaining command fields are ignored (valid only in the
            GUI).  A component name may be substituted for Line.

        lab
            Valid surface load label.  Load labels are listed under "Surface
            Loads" in the input table for each element type in the Element
            Reference.

        vali, valj
            Surface load values at the first keypoint (VALI) and at the second
            keypoint (VALJ) of the line, or table name for specifying tabular
            boundary conditions.  If VALJ is blank, it defaults to VALI.  If
            VALJ is zero, a zero is used.  If Lab = CONV, VALI and VALJ are the
            film coefficients and VAL2I and VAL2J are the bulk temperatures.
            To specify a table, enclose the table name in percent signs (%),
            e.g., %tabname%.  Use the *DIM command to define a table.  If Lab =
            CONV and VALI = -N, the film coefficient may be a function of
            temperature and is determined from the HF property table for
            material N [MP].  If Lab = RAD, VALI and VALJ values are surface
            emissivities and VAL2I and VAL2J are ambient temperatures.  The
            temperature used to evaluate the film coefficient is usually the
            average between the bulk and wall temperatures, but may be user
            defined for some elements.  If Lab = RDSF, VALI is the emissivity
            value; the following condition apply: If VALI = -N, the emissivity
            may be a function of the temperature and is determined from the
            EMISS property table for material N [MP]. If Lab = FSIN in a Multi-
            field solver (single or multiple code coupling) analysis, VALI is
            the surface interface number. If Lab = FSIN in a unidirectional
            ANSYS to CFX analysis, VALJ is the surface interface number (not
            available from within the GUI) and VALI is not used unless the
            ANSYS analysis is performed using the Multi-field solver.

        val2i, val2j
            Second surface load values (if any).  If Lab = CONV, VAL2I and
            VAL2J are the bulk temperatures. If Lab = RAD, VAL2I and VAL2J are
            the ambient temperatures. If Lab = RDSF, VAL2I is the enclosure
            number. Radiation will occur between surfaces flagged with the same
            enclosure numbers. If the enclosure is open, radiation will occur
            to the ambient. VAL2I and VAL2J are not used for other surface load
            labels.  If VAL2J is blank, it defaults to VAL2I.  If VAL2J is
            zero, a zero is used.  To specify a table (Lab = CONV), enclose the
            table name in percent signs (%), e.g., %tabname%.  Use the *DIM
            command to define a table.

        Notes
        -----
        Specifies surface loads on the selected lines of area regions.  The
        lines represent either the edges of area elements or axisymmetric shell
        elements themselves.  Surface loads may be transferred from lines to
        elements with the SFTRAN or SBCTRAN commands.  See the SFE command for
        a description of surface loads.  Loads input on this command may be
        tapered.  See the SFGRAD command for an alternate tapered load
        capability.

        You can specify a table name only when using structural (PRES) and
        thermal (CONV [film coefficient and/or bulk temperature], HFLUX), and
        surface emissivity and ambient temperature (RAD) surface load labels.
        VALJ and VAL2J are ignored for tabular boundary conditions.

        This command is also valid in PREP7.

        """
        command = "SFL,%s,%s,%s,%s,%s,%s" % (str(line), str(lab), str(vali), str(valj), str(val2i), str(val2j))
        self.RunCommand(command, **kwargs)

    def Filedisp(self, fname="", ext="", **kwargs):
        """
        APDL Command: FILEDISP

        Specifies the file containing the graphics data.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        Notes
        -----
        Specifies the input file containing the graphics data (defaults to
        File.GRPH).

        """
        command = "FILEDISP,%s,%s" % (str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Accoption(self, activate="", **kwargs):
        """
        APDL Command: ACCOPTION

        Specifies GPU accelerator capability options.

        Parameters
        ----------
        activate
            Activates the GPU accelerator capability within the equation
            solvers.

            Do not use GPU accelerator. - Use GPU accelerator.

        Notes
        -----
        The GPU accelerator capability requires specific hardware to be
        installed on the machine. See the appropriate ANSYS, Inc. Installation
        Guide (Windows or Linux) for a list of supported GPU hardware. Use of
        this capability also requires HPC licensing. For more information, see
        GPU Accelerator Capability in the Parallel Processing Guide.

        The GPU accelerator capability is available for the sparse direct
        solver and the PCG and JCG iterative solvers. Static, buckling, modal,
        full harmonic, and full transient analyses are supported. For buckling
        analyses, the Block Lanczos and Subspace eigensolvers are supported.
        For modal analyses, only the Block Lanczos, PCG Lanczos, Subspace,
        Unsymmetric, and Damped eigensolvers are supported. Activating this
        capability when using other equation solvers or other analysis types
        has no effect.

        The GPU accelerator capability is supported only on the Windows 64-bit
        and Linux 64-bit platforms.

        """
        command = "ACCOPTION,%s" % (str(activate))
        self.RunCommand(command, **kwargs)

    def Deact(self, **kwargs):
        """
        APDL Command: DEACT

        Specifies "Element birth and death" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "DEACT," % ()
        self.RunCommand(command, **kwargs)

    def Geometry(self, **kwargs):
        """
        APDL Command: GEOMETRY

        Specifies "Geometry" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "GEOMETRY," % ()
        self.RunCommand(command, **kwargs)

    def Naxis(self, action="", val="", **kwargs):
        """
        APDL Command: NAXIS

        Generates nodes for general axisymmetric element sections.

        Parameters
        ----------
        action
            Specifies one of the following command behaviors:

            GEN - Generates nodes around the axis of an axisymmetric section (default).

            CLEAR - Clears all nodes around the axis of an axisymmetric section.

            EFACET - Specifies the number of facets per edge between nodal planes and integration
                     planes in the circumferential direction to display using
                     PowerGraphics.  This option is only valid with /ESHAPE,1
                     and RSYS,SOLU commands.

        val
            Tolerance value or number of facets per edge:

            TOLER - When Action = GEN, the tolerance to use for merging the generated nodes around
                    the axis.

            NUM - When Action = EFACET, the number of facets per element edge for element plots:

            AUTO - Use program-chosen facets per edge (default).

            1 - Use 1 facet per edge (default for elements with 9, 10, 11, or 12 nodal planes).
                Shows nodal and integration planes only.

            2 - Use 2 facets per edge (default for elements with 5, 6, 7, or 8 nodal planes,
                and maximum for elements with 9, 10, 11, or 12 nodal planes).

            3 - Use 3 facets per edge (default for elements with 3 or 4 nodal planes, and
                maximum for elements with 6, 7, or 8 nodal planes).

            4 - Use 4 facets per edge (maximum for elements with 5 nodal planes).

            5 - Use 5 facets per edge (maximum for elements with 4 nodal planes).

            6 - Use 6 facets per edge (maximum for elements with 3 nodal planes).

        Notes
        -----
        The NAXIS command generates or clears the nodes for general
        axisymmetric element sections. The command applies to elements SURF159,
        SOLID272, and SOLID273.

        The generate option (Action = GEN) operates automatically on any
        current-technology axisymmetric element. Any nodes within the tolerance
        value (TOLER) of the axis are merged into a single node. The default
        tolerance is 1.0e-4.

        If you want to change the number of nodes, use the clear option (Action
        = CLEAR) before regenerating the nodes.

        To cause the 3-D element plot to appear more like the actual 3-D model,
        use NAXIS,EFACET,NUM, where NUM > 1. In this case, the coordinate
        system specified for displaying element and nodal results (RSYS) must
        be solution (RSYS,SOLU); otherwise, ANSYS resets NUM to 1.

        """
        command = "NAXIS,%s,%s" % (str(action), str(val))
        self.RunCommand(command, **kwargs)

    def Tbtemp(self, temp="", kmod="", **kwargs):
        """
        APDL Command: TBTEMP

        Defines a temperature for a material data table.

        Parameters
        ----------
        temp
            Temperature value (defaults to 0.0 if KMOD is blank).

        kmod
            If blank, TEMP defines a new temperature. (Issue TBLIST to list
            temperatures and data.)

        Notes
        -----
        The TBTEMP command defines a temperature to be associated with the data
        on subsequent TBPT or TBDATA commands.

        The defined temperature remains active until the next TBTEMP command is
        issued.

        Data values must be defined with the temperatures in ascending order.

        This command is also valid in SOLUTION.

        """
        command = "TBTEMP,%s,%s" % (str(temp), str(kmod))
        self.RunCommand(command, **kwargs)

    def Eddc(self, option="", ctype="", cont="", targ="", **kwargs):
        """
        APDL Command: EDDC

        Deletes or deactivates/reactivates contact surface specifications in an
        explicit dynamic analysis.

        Parameters
        ----------
        option
            Option to be performed for contact definition specified by Ctype,
            Cont, and Targ.

            DELE - Delete the specified contact definition (default); valid only in a new
                   analysis.

            DACT - Deactivate the specified contact definition; valid only in a small restart.

            RACT - Reactivate the specified contact definition (which was previously deactivated);
                   valid only in a small restart.

        ctype
            Contact behavior label (see EDCGEN command for valid labels).

        cont
            Component name or part number [EDPART] identifying the contact
            surface.

        targ
            Component name or part number [EDPART] identifying the target
            surface.

        Notes
        -----
        This command allows you to delete or deactivate/reactivate a particular
        contact specification that was defined by EDCGEN. The contact
        definition is identified by Ctype, Cont, and Targ (Note that Cont and
        Targ may not be required for Ctype = AG, SE, ASSC, ESS, and SS). The
        delete option (Option = DELE) permanently deletes the contact from the
        database. Any additional contact parameters defined with the EDCMORE
        command for the contact definition identified on this command will also
        be deleted or deactivated/reactivated.

        You cannot delete contact specifications in an explicit dynamic small
        restart (EDSTART,2). However, you can use Option = DACT to deactivate a
        contact definition that is not needed in the small restart. That
        contact definition may then be reactivated in a subsequent small
        restart by using Option = RACT.

        To delete or deactivate/reactivate all contact specifications for the
        entire model, use EDDC,Option,ALL.

        The EDDC command is not supported in an explicit dynamic full restart
        analysis (EDSTART,3). Thus, you cannot delete, deactivate, or
        reactivate contact specifications in a full restart that were defined
        in the previous analysis.

        This command is also valid in SOLUTION.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDDC,%s,%s,%s,%s" % (str(option), str(ctype), str(cont), str(targ))
        self.RunCommand(command, **kwargs)

    def Prrsol(self, lab="", **kwargs):
        """
        APDL Command: PRRSOL

        Prints the constrained node reaction solution.

        Parameters
        ----------
        lab
            Nodal reaction load type.  If blank, use the first ten of all
            available labels. Valid labels are:

        Notes
        -----
        Prints the constrained node reaction solution for the selected nodes in
        the sorted sequence.  For coupled nodes and nodes in constraint
        equations, the sum of all reactions in the coupled or constraint
        equation set  appears at the primary node of the set.  Results are in
        the global Cartesian coordinate directions unless transformed (RSYS).

        PRRSOL is not valid if any load is applied to a constrained node in the
        direction of the constraint and any of the following is true:

        LCOPER has been used.

        LCASE has been used to read from a load case file.

        The applied loads and constraints in the database are not the ones used
        to create the results data being processed.

        PRRSOL provides the total reaction solution (static, plus damping, plus
        inertial, as appropriate based on the analysis type); however, modal
        reactions include only the static contribution.

        Use PRRFOR instead of PRRSOL with the FORCE command to obtain only the
        static, damping, or inertial components.

        """
        command = "PRRSOL,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Wmore(self, node1="", node2="", ninc="", itime="", inc="", **kwargs):
        """
        APDL Command: WMORE

        Adds more nodes to the starting wave list.

        Parameters
        ----------
        node1, node2, ninc
            Add another node set to the previous starting wave list.  Set is
            NODE1 to NODE2 (defaults to NODE1) in steps of NINC (defaults to
            1).  If NODE1 is negative, delete (instead of add) this node set
            from previous starting wave list.

        itime, inc
            Add other node sets to the same starting wave list by repeating the
            previous node set with NODE1 and NODE2 incremented by INC (defaults
            to 1) each time after the first.  ITIME is the total number of sets
            (defaults to 1) defined with this command.

        Notes
        -----
        Adds more nodes to (or modifies) the previous starting wave list (if
        any) [WSTART].  Repeat WMORE command to add more nodes to the previous
        starting wave list.  Up to 10,000 nodes may be defined (total, for all
        starting waves).

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "WMORE,%s,%s,%s,%s,%s" % (str(node1), str(node2), str(ninc), str(itime), str(inc))
        self.RunCommand(command, **kwargs)

    def Dkdele(self, kpoi="", lab="", **kwargs):
        """
        APDL Command: DKDELE

        Deletes DOF constraints at a keypoint.

        Parameters
        ----------
        kpoi
            Keypoint for which constraint is to be deleted.  If ALL, delete for
            all selected keypoints [KSEL].  If KPOI = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be substituted for KPOI.

        lab
            Valid degree of freedom label.  If ALL, use all appropriate labels.
            Structural labels:  UX, UY, or UZ (displacements); ROTX, ROTY, or
            ROTZ (rotations); WARP (warping).  Thermal labels: TEMP, TBOT, TE2,
            TE3, . . ., TTOP (temperature). Acoustic labels:  PRES (pressure);
            UX, UY, or UZ (displacements for FSI coupled elements). Electric
            label:  VOLT (voltage).  Magnetic labels:  MAG (scalar magnetic
            potential); AX, AY, or AZ (vector magnetic potentials).  Diffusion
            label: CONC (concentration).

        Notes
        -----
        Deletes the degree of freedom constraints (and all corresponding finite
        element constraints) at a keypoint.  See the DDELE command for details.

        This command is also valid in PREP7.

        """
        command = "DKDELE,%s,%s" % (str(kpoi), str(lab))
        self.RunCommand(command, **kwargs)

    def Edrd(self, option="", part="", mrb="", **kwargs):
        """
        APDL Command: EDRD

        Switches a part from deformable to rigid or from rigid to deformable in
        an explicit dynamic analysis.

        Parameters
        ----------
        option
            Label identifying the option to be performed.

            D2R - Change specified part from deformable to rigid (default).

            R2D - Change specified part from rigid to deformable. Use this option to switch a
                  part back to a deformable state after it has been changed to
                  rigid using EDRD,D2R.

            LIST - List parts that are flagged to change from deformable to rigid or rigid to
                   deformable.

        part
            Part number for part to be changed (no default).

        mrb
            Part number of the master rigid body to which the part is merged.
            MRB is used only if Option = D2R. If MRB = 0 (which is the
            default), the part becomes an independent rigid body.

        Notes
        -----
        This command is valid in a new explicit dynamic analysis or in a
        restart. It is only possible to switch parts (D2R or R2D) in a restart
        if part switching is first activated in the original analysis. If part
        switching is not required in the original analysis but will be used in
        the restart, you must issue EDRD,D2R with no further arguments in the
        original analysis. You can use the EDRI command to define inertia
        properties for newly created rigid bodies (D2R).

        Parts that are defined as rigid using EDMP,RIGID are permanently rigid
        and cannot be changed to deformable.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDRD,%s,%s,%s" % (str(option), str(part), str(mrb))
        self.RunCommand(command, **kwargs)

    def Vcol(self, ncol1="", ncol2="", **kwargs):
        """
        APDL Command: *VCOL

        Specifies the number of columns in matrix operations.

        Parameters
        ----------
        ncol1
            Number of columns to be used for Par1 with *MXX operations.
            Defaults to whatever is needed to fill the result array.

        ncol2
            Number of columns to be used for Par2 with *MXX operations.
            Defaults to whatever is needed to fill the result array.

        Notes
        -----
        Specifies the number of columns to be used in array parameter matrix
        operations.  The size of the submatrix used is determined from the
        upper left starting array element (defined on the operation command) to
        the lower right array element (defined by the number of columns on this
        command and the number of rows on the *VLEN command).

        The default NCOL is calculated from the maximum number of columns of
        the result array (the *DIM column dimension) minus the starting
        location + 1.  For example, *DIM,R,,1,10 and a starting location of
        R(1,7) gives a default of 4 columns ( starting with R(1,7), R(1,8),
        R(1,9), and R(1,10)).  Repeat operations automatically terminate at the
        last column of the result array.  Existing values in the rows and
        columns of the results matrix remain unchanged where not overwritten by
        the requested input or operation values.

        The column control settings are reset to the defaults after each *MXX
        operation.  Use *VSTAT to list settings.

        This command is valid in any processor.

        """
        command = "*VCOL,%s,%s" % (str(ncol1), str(ncol2))
        self.RunCommand(command, **kwargs)

    def Dig(self, node1="", node2="", ninc="", **kwargs):
        """
        APDL Command: DIG

        Digitizes nodes to a surface.

        Parameters
        ----------
        node1, node2, ninc
            Digitize nodes NODE1 through NODE2 in steps of NINC.  NODE2
            defaults to NODE1 and NINC defaults to 1.

        Notes
        -----
        Digitizes nodes to the surface defined by the DSURF command.  The nodes
        indicated must be digitized from the tablet after this command is
        given.  The program must be in the interactive mode and the graphics
        terminal show option [/SHOW] must be active.  The global Cartesian
        coordinates of the nodes are stored.

        """
        command = "DIG,%s,%s,%s" % (str(node1), str(node2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Andscl(self, nfram="", delay="", ncycl="", **kwargs):
        """
        APDL Command: ANDSCL

        Produces an animated sequence of a deformed shape.

        Parameters
        ----------
        nfram
            Number of frames captured (defaults to 5).

        delay
            Time delay during animation (defaults to 0.1 seconds).

        ncycl
            Number of animation cycles (defaults to 5).  Available in non-UI
            mode only.

        Notes
        -----
        ANDSCL involves an ANSYS macro which produces an animation of
        displacement of the last plot action command (for example, PLDISP).
        This command operates only on graphic display platforms supporting the
        /SEG command.  After executing ANDSCL, you can replay the animated
        sequence by issuing the ANIM command.

        The command functions only in the postprocessor.

        """
        command = "ANDSCL,%s,%s,%s" % (str(nfram), str(delay), str(ncycl))
        self.RunCommand(command, **kwargs)

    def Lsba(self, nl="", na="", sepo="", keepl="", keepa="", **kwargs):
        """
        APDL Command: LSBA

        Subtracts areas from lines.

        Parameters
        ----------
        nl
            Line (or lines, if picking is used) to be subtracted from.  If ALL,
            use all selected lines.  If NL = P, graphical picking is enabled
            and all remaining command fields are ignored (valid only in the
            GUI).  A component name may also be substituted for NL.

        na
            Area (or areas, if picking is used) to be subtracted.  If ALL, use
            all selected areas.  A component name may also be substituted for
            NA.

        sepo
            Behavior if the intersection of the lines and the areas is a
            keypoint or keypoints:

            (blank) - The resulting lines will share keypoint(s) where they touch.

            SEPO - The resulting lines will have separate, but coincident keypoint(s) where they
                   touch.

        keepl
            Specifies whether NL lines are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete NL lines after LSBA operation (override BOPTN command settings).

            KEEP - Keep NL lines after LSBA operation (override BOPTN command settings).

        keepa
            Specifies whether NA areas are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete areas after LSBA operation (override BOPTN command settings).

            KEEP - Keep areas after LSBA operation (override BOPTN command settings).

        Notes
        -----
        Generates new lines by subtracting the regions common to both NL lines
        and NA areas (the intersection) from the NL lines.  The intersection
        can be a line(s) or keypoint(s).  If the intersection is a keypoint and
        SEPO is blank, the NL line is divided at the keypoint and the resulting
        lines will be connected, sharing a common keypoint where they touch.
        If SEPO is set to SEPO, NL is divided into two unconnected lines with
        separate keypoints where they touch.  See the Modeling and Meshing
        Guide for an illustration.  See the BOPTN command for an explanation of
        the options available to Boolean operations.  Element attributes and
        solid model boundary conditions assigned to the original entities will
        not be transferred to the new entities generated.

        """
        command = "LSBA,%s,%s,%s,%s,%s" % (str(nl), str(na), str(sepo), str(keepl), str(keepa))
        self.RunCommand(command, **kwargs)

    def Sed(self, sedx="", sedy="", sedz="", cname="", **kwargs):
        """
        APDL Command: SED

        Defines the excitation direction for response spectrum and PSD
        analyses.

        Parameters
        ----------
        sedx, sedy, sedz
            Global Cartesian coordinates of a point that defines a line
            (through the origin) corresponding to the excitation direction.
            For example: 0.0, 1.0, 0.0 defines global Y as the spectrum
            direction.

        cname
            The component name corresponding to the group of excited nodes.
            Only applies to base excitation multi-point response spectrum
            analysis (SPOPT, MPRS) and power spectral density analysis (SPOPT,
            PSD). Defaults to no component.

        Notes
        -----
        In single-point response spectrum analysis (SPOPT,SPRS), the excitation
        direction without rocking (ROCK) is normalized to one so that the SEDX,
        SEDY, and SEDZ values do not scale the spectrum.  The excitation
        direction with rocking is not normalized. The SEDX, SEDY, and SEDZ
        values must be consistent with the OMX, OMY, and OMZ values on the ROCK
        command. The calculated direction then scales the spectrum. For more
        information, see Participation Factors and Mode Coefficients.

        In multi-point response spectrum analysis (SPOPT,MPRS) and power
        spectral density analysis (SPOPT,PSD), the excitation direction is
        normalized to one so that the SEDX, SEDY, and SEDZ values do not scale
        the spectrum.  The component name (Cname) is required. The constraints
        corresponding to the excitation direction are applied to the component
        nodes.

        This command is also valid in PREP7.

        """
        command = "SED,%s,%s,%s,%s" % (str(sedx), str(sedy), str(sedz), str(cname))
        self.RunCommand(command, **kwargs)

    def Cycexpand(self, wn="", option="", value1="", value2="", **kwargs):
        """
        APDL Command: /CYCEXPAND

        Graphically expands displacements, stresses and strains of a cyclically
        symmetric model.

        Parameters
        ----------
        wn
            The window number to which the expansion applies. Valid values are
            1 through 5. The default value is 1. The window number applies only
            to the AMOUNT argument.

        option
            One of the following options:

            ON - Activates cyclic expansion using the previous settings (if any). If no previous
                 settings exist, this option activates the default settings.

            DEFAULT - Resets cyclic expansion to the default settings.

            OFF - Deactivates cyclic expansion. This option is the default.

            STATUS - Lists the current cyclic expansion settings.

            AMOUNT - The number of repetitions or the total angle.

            Value1 - NREPEAT

            Value2 - The number of repetitions. The default is the total number of sectors in 360
                     degrees.

            or - Value1

            ANGLE - Value2

            The total angle in degrees. The default is 360. - WHAT

            A specified portion or subset of the model to expand: - Value1

            The component name of the elements to expand. The default is all selected components. - EDGE

            Sector edge display key. - -1

            Suppresses display of edges between sectors even if the cyclic count varies between active windows. - Caution:  Plots with fewer than the maximum number of repetitions may have
                              missing element faces at the sector boundaries.

            0 or OFF - Averages stresses or strains across sector boundaries. This value is the
                       default (although the default reverts to 1 or ON if the
                       cyclic count varies between active windows).

            1 or ON - No averaging of stresses or strains occurs and sector boundaries are shown on
                      the plot.

            PHASEANG - The phase angle shift:

            Value1 - The phase angle shift in degrees. The valid range is 0 through 360. The default
                     is 0. For a full harmonic solution, this value is
                     typically the phase angle obtained via the CYCPHASE
                     command. If Value1 = AMPLITUDE (or if Value1 ≥ 360), the
                     amplitude is supplied. The amplitude solution for non-
                     component results (such as equivalent stress) are not
                     valid. For a mode-superposition harmonic solution, if
                     Value1 = SWEEP, the maximum values across a phase angle
                     sweep are supplied.

        Notes
        -----
        In preprocessing, the /CYCEXPAND command verifies a cyclically
        symmetric model by graphically expanding it partially or through the
        full 360 degrees.

        For the postprocessing plot nodal solution (PLNSOL) operation, the
        command graphically expands displacements, stresses and strains of a
        cyclically symmetric model partially or though the full 360 degrees by
        combining the real (original nodes and elements) and imaginary
        (duplicate nodes and elements) parts of the solution.

        For the print nodal solution (PRNSOL) operation, the command expands
        the printed output of displacements or stresses on a sector-by-sector
        basis.

        Use of the /CYCEXPAND command does not change the database. The command
        does not modify the geometry, nodal displacements or element stresses.

        The command affects element and result plots only. It has no effect on
        operations other than plot element solution (PLESOL), plot nodal
        solution (PLNSOL), print nodal solution (PRNSOL), and calculate
        harmonic solution (CYCCALC). Operations other than PLESOL, PLNSOL,
        PRNSOL, or CYCCALC work on the unprocessed real and imaginary parts of
        a cyclic symmetry solution

        If you issue a /CYCEXPAND,,OFF command, you cannot then expand the
        model by simply issuing another  /CYCEXPAND command  (for example, to
        specify an NREPEAT value for the number of repetitions). In such a
        case, you must specify /CYCEXPAND,,ON, which activates expansion using
        the previous settings (if any) or the default settings.

        The command requires PowerGraphics and will turn PowerGraphics on
        (/GRAPHICS,POWER) if not already active. Any setting which bypasses
        PowerGraphics (for example, /PBF) also bypasses cyclic expansion; in
        such cases, the /CYCEXPAND command displays unprocessed real and
        imaginary results.

        The CYCPHASE command uses full model graphics (/GRAPHICS,FULL) to
        compute peak values. Because of this, there may be slight differences
        between max/min values obtained with CYCPHASE, and those obtained via
        /CYCEXPAND, which uses power graphics (/GRAPHICS,POWER).

        For PHASEANG = AMPLITUDE (or 360) with a cyclic full harmonic solution,
        the only appropriate coordinate system is the solution coordinate
        system (RSYS,SOLU)

        To learn more about analyzing a cyclically symmetric structure, see the
        Cyclic Symmetry Analysis Guide.

        """
        command = "/CYCEXPAND,%s,%s,%s,%s" % (str(wn), str(option), str(value1), str(value2))
        self.RunCommand(command, **kwargs)

    def Esort(self, item="", lab="", order="", kabs="", numb="", **kwargs):
        """
        APDL Command: ESORT

        Sorts the element table.

        Parameters
        ----------
        item
            Label identifying the item:

            ETAB - (currently the only Item available)

        lab
            element table label:

            Lab - Any user-defined label from the ETABLE command (input in the Lab field of the
                  ETABLE command).

        order
            Order of sort operation:

            0 - Sort into descending order.

            1 - Sort into ascending order.

        kabs
            Absolute value key:

            0 - Sort according to real value.

            1 - Sort according to absolute value.

        numb
            Number of elements (element table rows) to be sorted in ascending
            or descending order (ORDER) before sort is stopped (remainder will
            be in unsorted sequence) (defaults to all elements).

        Notes
        -----
        The element table rows are sorted based on the column containing the
        Lab values. Use EUSORT to restore the original order. If ESORT is
        specified with PowerGraphics on [/GRAPHICS,POWER], then the nodal
        solution results listing [PRNSOL] will be the same as with the full
        graphics mode [/GRAPHICS,FULL].

        """
        command = "ESORT,%s,%s,%s,%s,%s" % (str(item), str(lab), str(order), str(kabs), str(numb))
        self.RunCommand(command, **kwargs)

    def Nooffset(self, label="", **kwargs):
        """
        APDL Command: NOOFFSET

        Prevents the CDREAD command from offsetting specified data items

        Parameters
        ----------
        label
            Specifies items not to be offset.

            NODE - Node numbers

            ELEM - Element numbers

            KP - Keypoint numbers

            LINE - Line numbers

            AREA - Area numbers

            VOLU - Volume numbers

            MAT - Material numbers

            TYPE - Element type numbers

            REAL - Real constant numbers

            CSYS - Coordinate system numbers

            SECN - Section numbers

            CP - Coupled set numbers

            CE - Constraint equation numbers

            CLEAR - All items will be offset

            STATUS - Shows which items are specified notto be offset.

        Notes
        -----
         The NOOFFSET command specifies data items not to be offset by a set of
        data read from a CDREAD command.

        """
        command = "NOOFFSET,%s" % (str(label))
        self.RunCommand(command, **kwargs)

    def Esla(self, type="", **kwargs):
        """
        APDL Command: ESLA

        Selects those elements associated with the selected areas.

        Parameters
        ----------
        type
            Label identifying the type of element select:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        Notes
        -----
        Selects area elements belonging to meshed [AMESH], selected [ASEL]
        areas.

        This command is valid in any processor.

        """
        command = "ESLA,%s" % (str(type))
        self.RunCommand(command, **kwargs)

    def Usrdof(self, action="", dof1="", dof2="", dof3="", dof4="", dof5="",
               dof6="", dof7="", dof8="", dof9="", dof10="", **kwargs):
        """
        APDL Command: USRDOF

        Specifies the degrees of freedom for the user-defined element USER300.

        Parameters
        ----------
        action
            One of the following command operations:

            DEFINE - Specify the degrees of freedom (DOFs). This value is the default.

            LIST - List all previously specified DOFs.

            DELETE -  Delete all previously specified DOFs.

        dof1, dof2, dof3, . . . , dof10
            The list of DOFs.

        Notes
        -----
        The USRDOF command specifies the degrees of freedom for the user-
        defined element USER300.

        Although you can intersperse other commands as necessary for your
        analysis, issue the USRDOF command as part of the following general
        sequence of commands:

        Issue the ET command for element USER300, followed by the related TYPE
        command.

        Issue both the USRELEM and USRDOF commands (in either order).

        Define your element using USER300.

        The DOF list (DOF1 through DOF10) can consist of up to 10 DOFs. Use any
        valid and appropriate DOF (such as UX, UY, UZ, ROTX, ROTY, ROTZ, AX,
        AY, AZ, VX, VY, VZ, PRES, WARP, TEMP, VOLT, MAG, EMF, and CURR).

        You can specify a maximum of 10 DOFs per USRDOF command. To define
        additional DOFs, issue the command again.

        The maximum number of DOFs for a user-defined element--the number of
        nodes times the number of DOFs per node--cannot exceed 480.

        To learn more about user-defined elements, see Creating a New Element
        in the Programmer's Reference.

        """
        command = "USRDOF,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(action), str(dof1), str(dof2), str(dof3), str(dof4), str(dof5), str(dof6), str(dof7), str(dof8), str(dof9), str(dof10))
        self.RunCommand(command, **kwargs)

    def Czdel(self, grp1="", grp2="", grp3="", **kwargs):
        """
        APDL Command: CZDEL

        Edits or clears cohesive zone sections.

        Parameters
        ----------
        grp1
            Initial group of cohesive zone elements to be deleted.

        grp2
            Final group of cohesive zone elements to be deleted.

        grp3
            Increment for selected groups.

        Notes
        -----
        The CZDEL command edits or deletes the interface elements and nodes,
        along with the associated changes made to the underlying plane or solid
        elements created during a previous CZMESH operation.

        Each CZMESH operation will create groups of elements and nodes with
        component names in the format CZME_EL01 (elements) and CZME_ND01
        (nodes). The final number of this format will be the number used for
        grp1 and grp2. If grp1 = ALL, all nodes and elements created by the
        CZMESH command will be deleted. After using CZDEL, all the user-defined
        components will be unselected.

        The CZDEL command is valid for structural analyses only.

        """
        command = "CZDEL,%s,%s,%s" % (str(grp1), str(grp2), str(grp3))
        self.RunCommand(command, **kwargs)

    def Nsle(self, type="", nodetype="", num="", **kwargs):
        """
        APDL Command: NSLE

        Selects those nodes attached to the selected elements.

        Parameters
        ----------
        type
            Label identifying the type of node select:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        nodetype
            Label identifying type of nodes to consider when selecting:

            ALL - Select all nodes of the selected elements (default).

            ACTIVE - Select only the active nodes. An active node is a node that contributes DOFs to
                     the model.

            INACTIVE - Select only inactive  nodes (such as orientation or radiation).

            CORNER - Select only corner nodes.

            MID - Select only midside nodes.

            POS - Select nodes in position Num.

            FACE - Select nodes on face Num.

        num
            Position or face number for NodeType = POS or FACE.

        Notes
        -----
        NSLE selects NodeType nodes attached to the currently-selected set of
        elements. Only nodes on elements in the currently-selected element set
        can be selected.

        Note:: : When using degenerate hexahedral elements, NSLE, U,CORNER and
        NSLE,S,MID will not select the same set of nodes because some nodes
        appear as both corner and midside nodes.

        This command is valid in any processor.

        """
        command = "NSLE,%s,%s,%s" % (str(type), str(nodetype), str(num))
        self.RunCommand(command, **kwargs)

    def Tsres(self, array="", **kwargs):
        """
        APDL Command: TSRES

        Defines an array of key times at which the time-stepping strategy
        changes.

        Parameters
        ----------
        array
            Identifies an Nx1x1 array parameter containing the key times at
            which the heat transfer time-stepping strategy changes (the time
            step is reset to the initial time step based on DELTIM or NSUBST
            settings).  The array name must be enclosed by % signs (e.g.,
            %array%).  See *DIM for more information on array parameters.

        Notes
        -----
        Time values in the array parameter must be in ascending order and must
        not exceed the time at the end of the load step as defined on the TIME
        command. The time increment between time points in the array list must
        be larger than the initial time step defined on the DELTIM or NSUBST
        command.  Time values must also fall between the beginning and ending
        time values of the load step.  For multiple load step problems, you
        must either change the parameter values to fall between the beginning
        and ending time values of the load step or reissue the command with a
        new array parameter.  To clear the array parameter specification, issue
        TSRES,ERASE.  Results can be output at the requested time points if the
        array or time values in the array are also specified in the OUTRES
        command using FREQ=%array%.  Use this command to reset the time-
        stepping strategy within a load step. You may need to reset the time-
        stepping strategy when using tabular time-varying boundary conditions.

        See Steady-State Thermal Analysis of the Thermal Analysis Guide  for
        more information on applying boundary conditions via tabular input.
        See Transient Thermal Analysis of the Thermal Analysis Guide for more
        information on defining the key time array.

        """
        command = "TSRES,%s" % (str(array))
        self.RunCommand(command, **kwargs)

    def Sfa(self, area="", lkey="", lab="", value="", value2="", **kwargs):
        """
        APDL Command: SFA

        Specifies surface loads on the selected areas.

        Parameters
        ----------
        area
            Area to which surface load applies.  If ALL, apply load to all
            selected areas [ASEL].  If Area = P, graphical picking is enabled
            and all remaining command fields are ignored (valid only in the
            GUI).  A component may be substituted for Area.

        lkey
            Load key associated with surface load (defaults to 1).  Load keys
            (1,2,3, etc.) are listed under "Surface Loads" in the input data
            table for each element type in the Element Reference.  LKEY is
            ignored if the area is the face of a volume region meshed with
            volume elements.

        lab
            Valid surface load label.  Load labels are listed under "Surface
            Loads" in the input table for each area type in the Element
            Reference.

        value
            Surface load value or table name reference for specifying tabular
            boundary conditions.

        value2
            Second surface load value (if any).

        Notes
        -----
        Surface loads may be transferred from areas to elements with the SFTRAN
        or SBCTRAN commands.  See the SFGRAD command for an alternate tapered
        load capability.

        Tabular boundary conditions (VALUE = %tabname% and/or VALUE2 =
        %tabname%) are available for the following surface load labels (Lab)
        only: : PRES (real and/or imaginary components), CONV (film coefficient
        and/or bulk temperature) or HFLUX, and RAD (surface emissivity and
        ambient temperature). Use the *DIM command to define a table.

        This command is also valid in PREP7.

        """
        command = "SFA,%s,%s,%s,%s,%s" % (str(area), str(lkey), str(lab), str(value), str(value2))
        self.RunCommand(command, **kwargs)

    def Usrcal(self, rnam1="", rnam2="", rnam3="", rnam4="", rnam5="",
               rnam6="", rnam7="", rnam8="", rnam9="", **kwargs):
        """
        APDL Command: USRCAL

        Allows user-solution subroutines to be activated or deactivated.

        Parameters
        ----------
        rnam1, rnam2, rnam3, . . . , rnam9
            User-defined solution subroutine names to be activated.  Up to nine
            may be defined on one command or multiple commands may be used.  If
            Rnam1 = ALL, activate all valid user subroutines.   If Rnam1 =
            NONE, deactivate all valid user subroutines.  All characters are
            required:

            USREFL - Allows user defined scalar field (body force) loads.

            USERCV - Allows user defined convection (surface) loads.

            USERPR - Allows user defined pressure (surface) loads.

            USERFX - Allows user-defined heat flux (surface) loads.

            USERCH - Allows user-defined charge density (surface) loads.

            USERFD - Computes the complex load vector for the frequency domain logic.

            USEROU - Allows user supplied element output.

            USERMC - Allows user control of the hygrothermal growth).

            USOLBEG - Allows user access before each solution.

            ULDBEG - Allows user access before each load step.

            USSBEG - Allows user access before each substep.

            UITBEG - Allows user access before each equilibrium iteration.

            UITFIN - Allows user access after each equilibrium iteration.

            USSFIN - Allows user access after each substep.

            ULDFIN - Allows user access after each load step.

            USOLFIN - Allows user access after each solution.

            UANBEG - Allows user access at start of run.

            UANFIN - Allows user access at end of run.

            UELMATX - Allows user access to element matrices and load vectors.

            UTIMEINC - Allows a user-defined time step, overriding the program-determined time step.

            UCNVRG - Allows user-defined convergence checking, overriding the program-determined
                     convergence.

        Notes
        -----
        Allows certain user-solution subroutines to be activated or deactivated
        (system-dependent).  This command only affects the subroutines named.
        Other user subroutines (such as user elements, user creep, etc.) have
        their own activation controls described with the feature.

        The routines are commented and should be listed after performing a
        custom installation from the distribution media for more details.  See
        also the Advanced Analysis Guide for a general description of user-
        programmable features.

        Users must have system permission, system access, and knowledge to
        write, compile, and link the appropriate subroutines into the program
        at the site where it is to be run.  All routines should be written in
        FORTRAN. (For more information on FORTRAN compilers please refer to
        either the ANSYS, Inc. Windows Installation Guide or the ANSYS, Inc.
        Linux Installation Guide for details specific to your platform or
        operating system.) Issue USRCAL,STAT to list the status of these user
        subroutines.  Since a user-programmed subroutine is a nonstandard use
        of the program, the verification of any ANSYS run incorporating these
        commands is entirely up to the user.  In any contact with ANSYS
        customer support regarding the performance of a custom version of the
        ANSYS program, you should explicitly state that a user programmable
        feature has been used.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "USRCAL,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(rnam1), str(rnam2), str(rnam3), str(rnam4), str(rnam5), str(rnam6), str(rnam7), str(rnam8), str(rnam9))
        self.RunCommand(command, **kwargs)

    def Cgrow(self, action="", par1="", par2="", **kwargs):
        """
        APDL Command: CGROW

        Defines crack-growth information

        Parameters
        ----------
        action
            Specifies the action for defining or manipulating crack-growth
            data:

            NEW - Initiate a new set of crack-growth simulation data (default).

            CID - Specify the crack-calculation (CINT) ID for energy-release rates to be used in
                  the fracture criterion calculation.

            FCOPTION - Specify the fracture criterion for crack-growth/delamination.

            CPATH - Specify the element component for crack growth.

            DTIME - Specify the initial time step for crack growth.

            DTMIN - Specify the minimum time step for crack growth.

            DTMAX - Specify the maximum time step for crack growth.

            FCRAT - Fracture criterion ratio (fc).

            STOP - Stops the analysis when the specified maximum crack extension is reached.

            METHOD - Define the method of crack propagation.

        Notes
        -----
        When Action = NEW, the CGROW command initializes a crack-growth
        simulation set. Subsequent CGROW commands define the parameters
        necessary for the simulation.

        For multiple cracks, issue multiple CGROW,NEW commands (and any
        subsequent CGROW commands necessary to define the parameters) for each
        crack.

        If the analysis is restarted (ANTYPE,,RESTART), the CGROW command must
        be re-issued.

        For VCCT crack growth:

        Crack-growth element components must use the crack tip nodes as the
        starting nodes of the crack path.

        Fracture criteria (Action = FCOPTION) use energy-release rates
        calculated via VCCT technology (CINT,TYPE,VCCT). For information about
        the fracture criteria available, see Fracture Criteria in the Fracture
        Analysis Guide or the documentation for the TB,CGCR command.

        For XFEM-based crack growth:

        When using XFEM to grow the crack (CGROW,METHOD,XFEM), the crack
        specification originates via the XFENRICH and XFDATA commands.

        In an XFEM-based crack growth analysis, Action = CPATH,DTMIN,DTMAX and
        STOP have no effect.

        """
        command = "CGROW,%s,%s,%s" % (str(action), str(par1), str(par2))
        self.RunCommand(command, **kwargs)

    def Bsm2(self, val1="", val2="", t="", **kwargs):
        """
        APDL Command: BSM2

        Specifies the bending curvature and moment relationship in plane XY for
        beam sections.

        Parameters
        ----------
        val1
            Curvature component (κ2).

        val2
            Bending moment component (M2).

        t
            Temperature.

        Notes
        -----
        The behavior of beam elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        The BSM2 command, one of several nonlinear general beam section
        commands, specifies the bending curvature and moment relationship for
        plane XY of a beam section. The section data defined is associated with
        the section most recently defined (via the SECTYPE command).

        Unspecified values default to zero.

        Related commands are BSAX, BSM1, BSTQ, BSS1, BSS2, BSMD, and BSTE.

        For complete information, see Using Nonlinear General Beam Sections.

        """
        command = "BSM2,%s,%s,%s" % (str(val1), str(val2), str(t))
        self.RunCommand(command, **kwargs)

    def Prfar(self, lab="", option="", phi1="", phi2="", nph1="", theta1="",
              theta2="", ntheta="", val1="", val2="", val3="", **kwargs):
        """
        APDL Command: PRFAR

        Prints pressure far fields and far field parameters.

        Parameters
        ----------
        lab
            Parameters to print:

            PRES - Acoustic parameters

            PROT - Acoustic parameters with the y-axis rotated extrusion

        option
            Print option, based on the specified print parameter type:

        phi1, phi2
            Starting and ending φ angles (degrees) in the spherical coordinate
            system. Defaults to 0.

        nphi
            Number of divisions between the starting and ending φ angles for
            data computations. Defaults to 0.

        theta1, theta2
            Starting and ending θ angles (degrees) in the spherical coordinate
            system. Defaults to 0 in 3-D and 90 in 2-D.

        ntheta
            Number of divisions between the starting and ending θ angles for
            data computations. Defaults to 0.

        val1
            Radius of the sphere surface. Used only when Option = SUMC, PHSC,
            SPLC, SPAC, PSCT, or TSCT.

        val2
            When Option = SPLC or SPAC: Reference rms sound pressure. Defaults
            to 2x10-5 Pa.

        val3
            When Lab = PRES: Thickness of 2-D model extrusion in the z
            direction (no default).

        Notes
        -----
        The PRFAR command prints pressure far fields and far field parameters
        as    determined by the equivalent source principle. Use this command
        to print pressure and acoustic    parameters. See the HFSYM command for
        the model symmetry and the     HFANG command for spatial radiation
        angles.

        To retrieve saved equivalent source data, issue the
        SET,Lstep,Sbstep,,REAL command.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PRFAR,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(lab), str(option), str(phi1), str(phi2), str(nph1), str(theta1), str(theta2), str(ntheta), str(val1), str(val2), str(val3))
        self.RunCommand(command, **kwargs)

    def Number(self, nkey="", **kwargs):
        """
        APDL Command: /NUMBER

        Specifies whether numbers, colors, or both are used for displays.

        Parameters
        ----------
        nkey
            Numbering style:

             0  - Color (terminal dependent) the numbered items and show numbers.

             1  - Color the numbered items.  Do not show the numbers.

             2  - Show the numbers.  Do not color the items.

            -1  - Do not color the items or show the numbers. For contour plots, the resulting
                  display will vary (see below).

        Notes
        -----
        Specifies whether numbers, colors, or both are used for numbering
        displays [/PNUM] of nodes, elements, keypoints, lines, areas, and
        volumes.

          Shading is also available for terminals configured with more than 4
        color planes [/SHOW].  Color automatically appears for certain items
        and may be manually controlled (off or on) for other items.

        When you suppress color (NKEY = -1) your contour plots will produce
        different results, depending on your graphics equipment. For non-3-D
        devices (X11, Win32, etc.) your contour plot will be white (no color).
        For 3-D devices, such as OpenGL, the resulting display will be in
        color.

        The following items are automatically given discrete colors:  Boundary
        condition symbols [/PBC], curves on graph displays, and distorted
        geometry on postprocessing displays.  Contour lines in postprocessing
        displays are automatically colored based upon a continuous, rather than
        a discrete, spectrum so that red is associated with the highest contour
        value.  On terminals with raster capability [/SHOW], the area between
        contour lines is filled with the color of the higher contour.

        Explicit entity colors or the discrete color mapping may be changed
        with the /COLOR command.

        This command is valid in any processor.

        """
        command = "/NUMBER,%s" % (str(nkey))
        self.RunCommand(command, **kwargs)

    def Vdrag(self, na1="", na2="", na3="", na4="", na5="", na6="", nlp1="",
              nlp2="", nlp3="", nlp4="", nlp5="", nlp6="", **kwargs):
        """
        APDL Command: VDRAG

        Generates volumes by dragging an area pattern along a path.

        Parameters
        ----------
        na1, na2, na3, . . . , na6
            List of areas in the pattern to be dragged (6 maximum if using
            keyboard entry).  If NA1 = P, graphical picking is enabled and all
            remaining command fields are ignored (valid only in the GUI).  If
            NA1 = ALL, all selected areas will be swept along the path.  A
            component name may also be substituted for NA1.

        nlp1, nlp2, nlp3, . . . , nlp6
            List of lines defining the path along which the pattern is to  be
            dragged (6 maximum if using keyboard entry).  Must be a continuous
            set of lines.  To be continuous, adjacent lines must share the
            connecting keypoint (the end keypoint of one line must also be
            first keypoint of the next line).

        Notes
        -----
        Generates volumes (and their corresponding keypoints, lines, and areas)
        by sweeping a given area pattern along a characteristic drag path.  If
        the drag path consists of multiple lines, the drag direction is
        determined by the sequence in which the path lines are input (NLP1,
        NLP2, etc.).  If the drag path is a single line (NLP1), the drag
        direction is from the keypoint on the drag line that is closest to the
        first keypoint of the given area pattern to the other end of the drag
        line.

        The magnitude of the vector between the keypoints of the given pattern
        and the first path keypoint remains constant for all generated keypoint
        patterns and the path keypoints.  The direction of the vector relative
        to the path slope also remains constant so that patterns may be swept
        around curves.  Lines are generated with the same shapes as the given
        pattern and the path lines.

        Keypoint, line, area, and volume numbers are automatically assigned
        (beginning with the lowest available values [NUMSTR]).  Adjacent lines
        use a common keypoint, adjacent areas use a common line, and adjacent
        volumes use a common area.  For best results, the entities to be
        dragged should be orthogonal to the start of the drag path.  Drag
        operations that produce an error message may create some of the desired
        entities prior to terminating.

        If element attributes have been associated with the input area via the
        AATT command, the opposite area generated by the VDRAG operation will
        also have those attributes (i.e., the element attributes from the input
        area are copied to the opposite area).  Note that only the area
        opposite the input area will have the same attributes as the input
        area; the areas adjacent to the input area will not.

        If the input areas are meshed or belong to a meshed volume, the area(s)
        can be extruded to a 3-D mesh.  Note that the NDIV argument of the
        ESIZE command should be set before extruding the meshed areas.
        Alternatively, mesh divisions can be specified directly on the drag
        line(s) (LESIZE).  See the Modeling and Meshing Guide for more
        information.

        You can use the VDRAG command to generate 3-D interface element meshes
        for elements INTER194 and INTER195. When generating interface element
        meshes using VDRAG, you must specify the line divisions to generate one
        interface element directly on the drag line using the LESIZE command.
        The source area to be extruded becomes the bottom surface of the
        interface element. Interface elements must be extruded in what will
        become the element's local x direction, that is, bottom to top.

        """
        command = "VDRAG,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(na1), str(na2), str(na3), str(na4), str(na5), str(na6), str(nlp1), str(nlp2), str(nlp3), str(nlp4), str(nlp5), str(nlp6))
        self.RunCommand(command, **kwargs)

    def Norl(self, line="", area="", ndir="", **kwargs):
        """
        APDL Command: NORL

        Rotates nodal coordinate systems perpendicular to line normal

        Parameters
        ----------
        line
            Line number containing the nodes to be rotated. If ALL, applies to
            all selected lines (see the LSEL command). If LINE = P, graphical
            picking is enabled.

        area
            The area number containing the selected lines. The normal of the
            line(s) selected is supposed to lie on this area. Defaults to the
            lowest numbered selected area containing the line number.

        ndir
            Direction of the normal. If NDIR = -1, the nodal coordinate system
            is rotated in the opposite direction of the line normal. The
            default is the same direction as the surface normal.

        Notes
        -----
        The NORL command rotates the X-axis of the nodal coordinate
        perpendicular to the line normal. The rotated nodal coordinate systems
        may be displayed through the /PSYMB command. In case multiple lines are
        selected, there could be conflicts at the boundaries. If a node belongs
        to two lines that have a different normal, its nodal coordinate system
        will be rotated to the line normal with the lowest number. Keep the
        following in mind when using the NORL command:

        If the nodal coordinate system is parallel to the global Cartesian
        system, it is not displayed through the /PSYMB command.

        Previously specified rotation on the selected nodes are overridden.

        """
        command = "NORL,%s,%s,%s" % (str(line), str(area), str(ndir))
        self.RunCommand(command, **kwargs)

    def Cs(self, kcn="", kcs="", norig="", nxax="", nxypl="", par1="", par2="",
           **kwargs):
        """
        APDL Command: CS

        Defines a local coordinate system by three node locations.

        Parameters
        ----------
        kcn
            Arbitrary reference number assigned to this coordinate system.
            Must be greater than 10.  A coordinate system previously defined
            with this number will be redefined.

        kcs
            Coordinate system type:

            0 or CART - Cartesian

            1 or CYLIN - Cylindrical (circular or elliptical)

            2 or SPHE - Spherical (or spheroidal)

            3 or TORO - Toroidal

        norig
            Node defining the origin of this coordinate system.  If NORIG = P,
            graphical picking is enabled and all remaining command fields are
            ignored (valid only in the GUI).

        nxax
            Node defining the positive x-axis orientation of this coordinate
            system.

        nxypl
            Node defining the x-y plane (with NORIG and NXAX) in the first or
            second quadrant of this coordinate system.

        par1
            Used for elliptical, spheroidal, or toroidal systems.  If KCS = 1
            or 2, PAR1 is the ratio of the ellipse Y-axis radius to X-axis
            radius (defaults to 1.0 (circle)).  If KCS  = 3, PAR1 is the major
            radius of the torus.

        par2
            Used for spheroidal systems.  If KCS = 2, PAR2 = ratio of ellipse
            Z-axis radius to X-axis radius (defaults to 1.0 (circle)).

        Notes
        -----
        Defines and activates a local right-handed coordinate system by
        specifying three existing nodes: to locate the origin, to locate the
        positive x-axis, and to define the positive x-y plane.  This local
        system becomes the active coordinate system.  See the CLOCAL, CSKP,
        CSWPLA, and LOCAL commands for alternate definitions.  Local coordinate
        systems may be displayed with the /PSYMB command.

        This command is valid in any processor.

        """
        command = "CS,%s,%s,%s,%s,%s,%s,%s" % (str(kcn), str(kcs), str(norig), str(nxax), str(nxypl), str(par1), str(par2))
        self.RunCommand(command, **kwargs)

    def Proein(self, name="", extension="", path="", proecomm="", **kwargs):
        """
        APDL Command: ~PROEIN

        Transfers a Creo Parametric part into the ANSYS program.

        Parameters
        ----------
        name
            The name of the Creo Parametric part to be imported, which cannot
            exceed 64 characters in length and must begin with an alphanumeric
            character. Special characters such as & - and * and spaces are not
            permitted in the part name.

        extension
            The general Creo Parametric extension format is prt for parts and
            asm for assemblies.

        path
            Full path name to the directory containing the part. The default is
            the current working directory.

        proecomm
            The start command for the version of Creo Parametric you are using.
            proe1 is the default command. Note that the full path name to the
            Creo Parametric command need not be used here if the path had been
            included in the PATH variable. The Creo Parametric command name is
            set by the PROE_START_CMD162 environment variable.

        Notes
        -----
        More information on importing Creo Parametric parts is available in
        Creo Parametric (formerly Pro/ENGINEER) in the Connection User's Guide.

        """
        command = "~PROEIN,%s,%s,%s,%s" % (str(name), str(extension), str(path), str(proecomm))
        self.RunCommand(command, **kwargs)

    def Meshing(self, **kwargs):
        """
        APDL Command: MESHING

        Specifies "Meshing" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "MESHING," % ()
        self.RunCommand(command, **kwargs)

    def Ask(self, par="", query="", dval="", **kwargs):
        """
        APDL Command: *ASK

        Prompts the user to input a parameter value.

        Parameters
        ----------
        par
            An alphanumeric name used to identify the scalar parameter.  See
            *SET for name restrictions.

        query
            Text string to be displayed on the next line as the query (32
            characters maximum).  Characters having special meaning (such as $
            ! ,) should not be included.

        dval
            Default value assigned to the parameter if the user issues a blank
            response.  May be a number or character string (up to 8 characters
            enclosed in single quotes).  If a default is not assigned, a blank
            response will delete the parameter.

        Notes
        -----
        Intended primarily for use in macros, the command prints the query
        (after the word ENTER) on the next line and waits for a response.  The
        response is read from the keyboard, except in batch mode [/BATCH], when
        the response(s) must be the next-read input line(s).  The response may
        be a number, a character string (up to 8 characters enclosed in single
        quotes), a parameter (numeric or character) or an expression that
        evaluates to a number.  The scalar parameter is then set to the
        response value.  For example,  *ASK,NN,PARAMETER NN  will set NN to the
        value entered on the next line (after the prompt ENTER PARAMETER NN).

        The *ASK command is not written to File.LOG,  but the responses are
        written there as follows:  If *ASK is contained in a macro, the
        response(s) (only) is written to File.LOG on the line(s) following the
        macro name.   If not contained in a macro, the response is written to
        File.LOG as a parameter assignment (i.e., Par = "user-response").

        If used within a do-loop that is executed interactively, *ASK should be
        contained in a macro.  If not contained in a macro, *ASK will still
        query the user as intended, but the resulting log file will not
        reproduce the effects of the original run.

        This command is valid in any processor.

        """
        command = "*ASK,%s,%s,%s" % (str(par), str(query), str(dval))
        self.RunCommand(command, **kwargs)

    def Partsel(self, type="", pmin="", pmax="", pinc="", **kwargs):
        """
        APDL Command: PARTSEL

        Selects a subset of parts in an explicit dynamic analysis.

        Parameters
        ----------
        type
            Label identifying type of select. Because PARTSEL is a command
            macro, the label must be enclosed in single quotes.

            'S' - Select a new set (default).

            'R' - Reselect a set from the current set.

            'A' - Additionally select a set and extend the current set.

            'U' - Unselect a set from the current set.

            'ALL' - Select all parts.

            'NONE' - Unselect all parts.

            'INVE' - Invert the current selected set.

        Notes
        -----
        PARTSEL invokes an ANSYS macro that selects parts in an explicit
        dynamic analysis. When PARTSEL is executed, an element component is
        automatically created for each existing part. For example, the elements
        that make up PART 1 are grouped into the element component _PART1. Each
        time the PARTSEL command is executed, components for unselected parts
        will be unselected. To plot selected parts, choose Utility Menu> Plot>
        Parts in the GUI or issue the command PARTSEL,'PLOT'.

        After selecting parts, if you change the selected set of nodes or
        elements and then plot parts, the nodes and elements associated with
        the previously selected parts (from the last PARTSEL command) will
        become the currently selected set.

        Note:: : A more efficient way to select and plot parts is to use the
        ESEL (with ITEM = PART) and EPLOT commands. We recommend using ESEL
        instead of PARTSEL since PARTSEL will be phased out in a future
        release. Note that the menu path mentioned above for plotting parts
        does not work with the ESEL command; use Utility Menu> Plot> Elements
        instead.

        In an explicit dynamic small restart analysis (EDSTART,2), PARTSEL can
        be used to unselect a part during the solution even if it is referenced
        in some way (such as in a contact definition). (Note that ESEL cannot
        be used for this purpose.) However, in a new analysis or a full restart
        analysis (EDSTART,3), all parts that are used in some type of
        definition must be selected at the time of solution.

        This command is valid in any processor.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PARTSEL,%s,%s,%s,%s" % (str(type), str(pmin), str(pmax), str(pinc))
        self.RunCommand(command, **kwargs)

    def Aptn(self, na1="", na2="", na3="", na4="", na5="", na6="", na7="",
             na8="", na9="", **kwargs):
        """
        APDL Command: APTN

        Partitions areas.

        Parameters
        ----------
        na1, na2, na3, . . . , na9
            Numbers of areas to be operated on.  If NA1 = ALL, NA2 to NA9 are
            ignored and all selected areas are used.  If NA1 = P, graphical
            picking is enabled and all remaining arguments are ignored (valid
            only in the GUI).  A component name may be substituted for NA1.

        Notes
        -----
        Partitions areas that intersect.  This command is similar to the
        combined functionality of the ASBA and AOVLAP commands.  If the
        intersection of two or more areas is an area (i.e., planar), new areas
        will be created with boundaries that conform to the area of
        intersection and to the boundaries of the non-intersecting portions of
        the input areas [AOVLAP].  If the intersection is a line (i.e., not
        planar), the areas will be subtracted, or divided, along the line(s) of
        intersection [ASBA].  Both types of intersection can occur during a
        single APTN operation.  Areas that do not intersect will not be
        modified.  See the Modeling and Meshing Guide for an illustration.  See
        the BOPTN command for an explanation of the options available to
        Boolean operations.  Element attributes and solid model boundary
        conditions assigned to the original entities will not be transferred to
        the new entities generated.

        """
        command = "APTN,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(na1), str(na2), str(na3), str(na4), str(na5), str(na6), str(na7), str(na8), str(na9))
        self.RunCommand(command, **kwargs)

    def Pmgtran(self, fname="", freq="", fcnam1="", fcnam2="", pcnam1="",
                pcnam2="", ecnam1="", ccnam1="", **kwargs):
        """
        APDL Command: PMGTRAN

        Summarizes electromagnetic results from a transient analysis.

        Parameters
        ----------
        fname
            File name (8 characters maximum) to which tabular data and plot
            files will be written.  Must be enclosed in single quotes when the
            command is manually typed in.  Defaults to MG_TRNS.  The data file
            extension is .OUT and the plot file extension is .PLT.

        freq
            Frequency of solution output.  Defaults to 1.  Every FREQth
            solution on the results file is output.

        fcnam1, fcnam2
            Names of element components for force calculation.  Must be
            enclosed in single quotes when the command is manually typed in.

        pcnam1, pcnam2
            Names of element components for power loss calculation.  Must be
            enclosed in single quotes when the command is manually typed in.

        ecnam1, ccnam1
            Names of element components for energy and total current
            calculations, respectively.  Must be enclosed in single quotes when
            the command is manually typed in.

        Notes
        -----
        PMGTRAN invokes an ANSYS macro which calculates and summarizes
        electromagnetic results from a transient analysis.  The results are
        summarized by element components and listed on the screen as well as
        written to a file (Fname.OUT).  Also, graph plots of results as a
        function of time are created and written to a file (Fname.PLT) for use
        in the DISPLAY program.

        Two components may be selected for the summary of electromagnetic
        forces (see FMAGSUM), two for power loss, and one each for stored
        energy (see SENERGY) and total current (see CURR2D).  See the
        referenced commands for other restrictions.

        PMGTRAN is restricted to MKSA units.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PMGTRAN,%s,%s,%s,%s,%s,%s,%s,%s" % (str(fname), str(freq), str(fcnam1), str(fcnam2), str(pcnam1), str(pcnam2), str(ecnam1), str(ccnam1))
        self.RunCommand(command, **kwargs)

    def Mpdele(self, lab="", mat1="", mat2="", inc="", lchk="", **kwargs):
        """
        APDL Command: MPDELE

        Deletes linear material properties.

        Parameters
        ----------
        lab
            Material property label (see MP command for valid labels).  If ALL,
            delete properties for all applicable labels.

        mat1, mat2, inc
            Delete materials from MAT1 to MAT2 (defaults to MAT1) in steps of
            INC (defaults to 1).  If MAT1 = ALL, MAT2 and INC are ignored and
            the properties for all materials are deleted.

        lchk
            Specifies the level of element-associativity checking:

            NOCHECK - No element-associativity check occurs. This option is the default.

            WARN - When a section, material, or real constant is associated with an element, ANSYS
                   issues a message warning that the necessary entity has been
                   deleted.

            CHECK - The command terminates, and no section, material, or real constant is deleted
                    if it is associated with an element.

        Notes
        -----
        This command is also valid in SOLUTION.

        The LCHK argument is valid only when Lab = ALL.

        """
        command = "MPDELE,%s,%s,%s,%s,%s" % (str(lab), str(mat1), str(mat2), str(inc), str(lchk))
        self.RunCommand(command, **kwargs)

    def Nlhist(self, key="", name="", item="", comp="", node="", elem="",
               shell="", layer="", stop_value="", stop_cond="", **kwargs):
        """
        APDL Command: NLHIST

        Specify result items to track during solution.

        Parameters
        ----------
        key
            Specifies the command operation:

            NSOL - Nodal solution data.

            ESOL - Element nodal data.

            PAIR  - Contact data (for pair-based contact).

            GCN - Contact data (for general contact).

            STAT - Displays a list of items to track.

            OFF or 0 - Deactivates tracking of all variables. This value is the default.

            ON or 1 - Activates tracking of all variables.  Tracking also activates whenever any
                      specification changes.

            DEL - Removes the specified variable from the set of result items to track. If Name =
                  ALL (default), all specifications are removed.

        name
            The 32-character user-specified name.

        item, comp
            Predetermined output item and component label for valid elements.
            See the Element Reference for more information.

        node
            Number identifying one of the following:

        elem
            Valid element number for element results. Used for ESOL items. If
            ELEM is specified, then a node number that belongs to the element
            must also be specified in the NODE field.

        shell
            Valid labels are TOP, MID or BOT. This field can specify the
            location on shell elements for which to retrieve data. Used only
            for element nodal data (ESOL).

        layer
            Layer number (for layered elements only). Used only for element
            nodal data (ESOL).

        stop_value
            Critical value of the tracked variable. This value is used to
            determine if the analysis should be terminated. This field is only
            valid for contact data (Key = PAIR or GCN).

        stop_cond
            Specifies the conditional relationship between the variable being
            tracked and the STOP_VALUE upon which the analysis will be
            terminated:

            -1 - Terminate the analysis when the tracked variable is less than or equal to
                 STOP_VALUE.

            0 - Terminate the analysis when the tracked variable equals STOP_VALUE.

            1 - Terminate the analysis when the tracked variable is greater than or equal to
                STOP_VALUE.

        Notes
        -----
        The NLHIST command is a nonlinear diagnostics tool that enables you to
        monitor diagnostics results of interest in real time during a solution.

        You can track a maximum of 50 variables during solution. The specified
        result quantities are written to the file Jobname.nlh. Nodal results
        and contact results are written for every converged substep
        (irrespective of the OUTRES command setting) while element results are
        written only at time points specified via the OUTRES command. For time
        points where element results data is not available, a very small number
        is written instead. If the conditions for contact to be established are
        not satisfied, 0.0 will be written for contact results.

        Results tracking is available only for a nonlinear structural analysis
        (static or transient), a nonlinear steady-state thermal analysis, or a
        transient thermal analysis (linear or nonlinear). All results are
        tracked in the Solution Coordinate System (that is, nodal results are
        in the nodal coordinate system and element results are in the element
        coordinate system).

        Contact results can be tracked for elements CONTA171 through CONTA177;
        they cannot be tracked for CONTA178.

        When contact results are tracked (Key = PAIR or GCN), the user-
        specified name (Name argument) is used to create a user-defined
        parameter. This enables you to monitor the parameter during solution.
        As an example, you can use a named parameter to easily convert the
        contact stiffness units from FORCE/LENGTH3 to FORCE/LENGTH based on the
        initial contact area CAREA. Be sure to specify Name using the APDL
        parameter naming convention.

        The STOP_VALUE and STOP_COND arguments enable you to automatically
        terminate the analysis when a desired value for a tracked contact
        result has been reached. This capability is only available for contact
        variables (Key = PAIR or GCN).

        The Jobname.nlh file is an ASCII file that lists each time point at
        which a converged solution occurs along with the values of the relevant
        result quantities.

        The GUI option Solution> Results tracking provides an interface to
        define the result items to be tracked. The GUI also allows you to graph
        one or more variables against time or against other variables during
        solution. You can use the interface to graph or list variables from any
        .nlh file generated by the ANSYS program.

        You can also track results during batch runs. Either access the ANSYS
        Launcher and select File Tracking from the Tools menu, or type
        nlhist162 at the command line. Use the supplied file browser to
        navigate to your Jobname.nlh file, and click on it to invoke the
        tracking utility. You can use this utility to read the file at any
        time, even after the solution is complete (the data in the file must be
        formatted correctly).

        Table: 205:: : NLHIST - Valid NSOL Item and Component Labels

        For SHELL131 and SHELL132 elements with KEYOPT(3) = 0 or 1, use the
        labels TBOT, TE2, TE3, . . ., TTOP instead of TEMP.

        For SHELL131 and SHELL132 elements with KEYOPT(3) = 0 or 1, use the
        labels HBOT, HE2, HE3, . . ., HTOP instead of HEAT.

        Table: 206:: : NLHIST - Valid ESOL Item and Component Labels

        ETABLE items are not supported for ESOL items.

        PAIR solution quantities are output on a "per contact pair" basis. GCN
        solution quantities are output on a “per general contact section”
        basis. As a consequence, the corresponding values listed in the
        Jobname.nlh file represent a minimum or a maximum over the associated
        contact pair or general contact surface, as detailed in the table
        below.

        Table: 207:: : NLHIST - Valid Contact (PAIR or GCN) Item and Component
        Labels

        For PENE, a positive value indicates a penetration, and a negative
        value indicates a gap. If the contact pair (or GCN surface) has a far-
        field contact status, penetration and gap are not available, and the
        value stored for PENE is the current pinball radius.

        """
        command = "NLHIST,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(key), str(name), str(item), str(comp), str(node), str(elem), str(shell), str(layer), str(stop_value), str(stop_cond))
        self.RunCommand(command, **kwargs)

    def Rmodif(self, nset="", stloc="", v1="", v2="", v3="", v4="", v5="",
               v6="", **kwargs):
        """
        APDL Command: RMODIF

        Modifies real constant sets.

        Parameters
        ----------
        nset
            Number of existing real constant set to be modified.

        stloc
            Starting location in table for modifying data.  For example, if
            STLOC = 1, data input in the V1 field is the first constant in the
            set.  If STLOC = 7, data input in the V1 field is the seventh
            constant in the set, etc.  Must be greater than zero.

        v1
            New value assigned to constant in location STLOC.  If zero (or
            blank), a zero value will be assigned.

        v2, v3, v4, . . . , v6
            New values assigned to constants in the next five locations.  If
            blank, the value remains unchanged.

        Notes
        -----
        Allows modifying (or adding) real constants to an existing set [R] at
        any location.

        Specify NSET = GCN to define/modify real constants for real constant
        sets that were previously assigned by the GCDEF command (that is, real
        constants used in general contact interactions).

        This command is also valid in SOLUTION. For important information about
        using this command within the solution phase, see What Are Nonstandard
        Uses? in the Advanced Analysis Guide.

        """
        command = "RMODIF,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nset), str(stloc), str(v1), str(v2), str(v3), str(v4), str(v5), str(v6))
        self.RunCommand(command, **kwargs)

    def Sload(self, secid="", plnlab="", kinit="", kfd="", fdvalue="",
              lsload="", lslock="", **kwargs):
        """
        APDL Command: SLOAD

        Load a pretension section.

        Parameters
        ----------
        secid
            Unique section number. The number must already be assigned to a
            pretension section.

        plnlab
            Label representing the pretension load sequence number in the
            format "PLnn" where nn is an integer from 1 through 99 (for
            example, PL01 through PL99).

        kinit
            Initial action key for pretension load PL01. (This field is omitted
            for PL02 and up.) Three scenarios are possible:

            LOCK - Constrains (connects) the cutting plane on the pretension section. This value
                   is the default.

            SLID - Unconstrains (disconnects) the cutting plane on the pretension section.

            TINY - Applies a very small pretension load (0.1% of FDVALUE) before the desired load
                   is established. The small load prevents convergence problems
                   which can occur when the desired load is not established in
                   the first load step. This value is valid only if KFD = FORC.

        kfd
            Force/Displacement key. Specifies whether FDVALUE is a force or a
            displacement:

            FORC - Apply a force on the specified pretension section. This value is the default.

            DISP - Apply a displacement (adjustment) on the specified pretension section.

        fdvalue
            Pretension load value. If KFD = FORC, this value is a pretension
            force. If KFD = DISP, this value is a pretension displacement
            (adjustment).

        lsload
            Load step in which to apply the FDVALUE.

        lslock
            The load step in which the displacement value resulting from the
            pretension force is locked. This value is valid only if KFD = FORC.

        Notes
        -----
        The SLOAD command applies pretension loads to specified pretension
        sections (created via the PSMESH command). A pretension load is ramp-
        applied (KBC = 0) if it is a force (KFD = FORC), and step-applied (KBC
        = 1) if it is a displacement (KFD = DISP).

        You can “lock” the load value at a specified load step. When locked,
        the load changes from a force to a displacement, and ANSYS applies the
        load as a constant displacement in all future load steps. Locking is
        useful when applying additional loadings. The additional loadings alter
        the effect of the initial load value, but because locking transforms
        the load into a displacement, it preserves the initial load's effect.

        In modal and harmonic analyses, any pretension load (force,
        displacement, or locked) is ignored and no load is produced.

        The following command shows how to establish loads on a pretension
        section:

        SLOAD,1,PL01,TINY,FORC,5000,2,3

        In this example, the load is applied to pretension section 1, and the
        sequence begins with the initial action key, KINIT, set to TINY. A
        small stabilization load (5 = 0.10% of 5000) is applied in the first
        load step, as the actual pretension force is not applied until the
        second load step. The next four fields set the actual load: the KFD
        value FORC specifies the type of load, FDVALUE defines the pretension
        load value (5000), LSLOAD specifies the load step in which the force is
        applied (2), and the LSLOCK field specifies the load step in which the
        force is locked (3). Additional sets of four fields can be used to
        define additional loads.

        You can use the SLOAD command to edit (overwrite) existing loads on a
        pretension section. This example changes the load on pretension section
        1 (set above) to 6000:

        SLOAD,1,PL01,,,6000,2,3

        Unspecified values (blank fields), as shown in this example, remain
        unchanged from prior settings. If no prior specifications exist, then
        default values (KINIT = LOCK and KFD = FORC) apply.

        The command can also delete all loads on a specified pretension
        section, as shown here:

        SLOAD,1,DELETE

        For a prestressed modal analysis, this command locks the pretension
        element:

        SLOAD,1,PL01,LOCK,DISP,0,1,2

        """
        command = "SLOAD,%s,%s,%s,%s,%s,%s,%s" % (str(secid), str(plnlab), str(kinit), str(kfd), str(fdvalue), str(lsload), str(lslock))
        self.RunCommand(command, **kwargs)

    def Cmsopt(self, cmsmeth="", nmode="", freqb="", freqe="", fbddef="",
               fbdval="", iokey="", **kwargs):
        """
        APDL Command: CMSOPT

        Specifies component mode synthesis (CMS) analysis options.

        Parameters
        ----------
        cmsmeth
            The component mode synthesis method to use. This value is required.

            FIX - Fixed-interface method.

            FREE - Free-interface method.

            RFFB - Residual-flexible free-interface method.

        nmode
            The number of normal modes extracted and used in the superelement
            generation. This value is required; the minimum is 1.

        freqb
            Beginning, or lower end, of frequency range of interest. This value
            is optional.

        freqe
            Ending, or upper end, of frequency range of interest. This value is
            optional.

        fbddef
            In a free-interface (CMSMETH = FREE) or residual-flexible free-
            interface (CMSMETH = RFFB) CMS analysis, the method to use for
            defining free body modes:

            FNUM - The number (FDBVAL) of rigid body modes in the calculation.

            FTOL - Employ a specified tolerance (FDBVAL) to determine rigid body modes in the
                   calculation.

            FAUTO - Automatically determine rigid body modes in the calculation. This method is the
                    default.

            RIGID - If no rigid body modes exist, define your own via the RIGID command.

        fbdval
            In a free-interface CMS analysis (CMSMETH = FREE), the number of
            rigid body modes if FBDDEF = FNUM (where the value is an integer
            from 0 through 6), or the tolerance to employ if FBDDEF = FTOL
            (where the value is a positive real number representing rad/sec).
            This value is required only when FBDDEF = FNUM or FBDDEF = FTOL;
            otherwise, any specified value is ignored.

        iokey
            Output key to control writing of the transformation matrix to the
            .TCMS file (FIX or FREE methods) or body properties to the .EXB
            file (FIX method).

            TCMS - Write the transformation matrix of the nodal component defined by the OUTPR
                   command to a .TCMS file. Refer to TCMS File Format in the
                   Programmer's Reference for more information on the this
                   file.

            EXB - Write a body property input file (.EXB file) containing the condensed
                  substructure matrices and other body properties for use with
                  AVL EXCITE. Refer to ANSYS Interface to AVL EXCITE in the
                  Substructuring Analysis Guide for more information.

        Notes
        -----
        CMS employs the Block Lanczos eigensolution method in the generation
        pass.

        CMS supports damping matrix reduction when a damping matrix exists. Set
        the matrix generation key to 3 (SEOPT,Sename,SEMATR) to generate and
        then reduce stiffness, mass, and damping matrices.

        CMS does not support the SEOPT,,,,,RESOLVE command. Instead, ANSYS sets
        the expansion method for the expansion pass (EXPMTH) to BACKSUB.

        For more information about performing a CMS analysis, see Component
        Mode Synthesis in the Substructuring Analysis Guide.

        If IOKEY = TCMS is used to output the transformation matrix, then only
        ITEM = NSOL is valid in the OUTPR command.  In the interactive
        sessions, the transformation matrix will not be output if the model has
        more than 10 elements.

        This command is also valid in /PREP7.

        """
        command = "CMSOPT,%s,%s,%s,%s,%s,%s,%s" % (str(cmsmeth), str(nmode), str(freqb), str(freqe), str(fbddef), str(fbdval), str(iokey))
        self.RunCommand(command, **kwargs)

    def Vinv(self, nv1="", nv2="", nv3="", nv4="", nv5="", nv6="", nv7="",
             nv8="", nv9="", **kwargs):
        """
        APDL Command: VINV

        Finds the intersection of volumes.

        Parameters
        ----------
        nv1, nv2, nv3, . . . , nv9
            Numbers of volumes to be intersected.  If NV1 = ALL, NV2 to NV9 are
            ignored, and the intersection of all selected volumes is found.  If
            NV1 = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).  A component name may
            also be substituted for NV1.

        Notes
        -----
        Finds the common (not pairwise) intersection of volumes.  The common
        intersection is defined as the regions shared (in common) by all
        volumes listed on this command.  New volumes will be generated where
        the original volumes intersect.  If the regions of intersection are
        only areas, new areas will be generated instead.  See the Modeling and
        Meshing Guide for an illustration.  See the BOPTN command for an
        explanation of the options available to Boolean operations.  Element
        attributes and solid model boundary conditions assigned to the original
        entities will not be transferred to the new entities generated.

        """
        command = "VINV,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nv1), str(nv2), str(nv3), str(nv4), str(nv5), str(nv6), str(nv7), str(nv8), str(nv9))
        self.RunCommand(command, **kwargs)

    def Uis(self, label="", value="", **kwargs):
        """
        APDL Command: /UIS

        Controls the GUI behavior.

        Parameters
        ----------
        label
            Behavior control key:

            BORD - Controls the functionality of the mouse buttons for dynamic viewing mode only.
                   When Label = BORD, the three values that follow control the
                   functionality of the LEFT, MIDDLE and RIGHT buttons,
                   respectively (see below).

            MSGPOP - Controls which messages from the ANSYS error message subroutine are displayed
                     in a message dialog box.

            REPLOT - Controls whether or not an automatic replot occurs after functions affecting
                     the model are executed.

            ABORT - Controls whether or not ANSYS displays dialog boxes to show the status of an
                    operation in progress and to cancel that operation.

            DYNA - Controls whether the dynamic mode preview is a bounding box or the edge outline
                   of the model.  This label only applies to 2-D display
                   devices (i.e., /SHOW,XII or /SHOW,WIN32).  This "model edge
                   outline" mode is only supported in PowerGraphics
                   [/GRAPHICS,POWER] and is intended for element, line,
                   results, area, or volume displays.

            PICK - Controls how graphical entities are highlighted from within the ANSYS Select
                   menu.

            POWER - Controls whether or not PowerGraphics is active when the GUI is initiated. The
                    ANSYS program default status is PowerGraphics “ON”;  this
                    command is used (placed in the start.ans file) when full
                    graphics is desired on start up.

            DPRO - Controls whether or not the ANSYS input window displays a dynamic prompt. The
                   dynamic prompt shows the correct command syntax for the
                   command, as you are entering it.

            UNDO - Controls whether or not the session editor includes nonessential commands or
                   comments in the file it creates. You can use this option to
                   include comments and other materials in the session editor
                   file.

            LEGE - Controls whether or not the multi-legend is activated when you start the GUI.
                   The multi-legend enables you to specify the location of your
                   legend items in each of the five graphics windows. You can
                   place this option in your start.ans file and have the GUI
                   start with the legend items in a pre-specified location.

            PBAK - Controls whether or not the background shading is activated when you start the
                   GUI. You can place this option in your start.ans file.

            ZPIC - Controls the sorting order for entities that are coincident (directly in front
                   of or behind each other) to a picked spot on your model.
                   When you pick a spot on your model that could indicate two
                   or more entities, a message warns you of this condition, and
                   a list of the coincident entities can be generated. The
                   VALUE term (below) will determine the sort order.

            HPOP - Controls the prioritization of your GUI windows when the contents are ported to
                   a plot or print file (/UI,COPY,SAVE). OpenGL (3D) graphics
                   devices require that the ANSYS Graphics Screen contents be
                   set in front of all overlying windows in order to port them
                   to a printer or a file. This operation can sometimes
                   conflict with /NOERASE settings. See the VALUE term (below)
                   to determine the available control options.

        value
            Values controlling behavior if Label = BORD:

            1 - PAN, controls dynamic translations.

            2 - ZOOM, controls zoom, and dynamic rotation about the view vector.

            3 - ROTATE, controls dynamic rotation about the screen X and Y axes.

        Notes
        -----
        Controls certain features of the Graphical User Interface (GUI),
        including whether the ANSYS program displays dialog boxes to show you
        the status of an operation (such as meshing or solution) in progress
        and to enable you to cancel that operation.  Issue /UIS,STAT for
        current status.  Issue /UIS,DEFA to reset default values for all
        labels.  Issue /UIS,Label,STAT and /UIS,Label,DEFA for status and to
        reset a specific Label item.

        A /UIS,HPOP,1 command employs a fast redraw method which does not allow
        entering the legend logic for a /PLOPTS,INFO,1 or /PLOPTS,INFO,2
        command. However, the legend is redrawn for /PLOPTS,INFO,3 because that
        command also allows a fast redraw.

        This command is valid in any processor.

        """
        command = "/UIS,%s,%s" % (str(label), str(value))
        self.RunCommand(command, **kwargs)

    def Spread(self, value="", **kwargs):
        """
        APDL Command: SPREAD

        Turns on a dashed tolerance curve for the subsequent curve plots.

        Parameters
        ----------
        value
            Amount of tolerance.  For example, 0.1 is ± 10%.

        """
        command = "SPREAD,%s" % (str(value))
        self.RunCommand(command, **kwargs)

    def Psmat(self, fname="", ext="", matrix="", color="", **kwargs):
        """
        APDL Command: PSMAT

        Writes an assembled global matrix to a postscript format that
        graphically displays nonzero matrix values.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        matrix
            Specify which matrix to write to the output postscript file:

            STIFF - Write stiffness matrix to output postscript file. Valid for all types of
                    analyses that write a .FULL file.

            MASS - Write mass matrix to output postscript file. Valid for buckling, substructure,
                   and modal analyses. If the .FULL file was generated in a
                   buckling analysis, then this label will write the stress
                   stiffening matrix to the output postscript file.

            DAMP - Write damping matrix to output postscript file. Only valid for damped modal
                   analyses.

        color
            Specifies whether to display the grid in black and white or in
            color:

            BLACK  - Each nonzero coefficient is symbolized by a black square (default).

            COLOR  - Each nonzero coefficient is symbolized by a colored square. The color depends
                     on the module of the coefficient; the range is from blue
                     for the smallest values to red for the largest values. The
                     color map is:

        Notes
        -----
        This command is used to copy a matrix from the assembled global matrix
        file (.FULL file) as specified on the FILE command and write it in a
        postscript format to a new file named Fname.Ext (defaults to
        Jobname.PS). The matrix is symbolized by a grid in which the black or
        colored squares represent the nonzero coefficients of the matrix. The
        .FULL file must be available for this command to work properly.

        If the matrix is large, it may be difficult to display the postscript
        file. In this case, use Color = BLACK to reduce the postscript file
        size.

        The assembled global matrix file is created during solution depending
        on the analysis type, equation solver, and other solution options. By
        default, the assembled global matrix file is never deleted at the end
        of solution. For most analysis types, the Sparse direct solver and the
        ICCG solver write a .FULL file. All mode extraction methods used for
        buckling and modal analyses write a properly formatted .FULL file to be
        used with the PSMAT command.

        When copying the stiffness matrix for transient and harmonic analyses,
        be aware that the element mass matrix values (and possibly element
        damping matrix values) are incorporated into the globally assembled
        stiffness matrix.  Thus, the globally assembled stiffness matrix
        represents more than the stiffness of the model for these analysis
        types.  Please refer to the Mechanical APDL Theory Reference for more
        details.

        The PSMAT command is not able to display a lumped mass matrix from a
        .FULL file generated by a harmonic analysis.

        When copying a .FULL file, the rows and columns corresponding to
        specified constraints (e.g., D commands) are eliminated from the system
        of equations and therefore are not written to the .PS file. In
        addition, rows and columns corresponding to eliminated (slave) degrees
        of freedom from coupling and/or constraint equations (e.g., CE, CP
        commands) are eliminated from the system of equations and are not
        written to the .PS file. The DOFs that are eliminated from any coupling
        and/or constraint equations are determined internally by the solution
        code and may not match what you specified via the CE/CP (or similar)
        commands.

        When copying a .FULL file, only the upper triangular part of the matrix
        will be written to the .PS  file if the matrix is symmetric; the full
        matrix is written if the matrix is unsymmetric. Only matrix
        coefficients that are greater than zero will be written.

        The WRFULL command, in conjunction with the SOLVE command, can be used
        to generate the assembled global matrix file and eliminate the equation
        solution process and results output process.

        The following command sequence shows typical usage of this command.

        Below is an example of an export of the stiffness matrix to a
        postscript format using the COLOR option.

        :

        """
        command = "PSMAT,%s,%s,%s,%s" % (str(fname), str(ext), str(matrix), str(color))
        self.RunCommand(command, **kwargs)

    def Nrrang(self, nmin="", nmax="", ninc="", **kwargs):
        """
        APDL Command: NRRANG

        Specifies the range of nodes to be read from the node file.

        Parameters
        ----------
        nmin, nmax, ninc
            Node range is defined from NMIN (defaults to 1) to NMAX (defaults
            to 99999999) in steps of NINC (defaults to 1).

        Notes
        -----
        Defines the range of nodes to be read [NREAD] from the node file.  Also
        implies an element range since only elements fully attached to these
        nodes will be read from the element file.

        """
        command = "NRRANG,%s,%s,%s" % (str(nmin), str(nmax), str(ninc))
        self.RunCommand(command, **kwargs)

    def Nwplan(self, wn="", norig="", nxax="", nplan="", **kwargs):
        """
        APDL Command: NWPLAN

        Defines the working plane using three nodes.

        Parameters
        ----------
        wn
            Window number whose viewing direction will be modified to be normal
            to the working plane (defaults to 1).  If WN is a negative value,
            the viewing direction will not be modified.  If fewer than three
            points are used, the viewing direction of window WN will be used
            instead to define the normal to the working plane.

        norig
            Node number defining the origin of the working plane coordinate
            system.  If NORIG = P, graphical picking is enabled and all
            remaining command fields are ignored (valid only in the GUI).

        nxax
            Node number defining the x-axis orientation (defaults to the x-axis
            being parallel to the global X-axis; or if the normal to the
            working plane is parallel to the global X-axis, then defaults to
            being parallel to the global Y-axis).

        nplan
            Node number defining the working plane (the normal defaults to the
            present display view [/VIEW] of window WN).

        Notes
        -----
        Defines a working plane to assist in picking operations using three
        nodes as an alternate to the WPLANE command.  The three nodes also
        define the working plane coordinate system.  A minimum of one node (at
        the working plane origin) is required.  Immediate mode may also be
        active.  See the WPSTYL command to set the style of the working plane
        display.

        This command is valid in any processor.

        """
        command = "NWPLAN,%s,%s,%s,%s" % (str(wn), str(norig), str(nxax), str(nplan))
        self.RunCommand(command, **kwargs)

    def Cmmod(self, cname="", keyword="", value="", **kwargs):
        """
        APDL Command: CMMOD

        Modifies the specification of a component.

        Parameters
        ----------
        cname
            Name of the existing component or assembly to be modified.

        keyword
            The label identifying the type of value to be modified.

        value
            If Keyword is NAME, then the value is the alphanumeric label to be
            applied. See the CM command for naming convention details. If a
            component named Value already exists, the command will be ignored
            and an error message will be generated.

        Notes
        -----
        The naming conventions for components, as specified in the CM command,
        apply for CMMOD (32 characters, “ALL”, “STAT” and “DEFA” are not
        allowed, etc.). However, if you choose a component name that is already
        designated for another component, an error message will be issued and
        the command will be ignored.

        This command is valid in any processor.

        """
        command = "CMMOD,%s,%s,%s" % (str(cname), str(keyword), str(value))
        self.RunCommand(command, **kwargs)

    def Dist(self, wn="", dval="", kfact="", **kwargs):
        """
        APDL Command: /DIST

        Specifies the viewing distance for magnifications and perspective.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        dval
            Distance along the view line from the observer to the focus point
            (defaults to value producing full-window display).  Distances "too
            close" to the object will produce excessive magnifications.  If
            DVAL = AUTO, zero, or blank, the program will calculate the
            distance automatically.  If DVAL = USER, the distance of last
            display will be used (useful when last display automatically
            calculated distance).

        kfact
            DVAL interpretation key:

            0 - Interpret numerical DVAL values as described above.

            1 - Interpret DVAL as a multiplier on the current distance (DVAL of 2 gives twice
                the current distance; 0.5 gives half the current distance,
                etc.).

        Notes
        -----
        The scale factor is relative to the window shape.  For example, for
        objects centered in a square window and with parallel projection (no
        perspective), a distance of :  /2 (+10%) produces a full window
        magnification, where :  is the largest in-plane vertical or horizontal
        dimension.  See also /AUTO and /USER commands.

        This command is valid in any processor.

        """
        command = "/DIST,%s,%s,%s" % (str(wn), str(dval), str(kfact))
        self.RunCommand(command, **kwargs)

    def Flst(self, nfield="", narg="", type="", otype="", leng="", **kwargs):
        """
        APDL Command: FLST

        Specifies data required for a picking operation (GUI).

        Parameters
        ----------
        nfield
            Field number on the command which uses the picking data.  (Count
            the command name as a field, so that a 2 indicates the first
            command argument, 3 for the second command argument, etc.)  The
            corresponding field on the command will have a P51X label.

        narg
            Number of items in the picked list.

        type
            Type of items picked:

            1 - Node numbers

            2 - Element numbers

            3 - Keypoint numbers

            4 - Line numbers

            5 - Area numbers

            6 - Volume numbers

            7 - Trace points

            8 - Coordinate locations (in Global Cartesian coordinates)

            9 - Screen picks (in X, Y screen coordinates (-1 to 1))

        otype
            Data order:

            NOOR - Data is not ordered (default).

            ORDER - Data is in an ordered list (such as for the E,P51X and A,P51X commands, in
                    which the order of the data items is significant for the
                    picking operation).

        leng
            Length of number of items describing the list (should equal NARG if
            Otype = NOOR; default).

        Notes
        -----
        Specifies data required for the FITEM command during a picking
        operation.  This is a command generated by the GUI and will appear in
        the log file (Jobname.LOG) if graphical picking is used.  This command
        is not intended to be typed in directly in an ANSYS session (although
        it can be included in an input file for batch input or for use with the
        /INPUT command).

        On the log file, FLST will always be followed by one or more FITEM
        commands which in turn are followed by the ANSYS command that contains
        a P51X label in one of its fields. This set of commands should not be
        edited.

        This command is valid in any processor.

        """
        command = "FLST,%s,%s,%s,%s,%s" % (str(nfield), str(narg), str(type), str(otype), str(leng))
        self.RunCommand(command, **kwargs)

    def Etcontrol(self, eltech="", eldegene="", **kwargs):
        """
        APDL Command: ETCONTROL

        Control the element technologies used in element formulation (for
        applicable elements).

        Parameters
        ----------
        eltech
            Element technology control:

            SUGGESTION - The program offers a suggestion for the best element technology before solving.
                         If necessary, mixed u-P (KEYOPT(6)) is also included
                         and reset. This behavior is the default.

            SET - The program informs you of the best settings and resets any applicable KEYOPT
                  settings automatically. This action overrides any previous
                  manual settings.

            OFF - Deactivates automatic selection of element technology. No suggestions are
                  issued, and no automatic resetting occurs.

        eldegene
            Element degenerated shape control:

            ON - If element shapes are degenerated, the degenerated shape function is employed
                 and enhanced strain, simplified enhanced strain, and B-bar
                 formulations are turned off (default).

            OFF - If element shapes are degenerated, regular shape functions are still used, and
                  the specified element technologies (e.g., enhanced strain,
                  B-bar, uniform reduced integration) are still used.

        Notes
        -----
        The command default is ETCONTROL,SUGGESTION,ON.

        This command is valid for elements SHELL181, PLANE182, PLANE183,
        SOLID185, SOLID186, SOLID187, BEAM188, BEAM189, SHELL208, SHELL209,
        PLANE223, SOLID226, SOLID227, REINF264, SOLID272, SOLID273, SHELL281,
        SOLID285, PIPE288, PIPE289, ELBOW290.

        For more information, see Automatic Selection of Element Technologies
        and Formulations in the Element Reference.

        """
        command = "ETCONTROL,%s,%s" % (str(eltech), str(eldegene))
        self.RunCommand(command, **kwargs)

    def Usrelem(self, nnodes="", ndim="", keyshape="", nreal="", nsavevars="",
                nrsltvar="", keyansmat="", nintpnts="", kestress="", keysym="",
                **kwargs):
        """
        APDL Command: USRELEM

        Specifies the characteristics of the user-defined element USER300.

        Parameters
        ----------
        nnodes
            The number of nodes.

        ndim
            The number of dimensions (of nodal coordinates). Valid values are 2
            or 3.

        keyshape
            One of the following element shape options:

            ANYSHAPE - Any shape (that is, no specified shape). This value is the default. (The ANSYS
                       MeshTool is unavailable.)

            POINT - Point.

            LINE - Straight line.

            TRIAN - Triangle.

            QUAD - Quadrilateral. This shape can be degenerated to a triangle.

            TET - Tetrahedron.

            BRICK - Brick. This shape can be degenerated to a wedge, pyramid, or tetrahedron.

        nreal
            The number of real constants.

        nsavevars
            The number of saved variables.

        nrsltvar
            The number of variables saved in results files.

        keyansmat
            Key for element formulation control:

            0 - Create your own material codes within the element formulation. In this case,
                the real constants are available to input material properties.
                You can also input linear material properties via MP and MPDATA
                commands .

            1 - Use ANSYS standard material routines or the USERMAT subroutine to form
                structural material data. ANSYS material properties must be
                input in the standard way (as you would for non-user-defined
                elements). This value is invalid when KeyShape = ANYSHAPE.

        nintpnts
            The maximum number of integration points (used when KEYANSMAT = 1).

        kestress
            Key for the element stress state (used when KEYANSMAT = 1):

            0 - Plane stress elements.

            1 - Axisymmetric elements.

            2 - Plane strain elements.

            3 - 3-D solid elements.

            4 - 3-D solid-shell elements.

            5 - Generalized plane strain elements.

            6 - Beam elements.

            7 - Link/truss elements.

            8 - 3-D shell elements.

            9 - Axisymmetric shell elements.

        keysym
            Key for specifying whether element stiffness matrices are symmetric
            or unsymmetric:

            0 - Symmetric.

            1 - Unsymmetric.

        Notes
        -----
        The USRELEM command specifies the characteristics of the user-defined
        element USER300.

        Although you can intersperse other commands as necessary for your
        analysis, issue the USRELEM command as part of the following general
        sequence of commands:

        Issue the ET command for element USER300, followed by the related TYPE
        command.

        Issue both the USRELEM and USRDOF commands (in either order).

        Define your element using USER300.

        The number of real constants (NREAL) can refer to geometry quantities,
        material quantities, or any parameters for element formulation.

        ANSYS saves variables in the .esav file to preserve element data when
        you specify a positive NSAVEVARS value. When KEYANSMAT = 0, all
        variables of both material and kinematic formulation are saved. When
        KEYANSMAT = 1, only the variables for kinematic formulation (such as
        deformation gradient tensor) are saved; in this case, the material
        routine saves all necessary material data automatically.

        Element data saved in results files (NRSLTVAR) are accessible only as
        nonsummable miscellaneous data.  ANSYS saves stress and total strain
        data for structural elements in the .rst file automatically (as it does
        for equivalent variables such as thermal gradient and thermal flux in
        thermal elements); therefore, NRSLTVAR does not need to include stress
        and total strain data.

        To learn more about creating user-defined elements, see Creating a New
        Element in the Programmer's Reference.

        """
        command = "USRELEM,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nnodes), str(ndim), str(keyshape), str(nreal), str(nsavevars), str(nrsltvar), str(keyansmat), str(nintpnts), str(kestress), str(keysym))
        self.RunCommand(command, **kwargs)

    def Tbpt(self, oper="", x1="", x2="", x3="", xn="", **kwargs):
        """
        APDL Command: TBPT

        Defines a point on a nonlinear data curve.

        Parameters
        ----------
        oper
            Operation to perform:

            DEFI - Defines a new data point (default).  The point is inserted into the table in
                   ascending order of X1.  If a point already exists with the
                   same X1 value, it is replaced.

            DELE - Deletes an existing point.  The X1 value must match the X1 value of the point
                   to be deleted (XN is ignored).

        x1, x2, ..., xn
            The N components of the point. N depends on the type of data table.
            Except for TB,EXPE all other TB Tables support only 2 components.

        Notes
        -----
        TBPT defines a point on a nonlinear data curve (such as a stress-strain
        curve, B-H curve, etc.) at the temperature specified on the last TBTEMP
        command. The meaning of the values depends on the type of data table
        specified on the last TB command (MISO, BH, etc.).

        This command is also valid in SOLUTION.

        """
        command = "TBPT,%s,%s,%s,%s,%s" % (str(oper), str(x1), str(x2), str(x3), str(xn))
        self.RunCommand(command, **kwargs)

    def Trplis(self, ntrp1="", ntrp2="", trpinc="", opt="", **kwargs):
        """
        APDL Command: TRPLIS

        Lists the particle flow or charged particle trace points.

        Parameters
        ----------
        ntrp1, ntrp2, trpinc
            List points from NTRP1 to NTRP2 (defaults to NTRP1) in steps of
            TRPINC (defaults to 1).  If NTRP1 = ALL, NTRP2 and TRPINC are
            ignored and all trace points are listed.  If NTRP1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).

        opt
            Opt = LOC lists the trace point number location (X, Y, Z). Default.

        Notes
        -----
        Lists the particle flow or charged particle trace points in the active
        display coordinate system [DSYS].  Trace points are defined with the
        TRPOIN command.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "TRPLIS,%s,%s,%s,%s" % (str(ntrp1), str(ntrp2), str(trpinc), str(opt))
        self.RunCommand(command, **kwargs)

    def Prnear(self, lab="", opt="", kcn="", val1="", val2="", val3="",
               val4="", val5="", val6="", val7="", val8="", val9="", **kwargs):
        """
        APDL Command: PRNEAR

        Prints the pressure in the near zone exterior to the equivalent source
        surface.

        Parameters
        ----------
        lab
            Print the maximum pressure or sound pressure level:

            POINT - at the point (x,y,z)

            SPHERE - on the spherical structure

            PATH - along the path

        opt
            PSUM

            PSUM  - Maximum complex pressure for acoustics.

            PHAS  - Phase angle of complex pressure for acoustics.

            SPL  - Sound pressure level for acoustics.

            SPLA - A-weighted sound pressure level for acoustics (dBA).

        kcn
            KCN is the coordinate system reference number. It may be 0
            (Cartesian) or any previously defined local coordinate system
            number (>10). Defaults to 0.

        val1, val2, val3, . . . , val9
            For Lab = POINT:

            VAL1 - x coordinate value

            VAL2 - y coordinate value

            VAL3 - z coordinate value

            VAL4 - VAL8 - not used

            VAL9 - Thickness of model in z direction (defaults to 0).

        Notes
        -----
        The command uses the equivalent source principle to calculate the
        pressure in the near zone exterior to the equivalent source surface
        (flagged with the Maxwell surface flag in the preprocessor) for one of
        the following locations:

        A point X, Y, Z in the KCN coordinate system

        A spherical surface in the KCN coordinate system

        A path defined by the PATH and PPATH commands

        To list the pressure results for a path, use the PRPATH command. See
        HFSYM command for the model symmetry.

        To retrieve saved equivalent source data, issue the
        SET,Lstep,Sbstep,,REAL command.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PRNEAR,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(lab), str(opt), str(kcn), str(val1), str(val2), str(val3), str(val4), str(val5), str(val6), str(val7), str(val8), str(val9))
        self.RunCommand(command, **kwargs)

    def Sspa(self, a11="", a21="", a31="", a22="", a32="", a33="", t="",
             **kwargs):
        """
        APDL Command: SSPA

        Specifies a preintegrated membrane stiffness for shell sections.

        Parameters
        ----------
        a11, a21, a31, a22, a32, a33
            Membrane stiffness component (symmetric lower part of submatrix
            [A]).

        t
            Temperature.

        Notes
        -----
        The behavior of shell elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        The SSPA command, one of several preintegrated shell section commands,
        specifies the membrane stiffness quantity (submatrix [A]) for a
        preintegrated shell section. The section data defined is associated
        with the section most recently defined (via the SECTYPE command).

        Unspecified values default to zero.

        Related commands are SSPB, SSPD, SSPE, SSMT, SSBT, and SSPM.

        For complete information, see Using Preintegrated General Shell
        Sections.

        """
        command = "SSPA,%s,%s,%s,%s,%s,%s,%s" % (str(a11), str(a21), str(a31), str(a22), str(a32), str(a33), str(t))
        self.RunCommand(command, **kwargs)

    def Outaero(self, sename="", timeb="", dtime="", **kwargs):
        """
        APDL Command: OUTAERO

        Outputs the superelement matrices and load vectors to formatted files
        for aeroelastic analysis.

        Parameters
        ----------
        sename
            Name of the superelement that models the wind turbine supporting
            structure. Defaults to the current Jobname.

        timeb
            First time at which the load vector is formed (defaults to be read
            from SENAME.sub).

        dtime
            Time step size of the load vectors (defaults to be read from
            SENAME.sub).

        Notes
        -----
        Both TIMEB and DTIME must be blank if the time data is to be read from
        the SENAME.sub file.

        The matrix file (SENAME.SUB) must be available from the substructure
        generation run before issuing this command. This superelement that
        models the wind turbine supporting structure must contain only one
        master node with six freedoms per node: UX, UY, UZ, ROTX, ROTY, ROTZ.
        The master node represents the connection point between the turbine and
        the supporting structure.

        This command will generate four files that are exported to the
        aeroelastic code for integrated wind turbine analysis. The four files
        are Jobname.GNK for the generalized stiffness matrix, Jobname.GNC for
        the generalized damping matrix, Jobname.GNM for the generalized mass
        matrix and Jobname.GNF for the generalized load vectors.

        For detailed information on how to perform a wind coupling analysis,
        see Coupling to External Aeroelastic Analysis of Wind Turbines in the
        Mechanical APDL Advanced Analysis Guide.

        """
        command = "OUTAERO,%s,%s,%s" % (str(sename), str(timeb), str(dtime))
        self.RunCommand(command, **kwargs)

    def Nlopt(self, **kwargs):
        """
        APDL Command: NLOPT

        Specifies "Nonlinear analysis options" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "NLOPT," % ()
        self.RunCommand(command, **kwargs)

    def Iclist(self, node1="", node2="", ninc="", lab="", **kwargs):
        """
        APDL Command: ICLIST

        Lists the initial conditions.

        Parameters
        ----------
        node1, node2, ninc
            List initial conditions for nodes NODE1 to NODE2 (defaults to
            NODE1) in steps of NINC (defaults to 1).  If NODE1 = ALL (default),
            NODE2 and NINC are ignored and initial conditions for all selected
            nodes [NSEL] are listed.  If NODE1 = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may be substituted for NODE1 (NODE2 and
            NINC are ignored).

        lab
            Velocity key:

            DISP - Specification is for first order degree of freedom value (displacements,
                   temperature, etc.) (default).

            VELO - Specification is for second order degree of freedom value (velocities).

        Notes
        -----
        Lists the initial conditions specified by the IC command.  Listing
        applies to all the selected nodes [NSEL] and DOF labels.  ICLIST is not
        the same as the DLIST command.  All the initial conditions including
        the default conditions are listed for the selected nodes.

        This command is valid in any processor.

        """
        command = "ICLIST,%s,%s,%s,%s" % (str(node1), str(node2), str(ninc), str(lab))
        self.RunCommand(command, **kwargs)

    def Rmrgenerate(self, **kwargs):
        """
        APDL Command: RMRGENERATE

        Performs fitting procedure for all ROM functions to generate response
        surfaces.

        Notes
        -----
        The fitting procedure uses modal analysis data and function data
        generated using the RMSMPLE command and specifications set forth in the
        RMROPTIONS command. The files jobname_ijk.pcs (modes i, j, k) will be
        generated containing the coefficients of the response surfaces. These
        files are needed for the ROM Use Pass along with a ROM data base file
        [RMSAVE].

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        Strain energy and capacitance data file jobname_ijk.dec

        Response surface coefficients jobname_ijk.pcs (modes i, j, k)

        """
        command = "RMRGENERATE," % ()
        self.RunCommand(command, **kwargs)

    def Wprota(self, thxy="", thyz="", thzx="", **kwargs):
        """
        APDL Command: WPROTA

        Rotates the working plane.

        Parameters
        ----------
        thxy
            First rotation about the working plane Z axis (positive X toward
            Y).

        thyz
            Second rotation about working plane X axis (positive Y toward Z).

        thzx
            Third rotation about working plane Y axis (positive Z toward X).

        Notes
        -----
        The specified angles (in degrees) are relative to the orientation of
        the working plane.

        This command is valid in any processor.

        """
        command = "WPROTA,%s,%s,%s" % (str(thxy), str(thyz), str(thzx))
        self.RunCommand(command, **kwargs)

    def Nrm(self, name="", normtype="", parr="", normalize="", **kwargs):
        """
        APDL Command: *NRM

        Computes the norm of the specified matrix or vector.

        Parameters
        ----------
        name
            Matrix or vector for which the norm will be computed. This can be a
            dense matrix (created by the *DMAT command), a sparse matrix
            (created by the *SMAT command) or a vector (created by the *VEC
            command)

        normtype
            Mathematical norm to use:

            L2 (Euclidian or SRSS) norm (default). - L1 (absolute sum) norm (vectors only).

        parr
            Parameter name that contains the result.

        normalize
            Normalization key; to be used only for vectors created by *VEC:

            Normalize the vector such that the norm is 1.0. - Do not normalize the vector (default).

        Notes
        -----
        The NRM2 option corresponds to the Euclidian or L2 norm and is
        applicable to either vectors or matrices. The NRM1 option corresponds
        to the L1 norm and is applicable to vectors only. The NRMINF option is
        the maximum norm and is applicable to either vectors or matrices.

        """
        command = "*NRM,%s,%s,%s,%s" % (str(name), str(normtype), str(parr), str(normalize))
        self.RunCommand(command, **kwargs)

    def Cbtmp(self, temp="", **kwargs):
        """
        APDL Command: CBTMP

        Specifies a temperature for composite-beam input.

        Parameters
        ----------
        temp
            Temperature value.

        Notes
        -----
        The CBTMP command, one of several composite beam-section commands,
        specifies a temperature to be associated with the data input via
        subsequent CBMX (preintegrated cross-section stiffness), CBMD
        (preintegrated section mass), or CBTE (thermal-expansion) commands.

        The specified temperature remains active until the next CBTMP command
        is issued.

        An unspecified temperature value defaults to zero.

        For complete information, see Using Preintegrated Composite Beam
        Sections.

        """
        command = "CBTMP,%s" % (str(temp))
        self.RunCommand(command, **kwargs)

    def Lsbac(self, enginename="", rhsvector="", solvector="", **kwargs):
        """
        APDL Command: *LSBAC

        Performs the solve (forward/backward substitution) of a factorized
        linear system.

        Parameters
        ----------
        enginename
            Name used to identify this engine. Must have been previously
            created using *LSENGINE and factorized using *LSFACTOR.

        rhsvector
            Name of vector containing the right-hand side (load) vectors as
            input. Must have been previously defined as a *VEC vector or a
            *DMAT matrix.

        solvector
            Name of vector that will contain the solution vectors upon
            completion. Must be predefined as a *VEC vector or *DMAT matrix.

        Notes
        -----
        This command performs forward and back substitution to obtain the
        solution to the linear matrix equation Ax = b. The matrix engine must
        have been previously defined using *LSENGINE, and the matrix factored
        using *LSFACTOR.

        You can use the *DMAT,,,COPY (or *VEC,,,COPY) command to copy the load
        vector to the solution vector in order to predefine it with the
        appropriate size.

        """
        command = "*LSBAC,%s,%s,%s" % (str(enginename), str(rhsvector), str(solvector))
        self.RunCommand(command, **kwargs)

    def Mflist(self, option="", value="", **kwargs):
        """
        APDL Command: MFLIST

        Lists the settings for an ANSYS Multi-field solver analysis.

        Parameters
        ----------
        option
            ALL

            ALL - Lists all ANSYS Multi-field solver analysis options.

            SOLU - Lists all solution-related ANSYS Multi-field solver options.

            FIELD - Lists all ANSYS Multi-field solver options related to the specified field
                    number.

            SINT - Lists all surface interface information for the specified surface interface
                   number.

            VINT - Lists all volumetric interface information for the specified volumetric
                   interface number.

        value
            Use only for Option = FIELD, SINT, or VINT.

        Notes
        -----
        This command is also valid in PREP7.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFLIST,%s,%s" % (str(option), str(value))
        self.RunCommand(command, **kwargs)

    def Mptres(self, lab="", mat="", **kwargs):
        """
        APDL Command: MPTRES

        Restores a temperature table previously defined.

        Parameters
        ----------
        lab
            Material property label [MP].

        mat
            Material reference number.

        Notes
        -----
        Restores into the database (from virtual space) a temperature table
        previously defined [MP] for a particular property.  The existing
        temperature table in the database is erased before this operation.

        This command is also valid in SOLUTION.

        """
        command = "MPTRES,%s,%s" % (str(lab), str(mat))
        self.RunCommand(command, **kwargs)

    def Pmlopt(self, esys="", lab="", xminus="", xplus="", yminus="", yplus="",
               zminus="", zplus="", **kwargs):
        """
        APDL Command: PMLOPT

        Defines perfectly matched layers (PMLs) for acoustic and structural
        analyses.

        Parameters
        ----------
        esys
            Element coordinate system number. ESYS may be 0 (global Cartesian)
            or any previously defined local Cartesian coordinate system number
            (>10). Defaults to 0.

        lab
            Label defining the number of dimensions:

            ONE - A one-dimensional PML region.

            THREE - A three-dimensional PML region (default).

        xminus
            For harmonic analysis, normal reflection coefficient in negative X
            direction of ESYS. Defaults to 1.E−3 (equivalent to -60 dB) for a
            harmonic analysis and 30 for a static structural analysis.

        xplus
            For harmonic analysis, normal reflection coefficient in positive X
            direction of ESYS. Defaults to 1.E−3 (equivalent to -60 dB) for a
            harmonic analysis and 30 for a static structural analysis.

        yminus
            For harmonic analysis, normal reflection coefficient in negative Y
            direction of ESYS. Defaults to 1.E−3 (equivalent to -60 dB) for a
            harmonic analysis and 30 for a static structural analysis.

        yplus
            For harmonic analysis, normal reflection coefficient in positive Y
            direction of ESYS. Defaults to 1.E−3 (equivalent to -60 dB) for a
            harmonic analysis and 30 for a static structural analysis.

        zminus
            For harmonic analysis, normal reflection coefficient in negative Z
            direction of ESYS. Defaults to 1.E−3 (equivalent to -60 dB) for a
            harmonic analysis and 30 for a static structural analysis.

        zplus
            For harmonic analysis, normal reflection coefficient in positive Z
            direction of ESYS. Defaults to 1.E−3 (equivalent to -60 dB) for a
            harmonic analysis and 30 for a static structural analysis.

        Notes
        -----
        PMLOPT defines perfectly matched layers (PML) for acoustic or
        structural analyses. Each PML region must have a uniquely defined
        element coordinate system. Normal reflection coefficient values for a
        harmonic analysis must be less than 1.

        Issue PMLOPT,STAT to list the current normal reflection coefficient or
        attenuation factor settings for a PML region. Issue PMLOPT,CLEAR to
        clear all normal reflection coefficient settings and restore them to
        the defaults. Issue PMLOPT,ESYS,CLEAR to clear all normal reflection
        coefficient settings for this element coordinate system and restore
        them to the defaults.

        """
        command = "PMLOPT,%s,%s,%s,%s,%s,%s,%s,%s" % (str(esys), str(lab), str(xminus), str(xplus), str(yminus), str(yplus), str(zminus), str(zplus))
        self.RunCommand(command, **kwargs)

    def Lccat(self, nl1="", nl2="", **kwargs):
        """
        APDL Command: LCCAT

        Concatenates multiple lines into one line for mapped meshing.

        Parameters
        ----------
        nl1, nl2
            Lines to be concatenated.  If NL1 = ALL, NL2 is ignored and all
            selected lines [LSEL] are concatenated.  If NL1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).  A component name may also be substituted
            for NL1 (NL2 is ignored).

        Notes
        -----
        Concatenates multiple, adjacent lines (the input lines) into one line
        (the output line) in preparation for mapped meshing.  An area that
        contains too many lines for mapped meshing can still be mapped meshed
        if some of the lines in that area are first concatenated (see Meshing
        Your Solid Model in the Modeling and Meshing Guide for details on
        mapped meshing restrictions).

        LCCAT is meant to be used solely for meshing and cannot be used for any
        other purposes.  Specifically, (a) the output line and any areas that
        have the output line on their line list [ALIST] cannot be used as input
        to any other solid modeling operation (not even another LCCAT command);
        and (b) the output line cannot accept solid model boundary conditions
        [DL, SFL].

        The output line will take on the element divisions of the input lines
        and will not accept element divisions that are directly assigned
        [LESIZE].  The output line from the LCCAT operation will be coincident
        with the input lines and the input lines will be retained.  Consider
        the LCOMB command instead of LCCAT if you wish to delete the input
        lines and if the lines to be combined have similar slopes at the common
        keypoint(s).  When an LCCAT command is issued, area line lists [ALIST]
        that contain all of the input lines will be updated so that the area
        line lists refer to the output line instead of the input lines.
        Deletion of the output line [LDELE] effectively reverses the LCCAT
        operation and restores area line lists to their original condition.

        You can use the LSEL command to select lines that were created by
        concatenation, and then follow it with an LDELE,ALL command to delete
        them.  Also see Meshing Your Solid Model in the Modeling and Meshing
        Guide for a discussion on how to easily select and delete concatenated
        lines in one step.

        """
        command = "LCCAT,%s,%s" % (str(nl1), str(nl2))
        self.RunCommand(command, **kwargs)

    def Soluopt(self, **kwargs):
        """
        APDL Command: SOLUOPT

        Specifies "Solution options" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "SOLUOPT," % ()
        self.RunCommand(command, **kwargs)

    def Sadd(self, labr="", lab1="", lab2="", fact1="", fact2="", const="",
             **kwargs):
        """
        APDL Command: SADD

        Forms an element table item by adding two existing items.

        Parameters
        ----------
        labr
            Label assigned to results.  If same as existing label, the existing
            values will be overwritten by these results.

        lab1
            First labeled result item in operation.

        lab2
            Second labeled result item in operation (may be blank).

        fact1
            Scale factor applied to Lab1. A (blank) or '0' entry defaults to
            1.0.

        fact2
            Scale factor applied to Lab2. A (blank) or '0' entry defaults to
            1.0.

        const
            Constant value.

        Notes
        -----
        Forms a labeled result (see ETABLE command) for the selected elements
        by adding two existing labeled result items according to the operation:

        LabR = (FACT1 x Lab1) + (FACT2 x Lab2) + CONST

        May also be used to scale results for a single labeled result item.  If
        absolute values are requested [SABS,1], absolute values of Lab1 and
        Lab2 are used.

        """
        command = "SADD,%s,%s,%s,%s,%s,%s" % (str(labr), str(lab1), str(lab2), str(fact1), str(fact2), str(const))
        self.RunCommand(command, **kwargs)

    def Mplist(self, mat1="", mat2="", inc="", lab="", tevl="", **kwargs):
        """
        APDL Command: MPLIST

        Lists linear material properties.

        Parameters
        ----------
        mat1, mat2, inc
            List materials from MAT1 to MAT2 (defaults to MAT1) in steps of INC
            (defaults to 1).  If MAT1= ALL (default), MAT2 and INC are ignored
            and properties for all material numbers are listed.

        lab
            Material property label (see the MP command for labels).  If ALL
            (or blank), list properties for all labels.  If EVLT, list
            properties for all labels evaluated at TEVL.

        tevl
            Evaluation temperature for Lab = EVLT listing (defaults to BFUNIF).

        Notes
        -----
        For Lab = EVLT, when the property is from tables, the MPPLOT command
        will not be valid because the property could be a function of more than
        temperature.

        This command is valid in any processor.

        """
        command = "MPLIST,%s,%s,%s,%s,%s" % (str(mat1), str(mat2), str(inc), str(lab), str(tevl))
        self.RunCommand(command, **kwargs)

    def Pdcdf(self, rlab="", name="", type="", conf="", nmax="", **kwargs):
        """
        APDL Command: PDCDF

        Plots the cumulative distribution function.

        Parameters
        ----------
        rlab
            Result set label. Identifies the result set to be used for
            postprocessing. A result set label can be the solution set label
            you defined in a PDEXE command (if you are directly postprocessing
            Monte Carlo Simulation results), or the response surface set label
            defined in an RSFIT command (for Response Surface Analyses).

        name
            Parameter name. The parameter must have been previously defined as
            a random input variable or a random output parameter with the PDVAR
            command.

        type
            Type of cumulative distribution curve to be displayed.

            EMP - Show an empirical cumulative distribution curve.

            GAUS - Show a cumulative distribution curve in a normal distribution plot. A random
                   variable based on a normal or Gaussian distribution appears
                   as a straight line in this type of plot.

            LOGN - Show a cumulative distribution curve in a log-normal plot. A random variable
                   based on a log-normal distribution appears as a straight
                   line in this type of plot.

            WEIB - Show a cumulative distribution curve in a Weibull distribution plot. A random
                   variable based on a Weibull distribution appears as a
                   straight line in this type of plot.

        conf
            Confidence level. The confidence level is used to plot confidence
            bounds around the cumulative distribution function. The value for
            the confidence level must be between 0.0 and 1.0 and it defaults to
            0.95 (95%). Plotting of the confidence bound is suppressed for CONF
            0.5. This parameter is ignored for the postprocessing of response
            surface methods results.

        nmax
            Maximum number of points to be plotted for the distribution curve.
            This must be a positive number and it defaults to 100. If the
            sample size is less than NMAX, all sample data is represented in
            the plot. If the sample size is larger than NMAX, the probabilistic
            design system classifies the sample into NMAX classes of
            appropriate size.

        Notes
        -----
        Plots the cumulative distribution function.

        The PDCDF command cannot be used to postprocess the results in a
        solution set that is based on Response Surface Methods, only Monte
        Carlo Simulations.

        If Rlab is left blank, then the result set label is inherited from the
        last PDEXE command (Slab), RSFIT command (RSlab), or the most recently
        used PDS postprocessing command where a result set label was explicitly
        specified.

        """
        command = "PDCDF,%s,%s,%s,%s,%s" % (str(rlab), str(name), str(type), str(conf), str(nmax))
        self.RunCommand(command, **kwargs)

    def Bfldele(self, line="", lab="", **kwargs):
        """
        APDL Command: BFLDELE

        Deletes body force loads on a line.

        Parameters
        ----------
        line
            Line at which body load is to be deleted.  If ALL, delete for all
            selected lines [LSEL].  A component name may also be substituted
            for LINE.

        lab
            Valid body load label. If ALL, use all appropriate labels. Load
            labels are listed under "Body Loads" in the input table for each
            element type in the Element Reference. See the BFL command for
            labels.

        Notes
        -----
        Deletes body force loads (and all corresponding finite element loads)
        for a specified line and label.  Body loads may be defined on a line
        with the BFL command.

        Graphical picking is available only via the listed menu paths.

        This command is also valid in PREP7.

        """
        command = "BFLDELE,%s,%s" % (str(line), str(lab))
        self.RunCommand(command, **kwargs)

    def Eslv(self, type="", **kwargs):
        """
        APDL Command: ESLV

        Selects elements associated with the selected volumes.

        Parameters
        ----------
        type
            Label identifying the type of element selected:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        Notes
        -----
        Selects volume elements belonging to meshed [VMESH], selected [VSEL]
        volumes.

        This command is valid in any processor.

        """
        command = "ESLV,%s" % (str(type))
        self.RunCommand(command, **kwargs)

    def Edread(self, nstart="", label="", num="", step1="", step2="",
               **kwargs):
        """
        APDL Command: EDREAD

        Reads explicit dynamics output into variables for time-history
        postprocessing.

        Parameters
        ----------
        nstart
            Starting reference number assigned to the first variable. Allowed
            range is 2 (the default) to NV [NUMVAR].  (NV defaults to 30 for an
            explicit dynamics analysis.)

        label
            Label identifying the output file to be read. No default.

            GLSTAT - Read data from the GLSTAT file.

            MATSUM - Read data from the MATSUM file.

            SPCFORC - Read data from the SPCFORC file.

            RCFORC - Read data from the RCFORC file.

            SLEOUT - Read data from the SLEOUT file.

            NODOUT - Read data from the NODOUT file.

            RBDOUT - Read data from the RBDOUT file.

        num
            Number identifying the data set to be read in (defaults to 1). If
            Label = GLSTAT, NUM is ignored. If Label = MATSUM or RBDOUT, NUM is
            the PART number [EDPART] for which output is desired. If Label =
            SPCFORC or NODOUT, NUM is the node number for which output is
            desired. If Label  = SLEOUT or RCFORC, NUM is the number of the
            contact entity for which output is desired.

        step1, step2
            Load step range of data to be read in. If STEP1 and STEP2 are
            blank, all load steps are read in.

        Notes
        -----
        EDREAD reads data from the specified ascii output file so that it may
        be used during postprocessing. After EDREAD, you must issue the STORE
        command to store the data in time history variables. Once stored, the
        variables can be viewed as plots of output item versus time.

        The number of variables stored depends on the file specified. The
        following table shows the items in each file and the order in which
        they are stored.  If data items were previously stored in variables
        NSTART to NSTART+15, they will be overwritten. If more variables are
        needed, change NV on the NUMVAR command. (Note that hourglass energy
        will not be available if it was not specified for the model
        [EDENERGY,1].)

        The following items under MATSUM are listed in the MATSUM ASCII file
        (in the Mat no. field) for each part number at time intervals specified
        by the EDHTIME command. Use EDREAD,,MATSUM,NUM to specify the part
        number that corresponds to the mat number in the MATSUM file.

        Resultant contact forces and sliding interface energies are available
        from the RCFORC and SLEOUT files, respectively. The RCFORC file is
        written for surface based contact types that include target and contact
        (master and slave) definitions.  You should ensure that this file
        contains valid force results before issuing EDREAD,,RCFORC. Only the
        resultant contact forces on the master surface are available for time-
        history postprocessing.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDREAD,%s,%s,%s,%s,%s" % (str(nstart), str(label), str(num), str(step1), str(step2))
        self.RunCommand(command, **kwargs)

    def Linv(self, nl="", nv="", **kwargs):
        """
        APDL Command: LINV

        Finds the intersection of a line with a volume.

        Parameters
        ----------
        nl
            Number of line to be intersected.  If NL = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).

        nv
            Number of volume to be intersected.

        Notes
        -----
        Finds the intersection of a line with a volume.  New lines will be
        generated where the lines intersect the volumes.  If the regions of
        intersection are only points, new keypoints will be generated instead.
        See the Modeling and Meshing Guide for an illustration.  See the BOPTN
        command for the options available to Boolean operations.  Element
        attributes and solid model boundary conditions assigned to the original
        entities will not be transferred to the new entities generated.

        """
        command = "LINV,%s,%s" % (str(nl), str(nv))
        self.RunCommand(command, **kwargs)

    def Msolve(self, numslv="", nrmtol="", nrmchkinc="", **kwargs):
        """
        APDL Command: MSOLVE

        Starts multiple solutions for random acoustics analysis with diffuse
        sound field.

        Parameters
        ----------
        numslv
            Number of multiple solutions (load steps) corresponding to the
            number of samplings. Default = 1.

        Notes
        -----
        The MSOLVE command starts multiple solutions (load steps) for random
        acoustics analysis with multiple samplings.

        The process is controlled by the norm convergence tolerance NRMTOL or
        the number of multiple solutions NUMSLV (if the solution steps reach
        the defined number).

        The program checks the norm convergence by comparing two averaged sets
        of radiated sound powers with the interval NRMCHKINC over the frequency
        range. For example, if NRMCHKINC = 5, the averaged values from 5
        solutions are compared with the averaged values from 10 solutions, then
        the averaged values from 10 solutions are compared with the averaged
        values from 15 solutions, and so on.

        The incident diffuse sound field is defined via the DFSWAVE command.

        The average result of multiple solutions with different samplings is
        calculated via the PLST command.

        """
        command = "MSOLVE,%s,%s,%s" % (str(numslv), str(nrmtol), str(nrmchkinc))
        self.RunCommand(command, **kwargs)

    def Dvmorph(self, volu="", xarea="", rmshky="", **kwargs):
        """
        APDL Command: DVMORPH

        Move nodes in selected volumes to conform to structural displacements.

        Parameters
        ----------
        volu
            Non-structural volume to which mesh movement (morph) applies.  If
            ALL, apply morphing to all selected volumes [VSEL]. If VOLU  = P,
            graphical picking is enabled. A component may be substituted for
            VOLU.

        xarea
            Areas to be excluded from morphing. If ALL, exclude all selected
            areas [ASEL].  If XAREA = P, graphical picking is enabled. A
            component may be substituted for XAREA. If XAREA is blank
            (default), allow morphing of nodes attached to areas of the
            selected volumes (VOLU) which are not shared by unselected volumes.
            (See Notes for clarification).

        rmshky
            Remesh flag option:

            0 - Remesh the selected non-structural volumes only if mesh morphing fails.

            1 - Remesh the selected non-structural volumes and bypass mesh morphing.

            2 - Perform mesh morphing only and do not remesh.

        Notes
        -----
        The selected volumes should include only non-structural regions
        adjacent to structural regions. DVMORPH will morph the non-structural
        volumes to coincide with the deflections of the structural regions.

        Nodes in the structural regions move in accordance with computed
        displacements. Displacements from a structural analysis must be in the
        database prior to issuing DVMORPH.

        By default, nodes attached to areas can move along the areas. You can
        use XAREA to restrain nodes on certain areas.

        By default (RMSHKY = 0), DVMORPH will remesh the selected non-
        structural volumes entirely if a satisfactory morphed mesh cannot be
        provided.

        If boundary conditions and loads are applied directly to nodes and
        elements, the DVMORPH command requires that these be removed before
        remeshing can take place.

        Exercise care with initial conditions defined by the IC command. Before
        a structural analysis is performed for a sequentially coupled analysis,
        the DVMORPH command requires that initial conditions be removed from
        all null element type nodes in the non-structural regions. Use ICDELE
        to delete the initial conditions.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "DVMORPH,%s,%s,%s" % (str(volu), str(xarea), str(rmshky))
        self.RunCommand(command, **kwargs)

    def Sumap(self, rsetname="", item="", comp="", **kwargs):
        """
        APDL Command: SUMAP

        Map results onto selected surface(s).

        Parameters
        ----------
        rsetname
            Eight-character name for the result being mapped.

        item
            Label identifying the item.

        comp
            Component label of item (if required).

        Notes
        -----
        The SUMAP command maps results in the current coordinate system (RSYS)
        using the selected set of elements.

        The command interpolates and stores the results data on to each of the
        selected surfaces.

        SUMAP,ALL,CLEAR deletes all results sets from all selected surfaces.

        """
        command = "SUMAP,%s,%s,%s" % (str(rsetname), str(item), str(comp))
        self.RunCommand(command, **kwargs)

    def Ematwrite(self, key="", **kwargs):
        """
        APDL Command: EMATWRITE

        Forces the writing of all the element matrices to File.EMAT.

        Parameters
        ----------
        key
            Write key:

            YES - Forces the writing of the element matrices to File.EMAT even if not normally
                  done.

            NO - Element matrices are written only if required. This value is the default.

        Notes
        -----
        The EMATWRITE command forces ANSYS to write the File.EMAT file. The
        file is necessary if you intend to follow the initial load step with a
        subsequent inertia relief calculation (IRLF). If used in the solution
        processor (/SOLU), this command is only valid within the first load
        step.

        This command is also valid in PREP7.

        """
        command = "EMATWRITE,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Pdef(self, lab="", item="", comp="", avglab="", **kwargs):
        """
        APDL Command: PDEF

        Interpolates an item onto a path.

        Parameters
        ----------
        lab
            Label assigned to the resulting path item (8 characters maximum).
            This item may be used as input for other path operations.

        item
            Label identifying the item for interpolation.  Valid item labels
            are shown in Table 216: PDEF - Valid Item and Component Labels
            below.  Some items also require a component label.

        comp
            Component of the item (if required).  Valid component labels are
            shown in Table 216: PDEF - Valid Item and Component Labels below.

        avglab
            Option to average across element boundaries:

            AVG - Average element results across elements (default).

            NOAV - Do not average element results across elements.  If the parameter DISCON = MAT
                   on the PMAP command, this option is automatically invoked.

        Notes
        -----
        Defines and interpolates a labeled path item along a predefined path
        (PATH).  Path item results are in the global Cartesian coordinate
        directions unless transformed (RSYS).  A path item must be defined
        before it can be used with other path operations.  Additional path
        items may be defined from the PVECT, PCALC, PDOT, and PCROSS commands.
        Path items may be listed (PRPATH) or displayed (PLPATH, PLPAGM).  A
        maximum number of path items permitted is established by the nSets
        argument specified with the PATH command.

        When you create the first path item (PDEF or PVECT), the program
        automatically interpolates four path items which are used to describe
        the geometry of the path.  These predefined items are the position of
        the interpolated path points (labels XG, YG, and ZG) in global
        Cartesian coordinates, and the path length (label S).  For alternate
        methods of mapping the path geometry (to include, for example, material
        discontinuity) see the PMAP command.  These items may also be listed or
        displayed with the PRPATH, PLPATH, and PLPAGM commands.

        If specifying that load case operations act on principal/equivalent
        stresses (SUMTYPE,PRIN), derived quantities (principal and equivalent
        stresses/strains) will be zero for path plots. A typical use for such a
        case involves mode combinations in a response spectrum analysis.

        The number of interpolation points on the path is defined by the nDiv
        argument on the PATH command.  See Mapping Nodal and Element Data onto
        the Path in the Mechanical APDL Theory Reference for details.  Use
        PDEF,STAT to list the path item labels.  Use PDEF,CLEAR to erase all
        labeled path items, except the path geometry items (XG, YG, ZG, S).

        Table: 216:: : PDEF - Valid Item and Component Labels

         For SHELL131 and SHELL132 elements with KEYOPT(3) = 0 or 1, use the
        labels TBOT, TE2, TE3, ..., TTOP instead of TEMP.

        For more information on the meaning of contact status and its possible
        values, see Reviewing Results in POST1 in the Contact Technology Guide.

        """
        command = "PDEF,%s,%s,%s,%s" % (str(lab), str(item), str(comp), str(avglab))
        self.RunCommand(command, **kwargs)

    def Mshmid(self, key="", **kwargs):
        """
        APDL Command: MSHMID

        Specifies placement of midside nodes.

        Parameters
        ----------
        key
            Key indicating placement of midside nodes:

            0 - Midside nodes (if any) of elements on a region boundary follow the curvature of
                the boundary line or area (the default).

            1 - Place midside nodes of all elements so that element edges are straight.  Allows
                coarse mesh along curves.

            2 - Do not create midside nodes (elements will have removed midside nodes).

        Notes
        -----
        The MSHMID, MSHAPE, and MSHKEY commands replace the functionality that
        was provided by the ESHAPE command in ANSYS 5.3 and earlier releases.

        This command is also valid for rezoning.

        """
        command = "MSHMID,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Mater(self, **kwargs):
        """
        APDL Command: MATER

        Specifies "Material properties" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "MATER," % ()
        self.RunCommand(command, **kwargs)

    def Upcoord(self, factor="", key="", **kwargs):
        """
        APDL Command: UPCOORD

        Modifies the coordinates of the active set of nodes, based on the
        current displacements.

        Parameters
        ----------
        factor
            Scale factor for displacements being added to nodal coordinates.
            If FACTOR = 1.0, the full displacement value will be added to each
            node, 0.5, half the displacement value will be added, etc.  If
            FACTOR = -1, the full displacement value will be subtracted from
            each node, etc.

        key
            Key for zeroing displacements in the database:

            OFF - Do not zero the displacements (default).

            ON - Zero the displacements.

        Notes
        -----
        The UPCOORD command uses displacements stored in the ANSYS database,
        and not those contained within the results file, Jobname.RST.  Nodal
        coordinates are updated each time the command is issued.  After
        updating, both the nodal displacements and rotations are set to zero if
        Key = ON.

        For structural solutions with an updated mesh, unless the coefficient
        matrix is otherwise reformed (e.g., a new analysis or NLGEOM,ON) it
        should first be reformed by issuing a KUSE,-1 command.

        UPCOORD should not be issued between load steps in structural analysis.

        For a multiphysics simulation where a CFD or electromagnetic field is
        being coupled to a structure undergoing large displacements, all (or a
        portion) of the surrounding field mesh may take part in the structural
        solution to "move" with the displacing structure.  You can use the
        UPCOORD command with a suitable FACTOR to update the coordinates of the
        nodes using the newly computed displacements.  The mesh will now
        conform with the displaced structure for subsequent field solutions.
        However, the mesh should always be restored to its original location by
        using an UPCOORD,FACTOR command before performing any subsequent
        structural solutions.  This is true for both repeated linear solutions,
        and for nonlinear restarts. (All saved displacements are relative to
        the original mesh location.)

        This command is not intended to replace either the large displacement
        or birth and death logic.

        This command is also valid in PREP7.

        """
        command = "UPCOORD,%s,%s" % (str(factor), str(key))
        self.RunCommand(command, **kwargs)

    def Pmap(self, form="", discon="", **kwargs):
        """
        APDL Command: PMAP

        Creates mapping of the path geometry by defining path interpolation
        division points.

        Parameters
        ----------
        form
            Defines the mapping method:

            UNIFORM - Maps uniform divisions (specified on the nDiv argument of the PATH command)
                      between specified points.  This is the default.

            ACCURATE - Map geometry using a small division at the beginning and end of each segment.
                       This gives you accurate derivatives, integrals,
                       tangents, and normals for curves which do not have
                       continuous slopes at the specified points. To create
                       nonuniform divisions, the nDiv argument of the PATH
                       command must be greater than 2.

        discon
            Sets mapping for discontinuities in the field.  The divisions are
            modified to put a point just before and just after the
            discontinuity.  The valid label is MAT, for a material
            discontinuity.  No discontinuity is the default.  Discontinuity
            mapping involves the NOAV option on the PDEF command.

        """
        command = "PMAP,%s,%s" % (str(form), str(discon))
        self.RunCommand(command, **kwargs)

    def Operate(self, **kwargs):
        """
        APDL Command: OPERATE

        Specifies "Operation data" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "OPERATE," % ()
        self.RunCommand(command, **kwargs)

    def Inquire(self, strarray="", func="", **kwargs):
        """
        APDL Command: /INQUIRE

        Returns system information to a parameter.

        Parameters
        ----------
        strarray
            Name of the "string array" parameter that will hold the returned
            values.  String array parameters are similar to character arrays,
            but each array element can be as long as 128 characters. If the
            string parameter does not exist, it will be created.

        Notes
        -----
        The /INQUIRE command is valid in any processor.

        """
        command = "/INQUIRE,%s,%s" % (str(strarray), str(func))
        self.RunCommand(command, **kwargs)

    def Al(self, l1="", l2="", l3="", l4="", l5="", l6="", l7="", l8="", l9="",
           l10="", **kwargs):
        """
        APDL Command: AL

        Generates an area bounded by previously defined lines.

        Parameters
        ----------
        l1, l2, l3, . . . , l10
            List of lines defining area.  The minimum number of lines is 3.
            The positive normal of the area is controlled by the direction of
            L1 using the right-hand rule.  A negative value of L1 reverses the
            normal direction.  If L1 = ALL, use all selected lines with L2
            defining the normal (L3 to L10 are ignored and L2 defaults to the
            lowest numbered selected line).  If L1 = P, graphical picking is
            enabled and all remaining arguments are ignored (valid only in the
            GUI).  A component name may also be substituted for L1.

        Notes
        -----
        Lines may be input (once each) in any order and must form a simply
        connected closed curve.  If the area is defined with more than four
        lines, the lines must also lie in the same plane or on a constant
        coordinate value in the active coordinate system (such as a plane or a
        cylinder).

        Note:: : Solid modeling in a toroidal coordinate system is not
        recommended.  Areas may be redefined only if not yet attached to a
        volume.

        This command is valid in any processor.

        """
        command = "AL,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(l1), str(l2), str(l3), str(l4), str(l5), str(l6), str(l7), str(l8), str(l9), str(l10))
        self.RunCommand(command, **kwargs)

    def Physics(self, option="", title="", fname="", ext="", **kwargs):
        """
        APDL Command: PHYSICS

        Writes, reads, or lists all element information

        Parameters
        ----------
        option
            Specifies what to do with element information:

            WRITE - Write all appropriate element types, key options, real constants, material
                    properties, solution analysis options, load step options,
                    constraint equations, coupled nodes, defined components,
                    and GUI preference settings to the file specified with the
                    Fname and Ext arguments.

            READ - Deletes all solution information (material properties, solution options, load
                   step options, constraint equations, coupled nodes, results,
                   and GUI preference settings) then reads all the information
                   listed above into the ANSYS database from the location
                   specified by the Fname and Ext arguments.

            LIST - Lists currently defined physics files and their titles.

            DELETE - Deletes a specified physics file and its title from the database.

            CLEAR - Deletes all material properties, solution options, load step options,
                    constraint equations, coupled nodes, results, and GUI
                    preference settings from the database. Does NOT clear the
                    active physics file title from the database.

            STATUS - Displays information about all active elements and settings.

        title
            A user-defined title that quickly identifies a set of physics
            settings.  For example, you might use "Fluid," "Structural," or
            "Magnetic" as titles.  A title can contain up to 64 characters. It
            can be entered in lower or upper case. Lower case is internally
            converted to upper case within the program.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        Use the PHYSICS command when you are performing a multiphysics analysis
        that involves two different disciplines (for example, magnetic and
        structural analyses) and you cannot solve both analyses simultaneously.
        Once you have set up physics environments for both analyses, you can
        use the PHYSICS,READ command to change between the defined physics
        environments.  For more information about doing multiphysics analyses,
        see Sequential Coupled-Field Analysis in the Coupled-Field Analysis
        Guide.

        The PHYSICS command outputs all solution information, including
        analysis options, to the Jobname.PHn file described above.  Although it
        also outputs components, the ANSYS program does not list entities
        (nodes, elements, lines, etc.).

        PHYSICS,WRITE will overwrite existing physics files with the same title
        (even if the name is different). In other words, if the directory has a
        physics file with the same title as the active physics file title, but
        a different name, the PHYSICS,WRITE command will overwrite the existing
        physics file and use the existing filename, not the filename specified
        on the PHYSICS,WRITE command.

        """
        command = "PHYSICS,%s,%s,%s,%s" % (str(option), str(title), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Demorph(self, elem="", dimn="", rmshky="", **kwargs):
        """
        APDL Command: DEMORPH

        Move nodes in selected elements to conform to structural displacements.

        Parameters
        ----------
        elem
             Non-structural elements to which mesh movement (morph) applies.
            If ALL, apply morphing to all selected elements [ESEL]. If ELEM =
            P, graphical picking is enabled.  A component may be substituted
            for ELEM.

        dimn
            Problem dimensionality.  Use "2" for a 2-D problem and "3" for a
            3-D problem (no default).

        rmshky
            Remesh flag option:

            0 - Remesh the selected non-structural regions only if mesh morphing fails.

            1 - Remesh the selected non-structural regions and bypass mesh morphing.

            2 - Perform mesh morphing only and do not remesh.

        Notes
        -----
        The selected elements should include only non-structural regions
        adjacent to structural regions. The exterior nodes of the selected
        elements will usually be on the boundary of the region which will have
        node positions displaced. For DIMN = 2, elements must lie on a flat
        plane. The DEMORPH command requires a single domain grouping of
        elements be provided (multiple domains of elements are not permitted).
        Exterior nodes will be assumed fixed (no nodes will be morphed) unless
        they coincide with structural nodes having nonzero displacements.

        Nodes in the structural regions move in accordance with computed
        displacements. Displacements from a structural analysis must be in the
        database prior to issuing DEMORPH.

        By default (RMSHKY = 0), DEMORPH will remesh the selected non-
        structural regions entirely if a satisfactory morphed mesh cannot be
        provided.

        If boundary conditions and loads are applied directly to nodes and
        elements, the DEMORPH command requires that these be removed before
        remeshing can take place.

        Exercise care with initial conditions defined by the IC command. Before
        a structural analysis is performed for a sequentially coupled analysis,
        the DEMORPH command requires that initial conditions be removed from
        all null element type nodes in the non-structural regions. Use ICDELE
        to delete the initial conditions.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "DEMORPH,%s,%s,%s" % (str(elem), str(dimn), str(rmshky))
        self.RunCommand(command, **kwargs)

    def Sscale(self, wn="", smult="", **kwargs):
        """
        APDL Command: /SSCALE

        Sets the contour multiplier for topographic displays.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        smult
            Contour multiplier that factors in results based on the product of
            the multiplier and the result being plotted. Defaults to 0.0 (no
            topographic effects).

        Notes
        -----
        Use this command to scale values to the geometry when the contours are
        shown elevated.  For section displays [/TYPE], the elevation is
        performed perpendicular to the section face.

        Nonzero contour multipliers factoring in large results (stresses or
        displacements) can produce very large distortion, causing images to
        disappear.  To bring a distorted image back into view, reduce the
        contour multiplier value.

        Portions of this command are not supported by PowerGraphics
        [/GRAPHICS,POWER].

        """
        command = "/SSCALE,%s,%s" % (str(wn), str(smult))
        self.RunCommand(command, **kwargs)

    def Bfescal(self, lab="", fact="", tb_ase="", **kwargs):
        """
        APDL Command: BFESCAL

        Scales element body force loads.

        Parameters
        ----------
        lab
            Valid body load label.  If ALL, use all appropriate labels.

        fact
            Scale factor for the element body load values.  Zero (or blank)
            defaults  to 1.0.  Use a small number for a "zero" scale factor.
            The scale factor is not applied to body load phase angles.

        tbase
            Base temperature for temperature difference.  Used only with Lab =
            TEMP.  Scale factor is applied to the temperature difference (T -
            TBASE)  and then added to TBASE.  T is the current temperature.

        Notes
        -----
        Scales element body force loads on the selected elements in the
        database.  Issue the BFELIST command to list the element body loads.
        Solid model boundary conditions are not scaled by this command, but
        boundary conditions on the FE model are scaled.  (Note that such scaled
        FE boundary conditions may still be overwritten by unscaled solid model
        boundary conditions if a subsequent boundary condition transfer
        occurs.)

        BFESCAL does not work for tabular boundary conditions.

        This command is also valid in PREP7.

        """
        command = "BFESCAL,%s,%s,%s" % (str(lab), str(fact), str(tb_ase))
        self.RunCommand(command, **kwargs)

    def Dldele(self, line="", lab="", **kwargs):
        """
        APDL Command: DLDELE

        Deletes DOF constraints on a line.

        Parameters
        ----------
        line
            Line for which constraints are to be deleted.  If ALL, delete for
            all selected lines [LSEL].  If LINE = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be substituted for LINE

        lab
            Constraint label:

            ALL - All constraints.

            SYMM - Symmetry constraints.

            ASYM - Antisymmetry constraints.

            UX - Displacement in X direction.

            UY - Displacement in Y direction.

            UZ - Displacement in Z direction.

            ROTX - Rotation about X axis.

            ROTY - Rotation about Y axis.

            ROTZ - Rotation about Z axis.

            WARP - Warping magnitude.

            PRES - Pressure.

            TEMP, TBOT, TE2, TE3, . . ., TTOP - Temperature.

            VOLT - Electric scalar potential.

            AX - Magnetic vector potential in X direction.

            AY - Magnetic vector potential in Y direction.

            AZ - Magnetic vector potential in Z direction.

            CONC - Concentration.

        Notes
        -----
        Deletes the degree of freedom constraints (and all corresponding finite
        element constraints) on a line previously specified with the DL
        command.  See the DDELE command for delete details.

        Warning:: : On previously meshed lines, all constraints on affected
        nodes will also be deleted, whether or not they were specified by the
        DL command.

        This command is also valid in PREP7.

        """
        command = "DLDELE,%s,%s" % (str(line), str(lab))
        self.RunCommand(command, **kwargs)

    def Ftcalc(self, nloc="", node="", **kwargs):
        """
        APDL Command: FTCALC

        Performs fatigue calculations for a particular node location.

        Parameters
        ----------
        nloc
            Location number of stress conditions to be used for fatigue
            calculation.

        node
            Node number (used only for convenience if NLOC is not input).

        """
        command = "FTCALC,%s,%s" % (str(nloc), str(node))
        self.RunCommand(command, **kwargs)

    def Fl(self, nloc="", node="", scfx="", scfy="", scfz="", title="",
           **kwargs):
        """
        APDL Command: FL

        Defines a set of fatigue location parameters.

        Parameters
        ----------
        nloc
            Reference number for this location (within MXLOC).  When defining a
            new location, defaults to lowest unused location.  If the specified
            NODE is already associated with a location, NLOC defaults to that
            existing location.

        node
            Node number corresponding to this location (must be unique).  Used
            only to associate a node with a new location or to find an existing
            location (if NLOC is not input).  If NODE = -1 (or redefined),
            erase all parameters and fatigue stresses for this location.

        scfx, scfy, scfz
            Stress concentration factors applied to the total stresses.
            Factors are applied in the global X, Y, and Z directions unless the
            axisymmetric option of the FSSECT is used (i.e., RHO is nonzero),
            in which case the factors are applied in the section x, y, and z
            (radial, axial, and hoop) directions.

        title
            User-defined title for this location (up to 20 characters).

        Notes
        -----
        Repeat FL command to define additional sets of location parameters
        (MXLOC limit), to redefine location parameters, or to delete location
        stress conditions.

        One location must be defined for each node of interest and only one
        node can be associated with each location.  See the FTSIZE command for
        the maximum locations (MXLOC) allowed.  A location will be
        automatically defined for a node not having a location when the FSSECT,
        FSNODE, or FS command is issued.  Automatically defined locations are
        assigned the lowest available location number, unity stress
        concentration factors, and no title.

        """
        command = "FL,%s,%s,%s,%s,%s,%s" % (str(nloc), str(node), str(scfx), str(scfy), str(scfz), str(title))
        self.RunCommand(command, **kwargs)

    def Rmmselect(self, nmode="", method="", dmin="", dmax="", **kwargs):
        """
        APDL Command: RMMSELECT

        Selects modes for the ROM method.

        Parameters
        ----------
        nmode
            Total number of modes to be selected

        method
            Method for mode selection. Valid methods are:

            TMOD - Automated selection using a test load. TMOD must be enclosed in single quotes.

        dmin
            Lower bound for total deflection range.

        dmax
            Upper bound for total deflection range.

        Notes
        -----
        Select pertinent modes for use in a ROM. Pertinent mode selection may
        be enhanced by using the deflection state of the structure
        representative of the operating nature of the device (Method = TMOD). A
        static analysis with an applied Test Load may be used.  The test load
        displacements must be extracted at the neutral plane of the device (if
        the device is stress-stiffened), or at any plane of the device (non-
        stress-stiffened). A node component "NEUN" must be defined for the
        plane of nodes, and the displacements extracted using the RMNDISP
        command prior to issuing this command. If Method = NMOD, use the first
        Nmode eigenmodes to select the pertinent modes for the ROM tool. Only
        those modes are selected that act in the operating direction of the
        structure [RMANL].

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        jobname.evx, jobname.evy, jobname.evz, jobname.evn, jobname.evl

        Test load and element load neutral plane displacement files:
        jobname.tld, jobname.eld

        """
        command = "RMMSELECT,%s,%s,%s,%s" % (str(nmode), str(method), str(dmin), str(dmax))
        self.RunCommand(command, **kwargs)

    def Lfsurf(self, sline="", tline="", **kwargs):
        """
        APDL Command: LFSURF

        Generates surface elements overlaid on the edge of existing solid
        elements and assigns the extra node as the closest fluid element node.

        Parameters
        ----------
        sline
            Component name for the surface lines of the meshed solid areas.

        tline
            Component name for the target lines meshed with fluid elements.

        Notes
        -----
        This command macro is used to generate surface effect elements overlaid
        on the surface of existing plane elements and, based on proximity, to
        determine and assign the extra node for each surface element.  The
        underlying areas of the solid region and the fluid lines must be meshed
        prior to calling this command macro. The active element type must be
        SURF151 with appropriate settings for KEYOPT(4), KEYOPT(5), KEYOPT(6),
        and KEYOPT(8).

        The surface lines of the solid and the target lines of the fluid are
        grouped into components and named using the CM command.  The names must
        be enclosed in single quotes (e.g., 'SLINE') when the LFSURF command is
        manually typed in.

        When using the GUI method, node and element components are created
        through the picking dialog boxes associated with this command.

        The macro is applicable for the SURF151 and FLUID116 element types.

        """
        command = "LFSURF,%s,%s" % (str(sline), str(tline))
        self.RunCommand(command, **kwargs)

    def Plf2d(self, ncont="", olay="", anum="", win="", **kwargs):
        """
        APDL Command: PLF2D

        Generates a contour line plot of equipotentials.

        Parameters
        ----------
        ncont
            Number of contour lines to display.  Issue in multiples of 9 (i.e.,
            9, 18, 27, etc.).  Default is 27 contour lines.

        olay
            Overlay:

            0 - Overlay edge outlines by material number.

            1 - Overlay edge outlines by real constant number.

        anum
            Highest material or real constant attribute number.  Command will
            cycle through ANUM element display overlays.  Defaults to 10.

        win
            Window number to which command applies.  Defaults to 1.

        Notes
        -----
        PLF2D invokes an ANSYS macro which plots equipotentials of the degree
        of freedom AZ.  These equipotential lines are parallel to flux lines
        and thus give a good representation of flux patterns.  In the
        axisymmetric case, the display is actually r*AZ where "r" is the node
        radius.  The macro overlays (OLAY) edge outlines by material number or
        real constant number (ANUM) and allows user control over the number of
        contour lines to display (NCONT).

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PLF2D,%s,%s,%s,%s" % (str(ncont), str(olay), str(anum), str(win))
        self.RunCommand(command, **kwargs)

    def Vmask(self, par="", **kwargs):
        """
        APDL Command: *VMASK

        Specifies an array parameter as a masking vector.

        Parameters
        ----------
        par
            Name of the mask parameter.  The starting subscript must also be
            specified.

        Notes
        -----
        Specifies the name of the parameter whose values are to be checked for
        each resulting row operation.  The mask vector usually contains only 0
        (for false) and 1 (for true) values.  For each row operation the
        corresponding mask vector value is checked.  A true value allows the
        operation to be done.  A false value skips the operation (and retains
        the previous results).  A mask vector can be created from direct input,
        such as M(1) = 1,0,0,1,1,0,1; or from the DATA function of the *VFILL
        command.  The NOT function of the *VFUN command can be used to reverse
        the logical sense of the mask vector.  The logical compare operations
        (LT, LE, EQ, NE, GE, and GT) of the *VOPER command also produce a mask
        vector by operating on two other vectors.  Any numeric vector can be
        used as a mask vector since the actual interpretation assumes values
        less than 0.0 are 0.0 (false) and values greater than 0.0 are 1.0
        (true).  If the mask vector is not specified (or has fewer values than
        the result vector), true (1.0) values are assumed for the unspecified
        values.  Another skip control may be input with NINC on the *VLEN
        command.  If both are present, operations occur only when both are
        true.  The mask setting is reset to the default (no mask) after each
        *VXX or *MXX operation.  Use *VSTAT to list settings.

        This command is valid in any processor.

        """
        command = "*VMASK,%s" % (str(par))
        self.RunCommand(command, **kwargs)

    def Toper(self, parr="", par1="", oper="", par2="", fact1="", fact2="",
              con1="", **kwargs):
        """
        APDL Command: *TOPER

        Operates on table parameters.

        Parameters
        ----------
        parr
            Name of the resulting table parameter. The command will create a
            table array parameter with this name.  Any existing parameter with
            this name will be overwritten.

        par1
            Name of the first table parameter.

        oper
            The operation to be performed: ADD.  The operation is:  ParR(i,j,k)
            =   FACT1*Par1(i,j,k) + FACT2 *Par2(i,j,k) +CON1

        par2
            Name of the second table parameter.

        fact1
            The first table parameter multiplying constant. Defaults to 1.

        fact2
            The second table parameter multiplying constant.  Defaults to 1.

        con1
            The constant increment for offset.  Defaults to 0.

        Notes
        -----
        *TOPER operates on table parameters according to: ParR(i,j,k) =
        FACT1*Par1(i,j,k) + FACT2 *Par2(i,j,k) +CON1

        Par1 and Par2 must have the same dimensions and the same variable names
        corresponding to those dimensions. Par1 and Par2 must also have
        identical index values for rows, columns, etc.

        If you want a local coordinate system for the resulting array, you must
        dimension it as such using the *DIM command before issuing *TOPER.

        This command is valid in any processor.

        """
        command = "*TOPER,%s,%s,%s,%s,%s,%s,%s" % (str(parr), str(par1), str(oper), str(par2), str(fact1), str(fact2), str(con1))
        self.RunCommand(command, **kwargs)

    def Atype(self, **kwargs):
        """
        APDL Command: ATYPE

        Specifies "Analysis types" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "ATYPE," % ()
        self.RunCommand(command, **kwargs)

    def Calc(self, **kwargs):
        """
        APDL Command: CALC

        Specifies "Calculation settings" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "CALC," % ()
        self.RunCommand(command, **kwargs)

    def Deriv(self, ir="", iy="", ix="", name="", facta="", **kwargs):
        """
        APDL Command: DERIV

        Differentiates a variable.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]).  If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.

        iy, ix
            Reference numbers of variables to be operated on.  IY is
            differentiated with respect to IX.

        --
            Unused field.

        name
            Thirty-two character name for identifying the variable on printouts
            and displays. Embedded blanks are compressed for output.

        --, --
            Unused fields.

        facta
            Scaling factor (positive or negative) applied as shown below
            (defaults to 1.0).

        Notes
        -----
        Differentiates variables according to the operation:

        IR = FACTA x d(IY)/d(IX)

        """
        command = "DERIV,%s,%s,%s,%s,%s" % (str(ir), str(iy), str(ix), str(name), str(facta))
        self.RunCommand(command, **kwargs)

    def Cmdomega(self, cm_name="", domegax="", domegay="", domegaz="", x1="",
                 y1="", z1="", x2="", y2="", z2="", **kwargs):
        """
        APDL Command: CMDOMEGA

        Specifies the rotational acceleration of an element component about a
        user-defined rotational axis.

        Parameters
        ----------
        cm_name,
            The name of the element component.

        domegax, domegay, domegaz
            If the X2, Y2, Z2 fields are not defined, DOMEGAX, DOMEGAY, and
            DOMEGAZ specify the components of the rotational acceleration
            vector in the global Cartesian X, Y, Z directions.

        x1, y1, z1
            If the X2, Y2, Z2 fields are defined, X1, Y1, and Z1 define the
            coordinates of the beginning point of the rotational axis vector.
            Otherwise, X1, Y1, and Z1 are the coordinates of a point through
            which the rotational axis passes.

        x2, y2, z2
            The coordinates of the end point of the rotational axis vector.

        Notes
        -----
        Specifies the rotational acceleration components DOMEGAX, DOMEGAY, and
        DOMEGAZ of an element component CM_NAME about a user-defined rotational
        axis. The rotational axis can be defined either as a vector passing
        through a single point, or a vector connecting two points.

        You can define the rotational acceleration and rotational axis with the
        CMDOMEGA command for STATIC, HARMIC, TRANS, and SUBSTR analyses.
        Rotational velocities are combined with the element mass matrices to
        form a body force load vector term. Units are radians/time2.

        The CMDOMEGA command supports tabular boundary conditions (%TABNAME_X%,
        %TABNAME_Y%, and %TABNAME_Z%) for CMDOMEGA_X, CMDOMEGA_Y, and
        CMDOMEGA_Z input values (*DIM) for full transient and harmonic
        analyses.

        Related commands are ACEL, CGLOC, CGLOC, OMEGA, CMOMEGA, DCGOMG,
        DOMEGA.

        See Analysis Tools in the Mechanical APDL Theory Reference for more
        information.

        You can use the CMDOMEGA command in conjunction with any one of the
        following two groups of commands, but not with both groups
        simultaneously:

        Components for which you want to specify rotational loading must
        consist of elements only. The elements you use cannot be part of more
        than one component, and elements that share nodes cannot exist in
        different element components. You cannot apply the loading to an
        assembly of element components.

        In a modal harmonic or transient analysis, you must apply the load in
        the modal portion of the analysis. Mechanical APDL calculates a load
        vector and writes it to the mode shape file, which you can apply via
        the LVSCALE command.

        See Acceleration Effect in the Mechanical APDL Theory Reference for
        more information.

        This command is also valid in PREP7.

        """
        command = "CMDOMEGA,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(cm_name), str(domegax), str(domegay), str(domegaz), str(x1), str(y1), str(z1), str(x2), str(y2), str(z2))
        self.RunCommand(command, **kwargs)

    def Keyopt(self, itype="", knum="", value="", **kwargs):
        """
        APDL Command: KEYOPT

        Sets element key options.

        Parameters
        ----------
        itype
            Element type number as defined on the ET command. The label GCN is
            also valid input for general contact elements (see Notes).

        knum
            Number of the KEYOPT to be defined (KEYOPT(KNUM)).

        value
            Value of this KEYOPT.

        Notes
        -----
        Alternative to inputting KEYOPT values on ET command.  Must be used if
        KEYOPT(7) or greater values are to be input. ITYPE must first be
        defined with the ET command.

        Specify ITYPE = GCN to set element key options for all contact elements
        types used in any existing general contact definitions (that is,
        contact elements having a real constant set number = 0).

        """
        command = "KEYOPT,%s,%s,%s" % (str(itype), str(knum), str(value))
        self.RunCommand(command, **kwargs)

    def Bcsoption(self, memory_option="", memory_size="", solve_info="",
                  **kwargs):
        """
        APDL Command: BCSOPTION

         Sets memory option for the sparse solver.

        Parameters
        ----------
        --
            Unused field

        memory_option
            Memory allocation option:

            DEFAULT - Use the default memory allocation strategy for the sparse solver. The default
                      strategy attempts to run in the INCORE memory mode. If
                      there is not enough available physical memory when the
                      solver starts to run in the INCORE memory mode, the
                      solver will then attempt to run in the OUTOFCORE memory
                      mode.

            INCORE -  Use a memory allocation strategy in the sparse solver that will attempt to
                     obtain enough memory to run with the entire factorized
                     matrix in memory. This option uses the most amount of
                     memory and should avoid doing any I/O. By avoiding I/O,
                     this option achieves optimal solver performance. However,
                     a significant amount of memory is required to run in this
                     mode, and it is only recommended on machines with a large
                     amount of memory. If the allocation for in-core memory
                     fails, the solver will automatically revert to out-of-core
                     memory mode.

            OUTOFCORE - Use a memory allocation strategy in the sparse solver that will attempt to
                        allocate only enough work space to factor each
                        individual frontal matrix in memory, but will store the
                        entire factorized matrix on disk. Typically, this
                        memory mode results in poor performance due to the
                        potential bottleneck caused by the I/O to the various
                        files written by the solver.

            FORCE - This option, when used in conjunction with the Memory_Size option, allows you
                    to force the sparse solver to run with a specific amount of
                    memory. This option is only recommended for the advanced
                    user who understands sparse solver memory requirements for
                    the problem being solved, understands the physical memory
                    on the system, and wants to control the sparse solver
                    memory usage.

        memory_size
            Initial memory size allocation for the sparse solver in MB. This
            argument allows you to tune the sparse solver memory and is not
            generally required. Although there is no upper limit for
            Memory_Size, the Memory_Size setting should always be well within
            the physical memory available, but not so small as to cause the
            sparse solver to run out of memory. Warnings and/or errors from the
            sparse solver will appear if this value is set too low. If the
            FORCE memory option is used, this value is the amount of memory
            allocated for the entire duration of the sparse solver solution.

        --, --
            Unused fields

        solve_info
            Solver output option:

            OFF - Turns off additional output printing from the sparse solver (default).

            PERFORMANCE - Turns on additional output printing from the sparse solver, including a
                          performance summary and a summary of file I/O for the
                          sparse solver. Information on memory usage during
                          assembly of the global matrix (that is, creation of
                          the Jobname.FULL file) is also printed with this
                          option.

        Notes
        -----
        This command controls options related to the sparse solver in all
        analysis types where the sparse solver can be used. It also controls
        the Block Lanczos eigensolver in a modal or buckling analysis.

        The sparse solver runs from one large work space (that is, one large
        memory allocation). The amount of memory required for the sparse solver
        is unknown until the matrix structure is preprocessed, including
        equation reordering. The amount of memory allocated for the sparse
        solver is then dynamically adjusted to supply the solver what it needs
        to compute the solution.

        If you have a very large memory system, you may want to try selecting
        the INCORE memory mode for larger jobs to improve performance. When
        running the sparse solver on a machine with very slow I/O performance
        (for example, slow hard drive speed), you may want to try using the
        INCORE memory mode to achieve better performance. However, doing so may
        require much more memory compared to running in the OUTOFCORE memory
        mode.

        Running with the INCORE memory mode is best for jobs which comfortably
        fit within the limits of the physical memory on a given system. If the
        sparse solver work space exceeds physical memory size, the system will
        be forced to use virtual memory (or the system page/swap file). In this
        case, it is typically more efficient to run with the OUTOFCORE memory
        mode. Assuming the job fits comfortably within the limits of the
        machine, running with the INCORE memory mode is often ideal for jobs
        where repeated solves are performed for a single matrix factorization.
        This occurs in a modal or buckling analysis or when doing multiple load
        steps in a linear, static analysis.

        For repeated runs with the sparse solver, you may set the initial
        sparse solver memory allocation to the amount required for
        factorization. This strategy reduces the frequency of allocation and
        reallocation in the run to make the INCORE option fully effective. If
        you have a very large memory system, you may use the Memory_Size
        argument to increase the maximum size attempted for in-core runs.

        """
        command = "BCSOPTION,%s,%s,%s" % (str(memory_option), str(memory_size), str(solve_info))
        self.RunCommand(command, **kwargs)

    def Filldata(self, ir="", lstrt="", lstop="", linc="", value="", dval="",
                 **kwargs):
        """
        APDL Command: FILLDATA

        Fills a variable by a ramp function.

        Parameters
        ----------
        ir
            Define data table as variable IR (2 to NV [NUMVAR]).

        lstrt
            Start at location LSTRT (defaults to 1).

        lstop
            Stop at location LSTOP (defaults to maximum location as determined
            from data previously stored.

        linc
            Fill every LINC location between LSTRT and LSTOP (defaults to 1).

        value
            Value assigned to location LSTRT.

        dval
            Increment value of previous filled location by DVAL and assign sum
            to next location to be filled (may be positive or negative.)

        Notes
        -----
        Locations may be filled continuously or at regular intervals (LINC).
        Previously defined data at a location will be overwritten.

        """
        command = "FILLDATA,%s,%s,%s,%s,%s,%s" % (str(ir), str(lstrt), str(lstop), str(linc), str(value), str(dval))
        self.RunCommand(command, **kwargs)

    def Psdgraph(self, tblno1="", tblno2="", **kwargs):
        """
        APDL Command: PSDGRAPH

        Displays input PSD curves

        Parameters
        ----------
        tblno1
            PSD table number to display.

        tblno2
            Second PSD table number to display. TBLNO2 is used only in
            conjunction with the COVAL or the QDVAL commands.

        Notes
        -----
        The input PSD tables are displayed in log-log format as dotted lines.
        The best-fit curves, used to perform the closed-form integration, are
        displayed as solid lines. If there is a significant discrepancy between
        the two, then you should add one or more intermediate points to the
        table to obtain a better fit.

        If TBLNO2 is zero, blank, or equal to TBLNO1, then the autospectra
        (PSDVAL) are displayed for TBLNO1. If TBLNO2 is also specified, then
        the autospectra for TBLNO1 and TBLNO2 are displayed, along with the
        corresponding cospectra (COVAL) and quadspectra (QDVAL), if they are
        defined.

        This command is valid in any processor.

        """
        command = "PSDGRAPH,%s,%s" % (str(tblno1), str(tblno2))
        self.RunCommand(command, **kwargs)

    def Bftran(self, **kwargs):
        """
        APDL Command: BFTRAN

        Transfers solid model body force loads to the finite element model.

        Notes
        -----
        Body loads are transferred from selected keypoints and lines to
        selected nodes and from selected areas and volumes to selected
        elements.  The BFTRAN operation is also done if the SBCTRAN command is
        either explicitly issued or automatically issued upon initiation of the
        solution calculations [SOLVE].

        This command is also valid in PREP7.

        """
        command = "BFTRAN," % ()
        self.RunCommand(command, **kwargs)

    def Taxis(self, parmloc="", naxis="", val1="", val2="", val3="", val4="",
              val5="", val6="", val7="", val8="", val9="", val10="", **kwargs):
        """
        APDL Command: *TAXIS

        Defines table index numbers.

        Parameters
        ----------
        parmloc
            Name and starting location in the table array parameter for
            indexing. Indexing occurs along the axis defined with nAxis.

        naxis
            Axis along which indexing occurs.  Valid labels are:

            Corresponds to Row. Default. - Corresponds to Column.

            Corresponds to Plane. - Corresponds to Book.

            Corresponds to Shelf. - Lists all index numbers. Valid only if Val1 = LIST.

        val1, val2, val3, . . . , val10
            Values of the index numbers for the axis nAxis, starting from the
            table array parameter location ParmLoc. You can define up to ten
            values.

        Notes
        -----
        *TAXIS is a convenient method to define table index values. These
        values reside in the zero column, row, etc. Instead of filling values
        in these zero location spots, use the *TAXIS command. For example,

         would fill index values 1.0, 2.2, 3.5, 4.7, and 5.9 in nAxis 2 (column
        location), starting at location 4.

        To list index numbers, issue *TAXIS,ParmLoc, nAxis, LIST, where nAxis =
        1 through 5 or ALL.

        """
        command = "*TAXIS,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(parmloc), str(naxis), str(val1), str(val2), str(val3), str(val4), str(val5), str(val6), str(val7), str(val8), str(val9), str(val10))
        self.RunCommand(command, **kwargs)

    def Pltrac(self, analopt="", item="", comp="", trpnum="", name="",
               mxloop="", toler="", option="", escl="", mscl="", **kwargs):
        """
        APDL Command: PLTRAC

        Displays a particle flow or charged particle trace on an element
        display.

        Parameters
        ----------
        analopt
            Analysis option

            FLUID - Particle trace in fluid flow (default)

            ELEC - Particle trace in electric field

            MAGN - Particle trace in magnetic field

            EMAG - Particle trace in presence of both electric and magnetic fields

        item
            Label identifying the item to be contoured.  Valid item labels are
            shown in Table 222: PLTRAC - Valid Item and Component Labels
            below.  Some items also require a component label.  If Item is
            blank, display only the path trajectory.

        comp
            Component of the item (if required).  Valid component labels are
            shown in Table 222: PLTRAC - Valid Item and Component Labels below.

        trpnum
            Trace point number for storing trajectory data for use with PATH
            logic. Defaults to 0 (no trajectory path data is stored for further
            processing with PATH logic).

        name
            Name of prefix of array variable. Defaults to TRAC. NamePOIN stores
            trajectory path points for trace point number TRPNum. If Analopt =
            ELEC, MAGN, or EMAG, two additional array parameters, NameDATA and
            NameLABL, store trajectory path data and labels for the same
            TRPNum.

        mxloop
            Maximum number of loops traced by a particle.  Defaults to 25 for
            Opt = FLUID; otherwise, defaults to 1000.

        toler
            Length tolerance used for particle trajectory geometry calculation.
            Valid only for Analopt = ELEC, MAGN, or EMAG.  If particle trace
            appears to terminate inside an element, adjusting the length
            tolerance may be necessary.  Defaults to 1.0 x 10-8.

        option
            Flow trace option:

            0 - Use the undeformed mesh for computing the flow trace.

            1 - Use the deformed mesh for computing the flow trace.

        escl
            Electric field scale factor. Setting this scale factor affects only
            the tracing, not the field solution results. A negative factor
            corresponds to the opposite vector direction. Valid only for
            Analopt = ELEC or EMAG. Defaults to 1.

        mscl
            Magnetic field scale factor. Setting this scale factor affects only
            the tracing, not the field solution results. A negative factor
            corresponds to the opposite vector direction. Valid only for
            Analopt = MAGN or EMAG. Defaults to 1.

        Notes
        -----
        For a specified item, the variation of the item is displayed along the
        particle trace as a color-contoured ribbon.  The TRPOIN command must be
        used to define a point on the trajectory path.  Multiple traces may be
        displayed simultaneously by defining multiple trace points.  Issue the
        TRPLIS command to list the current tracing points.  Issue the TRPDEL
        command to delete tracing points defined earlier.   Use the PAPUT
        command with the POIN option to retrieve the particle trajectory points
        as path points.

        The model must be 3-D for the ELEC, MAGN, and EMAG analysis options.

        Three array parameters are created at the time of the particle trace:
        TRACPOIN, TRACDATA and TRACLABL. These array parameters can be used to
        put the particle velocity and the elapsed time into path form.   The
        procedure to put the arrays into a path named PATHNAME is as follows:

        Not used if Analopt = FLUID.  If working in the GUI, use the "All
        information" option to retrieve information from all three arrays at
        once.

        If OPTION is set to 1, the deformed mesh is based on the displacement
        degrees of freedom UX, UY, and UZ, which must be available in the load
        step.

        Table: 222:: : PLTRAC - Valid Item and Component Labels

        See the Basic Analysis Guide for more information on particle flow and
        charged particle traces.  See Animation in the Basic Analysis Guide for
        information on particle trace animation.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PLTRAC,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(analopt), str(item), str(comp), str(trpnum), str(name), str(mxloop), str(toler), str(option), str(escl), str(mscl))
        self.RunCommand(command, **kwargs)

    def Kplot(self, np1="", np2="", ninc="", lab="", **kwargs):
        """
        APDL Command: KPLOT

        Displays the selected keypoints.

        Parameters
        ----------
        np1, np2, ninc
            Display keypoints from NP1 to NP2 (defaults to NP1) in steps of
            NINC (defaults to 1).  If NP1 = ALL (default), NP2 and NINC are
            ignored and all selected keypoints [KSEL] are displayed.

        lab
            Determines what keypoints are plotted (one of the following):

            (blank) - Plots all keypoints.

            HPT - Plots only those keypoints that are hard points.

        Notes
        -----
        This command is valid in any processor.

        """
        command = "KPLOT,%s,%s,%s,%s" % (str(np1), str(np2), str(ninc), str(lab))
        self.RunCommand(command, **kwargs)

    def Lsum(self, **kwargs):
        """
        APDL Command: LSUM

        Calculates and prints geometry statistics of the selected lines.

        Notes
        -----
        Calculates and prints geometry statistics (length, centroid, moments of
        inertia, etc.) associated with the selected lines.  Geometry items are
        reported in the global Cartesian coordinate system.  A unit density is
        assumed, irrespective of any material associations [LATT, MAT].  Items
        calculated by LSUM and later retrieved by a *GET or *VGET command are
        valid only if the model is not modified after the LSUM command is
        issued.

        """
        command = "LSUM," % ()
        self.RunCommand(command, **kwargs)

    def Bss2(self, val1="", val2="", t="", **kwargs):
        """
        APDL Command: BSS2

        Specifies the transverse shear strain and force relationship in plane
        XY for beam sections.

        Parameters
        ----------
        val1
            Transverse shear strain component (γ2).

        val2
            Transverse shear force component (S2).

        t
            Temperature.

        Notes
        -----
        The behavior of beam elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        The BSS1 command, one of several nonlinear general beam section
        commands, specifies the transverse shear strain and transverse shear
        force relationship for plane XY of a beam section. The section data
        defined is associated with the section most recently defined (via the
        SECTYPE command).

        Unspecified values default to zero.

        Related commands are BSAX, BSM1, BSM2, BSTQ, BSS1, BSMD, and BSTE.

        For complete information, see Using Nonlinear General Beam Sections.

        """
        command = "BSS2,%s,%s,%s" % (str(val1), str(val2), str(t))
        self.RunCommand(command, **kwargs)

    def Esln(self, type="", ekey="", nodetype="", **kwargs):
        """
        APDL Command: ESLN

        Selects those elements attached to the selected nodes.

        Parameters
        ----------
        type
            Label identifying the type of element selected:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        ekey
            Node set key:

            0 - Select element if any of its nodes are in the selected nodal set (default).

            1 - Select element only if all of its nodes are in the selected nodal set.

        nodetype
            Label identifying type of nodes to consider when selecting:

            ALL - Select elements considering all of their nodes (default).

            ACTIVE - Select elements considering only their active nodes. An active node is a node
                     that contributes DOFs to the model.

            INACTIVE - Select elements considering only their inactive nodes (such as orientation or
                       radiation nodes).

            CORNER - Select elements considering only their corner nodes.

            MID - Select elements considering only their midside nodes.

        Notes
        -----
        ESLN selects elements which have any (or all EKEY) NodeType nodes in
        the currently-selected set of nodes. Only elements having nodes in the
        currently-selected set can be selected.

        This command is valid in any processor.

        """
        command = "ESLN,%s,%s,%s" % (str(type), str(ekey), str(nodetype))
        self.RunCommand(command, **kwargs)

    def Extrem(self, nvar1="", nvar2="", ninc="", **kwargs):
        """
        APDL Command: EXTREM

        Lists the extreme values for variables.

        Parameters
        ----------
        nvar1, nvar2, ninc
            List extremes for variables NVAR1 through NVAR2 in steps of NINC.
            Variable range defaults to its maximum. NINC defaults to 1.

        Notes
        -----
        Lists the extreme values (and the corresponding times) for stored and
        calculated variables. Extremes for stored variables are automatically
        listed as they are stored. Only the real part of a complex number is
        used. Extreme values may also be assigned to parameters [*GET].

        """
        command = "EXTREM,%s,%s,%s" % (str(nvar1), str(nvar2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Mffname(self, fnumb="", fname="", **kwargs):
        """
        APDL Command: MFFNAME

        Specifies a file name for a field in an ANSYS Multi-field solver
        analysis.

        Parameters
        ----------
        fnumb
            Field number specified by the MFELEM command.

        fname
            File name. Defaults to field "FNUMB".

        Notes
        -----
        All files created for the field will have this file name with the
        appropriate extensions.

        This command is also valid in PREP7.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFFNAME,%s,%s" % (str(fnumb), str(fname))
        self.RunCommand(command, **kwargs)

    def Slashmap(self, **kwargs):
        """
        APDL Command: /MAP

        Enters the mapping processor.

        Notes
        -----
        Enters the mapping processor. This processor is used to read in source
        data from an external file and map it to the existing geometry.

        The current database is saved (to BeforeMapping.DB) upon entering the
        processor, and it is resumed upon exiting (FINISH command). Any nodes
        or elements not on the target surface are deleted for easier viewing of
        the mapping quantities. A database of this mapping geometry
        (Mapping.DB) is also saved at the FINISH command.

        This command is valid only at the Begin Level.

        """
        command = "/MAP," % ()
        self.RunCommand(command, **kwargs)

    def Gsave(self, fname="", ext="", **kwargs):
        """
        APDL Command: /GSAVE

        Saves graphics settings to a file for later use.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        This command does not save all graphics settings, but only those that
        may be reset by the /RESET command.  The database remains untouched.
        Use the /GRESUME command to read the file.  Repeated use of the /GSAVE
        command overwrites the previous data on the file.  The following
        commands are saved by /GSAVE:

        This command is valid in any processor.

        """
        command = "/GSAVE,%s,%s" % (str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Cdread(self, option="", fname="", ext="", fnamei="", exti="",
               **kwargs):
        """
        APDL Command: CDREAD

        Reads a file of solid model and database information into the database.

        Parameters
        ----------
        option
            Selects which data to read:

            ALL - Read all geometry, material property, load, and component data (default).
                  Solid model geometry and loads will be read from the file
                  Fnamei.Exti.  All other data will be read from the file
                  Fname.Ext.

            DB - Read all database information contained in file Fname.Ext. This file should
                 contain all information mentioned above except the solid model
                 loads. If reading a .CDB file written with the GEOM option of
                 the CDWRITE  command, element types [ET] compatible with the
                 connectivity of the elements on the file must be defined prior
                 to reading.

            SOLID - Read the solid model geometry and solid model loads from the file Fnamei.Exti.
                    This file could have been written by the CDWRITE or IGESOUT
                    command.

            COMB - Read the combined solid model and database information from the file Fname.Ext.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        fnamei
            Name of the IGES file and its directory path (248 characters
            maximum, including directory). If you do not specify a directory
            path, it will default to your working directory and you can use all
            248 characters for the file name.

        exti
            Filename extension (eight-character maximum).

        Notes
        -----
        This command causes coded files of solid model (in IGES format) and
        database (in command format) information to be read.  These files are
        normally written by the CDWRITE or IGESOUT command.  Note that the
        active coordinate system in these files has been reset to Cartesian
        (CSYS,0).

        If a set of data exists prior to the CDREAD operation, that data set is
        offset upward to allow the new data to fit without overlap. The
        NOOFFSET command allows this offset to be ignored on a set-by-set
        basis, causing the existing data set to be overwritten with the new
        data set.

        When you write the geometry data using the CDWRITE,GEOM option, you use
        the CDREAD,DB option to read the geometry information.

        Using the CDREAD,COMB option will not write NUMOFF commands to offset
        entity ID numbers if there is no solid model in the database.

        Multiple CDB file imports cannot have elements with real constants in
        one file and section definitions in another. The section attributes
        will override the real constant attributes.  If you use CDREAD to
        import multiple CDB files, define all of the elements using only real
        constants, or using only section definitions.  Combining real constants
        and section definitions is not recommended.

        This command is valid in any processor.

        """
        command = "CDREAD,%s,%s,%s,%s,%s" % (str(option), str(fname), str(ext), str(fnamei), str(exti))
        self.RunCommand(command, **kwargs)

    def Fsnode(self, node="", nev="", nlod="", **kwargs):
        """
        APDL Command: FSNODE

        Calculates and stores the stress components at a node for fatigue.

        Parameters
        ----------
        node
            Node number for which stress components are stored.

        nev
            Event number to be associated with these stresses (defaults to 1).

        nlod
            Loading number to be associated with these stresses (defaults to
            1).

        Notes
        -----
        Calculates and stores the total stress components at a specified node
        for fatigue.  Stresses are stored according to the event number and
        loading number specified.  The location is associated with that
        previously defined for this node [FL] or else it is automatically
        defined.  Stresses are stored as six total components (SX through SYZ).
        Temperature and current time are also stored along with the total
        stress components.  Calculations are made from the stresses currently
        in the database (last SET or LCASE command).  Stresses stored are in
        global Cartesian coordinates, regardless of the active results
        coordinate system [RSYS].  The FSLIST command may be used to list
        stresses.  The FS command can be used to modify stored stresses.

        """
        command = "FSNODE,%s,%s,%s" % (str(node), str(nev), str(nlod))
        self.RunCommand(command, **kwargs)

    def Inres(self, item1="", item2="", item3="", item4="", item5="", item6="",
              item7="", item8="", **kwargs):
        """
        APDL Command: INRES

        Identifies the data to be retrieved from the results file.

        Parameters
        ----------
        item1, item2, item3, . . . , item8
            Data to be read into the database from the results file.  May
            consist of any of the following labels:

            ALL - All solution items (default).

            BASIC - NSOL, RSOL, NLOAD, STRS, FGRAD, and FFLUX items.

            NSOL - Nodal DOF solution.

            RSOL - Nodal reaction loads.

            ESOL - Element solution items (includes all of the following):

            NLOAD - Element nodal loads.

            STRS - Element nodal stresses.

            EPEL - Element elastic strains.

            EPTH - Element thermal, initial, and swelling strains.

            EPPL - Element plastic strains.

            EPCR - Element creep strains.

            FGRAD - Element nodal gradients.

            FFLUX - Element nodal fluxes.

            MISC - Element miscellaneous data (SMISC and NMISC).

        Notes
        -----
        Identifies the type of data to be retrieved from the results file for
        placement into the database through commands such as SET, SUBSET, and
        APPEND.  INRES is a companion command to the OUTRES command controlling
        data written to the database and the results file.  Since the INRES
        command can only flag data that has already been written to the results
        file, care should be taken when using the OUTRES command to include all
        data you wish to retrieve for postprocessing later on.

        """
        command = "INRES,%s,%s,%s,%s,%s,%s,%s,%s" % (str(item1), str(item2), str(item3), str(item4), str(item5), str(item6), str(item7), str(item8))
        self.RunCommand(command, **kwargs)

    def Mfdtime(self, dtime="", dtmin="", dtmax="", carry="", **kwargs):
        """
        APDL Command: MFDTIME

        Sets time step sizes for an ANSYS Multi-field solver analysis.

        Parameters
        ----------
        dtime
            Multi-field time step size.  If automatic time stepping is being
            used [see Notes below], DTIME is the starting time step.

        dtmin
            Minimum time step. Defaults to DTIME.

        dtmax
            Maximum time step. Defaults to DTIME.

        carry
            Time step carryover key.

            OFF  - Use DTIME as the starting time step for the next restart run (default).

            ON  - Use the final time step from the previous run as the starting time step for the
                  next restart run.

        Notes
        -----
        This command specifies time step sizes for an ANSYS Multi-field solver
        analysis. If either DTMIN or DTMAX is not equal to DTIME, auto time-
        stepping is turned on for the multi-field loop. ANSYS will
        automatically adjust the time step size for the next multi-field step
        between DTMIN and DTMAX, based on the status of the current
        convergence, the number of target stagger iterations (specified by
        MFITER), and the actual number of iterations needed to reach
        convergence at the current time step.

        If auto time-stepping is turned off, the time step size must be evenly
        divisible into the end time (specified by MFTIME) minus the start time
        (0 for a new analysis or a restart time specified by MFRSTART).

        You can use a smaller time step within each ANSYS field analysis. This
        is called subcycling. Use the DELTIM and AUTOTS commands to subcycle a
        structural, thermal, or electromagnetic analysis.

        This command is also valid in PREP7.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFDTIME,%s,%s,%s,%s" % (str(dtime), str(dtmin), str(dtmax), str(carry))
        self.RunCommand(command, **kwargs)

    def If(self, val1="", oper1="", val2="", b_ase1="", val3="", oper2="",
           val4="", b_ase2="", **kwargs):
        """
        APDL Command: *IF

        Conditionally causes commands to be read.

        Parameters
        ----------
        val1
            First numerical value (or parameter which evaluates to a numerical
            value) in the conditional comparison operation.  VAL1, VAL2, VAL3,
            and VAL4 can also be character strings (enclosed in quotes) or
            parameters for Oper = EQ and NE only.

        oper1
            Operation label.  A tolerance of 1.0E-10 is used for comparisons
            between real numbers:

            Equal (for VAL1 = VAL2). - Not equal (for VAL1 ≠ VAL2).

            Less than (for VAL1 < VAL2). - Greater than (for VAL1 > VAL2).

            Less than or equal (for VAL1   VAL2). - Greater than or equal (for VAL1   VAL2).

            Absolute values of VAL1 and VAL2 before < operation. - Absolute values of VAL1 and VAL2 before > operation.

        val2
            Second numerical value (or parameter which evaluates to a numerical
            value) in the conditional comparison operation.

        base1
            Action based on the logical expression (Oper1) being true.  If
            false, continue reading at the next line. This is conditional,
            except for the IF-THEN-ELSE constructs described below; any of the
            following constructs (through Base1 = THEN) cause all subsequent
            fields to be ignored:

            A user-defined label (beginning with a colon (:), 8 characters maximum).  The command reader will skip (and wrap to the beginning of the file, if necessary) to the first line that begins with the matching :label. - Caution:   This label option may not be mixed with do-loop or if-then-else
                              constructs.

            This action will cause an exit from the ANSYS program at this line, unless running in interactive mode. In interactive mode, the program will not stop. - Exit the current do-loop [*EXIT].

            Skip to the end of the current do-loop [*CYCLE]. - Make this *IF an if-then-else construct (see below).

        val3
            Third numerical value (or parameter which evaluates to a numerical
            value).

        oper2
            Operation label. This will have the same labels as Oper1, except it
            uses Val3 and Val4. A tolerance of 1.0E-10 is used for comparisons
            between real numbers.

        val4
            Fourth numerical value (or parameter value which evaluates to a
            numerical value).

        base2
            Action based on the logical expression (Oper1 and Oper2) being
            true. They will be the same values as Base1, except as noted.

        Notes
        -----
        Conditionally causes commands to be read from a specific block or at a
        specific location.  Twenty levels of nested *IF blocks are allowed.
        Jumping to a :label line is not allowed with keyboard entry.  Jumping
        into, out of, or within a do-loop or an if-then-else construct to a
        :label line is not allowed. Using *IF interactively or from the command
        line prevents rereading the file to find a label. To do so, use batch
        mode or /INPUT.

        The following is an example of an if-then-else construct:

        *IF,VAL1,Oper,VAL2,THEN

        ----

        *ELSEIF,VAL1,Oper,VAL2

        ----

        *ELSEIF,VAL1,Oper,VAL2

        ----

        *ELSE

        ----

        *ENDIF

        where "----" represents a block of any number of commands.  Any number
        of *ELSEIF clauses (or none) may be included (in the location shown).
        One *ELSE clause (at most) may be included (in the location shown).
        The *IF command is executed by evaluating its logical expression.  If
        it is true, the block of commands following it is executed.  The
        construct is considered to be complete and the command following the
        *ENDIF is executed next.  If the logical expression is false, the next
        *ELSEIF command (if any) following the block is executed.  The
        execution logic is the same as for *IF.  The effect is that the logical
        expressions in the *IF and the *ELSEIF commands are sequentially tested
        until one is found to be true.  Then the block of commands immediately
        following the expression is executed, which completes the execution of
        the if-then-else construct.  If all *IF and *ELSEIF expressions are
        false, the block following the *ELSE command is executed, if there is
        one.  Only one block of commands (at most) is executed within the if-
        then-else construct.  If a batch input stream hits an end-of-file
        during a false *IF condition, the ANSYS run will not terminate
        normally. You will need to terminate it externally (use either the
        Linux “kill” function or the Windows task manager). The *IF, *ELSEIF,
        *ELSE, and *ENDIF commands for each if-then-else construct must all be
        read from the same file (or keyboard).

        This command is valid in any processor.

        """
        command = "*IF,%s,%s,%s,%s,%s,%s,%s,%s" % (str(val1), str(oper1), str(val2), str(b_ase1), str(val3), str(oper2), str(val4), str(b_ase2))
        self.RunCommand(command, **kwargs)

    def Get(self, par="", entity="", entnum="", item1="", it1num="", item2="",
            it2num="", **kwargs):
        """
        APDL Command: *GET

        Retrieves a value and stores it as a scalar parameter or part of an
        array parameter.

        Parameters
        ----------
        par
            The name of the resulting parameter. See *SET for name
            restrictions.

        entity
            Entity keyword. Valid keywords are NODE, ELEM, KP, LINE, AREA,
            VOLU, PDS, etc., as shown for Entity = in the tables below.

        entnum
            The number or label for the entity (as shown for ENTNUM = in the
            tables below). In some cases, a zero (or blank) ENTNUM represents
            all entities of the set.

        item1
            The name of a particular item for the given entity. Valid items are
            as shown in the Item1 columns of the tables below.

        it1num
            The number (or label) for the specified Item1 (if any). Valid
            IT1NUM values are as shown in the IT1NUM columns of the tables
            below. Some Item1 labels do not require an IT1NUM value.

        item2, it2num
            A second set of item labels and numbers to further qualify the item
            for which data are to be retrieved. Most items do not require this
            level of information.

        Notes
        -----
        *GET retrieves a value for a specified item and stores the value as a
        scalar parameter, or as a value in a user-named array parameter. An
        item is identified by various keyword, label, and number combinations.
        Usage is similar to the *SET command except that the parameter values
        are retrieved from previously input or calculated results. For example,
        *GET,A,ELEM,5,CENT,X returns the centroid x-location of element 5 and
        stores the result as parameter A. *GET command operations, along with
        the associated Get functions return values in the active coordinate
        system unless stated otherwise. A Get function is an alternative in-
        line function that can be used to retrieve a value instead of the *GET
        command (see Using In-line Get Functions for more information).

        Both *GET and *VGET retrieve information from the active data stored in
        memory. The database is often the source, and sometimes the information
        is retrieved from common memory blocks that the program uses to
        manipulate information. Although POST1 and POST26 operations use a
        *.rst file, *GET data is accessed from the database or from the common
        blocks. Get operations do not access the *.rst file directly. For
        repeated gets of sequential items, such as from a series of elements,
        see the *VGET command.

        Most items are stored in the database after they are calculated and are
        available anytime thereafter. Items are grouped according to where they
        are usually first defined or calculated. Preprocessing data will often
        not reflect the calculated values generated from section data. Do not
        use *GET to obtain data from elements that use calculated section data,
        such as beams or shells. Most of the general items listed below are
        available from all modules. Each of the sections for accessing *GET
        parameters are shown in the following order:

        *GET General Entity Items

        *GET Preprocessing Entity Items

        *GET Solution Entity Items

        *GET Postprocessing Entity Items

        *GET Probabilistic Design Entity Items

        The *GET command is valid in any processor.

        """
        command = "*GET,%s,%s,%s,%s,%s,%s,%s" % (str(par), str(entity), str(entnum), str(item1), str(it1num), str(item2), str(it2num))
        self.RunCommand(command, **kwargs)

    def Mail(self, address="", fname="", ext="", **kwargs):
        """
        APDL Command: /MAIL

        Mails file to the specified address.

        Parameters
        ----------
        --
            Unused field.

        address
            Email address (up to 64 characters) of the intended recipient of
            the file.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        Notes
        -----
        Issue the /MAIL command to alert someone when a long-running job has
        completed, as shown in this example:

        If you are running ANSYS in a Microsoft Windows environment, you must
        configure BLAT (a freeware program provided with ANSYS). Before issuing
        the /MAIL command, enter this command from the C:\Program Files\Ansys
        Inc\V162\Ansys\Bin\Intel directory:

        If you are running ANSYS in a Windows x64 environment, enter this
        command from the C:\Program Files\Ansys Inc\V162\Ansys\Bin\Winx64
        directory:

        where:

        <email server name> is the name of the user’s email server; for
        example, smtphost.xyz.com

        <username>@<domain> is the email address of the user; for example,
        john.doe@xyz.com

        Because Windows cannot operate on a busy file, you cannot mail standard
        ANSYS output.  Instead, redirect the output to a file and then mail
        that file, as shown in this example:

        """
        command = "/MAIL,%s,%s,%s" % (str(address), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Pwedge(self, xcentr="", ycentr="", xlrad="", angle1="", angle2="",
               **kwargs):
        """
        APDL Command: /PWEDGE

        Creates an annotation wedge (GUI).

        Parameters
        ----------
        xcentr
            Wedge X center location (-1.0 < X < 2.0).

        ycentr
            Wedge Y center location (-1.0 < Y < 1.0).

        xlrad
            Wedge radius length.

        angle1
            Starting angle of wedge.

        angle2
            Ending angle of wedge.  The wedge is drawn counterclockwise from
            the starting angle, ANGLE1, to the ending angle, ANGLE2.

        Notes
        -----
        Creates an annotation wedge to be written directly onto the display at
        a specified location.  This is a command generated by the Graphical
        User Interface (GUI) and will appear in the log file (Jobname.LOG) if
        annotation is used.  This command is not intended to be typed in
        directly in an ANSYS session (although it can be included in an input
        file for batch input or for use with the /INPUT command).

        All wedges are shown on subsequent displays unless the annotation is
        turned off or deleted.  Use the /LSPEC and the /PSPEC command to set
        the attributes of the wedge.

        This command is valid in any processor.

        """
        command = "/PWEDGE,%s,%s,%s,%s,%s" % (str(xcentr), str(ycentr), str(xlrad), str(angle1), str(angle2))
        self.RunCommand(command, **kwargs)

    def Kdele(self, np1="", np2="", ninc="", **kwargs):
        """
        APDL Command: KDELE

        Deletes unmeshed keypoints.

        Parameters
        ----------
        np1, np2, ninc
            Delete keypoints from NP1 to NP2 (defaults to NP1) in steps of NINC
            (defaults to 1).  If NP1 = ALL, NP2 and NINC are ignored and all
            selected keypoints [KSEL] are deleted.  If NP1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).  A component name may also be substituted
            for NP1 (NP2 and NINC are ignored).

        Notes
        -----
        Deletes selected keypoints.  A keypoint attached to a line cannot be
        deleted unless the line is first deleted.

        """
        command = "KDELE,%s,%s,%s" % (str(np1), str(np2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Cmap(self, fname="", ext="", kywrd="", ncntr="", **kwargs):
        """
        APDL Command: /CMAP

        Changes an existing or creates a new color mapping table.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        kywrd
            Keyword indicating the disposition of the color map file.

            (blank) - Loads existing color map file.

            CREATE - Starts the CMAP utility and modifies or creates the specified file.

            SAVE - Writes the active color map to the specified file, which can be imported into
                   future sessions.

        ncntr
            Number of contours to be defined. Default = 9 (even if an existing
            file is being modified). Maximum = 128.

        Notes
        -----
        Reads the color map file (RGB index specifications) to change from
        current specifications.  Only one color map may be active at a time.

        For 2-D drivers (especially Win32c), modifying the color map can
        produce anomalies, including legend/contour disagreement.

        When Kywrd equals CREATE, the 2-D drivers (X11c and Win32c) display the
        CMAP utility with an additional contour color picker called CONTOURS.
        Colors selected via the CONTOURS picker affect result contour displays
        (such as stresses). No other drivers offer the CONTOURS picker in the
        CMAP utility.

        Changing the color map using the /CMAP command changes the meaning of
        the color labels on the /COLOR command. See /COLOR for other color
        controls.

        This command is valid anywhere.

        """
        command = "/CMAP,%s,%s,%s,%s" % (str(fname), str(ext), str(kywrd), str(ncntr))
        self.RunCommand(command, **kwargs)

    def Shrink(self, ratio="", **kwargs):
        """
        APDL Command: /SHRINK

        Shrinks elements, lines, areas, and volumes for display clarity.

        Parameters
        ----------
        ratio
            Shrinkage ratio (input as a decimal (0.0 to 0.5)).  Defaults to 0.0
            (no shrinkage).  Values greater than 0.5 default to 0.1 (10%
            shrinkage).

        Notes
        -----
        Shrinks the elements, lines, areas, and volumes so that adjacent
        entities are separated for clarity.  Portions of this command are not
        supported by PowerGraphics [/GRAPHICS,POWER].

        If only the common lines of non-coplanar faces are drawn (as per the
        /EDGE command), then this command is ignored.

        This command is valid in any processor.

        """
        command = "/SHRINK,%s" % (str(ratio))
        self.RunCommand(command, **kwargs)

    def Waves(self, wopt="", oldmax="", oldrms="", **kwargs):
        """
        APDL Command: WAVES

        Initiates reordering.

        Parameters
        ----------
        wopt
            Option for comparison:

            MAX - Use maximum wavefront value for comparison (default).

            RMS - Use RMS wavefront value for comparison.

        oldmax, oldrms
            Maximum and RMS wavefront values of model to be used in place of
            the old values.  OLDRMS defaults to OLDMAX  (and vice versa).  If
            neither is specified, each defaults to its calculated old value.

        Notes
        -----
        Initiates the element reordering based upon the previously defined
        starting wave lists (if any).  Reordering affects only the element
        order for the solution phase and not the element numbers (input
        referring to element numbers, such as element pressures, is unaffected
        by reordering).

        Note:: : The new order is retained only if new the new maximum or RMS
        wavefront values are lower than the old values, as described below.
        See the WSORT command for another reordering procedure.  The resulting
        element ordering can be shown by listing the wavefront history
        [WFRONT,1] or by displaying elements with their element location
        numbers [/PNUM].

        The WAVES reordering procedure is based upon a list of nodes defining
        where the element reordering is to start (either input on the WSTART
        command, or automatically determined if not input).  If a list is
        input, additional starting wave lists may be defined to allow the user
        to block or guide the wave.  An input starting list usually consists of
        one node for a line element model, a line of nodes for an area element
        model, or a plane of nodes for a volume element model.  Elements are
        ordered from the first starting wave list in a direction that attempts
        to minimize the wavefront.  Nodes are deleted and added to the total
        wave list set as reordering progresses through the model.  Elements
        attached to nodes on succeeding starting lists, if any, are ignored at
        this time, thereby temporarily blocking that path.  Whenever no more
        elements can be reordered, the procedure is repeated starting from the
        nodes defined on the second starting list (if any).  The same node
        number should not appear on more than one list.  If no starting list is
        input, a starting node is automatically selected from an element
        weighting procedure (see Element Reordering in the Mechanical APDL
        Theory Reference).

        Reordering may be made from the previously reordered model or from the
        initial model (by issuing the SAVE command before reordering and then
        restoring that model back to PREP7 with the RESUME command after
        reordering).  The WFRONT command is useful for calculating and printing
        the current wavefront statistics at any time without causing any
        reordering of the elements.  If a model is to be used for both a
        thermal and a structural analysis, the reordered statistics should be
        based upon the structural model (because of the higher number of
        degrees of freedom per node).

        The reordering procedure treats separate portions of the model (i.e.,
        not connected by elements) as discontinuous.  Reordering automatically
        continues across a discontinuity as best as possible.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "WAVES,%s,%s,%s" % (str(wopt), str(oldmax), str(oldrms))
        self.RunCommand(command, **kwargs)

    def Sfe(self, elem="", lkey="", lab="", kval="", val1="", val2="", val3="",
            val4="", **kwargs):
        """
        APDL Command: SFE

        Specifies surface loads on elements.

        Parameters
        ----------
        elem
            Element to which surface load applies.  If ALL, apply load to all
            selected elements [ESEL].  If Elem = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may be substituted for Elem.

        lkey
            Load key or face number associated with surface load (defaults to
            1).  Load keys (1,2,3, etc.) are listed under "Surface Loads" in
            the input data table for each element type in the Element
            Reference.

        lab
            Valid surface load label.  Load labels are listed under "Surface
            Loads" in the input table for each element type in the Element
            Reference.

        kval
            Value key.  If Lab = PRES:

            0 or 1 - VAL1 through VAL4 are used as real components of pressures.

            2 - VAL1 through VAL4 are used as imaginary components of pressures.

        val1
            First surface load value (typically at the first node of the face)
            or the name of a table for specifying tabular boundary conditions.
            Face nodes are listed in the order given for "Surface Loads" in the
            input data table for each element type in the Element Reference.
            For example, for SOLID185, the item 1-JILK associates LKEY = 1
            (face 1) with nodes J, I, L, and K.  Surface load value VAL1 then
            applies to node J of face 1.  To specify a table, enclose the table
            name in percent signs (%), e.g., %tabname%.  Use the *DIM command
            to define a table.  Only one table can be specified, and it must be
            specified in the VAL1 position; tables specified in the VAL2, VAL3,
            or VAL4 positions will be ignored. VAL2 applies to node I, etc.

        val2, val3, val4
            Surface load values at the 2nd, 3rd, and 4th nodes (if any) of the
            face. If all three values are blank, all three default to VAL1,
            giving a constant load. Zero or other blank values are used as
            zero. If a table is specified in VAL1, then any tables or numerical
            values specified in VAL2, VAL3, or VAL4 will be ignored.

        Notes
        -----
        Specifies surface loads on selected elements.

        Caution:: : You cannot use the SFE command with the INFIN110 or
        INFIN111 elements without prior knowledge of element face orientation,
        i.e., you must  know which face is the exterior in order to flag it.
        Also, the surface effect elements, SURF153 and SURF154, use LKEY to
        allow the use of tangential and other loads (see SURF153and SURF154 of
        the Element Reference for more details).

        Tapered loads may be applied over the faces of most elements.

        For beam elements allowing lateral surface loads that may be offset
        from the nodes, use the SFBEAM command to specify the loads and
        offsets.  See the SF command for an alternate surface load definition
        capability based upon node numbers.  See the SFGRAD command for an
        alternate tapered load capability.  Use the SFCUM command to accumulate
        (add) surface loads applied with SFE.

        The SFE command can also be used to define fluid pressure penetration
        loads (Lab = PRES) at a contact interface. For this type of load, LKEY
        = 1 is used to specify the pressure values and LKEY = 2 is used to
        specify starting points and penetrating points. See Applying Fluid
        Penetration Pressure in the Contact Technology Guide for details on how
        to apply this type of load.

        Film effectiveness and free stream temperatures specified with Lab =
        CONV are only valid for SURF151 and SURF152. Film effectiveness must be
        between 0 and 1 and it defaults to 0. If film effectiveness is applied,
        bulk temperature is ignored. When film effectiveness and free stream
        temperatures are specified, the commands to specify a surface load
        gradient (SFGRAD) or surface load accumulation (SFCUM) are not valid.
        For more information on film effectiveness, see Conduction and
        Convection in the Mechanical APDL Theory Reference.

        You can specify a table name only when using structural (PRES) and
        thermal (CONV [film coefficient, bulk temperature, film effectiveness,
        and free stream temperature], HFLUX), diffusion flux (DFLUX), surface
        emissivity and ambient temperature (RAD), impedance (IMPD), normal
        velocity or acceleration (SHLD), attenuation coefficient (ATTN), and
        substructure (SELV) surface load labels.

        When a tabular function load is applied to an element, the load will
        not vary according to the positioning of the element in space.

        For cases where Lab=FSI, MXWF, FREE, and INF, VALUE is not needed.

        This command is also valid in PREP7 and in the /MAP processor.

        """
        command = "SFE,%s,%s,%s,%s,%s,%s,%s,%s" % (str(elem), str(lkey), str(lab), str(kval), str(val1), str(val2), str(val3), str(val4))
        self.RunCommand(command, **kwargs)

    def Crplim(self, crcr="", option="", **kwargs):
        """
        APDL Command: CRPLIM

        Specifies the creep criterion for automatic time stepping.

        Parameters
        ----------
        crcr
            Value of creep criteria for the creep limit ratio control.

        option
            Type of creep analysis for which the creep limit ratio is
            specified:

            1 (or ON) - Implicit creep analysis.

            0 (or OFF) - Explicit creep analysis.

        Notes
        -----
        The CUTCONTROL command can also be used to set the creep criterion and
        is preferred over this command for setting automatic time step
        controls.

        The creep ratio control can be used at the same time for implicit creep
        and explicit creep analyses. For implicit creep (Option = 1), the
        default value of CRCR is zero (i.e., no creep limit control), and you
        can specify any value. For explicit creep (Option = 0), the default
        value of CRCR is 0.1, and the maximum value allowed is 0.25.

        This command is also valid in PREP7.

        """
        command = "CRPLIM,%s,%s" % (str(crcr), str(option))
        self.RunCommand(command, **kwargs)

    def Prerr(self, **kwargs):
        """
        APDL Command: PRERR

        Prints SEPC and TEPC.

        Notes
        -----
        Prints the percent error in structural energy norm (SEPC) and the
        thermal energy norm percent error (TEPC).  Approximations of mesh
        discretization error associated with a solution are calculated for
        analyses having structural or thermal degrees of freedom.

        The structural approximation is based on the energy error (which is
        similar in concept to the strain energy) and represents the error
        associated with the discrepancy between the calculated stress field and
        the globally continuous stress field (see POST1 - Error Approximation
        Technique in the Mechanical APDL Theory Reference).  This discrepancy
        is due to the assumption in the elements that only the displacements
        are continuous at the nodes.  The stress field is calculated from the
        displacements and should also be continuous, but generally is not.

        Thermal analyses may use any solid and shell thermal element having
        only temperature degrees of freedom.  The thermal approximation is
        based on the total heat flow dissipation and represents the error
        associated with the discrepancy between the calculated nodal thermal
        flux within an element and a continuous global thermal flux.  This
        continuous thermal flux is calculated with the normal nodal averaging
        procedure.

        The volume (result label VOLU) is used to calculate the energy error
        per element (result label SERR for the structural energy error and TERR
        for the thermal energy error).  These energy errors, along with the
        appropriate energy, are then used to calculate the percent error in
        energy norm (SEPC for structural and TEPC for thermal). These
        percentages can be listed by the PRERR command, retrieved by the *GET
        command (with labels SEPC and TEPC) for further calculations, and shown
        on the displacement display (PLDISP), as applicable.

        For structural analyses, the maximum absolute value of nodal stress
        variation of any stress component for any node of an element (result
        item SDSG) is also calculated.  Similarly, for thermal gradient
        components, TDSG is calculated.  Minimum and maximum result bounds
        considering the possible effect of discretization error will be shown
        on contour displays (PLNSOL).  For shell elements, the top surface
        location is used to produce a meaningful percentage value.  SERR, TERR,
        SEPC, TEPC, SDSG, and TDSG will be updated whenever the nodal stresses
        or fluxes are recalculated.

        If the energy error is a significant portion of the total energy, then
        the analysis should be repeated using a finer mesh to obtain a more
        accurate solution.  The energy error is relative from problem to
        problem but will converge to a zero energy error as the mesh is
        refined.  An automated adaptive meshing procedure using this energy
        error is described with the ADAPT macro.

        The following element- and material-type limitations apply:

        Valid with most 2-D solid, 3-D solid, axisymmetric solid, or 3-D shell
        elements.

        The following element types are not valid: SHELL28, SHELL41, and
        SOLID65.

        The model should have only structural or thermal degrees of freedom.

        The analysis must be linear (for both material and geometry).

        Multi-material (for example, composite) elements are not valid.

        Transition regions from one material to another are not valid (that is,
        the entire model should consist of one material).

        Anisotropic materials (TB,ANEL) are not considered.

        """
        command = "PRERR," % ()
        self.RunCommand(command, **kwargs)

    def Edvel(self, option="", cname="", vx="", vy="", vz="", omegax="",
              omegay="", omegaz="", xc="", yc="", zc="", angx="", angy="",
              angz="", **kwargs):
        """
        APDL Command: EDVEL

        Applies initial velocities to nodes or node components in an explicit
        dynamic analysis.

        Parameters
        ----------
        option
            Label identifying the option to be performed.

            VGEN - Define initial velocities based on translational velocities (relative to global
                   Cartesian) and the rotational velocity about an arbitrary
                   axis. For this option, use the fields VX, VY, VZ to specify
                   the translational velocities, and use OMEGAX, XC, YC, ZC,
                   ANGX, ANGY, ANGZ to specify the rotational velocity and the
                   axis of rotation.

            VELO - Define initial velocity based on translational velocities and nodal rotational
                   velocities input relative to the global Cartesian axes. For
                   this option, use the following fields to define the initial
                   velocity: VX, VY, VZ, OMEGAX, OMEGAY, OMEGAZ.

            LIST - List initial velocity for the component or node specified by Cname. If Cname is
                   blank, all initial velocities defined on nodes and node
                   components are listed. Remaining fields are ignored for this
                   option.

            DELE - Delete initial velocity defined for the component or node specified by Cname.
                   If Cname is blank, all initial velocities defined on nodes
                   and node components are deleted. Remaining fields are
                   ignored for this option.

        cname
            Name of existing component [CM] or node number to which the initial
            velocity is to be applied. If a component is used, it must consist
            of nodes.

        vx
            Initial velocity in X direction. Defaults to 0.

        vy
            Initial velocity in Y direction. Defaults to 0.

        vz
            Initial velocity in Z direction. Defaults to 0.

        omegax
            For Option = VGEN, OMEGAX is the initial rotational velocity of the
            component (or node) about the specified rotational axis. For Option
            = VELO, OMEGAX is the initial nodal rotational velocity about the
            X-axis. OMEGAX defaults to 0.

        omegay
            Initial nodal rotational velocity about the Y-axis (used only if
            Option = VELO). Defaults to 0.

        omegaz
            Initial nodal rotational velocity about the Z-axis (used only if
            Option = VELO). Defaults to 0.

        Notes
        -----
        You cannot mix the two methods of initial velocity input (Option = VELO
        and Option = VGEN) in the same analysis. You must use only one method
        for all initial velocity definitions.

        The VGEN and VELO methods differ in how the rotational velocity is
        defined. Use Option = VGEN to input the initial velocities of a
        rotating component. Use Option = VELO to apply the rotations directly
        to the nodes' rotation degrees of freedom. Since only shell and beam
        elements have rotation degrees of freedom, the rotations input with
        Option = VELO are only applicable to SHELL163 and BEAM161 elements. The
        rotational velocities input with Option = VELO are ignored for nodes
        not having rotational degrees of freedom (such as nodes attached to a
        SOLID164 or SOLID168 element).

        It is normally acceptable to mix nodes belonging to deformable bodies
        and rigid bodies in the nodal component used in an initial velocity
        definition. However, when defining initial velocities in an implicit-
        to-explicit sequential solution, this is not an acceptable practice. In
        order for the initial velocities to be defined correctly in this type
        of analysis, you must define the initial velocities on the deformable
        body nodes separately from the initial velocities on the rigid body
        nodes.

        Issuing the EDVEL command again for the same component or node (Cname)
        will overwrite previous initial velocities defined for that component
        or node.

        To set the initial velocities to zero, issue the EDVEL command with
        only the Option (use VELO or VGEN) and Cname fields specified.

        In a small restart analysis (EDSTART,2), you can only use the Option =
        VELO method to change initial velocities. When used in a small restart,
        the command EDVEL,VELO changes the velocity of the specified nodes. If
        you don't change the velocity of the nodes, their velocity at the
        beginning of the restart will be the same as the velocity at the end of
        the previous analysis.

        Except for the LIST option, the EDVEL command is not supported in a
        full restart analysis (EDSTART,3). You can list initial velocities
        defined in the previous analysis with the command EDVEL,LIST. However,
        you cannot change initial velocities for nodes or node components that
        existed in the previous analysis; their velocity at the beginning of
        the analysis will be the same as the velocity at the end of the
        previous analysis. In addition, you cannot define initial velocities
        for any nodes that are added in the full restart; the velocity of new
        nodes will be zero.

        To apply initial velocities to parts or part assemblies, use the EDPVEL
        command.

        You can use EDPVEL and EDVEL in the same analysis. If a node or node
        component input on the EDVEL command shares common nodes with a part or
        part assembly input on the EDPVEL command, the initial velocities
        defined on the common nodes will be determined by the last command
        input.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDVEL,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(cname), str(vx), str(vy), str(vz), str(omegax), str(omegay), str(omegaz), str(xc), str(yc), str(zc), str(angx), str(angy), str(angz))
        self.RunCommand(command, **kwargs)

    def Trpdel(self, ntrp1="", ntrp2="", trpinc="", **kwargs):
        """
        APDL Command: TRPDEL

        Deletes particle flow or charged particle trace points.

        Parameters
        ----------
        ntrp1, ntrp2, trpinc
            Delete points from NTRP1 to NTRP2 (defaults to NTRP1) in steps of
            TRPINC (defaults to 1).  If NTRP1 = ALL, NTRP2 and TRPINC are
            ignored and all trace points are deleted.  If NTRP1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).

        Notes
        -----
        Deletes particle flow or charged particle trace points defined with the
        TRPOIN command.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "TRPDEL,%s,%s,%s" % (str(ntrp1), str(ntrp2), str(trpinc))
        self.RunCommand(command, **kwargs)

    def Rmclist(self, **kwargs):
        """
        APDL Command: RMCLIST

        Lists all lumped capacitance pairs defined.

        Notes
        -----
        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "RMCLIST," % ()
        self.RunCommand(command, **kwargs)

    def Csdele(self, kcn1="", kcn2="", kcinc="", **kwargs):
        """
        APDL Command: CSDELE

        Deletes local coordinate systems.

        Parameters
        ----------
        kcn1, kcn2, kcinc
            Delete coordinate systems from KCN1 (must be greater than 10) to
            KCN2 (defaults to KCN1) in steps of KCINC (defaults to 1).  If KCN1
            = ALL, KCN2 and KCINC are ignored and all coordinate systems are
            deleted.

        Notes
        -----
        This command is valid in any processor.

        """
        command = "CSDELE,%s,%s,%s" % (str(kcn1), str(kcn2), str(kcinc))
        self.RunCommand(command, **kwargs)

    def Campbell(self, action="", **kwargs):
        """
        APDL Command: CAMPBELL

        Prepares the result file for a subsequent Campbell diagram of a
        prestressed structure.

        Parameters
        ----------
        action
            Campbell action:

            NONE - Do not prepare the result file. This option is the default behavior.

            RSTP - Prepare the result file (Jobname.RSTP) for a subsequent Campbell diagram of a
                   prestressed structure.

        Notes
        -----
        For an analysis involving a prestressed structure, the CAMPBELL command
        specifies whether or not to prepare the result file to support  a
        Campbell diagram analysis (PRCAMP or PLCAMP).

        To prestress a structure, the program performs a static solution before
        the linear perturbation modal solution.

        The CAMPBELL command requires that modal and static analyses be
        performed alternately. It works only when the number of static analyses
        is the same as the number of modal analyses. Any number of analyses can
        be performed, but the same number of each (static and modal) is
        expected. The modal solutions are appended in the results file
        (Jobname.RSTP).

        For an example of PLCAMP command usage, see Example Campbell Diagram
        Analysis in the Advanced Analysis Guide.

        """
        command = "CAMPBELL,%s" % (str(action))
        self.RunCommand(command, **kwargs)

    def Bfe(self, elem="", lab="", stloc="", val1="", val2="", val3="",
            val4="", **kwargs):
        """
        APDL Command: BFE

        Defines an element body force load.

        Parameters
        ----------
        elem
            The element to which body load applies.  If ALL, apply to all
            selected elements (ESEL). A component name may also be substituted
            for Elem.

        lab
            Valid body load label. Valid labels are also listed for each
            element type in the Element Reference under "Body Loads" in the
            input table.

        stloc
            Starting location for entering VAL data, below.  For example, if
            STLOC = 1, data input in the VAL1 field applies to the first
            element body load item available for the element type, VAL2 applies
            to the second element item, etc. If STLOC = 5, data input in the
            VAL1 field applies to the fifth element item, etc.  Defaults to 1.

        val1, val2, val3, val4
            For Lab = TEMP, FLUE, DGEN, HGEN, and CHRGD, VAL1--VAL4 represent
            body load values at the starting location and subsequent locations
            (usually nodes) in the element.  VAL1 can also represent a table
            name for use with tabular boundary conditions. Enter only VAL1 for
            a uniform body load across the element.  For nonuniform loads, the
            values must be input in the same order as shown in the input table
            for the element type.  Values initially default to the BFUNIF value
            (except for CHRGD which defaults to zero).  For subsequent
            specifications, a blank leaves a previously specified value
            unchanged; if the value was not previously specified, the default
            value as described in the Element Reference is used.

        Notes
        -----
        Defines an element body force load (such as temperature in a structural
        analysis, heat generation rate in a thermal analysis, etc.). Body loads
        and element specific defaults are described for each element type in
        the Element Reference. If both the BF and BFE commands are used to
        apply a body load to an element, the BFE command takes precedence.

        For heat-generation (HGEN) loading on layered thermal solid elements
        SOLID278 / SOLID279 (KEYOPT(3) = 1 or 2), or layered thermal shell
        elements SHELL131 / SHELL132 (KEYOPT(3) = 1), STLOC refers to the layer
        number (not the node). In such cases, use VAL1 through VAL4 to specify
        the heat-generation values for the appropriate layers. Heat generation
        is constant over the layer.

        Specifying a Table

        You can specify a table name (VAL1) when using temperature (TEMP),
        diffusing substance generation rate (DGEN), heat generation rate
        (HGEN), and current density (JS) body load labels.

        Enclose the table name (tabname) in percent signs (%), as shown:

        BFE,Elem, Lab,STLOC,%tabname%

        Use the *DIM command to define a table.

        For Lab = TEMP, each table defines NTEMP temperatures, as follows:

        For layered elements, NTEMP is the number of layer interface corners
        that allow temperature input.

        For non-layered elements, NTEMP is the number of corner nodes.

        The temperatures apply to element items with a starting location of
        STLOC + n, where n is the value field location (VALn) of the table name
        input.

        For layered elements, a single BFE command returns temperatures for one
        layer interface. Multiple BFE commands are necessary for defining all
        layered temperatures.

        For beam, pipe and elbow elements that allow multiple temperature
        inputs per node, define the tabular load for the first node only (Node
        I), as loads on the remaining nodes are applied automatically. For
        example, to specify a tabular temperature load on a PIPE288 element
        with the through-wall-gradient option (KEYOPT(1) = 0), the BFE command
        looks like this:

        BFE,Elem,TEMP,1,%tabOut%, %tabIn%

        When a tabular function load is applied to an element, the load does
        not vary according to the positioning of the element in space.

        Graphical picking is available only via the listed menu paths.

        This command is also valid in PREP7.

        """
        command = "BFE,%s,%s,%s,%s,%s,%s,%s" % (str(elem), str(lab), str(stloc), str(val1), str(val2), str(val3), str(val4))
        self.RunCommand(command, **kwargs)

    def Emagerr(self, **kwargs):
        """
        APDL Command: EMAGERR

        Calculates the relative error in an electrostatic or electromagnetic
        field analysis.

        Notes
        -----
        The relative error is an approximation of the mesh discretization error
        associated with a solution. It is based on the discrepancy between the
        unaveraged, element-nodal field values and the averaged, nodal field
        values. The calculation is valid within a material boundary and does
        not consider the error in continuity of fields across dissimilar
        materials.

        For electrostatics, the field values evaluated are the electric field
        strength (EFSUM) and the electric flux density (DSUM). A relative error
        norm of each is calculated on a per-element basis and stored in the
        element table [ETABLE] with the labels EF_ERR and D_ERR. Normalized
        error values EFN_ERR and DN_ERR are also calculated and stored in the
        element table. Corresponding quantities for electromagnetics are H_ERR,
        B_ERR, HN_ERR, and BN_ERR, which are calculated from the magnetic field
        intensity (HSUM) and the magnetic flux density (BSUM).  The normalized
        error value is the relative error norm value divided by the peak
        element-nodal field value for the currently selected elements.

        Use the PLETAB and PRETAB commands to plot and list the error norms and
        normalized error values.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EMAGERR," % ()
        self.RunCommand(command, **kwargs)

    def Emft(self, **kwargs):
        """
        APDL Command: EMFT

        Summarizes electromagnetic forces and torques.

        Notes
        -----
        Use this command to summarize electromagnetic force and torque in both
        static electric and magnetic problems. To use this command, select the
        nodes in the region of interest and make sure that all elements are
        selected. If RSYS = 0, the force is reported in the global Cartesian
        coordinate system. If RSYS ≠ 0, force is reported in the specified
        coordinate system. However, for torque, if RSYS ≠ 0, this command will
        account for the shift and rotation as specified by RSYS, but will
        report only the Cartesian components.

        Forces are stored as items _FXSUM, _FYSUM, _FZSUM, and _FSSUM. Torque
        is stored as items _TXSUM, _TYSUM, _TZSUM, and _TSSUM.

        This command is valid only with PLANE121, SOLID122, SOLID123, PLANE233,
        SOLID236 and SOLID237 elements. For any other elements, you must use
        FMAGSUM.

        """
        command = "EMFT," % ()
        self.RunCommand(command, **kwargs)

    def Ceqn(self, **kwargs):
        """
        APDL Command: CEQN

        Specifies "Constraint equations" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "CEQN," % ()
        self.RunCommand(command, **kwargs)

    def Nsla(self, type="", nkey="", **kwargs):
        """
        APDL Command: NSLA

        Selects those nodes associated with the selected areas.

        Parameters
        ----------
        type
            Label identifying the type of node select:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        nkey
            Specifies whether only interior area nodes are to be selected:

            0 - Select only nodes interior to selected areas.

            1 - Select all nodes (interior to area, interior to lines, and at keypoints)
                associated with the selected areas.

        Notes
        -----
        Valid only if the nodes were generated by an area meshing operation
        [AMESH, VMESH] on a solid model that contains the selected areas.

        This command is valid in any processor.

        """
        command = "NSLA,%s,%s" % (str(type), str(nkey))
        self.RunCommand(command, **kwargs)

    def Time(self, time="", **kwargs):
        """
        APDL Command: TIME

        Sets the time for a load step.

        Parameters
        ----------
        time
            Time at the end of the load step.

        Notes
        -----
        Associates the boundary conditions at the end of the load step with a
        particular TIME value.

        TIME must be a positive, nonzero, monotonically increasing quantity
        that "tracks" the input history.  Units of time should be consistent
        with those used elsewhere (for properties, creep equations, etc.).

        Typically, for the first load step TIME defaults to 1. However, for the
        first load step of a mode-superposition transient analysis
        (ANTYPE,TRANS and TRNOPT,MSUP), the TIME command is ignored and a
        static solution is performed at TIME = 0.

        For a full transient analyses, the command's default behavior does not
        apply. You must specify a time for each load step and it must be
        greater than the time at the end of the prior load step.

        TIME does not apply to modal (ANTYPE,MODAL), harmonic (ANTYPE,HARMIC),
        or substructure (ANTYPE,SUBSTR) analyses.

        This command is also valid in PREP7.

        """
        command = "TIME,%s" % (str(time))
        self.RunCommand(command, **kwargs)

    def Int1(self, ir="", iy="", ix="", name="", facta="", factb="", const="",
             **kwargs):
        """
        APDL Command: INT1

        Integrates a variable.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]).  If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.  Table values represent integrated
            sum of IY to current table position of IX.

        iy, ix
            Integrate variable IY with respect to IX.

        --
            Unused field.

        name
            Thirty-two character name for identifying the variable on the
            printout and displays.  Embedded blanks are compressed upon output.

        --, --
            Unused fields.

        facta, factb
            Scaling factors (positive or negative) applied to the corresponding
            variables (default to 1.0).

        const
            Initial value.

        Notes
        -----
        Integrates variables according to the operation:

        IR = ∫ (FACTA x IY) d(FACTB x IX) + CONST

        """
        command = "INT1,%s,%s,%s,%s,%s,%s,%s" % (str(ir), str(iy), str(ix), str(name), str(facta), str(factb), str(const))
        self.RunCommand(command, **kwargs)

    def Cpcyc(self, lab="", toler="", kcn="", dx="", dy="", dz="", knonrot="",
              **kwargs):
        """
        APDL Command: CPCYC

        Couples the two side faces of a cyclically symmetric model for loadings
        that are the same on every segment.

        Parameters
        ----------
        lab
            Degree of freedom label for coupled nodes (in the nodal coordinate
            system). If ALL, use all appropriate labels.  Valid labels are:
            Structural labels:  UX, UY, or UZ (displacements); ROTX, ROTY, or
            ROTZ (rotations, in radians).

        toler
            Tolerance for coincidence (based on maximum coordinate difference
            in each global Cartesian direction for node locations and on angle
            differences for node orientations).  Defaults to 0.0001. Only nodes
            within the tolerance are considered to be coincident for coupling.

        kcn
             In coordinate system KCN, node 1 of CP + dx dy dz = node 2 of CP.

        dx, dy, dz
            Node location increments in the active coordinate system (DR, Dθ,
            DZ for cylindrical, DR, D θ, DΦ for spherical or toroidal).

        knonrot
            When KNONROT = 0, the nodes on coupled sets are rotated into
            coordinate system KCN (see NROTAT command description). When
            KNONROT = 1, the nodes are not rotated, and you should make sure
            that coupled nodal DOF directions are correct.

        Notes
        -----
         Cyclic coupling requires identical node and element patterns on the
        low and high sector boundaries. The MSHCOPY operation allows convenient
        generation of identical node and element patterns. See Using CPCYC and
        MSHCOPY Commands in the Modeling and Meshing Guide for more
        information.

        Although developed primarily for use with cyclically symmetric models,
        your use of the CPCYC command is not limited to cyclic symmetry
        analyses.

        """
        command = "CPCYC,%s,%s,%s,%s,%s,%s,%s" % (str(lab), str(toler), str(kcn), str(dx), str(dy), str(dz), str(knonrot))
        self.RunCommand(command, **kwargs)

    def Spoint(self, node="", x="", y="", z="", **kwargs):
        """
        APDL Command: SPOINT

        Defines a point for moment summations.

        Parameters
        ----------
        node
            Node number of the desired point.  If zero, use X,Y,Z to describe
            point.

        x, y, z
            Global Cartesian coordinates of the desired summation point.  Used
            if NODE is 0.  Defaults to (0,0,0).

        Notes
        -----
        Defines a point (any point other than the origin) about which the
        tabular moment summations are computed [NFORCE, FSUM].  If force
        summations are desired in other than the global Cartesian directions, a
        node number must be specified on the NODE field, and the desired
        coordinate system must be activated with RSYS.

        """
        command = "SPOINT,%s,%s,%s,%s" % (str(node), str(x), str(y), str(z))
        self.RunCommand(command, **kwargs)

    def Modify(self, set="", lstep="", iter="", cumit="", time="", ktitle="",
               **kwargs):
        """
        APDL Command: MODIFY

        Changes the listed values of the data in a set.

        Parameters
        ----------
        set
            Set of data in results file to be modified.

        lstep
            The new load step number.

        iter
            The new load substep number.

        cumit
            The new cumulative iteration.

        time
            The new time/frequency value.

        ktitle
            Indicates if the set title should be modified.

            0 - Keep the original title.

            1 - Change the title to the title specified with the most current /TITLE command.

        Notes
        -----
        Use this command to change the listed values in a data set in a results
        file. Using this command does not change any actual model data; it
        affects only the values listed in the results file.

        For example, if you start with the following results file:

        and you then issue the following commands:

        The modified results file would look like this:

        """
        command = "MODIFY,%s,%s,%s,%s,%s,%s" % (str(set), str(lstep), str(iter), str(cumit), str(time), str(ktitle))
        self.RunCommand(command, **kwargs)

    def Rsprnt(self, rslab="", yname="", xout="", **kwargs):
        """
        APDL Command: RSPRNT

        Print a response surface.

        Parameters
        ----------
        rslab
            Response Surface set label. Identifies the response surfaces
            generated by the RSFIT command.

        yname
            Parameter name. The parameter must have been previously defined as
            a random output parameter with the PDVAR command. Identifies the
            parameter for which a response surface has been generated by the
            RSFIT command.

        xout
            An option if an extended print-out of more feedback about goodness-
            of-fit and the details of the regression analysis of the response
            surface is requested.

            No - Use the standard print-out (default).

            Yes - Use the extended print-out.

        Notes
        -----
        Prints the results and details of a response surface analysis generated
        by the RSFIT command. For the specified output parameter Yname, the
        fitting details such as the individual terms of the response surface
        model and their corresponding coefficients are listed. The command also
        produces a comparison of the original values of Yname used for the
        fitting process and the approximate values derived from the fitting,
        and some goodness of fit measures.

        If Xout = Yes, then more information about the regression analysis of
        the response surface will be printed. For example, the confidence
        intervals on the regression coefficients and the correlation between
        the regression coefficients among others.

        """
        command = "RSPRNT,%s,%s,%s" % (str(rslab), str(yname), str(xout))
        self.RunCommand(command, **kwargs)

    def Btol(self, ptol="", **kwargs):
        """
        APDL Command: BTOL

        Specifies the Boolean operation tolerances.

        Parameters
        ----------
        ptol
            Point coincidence tolerance.  Points within this distance to each
            other will be assumed to be coincident during Boolean operations.
            Loosening the tolerance will increase the run time and storage
            requirements, but will allow more Boolean intersections to succeed.
            Defaults to 0.10E-4.

        Notes
        -----
        Use BTOL,DEFA to reset the setting to its default value.  Use BTOL,STAT
        to list the status of the present setting.

        """
        command = "BTOL,%s" % (str(ptol))
        self.RunCommand(command, **kwargs)

    def Prism(self, z1="", z2="", **kwargs):
        """
        APDL Command: PRISM

        Creates a prism volume based on working plane coordinate pairs.

        Parameters
        ----------
        z1, z2
            Working plane Z coordinates of the top and bottom of the prism.

        Notes
        -----
        Defines a prism volume based on the working plane. The top and bottom
        areas will each be defined with NPT keypoints and NPT lines, where NPT
        (must be at least 3) is the number of coordinate pairs defined with
        PTXY command.  Also, a line will be defined between each point pair on
        the top and bottom face.  See the RPRISM and RPR4 commands for other
        ways to create prisms.

        """
        command = "PRISM,%s,%s" % (str(z1), str(z2))
        self.RunCommand(command, **kwargs)

    def Pras(self, quantity="", loadstep="", substep="", **kwargs):
        """
        APDL Command: PRAS

        Calculates a specified acoustic quantity on the selected exterior
        surface or the frequency-band sound pressure level (SPL).

        Parameters
        ----------
        quantity
            The acoustic quantity to calculate:

            SIMP - Specific acoustic impedance on the selected surface.

            AIMP  - Acoustic impedance on the selected surface.

            MIMP  - Mechanical impedance on the selected surface.

            PRES  - Average pressure on the selected surface.

            FORC  - Force on the selected surface.

            POWE  - Acoustic power on the selected surface.

            BSPL  - Frequency-band sound pressure level.

            BSPA  - A-weighted frequency-band sound pressure level.

        loadstep
            Specified load step. Default = 1.

        substep
            Specified substep. Default = All substeps at the specified load
            step. Not valid for Quantity = BSPL or BSPA.

        Notes
        -----
        The PRAS command calculates a specified acoustic quantity on the
        selected exterior surface in postprocessing. The calculation is based
        on the pressure and velocity solution or the frequency-band sound
        pressure level (SPL).

        The total pressure and velocity are used if the selected surface is the
        excitation source surface. To calculate the incoming and outgoing
        acoustic power, and other sound power parameters, on the excitation
        source surface, issue the SF,,PORT and SPOWER commands.

        The sound pressure level of the octave bands and general frequency band
        (defined via the HARFRQ command) is calculated at the selected nodes in
        the model.

        """
        command = "PRAS,%s,%s,%s" % (str(quantity), str(loadstep), str(substep))
        self.RunCommand(command, **kwargs)

    def Kuse(self, key="", **kwargs):
        """
        APDL Command: KUSE

        Specifies whether or not to reuse the factorized matrix.

        Parameters
        ----------
        key
            Reuse key:

            0  - Program decides whether or not to reuse the previous factorized stiffness
                 matrix.

            1  - Force the previous factorized stiffness matrix to be reused.  Used mainly in a
                 restart.  Forcing reuse of the matrix is a nonstandard use of
                 the program, and should be done with caution.  For instance,
                 using this option and changing the number of elements, or the
                 number or type of degrees of freedom, may cause an abort.

            -1  - All element matrices are reformed and are used to reform a new factorized
                  stiffness matrix.

        Notes
        -----
        Overrides the program logic to determine whether or not to reuse the
        previous factorized stiffness matrix for each substep of this load
        step.  Applies only to static or full transient analyses and to full
        harmonic analyses if the frequency is not changed for continuing
        loadsteps. For full harmonic analyses, only KEY = 1 or KEY = 0 is
        valid.

        This command is also valid in PREP7.

        """
        command = "KUSE,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Local(self, kcn="", kcs="", xc="", yc="", zc="", thxy="", thyz="",
              thzx="", par1="", par2="", **kwargs):
        """
        APDL Command: LOCAL

        Defines a local coordinate system by a location and orientation.

        Parameters
        ----------
        kcn
            Arbitrary reference number assigned to this coordinate system.
            Must be greater than 10.  A coordinate system previously defined
            with this number will be redefined.

        kcs
            Coordinate system type:

            0 or CART - Cartesian

            1 or CYLIN - Cylindrical (circular or elliptical)

            2 or SPHE - Spherical (or spheroidal)

            3 or TORO - Toroidal

        xc, yc, zc
            Location (in the global Cartesian coordinate system) of the origin
            of the new coordinate system.

        thxy
            First rotation about local Z (positive X toward Y).

        thyz
            Second rotation about local X (positive Y toward Z).

        thzx
            Third rotation about local Y (positive Z toward X).

        par1
            Used for elliptical, spheroidal, or toroidal systems.  If KCS = 1
            or 2, PAR1 is the ratio of the ellipse Y-axis radius to X-axis
            radius (defaults to 1.0 (circle)).  If KCS = 3, PAR1 is the major
            radius of the torus.

        par2
            Used for spheroidal systems.  If KCS = 2, PAR2 = ratio of ellipse
            Z-axis radius to X-axis radius (defaults to 1.0 (circle)).

        Notes
        -----
        Defines a local coordinate system by origin location and orientation
        angles.  The local coordinate system is parallel to the global
        Cartesian system unless rotated.  Rotation angles are in degrees and
        redefine any previous rotation angles.  See the CLOCAL, CS, CSWPLA, and
        CSKP commands for alternate definitions.  This local system becomes the
        active coordinate system [CSYS].  Local coordinate systems may be
        displayed with the /PSYMB command.

        This command is valid in any processor.

        """
        command = "LOCAL,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(kcn), str(kcs), str(xc), str(yc), str(zc), str(thxy), str(thyz), str(thzx), str(par1), str(par2))
        self.RunCommand(command, **kwargs)

    def Eintf(self, toler="", k="", tlab="", kcn="", dx="", dy="", dz="",
              knonrot="", **kwargs):
        """
        APDL Command: EINTF

        Defines two-node elements between coincident or offset nodes.

        Parameters
        ----------
        toler
            Tolerance for coincidence (based on maximum Cartesian coordinate
            difference for node locations and on angle differences for node
            orientations). Defaults to 0.0001. Only nodes within the tolerance
            are considered to be coincident.

        k
            Only used when the type of the elements to be generated is
            PRETS179. K is the pretension node that is common to the pretension
            section that is being created. If K is not specified, it will be
            created by ANSYS automatically and will have an ANSYS-assigned node
            number. If K is specified but does not already exist, it will be
            created automatically but will have the user-specified node number.
            K cannot be connected to any existing element.

        tlab
            Nodal number ordering. Allowable values are:

            LOW - The 2-node elements are generated from the lowest numbered node to the highest
                  numbered node.

            HIGH - The 2-node elements are generated from the highest numbered node to the lowest
                   numbered node.

            REVE - Reverses the orientation of the selected 2-node element.

        kcn
            In coordinate system KCN, elements are created between node 1 and
            node 2 (= node 1 + dx dy dz).

        dx, dy, dz
            Node location increments that define the node offset in the active
            coordinate system (DR, Dθ, DZ for cylindrical and DR, Dθ, DΦ for
            spherical or toroidal).

        knonrot
            When KNONROT = 0, the nodes coordinate system is not rotated. When
            KNONROT = 1, the nodes belonging to the elements created are
            rotated into coordinate system KCN (see NROTAT command
            description).

        Notes
        -----
        Defines 2-node elements (such as gap elements) between coincident or
        offset nodes (within a tolerance). May be used, for example, to "hook"
        together elements interfacing at a seam, where the seam consists of a
        series of node pairs. One element is generated for each set of two
        coincident nodes. For more than two coincident or offset nodes in a
        cluster, an element is generated from the lowest numbered node to each
        of the other nodes in the cluster. If fewer than all nodes are to be
        checked for coincidence, use the NSEL command to select the nodes.
        Element numbers are incremented by one from the highest previous
        element number. The element type must be set [ET] to a 2-node element
        before issuing this command. Use the CPINTF command to connect nodes by
        coupling instead of by elements. Use the CEINTF command to connect the
        nodes by constraint equations instead of by elements.

        For contact element CONTA178, the tolerance is based on the maximum
        Cartesian coordinate difference for node locations only. The angle
        differences for node orientations are not checked.

        """
        command = "EINTF,%s,%s,%s,%s,%s,%s,%s,%s" % (str(toler), str(k), str(tlab), str(kcn), str(dx), str(dy), str(dz), str(knonrot))
        self.RunCommand(command, **kwargs)

    def Type(self, itype="", **kwargs):
        """
        APDL Command: TYPE

        Sets the element type attribute pointer.

        Parameters
        ----------
        itype
            Assign this type number to the elements (defaults to 1).

        Notes
        -----
        Activates an element type number to be assigned to subsequently defined
        elements.  This number refers to the element type number (ITYPE)
        defined with the ET command.  Type numbers may be displayed [/PNUM].

        In some cases, ANSYS can proceed with a meshing operation even when no
        logical element type has been assigned via TYPE or XATT,,,TYPE.  For
        more information, see the discussion on setting element attributes in
        Meshing Your Solid Model in the Modeling and Meshing Guide.

        """
        command = "TYPE,%s" % (str(itype))
        self.RunCommand(command, **kwargs)

    def Octable(self, val1="", val2="", val3="", val4="", val5="", val6="",
                val7="", **kwargs):
        """
        APDL Command: OCTABLE

        Defines an ocean load using table data.

        Parameters
        ----------
        val1, val2, val3, . . . , val6
            Values describing the basic ocean load, a current condition, or a
            wave condition.

        Notes
        -----
        The OCTABLE specifies table data that defines the ocean load. The terms
        VAL1, VAL2, etc. are specialized according to the input set required
        for the given ocean load.

        The program interprets the data input via the OCTABLE command within
        the context of the most recently issued OCTYPE command.

        There is no limit to the number of data input.

        Input values in the order indicated.

        This command is also valid in PREP7.

        You can define the following ocean data types:

        If the current is constant, only one OCTABLE command is necessary and
        Dep  is not required.

        For waves, the current profile is stretched or compressed linearly up
        to 10 percent.

        The first Dep value (representing the mean sea level) must be zero. The
        last Dep value (representing the mud line) must be equal to the DEPTH
        value input on the OCDATA command.

        The Cartesian Z values used to locate nodes, etc. decrease as one moves
        from the ocean surface to the sea floor, but the Dep values increase.
        See Figure: 5:: Basic Ocean Data Type Components .

        Dep is not affected by changes to Zmsl on the OCDATA command, as that
        value simply relocates the origin.

        When specifying an ocean wave type, issue the OCTABLE command to input
        either wave location data or wave spectrum data.

        Hints for Wave Location Input:

        The TIME command is not used, except perhaps to identify the load case.

                The phase shift (Ps) determines the wave position (that is, the
        point at which the load is to be applied).

        When using the Stokes fifth-order (KWAVE = 2) or stream function (KWAVE
        = 3) wave type, issue only one OCTABLE command.

        The valid range of the order of the stream function (NORDER) is 3
        through 50. If no value is specified, the program determines a value
        automatically.

        When using the diffracted wave type (KWAVE = 8), an OCREAD command is
        also required to read in the hydrodynamic data from the hydrodynamic
        analysis.

        Hints for Wave Spectrum Input:

        When defining a Pierson-Moskowitz or JONSWAP spectrum (SPECTRUM = 0 or
        1, respectively, on the OCDATA command), issue only one OCTABLE
        command.

        When defining a Pierson-Moskowitz or JONSWAP spectrum for Shell new
        wave (KWAVE = 6 on the OCDATA command), HS is calculated from the
        maximum wave crest amplitude (AMPMAX on the OCDATA command) if no value
        is specified. For further information, see Hydrodynamic Loads in the
        Mechanical APDL Theory Reference.

        For a user-defined spectrum (SPECTRUM = 2 on the OCDATA command), issue
        an OCTABLE command for each frequency data point defining the spectrum.
        Specify the frequency data in ascending order. The number of wave
        components (NWC) is required on the first OCTABLE command only.

        An ocean zone is a local space where you can override global ocean-
        loading parameters.

        Ocean zone data to provide in the value fields:

        where

        Ocean zone values specified via the OCTABLE command override global
        ocean-loading parameters.

        Arguments not specified default to the global values specified for the
        basic ocean type (OCTYPE,BASIC). Therefore, the relationship between Ca
        and CM values (Ca = CM + 1.0) is not applied to ocean zones.

        The OCTABLE command is not valid for a pipe-type ocean zone
        (OCZONE,PIP).

        """
        command = "OCTABLE,%s,%s,%s,%s,%s,%s,%s" % (str(val1), str(val2), str(val3), str(val4), str(val5), str(val6), str(val7))
        self.RunCommand(command, **kwargs)

    def Fe(self, nev="", cycle="", fact="", title="", **kwargs):
        """
        APDL Command: FE

        Defines a set of fatigue event parameters.

        Parameters
        ----------
        nev
            Reference number for this event (within MXEV).

        cycle
            Number of required cycles (defaults to 1).  If -1, erase all
            parameters and fatigue stresses for this event.

        fact
            Scale factor to be applied to all loadings in this event (defaults
            to 1.0).

        title
            User defined identification title for this event (up to 20
            characters).

        Notes
        -----
        Repeat FE command to define additional sets of event parameters (MXEV
        limit), to redefine event parameters, or to delete event stress
        conditions.

        The set of fatigue event parameters is associated with all loadings and
        all locations.  See the FTSIZE command for the maximum set of events
        (MXEV) allowed.

        """
        command = "FE,%s,%s,%s,%s" % (str(nev), str(cycle), str(fact), str(title))
        self.RunCommand(command, **kwargs)

    def Mfbucket(self, key="", value="", **kwargs):
        """
        APDL Command: MFBUCKET

        Turns a bucket search on or off.

        Parameters
        ----------
        key
            Bucket search key:

            ON - Activates a bucket search (default).

            OFF - Deactivates a bucket search. A global search is then activated.

        value
            Scaling factor (%) used to determine the number of buckets for a
            bucket search. Defaults to 50%.

        Notes
        -----
        A bucket search will more efficiently compute the mapping of surface
        and volumetric interpolation data across field interfaces (flagged by
        the FSIN label using SF, SFA, SFE, or SFL or the FVIN label using BFE).

        The number of buckets used to partition a flagged interface is equal to
        the scaling factor (%) times the total number of interface elements.
        For example, for the default scaling factor of 50% and a 10,000 element
        interface, 5,000 buckets are used.

        This command is also valid in PREP7.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFBUCKET,%s,%s" % (str(key), str(value))
        self.RunCommand(command, **kwargs)

    def Remesh(self, action="", filename="", ext="", opt1="", opt2="",
               **kwargs):
        """
        APDL Command: REMESH

        Specifies the starting and ending remeshing points, and other options,
        for rezoning.

        Parameters
        ----------
        action
            START

            START - Starts the remeshing operation.

            FINISH - Ends the remeshing operation.

            READ - Reads in a generic (.cdb format) new mesh file generated by a third-party
                   application. This remeshing option applies to both 2-D and
                   3-D rezoning.

            SPLIT - Splits selected elements of an existing 2-D or 3-D mesh such that a
                    quadrilateral element is split into four quadrilaterals, a
                    degenerate quadrilateral is split into three
                    quadrilaterals, and a quadratic triangular element is split
                    into four quadratic triangles.  A tetrahedral element is
                    split into eight tetrahedra.

        filename
            Name of a .cdb generic mesh file. The default value is jobname.
            Valid only when Action = READ.

        ext
            File name extension. The only valid (and the default) extension is
            CDB. Valid only when Action = READ.

        --
            Unused field.

        opt1
            Specifies options for the new mesh when using a generic imported
            mesh file or the mesh-splitting remeshing method. Valid only when
            Action = READ or Action = SPLIT.

            REGE  - Regenerates all node and element numbers on the new mesh using an offset of the
                    highest existing node and element numbers. This is the
                    default behavior when Action = READ; otherwise, this value
                    is ignored.

            KEEP  - Keeps the similarly numbered nodes and elements in the new and the old meshes
                    unchanged. Valid only when Action = READ.

            TRAN  - Generates transition elements to ensure nodal compatibility between split and
                    unsplit parts of the mesh. Valid only when Action = SPLIT
                    for 2-D analyses.

        opt2
            Specifies transition options for the mesh when elements are split.
            These options are valid only when Action = SPLIT for 2-D analyses.

            QUAD  - Minimizes the number of degenerate elements in the transition mesh and tries to
                    maximize the number of quadrilateral transition elements
                    across several layers of elements from the split regions.
                    This is the default behavior.

            DEGE  - Creates transition zones between the split and unsplit parts of the mesh using
                    mostly degenerate elements with a single element layer.

        Notes
        -----
        The REMESH command is valid only during the rezoning (REZONE) process.

        In rezoning, a REMESH,START command temporarily exits the /SOLU
        solution processor and enters a special mode of the /PREP7
        preprocessor, after which a limited number of preprocessing commands
        are available for mesh control, but no solution commands are valid.

        A REMESH,FINISH command exits the remeshing process and reenters the
        solution processor, at which point no preprocessing commands are
        available. If the new mesh exists, the command creates contact elements
        if needed, and transfers all boundary conditions (BCs) and loads from
        the original mesh to the new mesh. You can issue any list or plot
        command to verify the created contact elements, transferred BCs, and
        loads. A REMESH,FINISH command is valid only after a previously issued
        REMESH,START command, and is the only way to safely end the remeshing
        operation (and exit the special mode of the /PREP7 preprocessor).

        A REMESH,READ command is valid only when you want to perform a rezoning
        operation using a generic new mesh generated by a third-party
        application (rather than a new mesh generated internally by the ANSYS
        program). The command is valid between REMESH,START and REMESH,FINISH
        commands. In this case, the only valid file extension is .cdb (Ext =
        CDB). When Option = KEEP, ANSYS assumes that the common node and
        element numbers between the old and the new mesh are topologically
        similar (that is, these commonly numbered areas have the same element
        connectivity and nodal coordinates).

        A REMESH,SPLIT command is valid only when you wish to perform a
        rezoning operation by splitting the existing mesh. The command is valid
        between REMESH,START and REMESH,FINISH commands.

        You can use REMESH,READ and REMESH,SPLIT commands for horizontal
        multiple rezoning provided that the meshes used in REMESH,READ do not
        intersect. (ANSYS recommends against issuing an AREMESH command after
        issuing either of these commands.)

        For more detailed about the remeshing options available to you during a
        rezoning operation, see Rezoning in the Advanced Analysis Guide.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "REMESH,%s,%s,%s,%s,%s" % (str(action), str(filename), str(ext), str(opt1), str(opt2))
        self.RunCommand(command, **kwargs)

    def Map(self, kdim="", kout="", limit="", **kwargs):
        """
        APDL Command: MAP

        Maps pressures from source points to target surface elements.

        Parameters
        ----------
        --
            Unused field.

        kdim
            Interpolation key:

            0 or 2 - Interpolation is done on a surface (default).

            3 - Interpolation is done within a volume. This option is useful if the supplied
                source data is volumetric field data rather than surface data.

        --
            Unused field.

        kout
            Key to control how pressure is applied when a target node is
            outside of the source region:

            0 - Use the pressure(s) of the nearest source point for target nodes outside of the
                region (default).

            1 - Set pressures outside of the region to zero.

        limit
            Number of nearby points considered for interpolation. The minimum
            is 5; the default is 20. Lower values reduce processing time.
            However, some distorted or irregular meshes will require a higher
            LIMIT value to find the points encompassing the target node in
            order to define the region for interpolation.

        Notes
        -----
        Maps pressures from source points to target surface elements.

        """
        command = "MAP,%s,%s,%s" % (str(kdim), str(kout), str(limit))
        self.RunCommand(command, **kwargs)

    def File(self, fname="", ext="", **kwargs):
        """
        APDL Command: FILE

        Specifies the data file where results are to be found.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        Specifies the ANSYS data file where the results are to be found for
        postprocessing.

        """
        command = "FILE,%s,%s" % (str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Dflx(self, node="", bx="", by="", bz="", bx2="", by2="", bz2="",
             **kwargs):
        """
        APDL Command: DFLX

        Imposes a uniform magnetic flux B on an edge-element electromagnetic
        model.

        Parameters
        ----------
        node
            Nodes at which the edge-flux (AZ) constraints corresponding to the
            uniform magnetic flux are to be specified.  Valid options are ALL
            (default) or Component Name. If ALL, constraints are applied to all
            selected nodes (NSEL).

        bx, by, bz
            Real components of magnetic flux B.

        bx2, by2, bz2
            Imaginary components of magnetic flux B.

        Notes
        -----
        The DFLX command sets the constraints on the edge-flux (AZ) degrees of
        freedom to produce a uniform magnetic flux B in an edge-based
        electromagnetic analysis using elements SOLID236 and SOLID237. The
        command ignores the corner nodes of the elements (even if they were
        selected) and imposes the AZ-constraints on the mid-side nodes only.
        The AZ-constraints are imposed in the active Cartesian coordinate
        system. A non-Cartesian coordinate system will be ignored by the DFLX
        command.

        The edge-flux constraints at the mid-side nodes are derived from  the
        magnetic vector potential A, which is related to the imposed magnetic
        flux B as follows:

        where r is the position of the mid-side node.

        The DFLX command creates a component named _DFLX for the constrained
        midside nodes. You can use this component to delete the constraints
        imposed by the DFLX command.

        This command is also valid in PREP7.

        """
        command = "DFLX,%s,%s,%s,%s,%s,%s,%s" % (str(node), str(bx), str(by), str(bz), str(bx2), str(by2), str(bz2))
        self.RunCommand(command, **kwargs)

    def Secplot(self, secid="", val1="", val2="", val3="", **kwargs):
        """
        APDL Command: SECPLOT

        Plots the geometry of a beam, pipe, shell, or reinforcing section to
        scale.

        Parameters
        ----------
        secid
            The section ID number (as defined via the SECTYPE command).

        val1, val2, val3
            Values that control the information to be plotted. See the "Notes"
            section of this command description for more information. For
            clarity, the labels VAL1, VAL2, and VAL3 are renamed according to
            the section type.

        Notes
        -----
        The SECPLOT command is valid only for "Beams and Pipes", "Shells", and
        "Reinforcings".

        SECPLOT cannot display the plot of an ASEC (arbitrary section) subtype.

        Plots the geometry of the beam or pipe section to scale depicting the
        centroid, shear center, and origin.  SECPLOT also lists various section
        properties such as Iyy, Iyz, and Izz.

        Data to be supplied in the value fields:

        Beam or pipe section mesh display options:

        Display section outline only.

        Display beam or pipe section mesh.

        Display the section mesh with node numbers.

        Display the section mesh with cell numbers.

        Display the section mesh with material numbers and colors.

        Display the section mesh with material colors only.

        Display the section mesh with the RST node numbers. RST nodes are
        section corner nodes where results are available. This is applicable
        when the averaged results format (KEYOPT(15) = 0 for BEAM188, BEAM189,
        PIPE288, and PIPE289) is used.

        Display the section mesh with the RST cell numbers. RST cells are
        section cells where results are available. This is applicable when the
        non-averaged results format (KEYOPT(15) = 1 for BEAM188, BEAM189,
        PIPE288, and PIPE289) is used.

        Options 2 through 6 do not depict centroid and shear center, nor do
        they list section properties.

        Following is a sample section plot for the beam section type:

        Plots the layer arrangement of the shell section showing the layer
        material and orientation.

        Data to be supplied in the value fields:

        The range of layer numbers to be displayed. If LAYR1 is greater than
        LAYR2, a reversed order display is produced. Up to 20 layers may be
        displayed at the same time. LAYR1 defaults to 1. LAYR2 defaults to
        LAYR1 if LAYR1 is input or to the number of layers (or to 19+LAYR1, if
        smaller) if LAYR1 is not input.

        Following is a sample section plot for the shell section type:

        Plots the arrangement of a reinforcing section within the base element.

        Data to be supplied in the value fields:

        REINF1, REINF2 -- The numerical range of reinforcings to be displayed.
        The default REINF1 value is 1. The default REINF2 value is the number
        of reinforcings.

        OVERLAY -- The section ID of the base element within which to display
        the reinforcing section. The section appears translucent and the
        reinforcing section is solid. Valid values are:

        SOLID -- Display a translucent solid block over the reinforcing section

        SECID -- A number corresponding to a specific section ID of the base
        element.

        If no OVERLAY value is specified, ANSYS displays the reinforcing
        section only.

        Following is a sample section plot for the reinforcing section type:

        For more information about reinforcing, see the documentation for the
        SECDATA command, and the REINF264 and REINF265 elements.

        """
        command = "SECPLOT,%s,%s,%s,%s" % (str(secid), str(val1), str(val2), str(val3))
        self.RunCommand(command, **kwargs)

    def Et(self, itype="", ename="", kop1="", kop2="", kop3="", kop4="",
           kop5="", kop6="", inopr="", **kwargs):
        """
        APDL Command: ET

        Defines a local element type from the element library.

        Parameters
        ----------
        itype
            Arbitrary local element type number. Defaults to 1 + current
            maximum.

        ename
            Element name (or number) as given in the element library in Chapter
            4 of the Element Reference. The name consists of a category prefix
            and a unique number, such as PIPE288.  The category prefix of the
            name (PIPE for the example) may be omitted but is displayed upon
            output for clarity. If Ename = 0, the element is defined as a null
            element.

        kop1, kop2, kop3, . . . , kop6
            KEYOPT values (1 through 6) for this element, as described in the
            Element Reference.

        inopr
            If 1, suppress all element solution printout for this element type.

        Notes
        -----
        The ET command selects an element type from the element library and
        establishes it as a local element type for the current model.
        Information derived from the element type is used for subsequent
        commands, so the ET command(s) should be issued early. (The Element
        Reference describes the available elements.)

        A special option, Ename = 0, permits the specified element type to be
        ignored during solution without actually removing the element from the
        model. Ename may be set to zero only after the element type has been
        previously defined with a nonzero Ename.  The preferred method of
        ignoring elements is to use the select commands (such as ESEL).

        KOPn are element option keys. These keys (referred to as KEYOPT(n)) are
        used to turn on certain element options for this element. These options
        are listed under "KEYOPT" in the input table for each element type in
        the Element Reference.  KEYOPT values include stiffness formulation
        options, printout controls, and various other element options. If
        KEYOPT(7) or greater is needed, input their values with the KEYOPT
        command.

        The ET command only defines an element type local to your model (from
        the types in the element library). The TYPE or similar [KATT, LATT,
        AATT, or VATT] command must be used to point to the desired local
        element type before meshing.

        To activate the ANSYS program's LS-DYNA explicit dynamic analysis
        capability,  use the ET command or its GUI equivalent to choose an
        element that works only with LS-DYNA (such as SHELL163).  Choosing LS-
        DYNA in the Preferences dialog box does not activate LS-DYNA; it simply
        makes items and options related to LS-DYNA accessible in the GUI.

        """
        command = "ET,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(itype), str(ename), str(kop1), str(kop2), str(kop3), str(kop4), str(kop5), str(kop6), str(inopr))
        self.RunCommand(command, **kwargs)

    def Cmatrix(self, symfac="", condname="", numcond="", grndkey="",
                capname="", **kwargs):
        """
        APDL Command: CMATRIX

        Performs electrostatic field solutions and calculates the self and
        mutual capacitances between multiple conductors.

        Parameters
        ----------
        symfac
            Geometric symmetry factor.  Capacitance values are scaled by this
            factor which represents the fraction of the total device modeled.
            Defaults to 1.

        condname
            Alpha-numeric prefix identifier used in defining named conductor
            components.

        numcond
            Total Number of Components.  If a ground is modeled, it is to be
            included as a component.  If a ground is not modeled, but infinite
            elements are used to model the far-field ground, a named component
            for the far-field ground is not required.

        grndkey
            Ground key:

            0 - Ground is one of the components, which is not at infinity.

            1 - Ground is at infinity (modeled by infinite elements).

        capname
            Array name for computed capacitance matrix.  Defaults to CMATRIX.

        Notes
        -----
        To invoke the CMATRIX macro, the exterior nodes of each conductor must
        be grouped into individual components using the CM command.  Each set
        of  independent components is assigned a component name with a common
        prefix followed by the conductor number.  A conductor system with a
        ground must also include the ground nodes as a component.  The ground
        component is numbered last in the component name sequence.

        A ground capacitance matrix relates charge to a voltage vector.  A
        ground matrix cannot be applied to a circuit modeler. The lumped
        capacitance matrix is a combination of  lumped "arrangements" of
        voltage differences between conductors.  Use the lumped capacitance
        terms in a circuit modeler to represent capacitances between
        conductors.

        Enclose all name-strings in single quotes in the CMATRIX command line.

        See the Mechanical APDL Theory Reference and HMAGSOLV in the Low-
        Frequency Electromagnetic Analysis Guide for details.

        This command does not support multiframe restarts.

        """
        command = "CMATRIX,%s,%s,%s,%s,%s" % (str(symfac), str(condname), str(numcond), str(grndkey), str(capname))
        self.RunCommand(command, **kwargs)

    def Nprint(self, n="", **kwargs):
        """
        APDL Command: NPRINT

        Defines which time points stored are to be listed.

        Parameters
        ----------
        n
            List data associated with every N time (or frequency) point(s),
            beginning with the first point stored (defaults to 1).

        Notes
        -----
        Defines which time (or frequency) points within the range stored are to
        be listed.

        """
        command = "NPRINT,%s" % (str(n))
        self.RunCommand(command, **kwargs)

    def Aclear(self, na1="", na2="", ninc="", **kwargs):
        """
        APDL Command: ACLEAR

        Deletes nodes and area elements associated with selected areas.

        Parameters
        ----------
        na1, na2, ninc
            Delete mesh for areas NA1 to NA2 (defaults to NA1) in steps of NINC
            (defaults to 1).  If NA1 = ALL, NA2 and NINC are ignored and the
            mesh for all selected areas [ASEL] is deleted.  If NA1 = P,
            graphical picking is enabled and all remaining arguments are
            ignored (valid only in the GUI).  A component name may also be
            substituted for NA1 (NA2 and NINC are ignored).

        Notes
        -----
        Deletes all nodes and area elements associated with selected areas
        (regardless of whether the nodes or elements are selected).  Nodes
        shared by adjacent meshed areas and nodes associated with non-area
        elements will not be deleted.  Attributes assigned as a result of AATT
        are maintained.  In the program's response to the command, if an area,
        line, or keypoint is tallied as "cleared," it means either its node or
        element reference was deleted.

        This command is also valid for rezoning. When issued during rezoning
        (after the REMESH,START command and before the REMESH,FINISH command),
        ACLEAR clears only the area generated by the AREMESH command.

        """
        command = "ACLEAR,%s,%s,%s" % (str(na1), str(na2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Vtran(self, kcnto="", nv1="", nv2="", ninc="", kinc="", noelem="",
              imove="", **kwargs):
        """
        APDL Command: VTRAN

        Transfers a pattern of volumes to another coordinate system.

        Parameters
        ----------
        kcnto
            Reference number of coordinate system where the pattern is to be
            transferred.  Transfer occurs from the active coordinate system.
            The coordinate system type and parameters of KCNTO must be the same
            as the active system.

        nv1, nv2, ninc
            Transfer volumes from pattern beginning with NV1 to NV2 (defaults
            to NV1) in steps of NINC (defaults to 1).  If NV1 = ALL, NV2 and
            NINC are ignored and the pattern is all selected volumes [VSEL].
            If NV1 = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).  A component name may
            also be substituted for NV1 (NV2 and NINC are ignored).

        kinc
            Keypoint increment between sets.  If zero, the lowest available
            keypoint numbers are assigned [NUMSTR].

        noelem
            Specifies whether elements and nodes are also to be generated:

            0 - Generate nodes and elements associated with the original volumes, if they
                exist.

            1 - Do not generate nodes and elements.

        imove
            Specifies whether to redefine the existing volumes:

            0 - Generate additional volumes.

            1 - Move original volumes to new position retaining the same keypoint numbers (KINC
                and NOELEM are ignored).  Corresponding meshed items are also
                moved if not needed at their original position.

        Notes
        -----
        Transfers a pattern of volumes (and their corresponding keypoints,
        lines, areas and mesh) from one coordinate system to another (see
        analogous node transfer command, TRANSFER).  The MAT, TYPE, REAL, and
        ESYS attributes are based upon the volumes in the pattern and not upon
        the current settings.  Coordinate systems may be translated and rotated
        relative to each other.  Initial pattern may be generated in any
        coordinate system.  However, solid modeling in a toroidal coordinate
        system is not recommended.  Coordinate and slope values are interpreted
        in the active coordinate system and are transferred directly.  Volumes
        are generated as described in the VGEN command.

        """
        command = "VTRAN,%s,%s,%s,%s,%s,%s,%s" % (str(kcnto), str(nv1), str(nv2), str(ninc), str(kinc), str(noelem), str(imove))
        self.RunCommand(command, **kwargs)

    def Edlcs(self, option="", cid="", x1="", y1="", z1="", x2="", y2="",
              z2="", x3="", y3="", z3="", **kwargs):
        """
        APDL Command: EDLCS

        Defines a local coordinate system for use in explicit dynamics
        analysis.

        Parameters
        ----------
        option
            Label identifying the option to be performed:

            ADD - Define a coordinate system (default).

            DELE - Delete a coordinate system. If CID is blank, all coordinate systems are
                   deleted.

            LIST - List defined coordinate systems. If CID is blank, all coordinate systems are
                   listed.

        cid
            Coordinate system ID.

        x1, y1, z1
            X, Y, and Z coordinates of a point on the local x-axis.

        x2, y2, z2
            X, Y, and Z coordinates of a point on the local x-y plane.

        x3, y3, z3
            X, Y, and Z coordinates of the origin. X3, Y3, and Z3 all default
            to zero.

        Notes
        -----
        Local coordinate systems defined by this command are used in an
        explicit dynamic analysis. For example, a local coordinate system may
        be used when defining orthotropic material properties (see EDMP).

        The coordinate system is defined by 2 vectors, one from the origin (X3,
        Y3, Z3) to a point on the x-axis (X1, Y1, Z1), and one from the origin
        to a point on the x-y plane (X2, Y2, Z2).  The cross product of these
        two vectors determines the z-axis, and the cross product of the z-axis
        vector and x-axis vector determines the y-axis. If X3, Y3, and Z3 are
        not specified, the global origin (0,0,0) is used by default (as shown
        in the figure below).

        The x-axis vector and the xy vector should be separated by a reasonable
        angle to avoid numerical inaccuracies.

        When you use the local coordinate system (defined by the EDLCS command)
        to define a load (EDLOAD command), the direction of the load will
        depend on the load type. For force and moment loads (Lab = FX, MX, etc.
        on EDLOAD), the load will be applied in the direction of the local
        coordinate system defined by EDLCS. For prescribed motion degrees of
        freedom (Lab = UX, ROTX, VX, AX, etc. on EDLOAD), the motion will act
        in the direction of a vector from point (X1, Y1, Z1) to point (X2, Y2,
        Z2) as input on EDLCS. See the EDLOAD command for more information.

        This command is also valid in SOLUTION.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDLCS,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(cid), str(x1), str(y1), str(z1), str(x2), str(y2), str(z2), str(x3), str(y3), str(z3))
        self.RunCommand(command, **kwargs)

    def Aina(self, na1="", na2="", na3="", na4="", na5="", na6="", na7="",
             na8="", na9="", **kwargs):
        """
        APDL Command: AINA

        Finds the intersection of areas.

        Parameters
        ----------
        na1, na2, na3, . . . , na9
            Numbers of areas to be intersected.  If NA1 = ALL, NA2 to NA9 are
            ignored and the intersection of all selected areas is found.  If
            NA1 = P, graphical picking is enabled and all remaining arguments
            are ignored (valid only in the GUI).  A component name may also be
            substituted for NA1.

        Notes
        -----
        Finds the common (not pairwise) intersection of areas.  The common
        intersection is defined as the regions shared (in common) by all areas
        listed on this command.  New areas will be generated where the original
        areas intersect.  If the regions of intersection are only lines, new
        lines will be generated instead.  See the Modeling and Meshing Guide
        for an illustration.  See the BOPTN command for the options available
        to Boolean operations.  Element attributes and solid model boundary
        conditions assigned to the original entities will not be transferred to
        the new entities generated.

        """
        command = "AINA,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(na1), str(na2), str(na3), str(na4), str(na5), str(na6), str(na7), str(na8), str(na9))
        self.RunCommand(command, **kwargs)

    def Rlist(self, nset1="", nset2="", ninc="", **kwargs):
        """
        APDL Command: RLIST

        Lists the real constant sets.

        Parameters
        ----------
        nset1, nset2, ninc
            List real constant sets from NSET1 to NSET2 (defaults to NSET1) in
            steps of NINC (defaults to 1).  If NSET1 = ALL (default), ignore
            NSET2  and NINC and list all real constant sets [R].

        Notes
        -----
        The real constant sets listed contain only those values specifically
        set by the user.  Default values for real constants set automatically
        within the various elements are not listed.

        This command is valid in any processor.

        """
        command = "RLIST,%s,%s,%s" % (str(nset1), str(nset2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Pri2(self, p51x="", z1="", z2="", **kwargs):
        """
        APDL Command: PRI2

        Creates a polygonal area or a prism volume by vertices (GUI).

        Notes
        -----
        Creates a polygonal area or a prism volume using the vertices as input.
        This is a command generated by the Graphical User Interface (GUI) and
        will appear in the log file (Jobname.LOG) if graphical picking is used.
        This command is not intended to be typed in directly in an ANSYS
        session (although it can be included in an input file for batch input
        or for use with the /INPUT command).

        For polygons, the PRI2 command will appear in the log file as
        PRI2,P51X,0.0,0.0 preceded by FITEM commands that define the vertices
        (in global Cartesian coordinates).  For prisms, the PRI2 command will
        appear in the log file as PRI2,P51X preceded by FITEM commands that
        define the vertices and the Z-end of the prism.

        See the RPOLY, POLY, RPRISM, PRISM, and RPR4 commands for other ways to
        create polygons and prisms.

        """
        command = "PRI2,%s,%s,%s" % (str(p51x), str(z1), str(z2))
        self.RunCommand(command, **kwargs)

    def Boptn(self, lab="", value="", **kwargs):
        """
        APDL Command: BOPTN

        Specifies Boolean operation options.

        Parameters
        ----------
        lab
            Default/status key:

            DEFA  - Resets settings to default values.

            STAT  - Lists status of present settings.

        value
            Option settings if Lab = KEEP:

            NO  - Delete entities used as input with a Boolean operation (default).  Entities
                  will not be deleted if meshed or if attached to a higher
                  entity.

            YES  - Keep input solid modeling entities.

        Notes
        -----
        Boolean operations at Revision 5.2 may produce a different number of
        entities than previous revisions of ANSYS.   When running input files
        created at earlier revisions of ANSYS, match the Boolean compatibility
        option (VERSION) to the revision originally used. For instance,  if you
        are running Revision 5.2 and are reading an input file (/INPUT) created
        at Revision 5.1, it is recommended that you set VERSION to RV51 before
        reading the input.

        See the Modeling and Meshing Guide for further details on the functions
        of the RV51 and RV52 labels.

        This command is valid in any processor.

        """
        command = "BOPTN,%s,%s" % (str(lab), str(value))
        self.RunCommand(command, **kwargs)

    def Edasmp(self, option="", _asmid="", part1="", part2="", part3="",
               part4="", part5="", part6="", part7="", part8="", part9="",
               part10="", part11="", part12="", part13="", part14="",
               part15="", part16="", **kwargs):
        """
        APDL Command: EDASMP

        Creates a part assembly to be used in an explicit dynamic analysis.

        Parameters
        ----------
        option
            Label identifying the part assembly option to be performed.

            ADD - Adds a part assembly (default).

            DELETE - Deletes a part assembly.

            LIST - Lists each part assembly number, and the part numbers that make up each part
                   assembly.

        asmid
             User defined part assembly ID number. The part assembly number
            cannot be the same as any currently defined part ID number.

        part1, part2, part3, . . . , part16
            Part numbers to be included in the assembly (up to 16 different
            parts).

        Notes
        -----
        Several ANSYS LS-DYNA commands (such as EDCGEN, EDPVEL, and EDIS) refer
        to assembly ID numbers. If you intend to use assembly ID numbers with
        these commands, you must first define the assembly ID numbers using
        EDASMP.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDASMP,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(_asmid), str(part1), str(part2), str(part3), str(part4), str(part5), str(part6), str(part7), str(part8), str(part9), str(part10), str(part11), str(part12), str(part13), str(part14), str(part15), str(part16))
        self.RunCommand(command, **kwargs)

    def Data(self, ir="", lstrt="", lstop="", linc="", name="", kcplx="",
             **kwargs):
        """
        APDL Command: DATA

        Reads data records from a file into a variable.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]).  If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.

        lstrt
            Start at location LSTRT (defaults to 1).

        lstop
            Stop at location LSTOP (defaults to LSTRT).  Maximum location
            available is determined from data previously stored.

        linc
            Fill every LINC location between LSTRT and LSTOP (defaults to 1).

        name
            Eight character name for identifying the variable on the printout
            and displays.  Embedded blanks are compressed upon output.

        kcplx
            Complex number key:

            0 - Data stored as the real part of the complex number.

            1 - Data stored as the imaginary part of the complex number.

        Notes
        -----
        This command must be followed by a format statement (on the next line)
        and the subsequent data records, and all must be on the same file (that
        may then be read with the /INPUT command).  The format specifies the
        number of fields to be read per record, the field width, and the
        placement of the decimal point (if one is not included in the data
        value).  The read operation follows the available FORTRAN FORMAT
        conventions of the system.  See the system FORTRAN manual for details.
        Any standard FORTRAN real format (such as (4F6.0), (F2.0,2X,F12.0),
        etc.) may be used.  Integer (I), character (A), and list-directed (*)
        descriptors may not be used.  The parentheses must be included in the
        format.  Up to 80 columns per record may be read.  Locations may be
        filled within a range.  Previous data in the range will be overwritten.

        """
        command = "DATA,%s,%s,%s,%s,%s,%s" % (str(ir), str(lstrt), str(lstop), str(linc), str(name), str(kcplx))
        self.RunCommand(command, **kwargs)

    def Mopt(self, lab="", value="", **kwargs):
        """
        APDL Command: MOPT

        Specifies meshing options.

        Parameters
        ----------
        lab
            Meshing option to be specified (determines the meaning of Value):

            AORDER - Mesh by ascending area size order.  Set Value to ON to mesh smaller areas
                     first. Using this results in finer meshes in critical
                     areas for volume meshes; this can be used for cases where
                     SMRTSIZE does not mesh as needed. Default is OFF.

            EXPND - Area mesh expansion (or contraction) option.  (This option is the same as
                    SMRTSIZE,,,EXPND.) This option is used to size internal
                    elements in an area based on the size of the elements on
                    the area's boundaries.

            Value is the expansion (or contraction) factor. For example, issuing MOPT,EXPND,2 before meshing an area will allow a mesh with elements that are approximately twice as large in the interior of an area as they are on the boundary. If Value is less than 1, a mesh with smaller elements on the interior of the area will be allowed. Value for this option should be greater than 0.5 but less than 4.  - Value defaults to 1, which does not allow expansion or contraction of internal
                              element sizes (except when using AESIZE sizing).
                              If Value = 0, the default value of 1 will be
                              used. The actual size of the internal elements
                              will also depend on the TRANS option (or upon
                              AESIZE or ESIZE sizing, if used).

            TETEXPND - Tet-mesh expansion (or contraction) option. This option is used to size
                       internal elements in a volume based on the size of the
                       elements on the volume's boundaries.

            Value is the expansion (or contraction) factor.  For example, issuing MOPT,TETEXPND,2 before meshing a volume will allow a mesh with elements that are approximately twice as large in the interior of the volume as they are on the boundary. If Value is less than 1, a mesh with smaller elements on the interior of the volume will be allowed. Value for this option should be greater than 0.1 but less than 3.   - Value defaults to 1, which does not allow expansion or contraction of internal
                              element sizes.  If Value = 0, the default value
                              of 1 will be used. If Value is greater than 2,
                              mesher robustness may be affected.

            The TETEXPND option is supported for both the VMESH and FVMESH commands.  Tet-mesh expansion is the only mesh control supported by FVMESH. - TRANS

            Mesh-transition option. Controls how rapidly elements are permitted to change in size from the boundary to the interior of an area. (This option performs the same operation as SMRTSIZE,,,,TRANS.)   - Value is the transitioning factor. Value defaults to 2.0, which permits
                              elements to approximately double in size as they
                              approach the interior of the area.  (If Value =
                              0, the default value of 2 will be used.) Value
                              must be greater than 1 and, for best results,
                              should be less than 4.  The actual size of the
                              internal elements will also depend on the EXPND
                              option (or upon AESIZE or ESIZE sizing, if used).

            For a quad mesh with any element size, this option has no effect, as the program strictly respects any face size to ensure the most uniform quad mesh possible. To obtain a graded mesh using this option, apply LESIZE to the lines of the desired face. - AMESH

            Triangle surface-meshing option.  Valid inputs for Value are:     - DEFAULT

            Allows the program to choose which triangle mesher to use.  In most cases, the program chooses the main triangle mesher, which is the Riemann space mesher.  If the chosen mesher fails for any reason, the program invokes the alternate mesher and retries the meshing operation. - MAIN

            The program uses the main triangle mesher (Riemann space mesher), and it does not invoke an alternate mesher if the main mesher fails.  The Riemann space mesher is well suited for most surfaces. - ALTERNATE

            The program uses the first alternate triangle mesher (3-D tri-mesher), and it does not invoke another mesher if this mesher fails.  This option is not recommended due to speed considerations.  However, for surfaces with degeneracies in parametric space, this mesher often provides the best results. - ALT2

            The program uses the second alternate triangle mesher (2-D parametric space mesher), and it does not invoke another mesher if this mesher fails.  This option is not recommended for use on surfaces with degeneracies (spheres, cones, etc.) or poorly parameterized surfaces because poor meshes may result. - QMESH

            Quadrilateral surface meshing option.  (Quadrilateral surface meshes will differ based on which triangle surface mesher is selected.  This is true because all free quadrilateral meshing algorithms use a triangle mesh as a starting point.)  Valid inputs for Value are:     - DEFAULT

            Let the program choose which quadrilateral mesher to use.  In most cases, the program will choose the main quadrilateral mesher, which is the Q-Morph (quad-morphing) mesher.  For very coarse meshes, the program may choose the alternate quadrilateral mesher instead.  In most cases, the Q-Morph mesher results in higher quality elements.  If either mesher fails for any reason, the program invokes the other mesher and retries the meshing operation.  (Default.) - MAIN

            The program uses the main quadrilateral mesher (Q-Morph mesher), and it does not invoke the alternate mesher if the main mesher fails. - ALTERNATE

            The program uses the alternate quadrilateral mesher, and it does not invoke the Q-Morph mesher if the alternate mesher fails.  To use the alternate quadrilateral mesher, you must also select MOPT,AMESH,ALTERNATE or MOPT,AMESH,ALT2. - VMESH

            Tetrahedral element meshing option.  Valid inputs for Value are:     - DEFAULT

            Let the program choose which tetrahedra mesher to use.   - MAIN

            Use the main tetrahedra mesher (Delaunay technique mesher).  (GHS3D meshing technology by P. L. George, INRIA, France.)  For most models, this mesher is significantly faster than the alternate mesher. - ALTERNATE

            Use the alternate tetrahedra mesher (advancing front mesher).  This mesher does not support the generation of a tetrahedral volume mesh from facets (FVMESH).  If this mesher is selected and you issue the FVMESH command, the program uses the main tetrahedra mesher to create the mesh from facets and issues a warning message to notify you. - SPLIT

            Quad splitting option for non-mapped meshing.  If Value = 1, ON, or ERR, quadrilateral elements in violation of shape error limits are split into triangles (default).  If Value = 2 or WARN, quadrilateral elements in violation of either shape error or warning limits are split into triangles.  If Value = OFF, splitting does not occur, regardless of element quality. - LSMO

            Line smoothing option. Value can be ON or OFF.  If Value = ON, smoothing of nodes on area boundaries is performed during smoothing step of meshing.  During smoothing, node locations are adjusted to achieve a better mesh.  If Value = OFF (default), no smoothing takes place at area boundaries. - CLEAR

            This option affects the element and node numbering after clearing a mesh.  If Value = ON (default), the starting node and element numbers will be the lowest available number after the nodes and elements are cleared.  If Value = OFF, the  starting node and element numbers are not reset after the clear operation. - PYRA

            Transitional pyramid elements option.  Value can be ON or OFF.  If Value = ON (default), the program automatically creates transitional pyramid elements, when possible.  Pyramids may be created at the interface of tetrahedral and hexahedral elements, or directly from quadrilateral elements.  For pyramids to be created, you must also issue the command MSHAPE,1,3D (degenerate 3-D elements).  If Value = OFF, the program does not create transitional pyramid elements. - TIMP

            Identifies the level of tetrahedra improvement to be performed when the next free volume meshing operation is initiated (VMESH, FVMESH).  (For levels 2-5, improvement occurs primarily through the use of face swapping and node smoothing techniques.)  Valid inputs for Value are:     - 0

            Turn off tetrahedra improvement.  Although this value can lead to faster tetrahedral mesh creation, it is not recommended because it often leads to poorly shaped elements and mesh failures. - 1

            Do the minimal amount of tetrahedra improvement.  (Default.)  This option is supported by the main tetrahedra mesher only [MOPT,VMESH,MAIN].  If the alternate tetrahedra mesher (MOPT,VMESH,ALTERNATE) is invoked with this setting, the program automatically performs tetrahedra improvement at level 3 instead (MOPT,TIMP,3). - 2

            Perform the least amount of swapping/smoothing.  No improvement occurs if all tetrahedral elements are within acceptable limits. - 3

            Perform an intermediate amount of swapping/smoothing.  Some improvement is always done. - 4

            Perform the greatest amount of swapping/smoothing.  Meshing takes longer with this level of improvement, but usually results in a better mesh. - 5

            Perform the greatest amount of swapping/smoothing, plus additional improvement techniques.  This level of improvement usually produces results that are similar to those at level 4, except for very poor meshes. - 6

            For linear tetrahedral meshes, this value provides the same level of improvement as MOPT,TIMP,5.  For quadratic tetrahedral meshes, this value provides an additional pass of cleanup.  This value is supported for both the main (MOPT,VMESH,MAIN) and alternate (MOPT,VMESH,ALTERNATE) tetrahedra meshers. - STAT

            Display status of MOPT settings. Value is ignored. - DEFA

        value
            Value, as described for each different Lab above.

        Notes
        -----
        See the Modeling and Meshing Guide for more information on the MOPT
        command and its options.

        This command is also valid for rezoning.

        """
        command = "MOPT,%s,%s" % (str(lab), str(value))
        self.RunCommand(command, **kwargs)

    def Cmgrp(self, aname="", cnam1="", cnam2="", cnam3="", cnam4="", cnam5="",
              cnam6="", cnam7="", cnam8="", **kwargs):
        """
        APDL Command: CMGRP

        Groups components and assemblies into an assembly.

        Parameters
        ----------
        aname
            An alphanumeric name used to identify this assembly.  Aname may be
            up to 32 characters, beginning with a letter and containing only
            letters, numbers, and underscores.  Overwrites a previously defined
            Aname (and removes it from higher level assemblies, if any).

        cnam1, cnam2, cnam3, . . . , cnam8
            Names of existing components or other assemblies to be included in
            this assembly.

        Notes
        -----
        Groups components and other assemblies into an assembly identified by a
        name.  CMGRP is used for the initial definition of an assembly.  An
        assembly is used in the same manner as a component.  Up to 5 levels of
        assemblies within assemblies may be used.

        An assembly is a convenient grouping of previously defined components
        and other assemblies.  Assemblies may contain components only, other
        assemblies, or any combination.  A component may belong to any number
        of assemblies.  Up to 5 levels of nested assemblies may be defined.
        Components and assemblies may be added to or deleted from an existing
        assembly by the CMEDIT command.  Once defined, an assembly may be
        listed, deleted, selected, or unselected using the same commands as for
        a component.  Assemblies are automatically updated to reflect deletions
        of one or more of their components or lower-level assemblies.
        Assemblies are not automatically deleted when all their components or
        subassemblies are deleted.

        This command is valid in any processor.

        """
        command = "CMGRP,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(aname), str(cnam1), str(cnam2), str(cnam3), str(cnam4), str(cnam5), str(cnam6), str(cnam7), str(cnam8))
        self.RunCommand(command, **kwargs)

    def Rcon(self, **kwargs):
        """
        APDL Command: RCON

        Specifies "Real constants" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "RCON," % ()
        self.RunCommand(command, **kwargs)

    def Nrlsum(self, signif="", label="", labelcsm="", forcetype="", **kwargs):
        """
        APDL Command: NRLSUM

        Specifies the Naval Research Laboratory (NRL) sum mode combination
        method.

        Parameters
        ----------
        signif
            Combine only those modes whose significance level exceeds the
            SIGNIF threshold. For single point, multipoint, or DDAM response
            (SPOPT,SPRS, MPRS or DDAM), the significance level of a mode is
            defined as the mode coefficient of the mode, divided by the maximum
            mode coefficient of all modes. Any mode whose significance level is
            less than SIGNIF is considered insignificant and is not contributed
            to the mode combinations. The higher the SIGNIF threshold, the
            fewer the number of modes combined. SIGNIF defaults to 0.001. If
            SIGNIF is specified as 0.0, it is taken as 0.0. (This mode
            combination method is not valid for SPOPT,PSD.)

        label
            Label identifying the combined mode solution output.

            DISP - Displacement solution (default). Displacements, stresses, forces, etc., are
                   available.

            VELO - Velocity solution. Velocities, "stress velocities," "force velocities," etc.,
                   are available.

            ACEL - Acceleration solution. Accelerations, "stress accelerations," "force
                   accelerations," etc., are available.

        labelcsm
            Label identifying the CSM (Closely Spaced Modes) method.

            CSM - Use the CSM method.

            Blank  - Do not use the CSM method (default).

        forcetype
            Label identifying the forces to be combined:

            STATIC - Combine the modal static forces (default).

            TOTAL - Combine the modal static plus inertial forces.

        Notes
        -----
        This command is also valid in PREP7. This mode combination method is
        usually used for SPOPT,DDAM.

        This CSM method is only applicable in a DDAM analysis (SPOPT,DDAM). The
        CSM method combines two closely spaced modes into one mode when their
        frequencies are within 10 percent of the common mean frequency and
        their responses are opposite in sign. The contribution of these closely
        spaced modes is then included in the NRL sum as a single effective
        mode. Refer to Closely Spaced Modes (CSM) Method in the Mechanical APDL
        Theory Reference for more information.

        NRLSUM is not allowed in ANSYS Professional.

        """
        command = "NRLSUM,%s,%s,%s,%s" % (str(signif), str(label), str(labelcsm), str(forcetype))
        self.RunCommand(command, **kwargs)

    def Eddump(self, num="", dt="", **kwargs):
        """
        APDL Command: EDDUMP

        Specifies output frequency for the explicit dynamic restart file
        (d3dump).

        Parameters
        ----------
        num
            Number of d3dump (restart) files written during the analysis
            (defaults to 1). When you specify NUM, the time interval between
            restart files is TIME / NUM, where TIME is the analysis end-time
            specified on the TIME command.

        dt
            Time interval at which the d3dump (restart) files are written. If
            NUM is input, DT is ignored.

        Notes
        -----
        You can use NUM or DT to specify the time interval at which d3dump
        restart files will be written. You should not specify both quantities;
        if both are input, NUM will be used. The restart files are written
        sequentially as d3dump01, d3dump02, etc.

        In LS-DYNA, the restart file output is specified in terms of number of
        time steps. Because the total number of time steps is not known until
        the LS-DYNA solution finishes, Mechanical APDL calculates an
        approximate number of time steps for the solution, and then uses NUM or
        DT to calculate the required LS-DYNA input. This approximated number of
        time steps may be different from the total number reached in LS-DYNA
        after the solution finishes. Therefore, the number of restart dump
        files or the output interval may differ slightly from what you
        requested using NUM or DT.

        In an explicit dynamic small restart (EDSTART,2) or full restart
        analysis (EDSTART,3), the EDDUMP setting will default to the NUM or DT
        value used in the original analysis. You can issue EDDUMP in the
        restart to change this setting.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDDUMP,%s,%s" % (str(num), str(dt))
        self.RunCommand(command, **kwargs)

    def Harfrq(self, freqb="", freqe="", logopt="", freqarr="", toler="",
               **kwargs):
        """
        APDL Command: HARFRQ

        Defines the frequency range in a harmonic analysis.

        Parameters
        ----------
        freqb
            Frequency (Hz) at the beginning of the FREQB to FREQE range (if
            FREQE > FREQB).  If FREQE is blank, the solution is done only at
            frequency FREQB (the central frequency of octave bands, when LogOpt
            = OB1, OB2, OB3, OB6, OB12 or OB24).

        freqe
            Frequency at end of this range.  Solutions are done at an interval
            of (FREQE-FREQB) / NSBSTP, ending at FREQE.  No solution is done at
            the beginning of the frequency range.  NSBSTP is input via the
            NSUBST command.  See the EXPSOL command documentation for expansion
            pass solutions.

        --
            Reserved.

        logopt
            Logarithm frequency span. Solutions are done at an interval of
            (log(FREQE) - log(FREQB)) / (NSBSTP-1), (NSBSTP>1). The central
            frequency or beginning frequency is used for NSBSTP = 1. Valid
            values are:

            OB1 - Octave band.

            OB2 - 1/2 octave band.

            OB3 - 1/3 octave band.

            OB6 - 1/6 octave band.

            OB12 - 1/12 octave band.

            OB24 - 1/24 octave band.

            LOG - General logarithm frequency span.

        freqarr
            An array containing frequency values (Hz). Combined with the
            tolerance argument, Toler, these values are merged with values
            calculated based on the specifications from FREQB, FREQE, and
            LogOpt, as well NSBSTP on the NSUBST command and Clust on the HROUT
            command. Enclose the array name in percent (%) signs (for example,
            HARFRQ,,,,,%arrname%). Use *DIM to define the array.

        toler
            Tolerance to determine if a user input frequency value in FREQARR
            is a duplicate and can be ignored. Two frequency values are
            considered duplicates if their difference is less than the
            frequency range multiplied by the tolerance. The default value is 1
            x 10-5.

        Notes
        -----
        Defines the frequency range for loads in the harmonic analysis
        (ANTYPE,HARMIC).

        Do not use this command for a harmonic ocean wave analysis (HROCEAN).

        When frequencies are user-defined, the array FREQARR must be one-
        dimensional and contain positive values. User-defined frequency input
        is not supported in the following cases:

        in a cyclic symmetry harmonic analysis

        when the Variational Technology method is used (Method = VT on the
        HROPT command)

        This command is also valid in PREP7.

        """
        command = "HARFRQ,%s,%s,%s,%s,%s" % (str(freqb), str(freqe), str(logopt), str(freqarr), str(toler))
        self.RunCommand(command, **kwargs)

    def Seexp(self, sename="", usefil="", imagky="", expopt="", **kwargs):
        """
        APDL Command: SEEXP

        Specifies options for the substructure expansion pass.

        Parameters
        ----------
        sename
            The name (case-sensitive) of the superelement matrix file created
            by the substructure generation pass (Sename.SUB).  Defaults to the
            initial jobname File.  If a number, it is the element number of the
            superelement as used in the use pass.

        usefil
            The name of the file containing the superelement degree-of-freedom
            (DOF) solution created by the substructure use pass (Usefil.DSUB).

        imagky
            Key to specify use of the imaginary component of the DOF solution.
            Applicable only if the use pass is a harmonic (ANTYPE,HARMIC)
            analysis:

            OFF - Use real component of DOF solution (default).

            ON - Use imaginary component of DOF solution.

        expopt
            Key to specify whether the superelement (ANTYPE,SUBSTR) expansion
            pass (EXPASS,ON) should transform the geometry:

            OFF - Do not transform node or element locations (default).

            ON - Transform node or element locations in the FE geometry record of the .rst
                 results file.

        Notes
        -----
        Specifies options for the expansion pass of the substructure analysis
        (ANTYPE,SUBSTR).  If used in SOLUTION, this command is valid only
        within the first load step.

        If you specify geometry transformation (Expopt = ON), you must retrieve
        the transformation matrix (if it exists) from the specified .SUB file.
        The command updates the nodal X, Y, and Z coordinates to represent the
        transformed node locations. The Expopt option is useful when you want
        to expand superelements created from other superelements (via SETRAN or
        SESYMM commands). For more information, see Superelement Expansion in
        Transformed Locations and Plotting or Printing Mode Shapes.

        This command is also valid in /PREP7.

        """
        command = "SEEXP,%s,%s,%s,%s" % (str(sename), str(usefil), str(imagky), str(expopt))
        self.RunCommand(command, **kwargs)

    def Emsym(self, nsect="", **kwargs):
        """
        APDL Command: EMSYM

        Specifies circular symmetry for electromagnetic sources.

        Parameters
        ----------
        nsect
            The number of circular symmetry sections (defaults to 1).

        Notes
        -----
        Specifies the number of times to repeat electromagnetic sources for
        circular symmetry. Applies to SOURC36 elements and to coupled-field
        elements with electric current conduction results in the database.
        Sources are assumed to be equally spaced over 360° about the global
        Cartesian Z axis.

        This command is also valid in SOLUTION.

        """
        command = "EMSYM,%s" % (str(nsect))
        self.RunCommand(command, **kwargs)

    def Cval(self, wn="", v1="", v2="", v3="", v4="", v5="", v6="", v7="",
             v8="", **kwargs):
        """
        APDL Command: /CVAL

        Specifies nonuniform contour values on stress displays.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        v1, v2, v3, . . . , v8
            Up to 8 contour values may be specified (in ascending order).  The
            0.0 value (if any) must not be the last value specified.  If no
            values are specified, all contour specifications are erased and
            contours are automatically calculated.

        Notes
        -----
        This command is similar to the /CONTOUR command. With /CVAL, however,
        you define the upper level of each contour band instead of having the
        contours uniformly distributed over the range. The minimum value
        (including a zero value for the first band) for a contour band cannot
        be specified. If you use both /CONTOUR and /CVAL, the last command
        issued takes precedence.

        This command is valid in any processor.

        """
        command = "/CVAL,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(wn), str(v1), str(v2), str(v3), str(v4), str(v5), str(v6), str(v7), str(v8))
        self.RunCommand(command, **kwargs)

    def Midtol(self, key="", tolerb="", resfq="", **kwargs):
        """
        APDL Command: MIDTOL

        Sets midstep residual criterion values for structural transient
        analyses.

        Parameters
        ----------
        key
            Midstep residual criterion activation key.

            ON or 1 - Activate midstep residual criterion in a structural transient analysis
                      (default).

            OFF or 0 - Deactivate midstep residual criterion in a structural transient analysis.

            STAT  - List the current midstep residual criterion setting.

        tolerb
            Midstep residual tolerance or reference value for bisection.
            Defaults to 100 times the TOLER setting of the CNVTOL command.

        resfq
            Key to use response frequency computation along with midstep
            residual criterion for automatic time stepping (AUTOTS,ON).

            OFF or 0 - Do not calculate response frequency and do not consider it in the automatic
                       time stepping (default).

            ON or 1 - Calculate response frequency and consider it in the automatic time stepping.

        Notes
        -----
        When TOLERB is input as a tolerance value (TOLERB > 0), the typical
        force and/or moment from the regular time step is used in the midstep
        residual force and/or moment comparison.

        In a structural transient analysis, the suggested tolerance range of
        TOLERB (TOLERB > 0) is as follows:

        If the structural transient analysis is elastic and linear, and the
        load is constant or changes slowly, use a smaller value of TOLERB to
        achieve an accurate solution. If the analysis involves large amounts of
        energy dissipation, such as elastic-plastic material, TOLERB can be
        larger. If the analysis includes contact or rapidly varying loads, a
        smaller value of TOLERB should be used if high frequency response is
        important; otherwise, a larger value of TOLERB may be used to enable
        faster convergence with larger time step sizes.

        For more information on how the midstep criterion is used by the
        program, see Midstep Residual for Structural Dynamic Analysis in the
        Mechanical APDL Theory Reference.

        This command is also valid in PREP7.

        """
        command = "MIDTOL,%s,%s,%s" % (str(key), str(tolerb), str(resfq))
        self.RunCommand(command, **kwargs)

    def Exp(self, ir="", ia="", name="", facta="", factb="", **kwargs):
        """
        APDL Command: EXP

        Forms the exponential of a variable.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]). If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.

        ia
            Reference number of the variable to be operated on.

        --, --
            Unused fields.

        name
            Thirty-two character name for identifying the variable on the
            printout and displays. Embedded blanks are compressed upon output.

        --, --
            Unused fields.

        facta
            Scaling factor applied to variable IA (defaults to 1.0).

        factb
            Scaling factor (positive or negative) applied to the operation
            (defaults to 1.0).

        Notes
        -----
        Forms the exponential of a variable according to the operation:

        IR = FACTB*EXP(FACTA x IA)

        """
        command = "EXP,%s,%s,%s,%s,%s" % (str(ir), str(ia), str(name), str(facta), str(factb))
        self.RunCommand(command, **kwargs)

    def Dsym(self, lab="", normal="", kcn="", **kwargs):
        """
        APDL Command: DSYM

        Specifies symmetry or antisymmetry degree-of-freedom constraints on
        nodes.

        Parameters
        ----------
        lab
            Symmetry label:

            SYMM - Generate symmetry constraints as described below (default).

            ASYM - Generate antisymmetry constraints as described below.

        normal
            Surface orientation label to determine the constraint set (surface
            is assumed to be perpendicular to this coordinate direction in
            coordinate system KCN):

            X - Surface is normal to coordinate X direction (default).  Interpreted as R
                direction for non-Cartesian coordinate systems.

            Y - Surface is normal to coordinate Y direction.   θ direction for non-Cartesian
                coordinate systems.

            Z - Surface is normal to coordinate Z direction.   Φ direction for spherical or
                toroidal coordinate systems.

        kcn
            Reference number of global or local coordinate system used to
            define surface orientation.

        Notes
        -----
        Specifies symmetry or antisymmetry degree-of-freedom constraints on the
        selected nodes. The nodes are first automatically rotated (any
        previously defined rotations on these nodes are redefined) into
        coordinate system KCN, then zero-valued constraints are generated, as
        described below, on the selected degree-of-freedom set (limited to
        displacement, velocity, and magnetic degrees of freedom) [DOFSEL].
        Constraints are defined in the (rotated) nodal coordinate system, as
        usual. See the D and NROTAT commands for additional details about
        constraints and nodal rotations.

        This command is also valid in PREP7.

        Symmetry or antisymmetry constraint generations are based upon the
        valid degrees of freedom in the model, i.e., the degrees of freedom
        associated with the elements attached to the nodes.  The labels for
        degrees of freedom used in the generation depend on the Normal label.

        For displacement degrees of freedom, the constraints generated are:

        For velocity degrees of freedom, the constraints generated are:

        For magnetic degrees of freedom, the SYMM label generates flux normal
        conditions (flux flows normal to the surface).  Where no constraints
        are generated, the flux normal condition is "naturally" satisfied.  The
        ASYM label generates flux parallel conditions (flux flows parallel to
        the surface).

        """
        command = "DSYM,%s,%s,%s" % (str(lab), str(normal), str(kcn))
        self.RunCommand(command, **kwargs)

    def Ngen(self, itime="", inc="", node1="", node2="", ninc="", dx="", dy="",
             dz="", space="", **kwargs):
        """
        APDL Command: NGEN

        Generates additional nodes from a pattern of nodes.

        Parameters
        ----------
        itime, inc
            Do this generation operation a total of ITIME times, incrementing
            all nodes in the given pattern by INC each time after the first.
            ITIME must be > 1 for generation to occur.

        node1, node2, ninc
            Generate nodes from the pattern of nodes beginning with NODE1 to
            NODE2 (defaults to NODE1) in steps of NINC (defaults to 1).  If
            NODE1 = ALL, NODE2 and NINC are ignored and the pattern is all
            selected nodes [NSEL].  If NODE1 = P, graphical picking is enabled
            and all remaining command fields are ignored (valid only in the
            GUI).  A component name may also be substituted for NODE1 (NODE2
            and NINC are ignored).

        dx, dy, dz
            Node location increments in the active coordinate system (DR, Dθ,
            DZ for cylindrical, DR, Dθ, DΦ for spherical or toroidal).

        space
            Spacing ratio.  Ratio of last division size to first division size.
            If > 1.0, divisions increase.  If < 1.0, divisions decrease.  Ratio
            defaults to 1.0 (uniform spacing).

        Notes
        -----
        Generates additional nodes from a given node pattern.  Generation is
        done in the active coordinate system.  Nodes in the pattern may have
        been generated in any coordinate system.

        This command is also valid in the /MAP processor.

        """
        command = "NGEN,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(itime), str(inc), str(node1), str(node2), str(ninc), str(dx), str(dy), str(dz), str(space))
        self.RunCommand(command, **kwargs)

    def Fssparm(self, port1="", port2="", **kwargs):
        """
        APDL Command: FSSPARM

        Calculates reflection and transmission properties of a frequency
        selective surface.

        Parameters
        ----------
        port1
            Port number of input port. Defaults to 1.

        port2
            Port number of output port. Defaults to 1.

        Notes
        -----
        FSSPARM calculates reflection and transmission coefficients, power
        reflection and transmission coefficients, and return and insertion
        losses of a frequency selective surface.

        """
        command = "FSSPARM,%s,%s" % (str(port1), str(port2))
        self.RunCommand(command, **kwargs)

    def Thexpand(self, key="", **kwargs):
        """
        APDL Command: THEXPAND

        Enables or disables thermal loading

        Parameters
        ----------
        key
            Activation key:

            ON  - Thermal loading is included in the load vector (default).

            OFF - Thermal loading is not included in the load vector.

        Notes
        -----
        Temperatures applied in the analysis are used by default to evaluate
        material properties and contribute to the load vector if the
        temperature does not equal the reference temperature and a coefficient
        of thermal expansion is specified.

        Use THEXPAND,OFF to evaluate the material properties but not contribute
        to the load vector. This capability is particularly useful when
        performing a harmonic analysis where you do not want to include
        harmonically varying thermal loads. It is also useful in a modal
        analysis when computing a modal load vector but excluding the thermal
        load.

        This command is valid for all analysis types except linear perturbation
        modal and linear perturbation harmonic analyses. For these two linear
        perturbation analysis types, the program internally sets THEXPAND,OFF,
        and it cannot be set to ON by using this command (THEXPAND,ON is
        ignored).

        """
        command = "THEXPAND,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Bstq(self, val1="", val2="", t="", **kwargs):
        """
        APDL Command: BSTQ

        Specifies the cross section twist and torque relationship for beam
        sections.

        Parameters
        ----------
        val1
            Twist component (χ).

        val2
            Torque component (τ).

        t
            Temperature.

        Notes
        -----
        The behavior of beam elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        The BSTQ command, one of several nonlinear general beam section
        commands, specifies the cross section twist and torque relationship for
        a beam section. The section data defined is associated with the section
        most recently defined (via the SECTYPE command).

        Unspecified values default to zero.

        Related commands are BSAX, BSM1, BSM2, BSS1, BSS2, BSMD, and BSTE.

        For complete information, see Using Nonlinear General Beam Sections.

        """
        command = "BSTQ,%s,%s,%s" % (str(val1), str(val2), str(t))
        self.RunCommand(command, **kwargs)

    def Vfopt(self, opt="", filename="", ext="", dir="", filetype="",
              fileformat="", **kwargs):
        """
        APDL Command: VFOPT

        Specifies options for the view factor file and calculates view factors.

        Parameters
        ----------
        opt
            View factor option:

            NEW - Calculate view factors and write them to a file.

            OFF - Do not recalculate view factors it they already exist in the database,
                  otherwise calculate compute them. This option is the default
                  behavior.

            READ - Read view factors from a binary file. For subsequent SOLVE commands, switch to
                   the default option (OFF).

            NONE - Do not write view factors to a file.

        fname
            File name for view factor matrix. Default = Jobname.

        ext
            Filename extension for view factor matrix. Default = .vf.

        dir
            Directory path for view factor matrix. If you do not specify a
            directory path, it will default to your working directory.

        filetype
            View factor file type:

            BINA - Binary (default).

            ASCI - ASCII.

        fileformat
            Format for the specified Filetype:

            Binary files (Filetype = BINA): - 0

            No compression. (View factor file size may be very large.) - 1

        Notes
        -----
        The VFOPT command allows you to deactivate the view factor computation
        (Opt = OFF) if the view factors already exist in the database. The
        default behavior is OFF upon encountering the second and subsequent
        SOLVE commands in the solution processor.

        When Opt = READ, only a previously calculated view factor binary file
        is valid.

        For 3-D analyses, two options are available for calculating view
        factors when running Distributed ANSYS:

        Issue a SOLVE command -- View factors are calculated in parallel mode
        if no view factors were previously calculated.

        Issue a VFOPT,NEW command -- View factors are calculated in serial
        mode.

        For 2-D analyses, view factors are calculated in serial mode.

        """
        command = "VFOPT,%s,%s,%s,%s,%s,%s" % (str(opt), str(filename), str(ext), str(dir), str(filetype), str(fileformat))
        self.RunCommand(command, **kwargs)

    def Mfvolume(self, inumb="", fnumb1="", label="", fnumb2="", **kwargs):
        """
        APDL Command: MFVOLUME

        Defines a volume load transfer for an ANSYS Multi-field solver
        analysis.

        Parameters
        ----------
        inumb
            Interface number for load transfer. The interface number
            corresponds to the interface number specified by the volume flag
            FVIN (BFE command).

        fnumb1
            Field number of sending field.

        label
            Valid volume load labels:

        fnumb2
            Field number for receiving field.

        Notes
        -----
        This command is also valid in PREP7.

        The ANSYS Multi-field solver solver does not allow you to switch the
        load transfer direction for the same load quantity across the same
        interfaces for a restart run. For example, if Field1 sends temperature
        to and receives heat flow from Field2 across Interface 1 in a previous
        solution, then you cannot make Field1 send heat flow to and receive
        temperatures from Field2 across the same interface in a restart run,
        even if you cleared the corresponding load transfer command.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFVOLUME,%s,%s,%s,%s" % (str(inumb), str(fnumb1), str(label), str(fnumb2))
        self.RunCommand(command, **kwargs)

    def Wpcsys(self, wn="", kcn="", **kwargs):
        """
        APDL Command: WPCSYS

        Defines the working plane location based on a coordinate system.

        Parameters
        ----------
        wn
            Window number whose viewing direction will be modified to be normal
            to the working plane (defaults to 1).  If WN is a negative value,
            the viewing direction will not be modified.

        kcn
            Coordinate system number.  KCN may be 0,1,2 or any previously
            defined local coordinate system number (defaults to the active
            system).

        Notes
        -----
        Defines a working plane location and orientation based on an existing
        coordinate system.  If a Cartesian system is used as the basis (KCN)
        for the working plane, the working plane will also be Cartesian, in the
        X-Y plane of the base system.  If a cylindrical, spherical, or toroidal
        base system is used, the working plane will be a polar system in the
        R-θ plane of the base system.

        If working plane tracking has been activated (CSYS,WP or CSYS,4), the
        updated active coordinate system will be of a similar type, except that
        a toroidal system will be updated to a cylindrical system.  See the
        Modeling and Meshing Guide for more information on working plane
        tracking.

        This command is valid in any processor.

        Some primitive generation commands will not honor R-theta
        transformations for non-cartesian coordinate systems. Refer to the
        primitive commands table for more information.

        """
        command = "WPCSYS,%s,%s" % (str(wn), str(kcn))
        self.RunCommand(command, **kwargs)

    def Cmplot(self, label="", entity="", keyword="", **kwargs):
        """
        APDL Command: CMPLOT

        Plots the entities contained in a component or assembly.

        Parameters
        ----------
        label
            Name of the component or assembly to be plotted.

            (blank) - All selected components and assemblies are plotted (default).  If fewer than 11
                      components are selected, then all  are plotted.  If more
                      than 11 components are selected, then only the first 11
                      are plotted.

            ALL - All selected components are plotted. If number of selected components is
                  greater than 11, then the legend showing component names will
                  not be shown.

            N - Next set of defined components and assemblies is plotted.

            P - Previous set of defined components and assemblies is plotted.

            Cname - The specified component or assembly is plotted.

            SetNo. - The specified set number is plotted.

        entity
            If Label is BLANK or ALL, then the following entity types can be
            specified:

            VOLU - Plot the volume components only.

            AREA - Plot the area components only.

            LINE - Plot the line components only.

            KP - Plot the keypoint components only.

            ELEM - Plot the element components only.

            NODE - Plot the node components only.

        keyword
            For Keyword = ALL, plot the specified component name in the Label
            field in the context of all entities of the same type. Not valid if
            Label field is BLANK or ALL.

        Notes
        -----
        Components are plotted with their native entities.  For assemblies, all
        native entities for the underlying component types are plotted
        simultaneously.  Although more components can be plotted, the legend
        displays only 11 at a time. When more than eleven are plotted, the
        legend is not displayed.

        Possible usage:

        This command is valid in any processor.

        """
        command = "CMPLOT,%s,%s,%s" % (str(label), str(entity), str(keyword))
        self.RunCommand(command, **kwargs)

    def Rectng(self, x1="", x2="", y1="", y2="", **kwargs):
        """
        APDL Command: RECTNG

        Creates a rectangular area anywhere on the working plane.

        Parameters
        ----------
        x1, x2
            Working plane X coordinates of the rectangle.

        y1, y2
            Working plane Y coordinates of the rectangle.

        Notes
        -----
        The area will be defined with four keypoints and four lines.  See the
        BLC4 and BLC5 commands for alternate ways to create rectangles.

        """
        command = "RECTNG,%s,%s,%s,%s" % (str(x1), str(x2), str(y1), str(y2))
        self.RunCommand(command, **kwargs)

    def Vsba(self, nv="", na="", sepo="", keepv="", keepa="", **kwargs):
        """
        APDL Command: VSBA

        Subtracts areas from volumes.

        Parameters
        ----------
        nv
            Volume (or volumes, if picking is used) to be subtracted from.  If
            ALL, use all selected volumes.  If P, graphical picking is enabled
            (valid only in the GUI) and remaining fields are ignored.  A
            component name may also be substituted for NV.

        na
            Area (or areas, if picking is used) to subtract.  If ALL, use all
            selected areas.  A component name may also be substituted for NA.

        sepo
            Behavior of the touching boundary:

            (blank) - The resulting volumes will share area(s) where they touch.

            SEPO - The resulting volumes will have separate, but coincident area(s) where they
                   touch.

        keepv
            Specifies whether NV volumes are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete NV volumes after VSBA operation (override BOPTN command settings).

            KEEP - Keep NV volumes after VSBA operation (override BOPTN command settings).

        keepa
            Specifies whether NA areas are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete NA areas after VSBA operation (override BOPTN command settings).

            KEEP - Keep NA areas after VSBA operation (override BOPTN command settings).

        Notes
        -----
        Generates new volumes by subtracting the regions common to both the
        volumes and areas (the intersection) from the NV volumes.  The
        intersection will be an area(s).  If SEPO is blank, the volume is
        divided at the area and the resulting volumes will be connected,
        sharing a common area where they touch.  If SEPO is set to SEPO, the
        volume is divided into two unconnected volumes with separate areas
        where they touch.  See the Modeling and Meshing Guide for an
        illustration.  See the BOPTN command for an explanation of the options
        available to Boolean operations.  Element attributes and solid model
        boundary conditions assigned to the original entities will not be
        transferred to the new entities generated.

        """
        command = "VSBA,%s,%s,%s,%s,%s" % (str(nv), str(na), str(sepo), str(keepv), str(keepa))
        self.RunCommand(command, **kwargs)

    def Esys(self, kcn="", **kwargs):
        """
        APDL Command: ESYS

        Sets the element coordinate system attribute pointer.

        Parameters
        ----------
        kcn
            Coordinate system number:

            0 - Use element coordinate system orientation as defined (either by default or by
                KEYOPT setting) for the element (default).

            N - Use element coordinate system orientation based on local coordinate system N
                (where N must be greater than 10). For global system 0, 1, or
                2, define a local system N parallel to appropriate system with
                the LOCAL or CS command (for example: LOCAL,11,1).

        Notes
        -----
        Identifies the local coordinate system to be used to define the element
        coordinate system of subsequently defined elements. Used only with area
        and volume elements. For non-layered volume elements, the local
        coordinate system N is simply assigned to be the element coordinate
        system. For shell and layered volume elements, the x and y axes of the
        local coordinate system N are projected onto the shell or layer plane
        to determine the element coordinate system. See Understanding the
        Element Coordinate System for more details. N refers to the coordinate
        system reference number (KCN) defined using the LOCAL (or similar)
        command. Element coordinate system numbers may be displayed [/PNUM].

        """
        command = "ESYS,%s" % (str(kcn))
        self.RunCommand(command, **kwargs)

    def Rmporder(self, ord1="", ord2="", ord3="", ord4="", ord5="", ord6="",
                 ord7="", ord8="", ord9="", **kwargs):
        """
        APDL Command: RMPORDER

        Defines polynomial orders for ROM functions.

        Parameters
        ----------
        ord1, ord2, ord3, . . . , ord9
            Polynomial orders for modes. Ordi specifies the polynomial order
            for modei. Modes are ordered as extracted from a modal analysis
            using the RMNEVEC command.  Defaults to 0 if mode i is unused;
            default to nstep(i) -1 for dominant or relevant modes, where
            nstep(i) is the number of equidistant steps in fit range of mode i.
            nstep(i) is automatically set by RMMSELECT or modified by the
            RMMRANGE command.

        Notes
        -----
        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "RMPORDER,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(ord1), str(ord2), str(ord3), str(ord4), str(ord5), str(ord6), str(ord7), str(ord8), str(ord9))
        self.RunCommand(command, **kwargs)

    def Contour(self, wn="", ncont="", vmin="", vinc="", vmax="", **kwargs):
        """
        APDL Command: /CONTOUR

        Specifies the uniform contour values on stress displays.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        ncont
            Number of contour values. NCONT defaults to 9 for X11 or WIN32 and
            to 128 for X11c or WIN32C.  The default graphics window display for
            3-D devices is a smooth continuous shading effect that spans the
            maximum of 128 contours available. Use the /DV3D command to create
            defined banding for your contour values (values of 9 and 128 are
            displayed in smooth shading only). The legend, however, will
            display only nine color boxes, which span the full range of colors
            displayed in the graphics window.

        vmin
            Minimum contour value.  If VMIN = AUTO, automatically calculate
            contour values based upon NCONT uniformly spaced values over the
            min-max extreme range.  Or, if VMIN = USER, set contour values to
            those of the last display (useful when last display automatically
            calculated contours).

        vinc
            Value increment (positive) between contour values.  Defaults to
            (VMAX-VMIN)/NCONT.

        vmax
            Maximum contour value.  Ignored if both VMIN and VINC are
            specified.

        Notes
        -----
        See the /CVAL command for alternate specifications.  Values represent
        contour lines in vector mode, and the algebraic maximum of contour
        bands in raster mode.

        Note:: : No matter how many contours (NCONT) are specified by /CONTOUR,
        the actual number of contours that appear on your display depends also
        on the device name, whether the display is directed to the screen or to
        a file, the display mode (vector or raster), and the number of color
        planes.  (All these items are controlled by /SHOW settings.) In any
        case, regardless of whether they are smoothed or banded, only 128
        contours can be displayed. See Creating Geometric Results Displays in
        the Basic Analysis Guide for more information on changing the number of
        contours.

        If the current ANSYS graphics are not displayed as Multi-Plots, then
        the following is true:  If the current device is a 3-D device
        [/SHOW,3D], the model contours in all active windows will be the same,
        even if separate /CONTOUR commands are issued for each active window.
        For efficiency, ANSYS 3-D graphics logic maintains a single data
        structure (segment), which contains precisely one set of contours.  The
        program displays the same segment in all windows.  The view settings of
        each window constitute the only differences in the contour plots in the
        active windows.

        This command is valid in any processor.

        """
        command = "/CONTOUR,%s,%s,%s,%s,%s" % (str(wn), str(ncont), str(vmin), str(vinc), str(vmax))
        self.RunCommand(command, **kwargs)

    def Rmmlist(self, **kwargs):
        """
        APDL Command: RMMLIST

        Lists all mode specifications for the ROM method.

        Notes
        -----
        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "RMMLIST," % ()
        self.RunCommand(command, **kwargs)

    def Hfsym(self, kcn="", xkey="", ykey="", zkey="", **kwargs):
        """
        APDL Command: HFSYM

        Indicates the presence of symmetry planes for the computation of
        acoustic fields in the near and far field domains (beyond the finite
        element region).

        Parameters
        ----------
        kcn
            Coordinate system reference number. KCN may be 0 (Cartesian), or
            any previously defined local Cartesian coordinate system number
            (>10). Defaults to 0.

        xkey
            Key for acoustic field boundary condition, as prescribed for the
            solution, corresponding to the x = constant plane:

            None - No sound soft or sound hard boundary conditions (default).

            SSB - Sound soft boundary (pressure = 0).

            SHB - Sound hard boundary (normal velocity = 0).

        ykey
            Key for acoustic field boundary condition, as prescribed for the
            solution, corresponding to the y = constant plane:

            None - No sound soft or sound hard boundary conditions (default).

            SSB - Sound soft boundary (pressure = 0).

            SHB - Sound hard boundary (normal velocity = 0).

        zkey
            Key for acoustic field boundary condition, as prescribed for the
            solution, corresponding to the z = constant plane:

            None - No sound soft or sound hard boundary conditions (default).

            SSB - Sound soft boundary (pressure = 0).

            SHB - Sound hard boundary (normal velocity = 0).

        Notes
        -----
        HFSYM uses the image principle to indicate symmetry planes (x, y, or z
        = constant plane) for acoustic field computations outside the modeled
        domain. A sound hard boundary condition must be indicated even though
        it occurs as a natural boundary condition.

        No menu paths are available for acoustic applications.

        """
        command = "HFSYM,%s,%s,%s,%s" % (str(kcn), str(xkey), str(ykey), str(zkey))
        self.RunCommand(command, **kwargs)

    def Sexp(self, labr="", lab1="", lab2="", exp1="", exp2="", **kwargs):
        """
        APDL Command: SEXP

        Forms an element table item by exponentiating and multiplying.

        Parameters
        ----------
        labr
            Label assigned to results.  If same as existing label, the existing
            values will be overwritten by these results.

        lab1
            First labeled result item in operation.

        lab2
            Second labeled result item in operation (may be blank).

        exp1
            Exponent applied to Lab1.

        exp2
            Exponent applied to Lab2.

        Notes
        -----
        Forms a labeled result item (see ETABLE command) for the selected
        elements by exponentiating and multiplying two existing labeled result
        items according to the operation:

        LabR = (|Lab1|EXP1) x (|Lab2|EXP2)

        Roots, reciprocals, and divides may also be done with this command.

        """
        command = "SEXP,%s,%s,%s,%s,%s" % (str(labr), str(lab1), str(lab2), str(exp1), str(exp2))
        self.RunCommand(command, **kwargs)

    def Nstore(self, tinc="", **kwargs):
        """
        APDL Command: NSTORE

        Defines which time points are to be stored.

        Parameters
        ----------
        tinc
            Store data associated with every TINC time (or frequency) point(s),
            within the previously defined range of TMIN to TMAX [TIMERANGE].
            (Defaults to 1)

        Notes
        -----
        Defines which time (or frequency) points within the range are to be
        stored.

        """
        command = "NSTORE,%s" % (str(tinc))
        self.RunCommand(command, **kwargs)

    def Swdel(self, ecomp="", **kwargs):
        """
        APDL Command: SWDEL

        Deletes spot weld sets.

        Parameters
        ----------
        ecomp
            Name of an existing spot weld set that was previously defined using
            SWGEN. If Ecomp = ALL (default) all spot welds are deleted.

        Notes
        -----
        This command deletes spot weld sets previously defined by the SWGEN
        command.

        """
        command = "SWDEL,%s" % (str(ecomp))
        self.RunCommand(command, **kwargs)

    def Vsbv(self, nv1="", nv2="", sepo="", keep1="", keep2="", **kwargs):
        """
        APDL Command: VSBV

        Subtracts volumes from volumes.

        Parameters
        ----------
        nv1
            Volume (or volumes, if picking is used) to be subtracted from.  If
            ALL, use all selected volumes.  Volumes specified in set NV2 are
            removed from set NV1.  If P, graphical picking is enabled (valid
            only in the GUI) and remaining fields are ignored.  A component
            name may also be substituted for NV1.

        nv2
            Volume (or volumes, if picking is used) to subtract.  If ALL, use
            all selected volumes (except those included in the NV1 argument).
            A component name may also be substituted for NV2.

        sepo
            Behavior if the intersection of the NV1 volumes and the NV2 volumes
            is an area or areas:

            (blank) - The resulting volumes will share area(s) where they touch.

            SEPO - The resulting volumes will have separate, but coincident area(s) where they
                   touch.

        keep1
            Specifies whether NV1 volumes are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete NV1 volumes after VSBV operation (override BOPTN command settings).

            KEEP - Keep NV1 volumes after VSBV operation (override BOPTN command settings).

        keep2
            Specifies whether NV2 volumes are to be deleted:

            (blank) - Use the setting of KEEP on the BOPTN command.

            DELETE - Delete NV2 volumes after VSBV operation (override BOPTN command settings).

            KEEP - Keep NV2 volumes after VSBV operation (override BOPTN command settings).

        Notes
        -----
        Generates new volumes by subtracting the regions common to both NV1 and
        NV2 volumes (the intersection) from the NV1 volumes.  The intersection
        can be a volume(s) or area(s).  If the intersection is an area and SEPO
        is blank, the NV1 volume is divided at the area and the resulting
        volumes will be connected, sharing a common area where they touch.  If
        SEPO is set to SEPO, NV1 is divided into two unconnected volumes with
        separate areas where they touch.  See the Modeling and Meshing Guide
        for an illustration.  See the BOPTN command for an explanation of the
        options available to Boolean operations.  Element attributes and solid
        model boundary conditions assigned to the original entities will not be
        transferred to the new entities generated.  VSBV,ALL,ALL will have no
        effect because all the volumes in set NV1will have been moved to set
        NV2.

        """
        command = "VSBV,%s,%s,%s,%s,%s" % (str(nv1), str(nv2), str(sepo), str(keep1), str(keep2))
        self.RunCommand(command, **kwargs)

    def Dsum(self, signif="", label="", td="", forcetype="", **kwargs):
        """
        APDL Command: DSUM

        Specifies the double sum mode combination method.

        Parameters
        ----------
        signif
            Combine only those modes whose significance level exceeds the
            SIGNIF threshold.  For single point, multipoint, or DDAM response
            (SPOPT, SPRS, MPRS, or DDAM), the significance level of a mode is
            defined as the mode coefficient of the mode, divided by the maximum
            mode coefficient of all modes.  Any mode whose significance level
            is less than SIGNIF is considered insignificant and is not
            contributed to the mode combinations.  The higher the SIGNIF
            threshold, the fewer the number of modes combined. SIGNIF defaults
            to 0.001.  If SIGNIF is specified as 0.0, it is taken as 0.0.
            (This mode combination method is not valid for SPOPT, PSD.)

        label
            Label identifying the combined mode solution output.

            DISP - Displacement solution (default).  Displacements, stresses, forces, etc., are
                   available.

            VELO - Velocity solution.  Velocities, "stress velocities," "force velocities," etc.,
                   are available.

            ACEL - Acceleration solution.  Accelerations, "stress accelerations," "force
                   accelerations," etc., are available.

        td
            Time duration for earthquake or shock spectrum.  TD defaults to 10.

        forcetype
            Label identifying the forces to be combined:

            STATIC - Combine the modal static forces (default).

            TOTAL - Combine the modal static plus inertial forces.

        Notes
        -----
        This command is also valid for PREP7.

        """
        command = "DSUM,%s,%s,%s,%s" % (str(signif), str(label), str(td), str(forcetype))
        self.RunCommand(command, **kwargs)

    def Cbdof(self, fname1="", ext1="", fname2="", ext2="", kpos="", clab="",
              kshs="", tolout="", tolhgt="", tolthk="", **kwargs):
        """
        APDL Command: CBDOF

        Activates cut-boundary interpolation (for submodeling).

        Parameters
        ----------
        fname1
            File name and directory path (248 characters maximum, including
            directory) from which to read boundary node data. If no specified
            directory path exists, the path defaults to your working directory
            and you can use all 248 characters for the file name.

        ext1
            Filename extension (eight-character maximum).

        --
            Unused field.

        fname2
            File name and directory path (248 characters maximum, including
            directory) to which cut-boundary D commands are written. If no
            specified directory path exists, the path defaults to your working
            directory and you can use all 248 characters for the file name.

        ext2
            Filename extension (eight-character maximum).

        --
            Unused field.

        kpos
            Position on Fname2 to write block of D commands:

            0 - Beginning of file (overwrite existing file).

            1 - End of file (append to existing file).

        clab
            Label (eight characters maximum, including the colon) for this
            block of D commands on Fname2.  his label is appended to the colon
            (:).  Defaults to CBn, where n is the cumulative iteration number
            for the data set currently in the database.  For imaginary data
            (see KIMG on the *SET command), Clab defaults to CIn.

        kshs
            Shell-to-solid submodeling key:

            0 - Solid-to-solid or shell-to-shell submodel.

            1 - Shell-to-solid submodel.

        tolout
            Extrapolation tolerance about elements, based on a fraction of the
            element dimension. Submodel nodes outside the element by more than
            TOLOUT are not accepted as candidates for DOF extrapolation.
            Defaults to 0.5 (50 percent).

        tolhgt
            Height tolerance above or below shell elements, in units of length.
            Used only for shell-to-shell submodeling (KSHS = 0). Submodel nodes
            off the element surface by more than TOLHGT are not accepted as
            candidates for degree-of-freedom interpolation or extrapolation.
            Defaults to 0.0001 times the maximum element dimension.

        tolthk
            Height tolerance above or below shell elements, based on a fraction
            of the shell element thickness. Used only for shell-to-solid
            submodeling (KSHS = 1). Submodel nodes off the element surface by
            more than TOLTHK are not accepted as candidates for DOF
            interpolation or extrapolation. Defaults to 0.1 times the average
            shell thickness.

        Notes
        -----
        File Fname1 should contain a node list for which boundary conditions
        are to be interpolated (NWRITE).  File Fname2 is created to contain
        interpolated boundary conditions written as a block of D commands.

        Boundary conditions are written for the active degree-of-freedom set
        for the element from which interpolation is performed. Interpolation
        occurs on the selected set of elements.  The block of D commands begins
        with an identifying colon label and ends with a /EOF command.  The
        colon label is of the form :Clab (described above).

        Interpolation from multiple results sets can be performed by looping
        through the results file in a user-defined macro.  Additional blocks
        can be appended to Fname2 by using KPOS and unique colon labels.  To
        read the block of commands, issue the /INPUT command with the
        appropriate colon label.

        If the model has coincident (or very close) nodes, the CBDOF must be
        applied to each part of the model separately to ensure that the mapping
        of the nodes is correct.  For example, if nodes belonging to two
        adjacent parts linked by springs are coincident, the operation should
        be performed on each part of the model separately.

        Resume the coarse model database at the beginning of the cut-boundary
        procedure. The database should have been saved after the first coarse
        model solution, as the number of nodes in the database and the results
        file must match, and internal nodes are sometimes created during the
        solution.

        Caution:: : Relaxing the TOLHGT or TOLTHK tolerances to allow submodel
        nodes to be “found” can produce poor submodel results.

        """
        command = "CBDOF,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(fname1), str(ext1), str(fname2), str(ext2), str(kpos), str(clab), str(kshs), str(tolout), str(tolhgt), str(tolthk))
        self.RunCommand(command, **kwargs)

    def Slasherase(self, **kwargs):
        """
        APDL Command: /ERASE

        Specifies that the screen is to be erased before each display.

        Notes
        -----
        Erase occurs with the next display request, but before the display is
        actually started. /NOERASE can be used to suppress the automatic screen
        erase.

        This command is valid in any processor.

        """
        command = "/ERASE," % ()
        self.RunCommand(command, **kwargs)

    def Werase(self, **kwargs):
        """
        APDL Command: WERASE

        Erases all reordering wave lists.

        Notes
        -----
        The REORDER then STAT commands will display the current wave lists.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "WERASE," % ()
        self.RunCommand(command, **kwargs)

    def Dfswave(self, kcn="", radius="", psdref="", dens="", sonic="",
                incang="", npara="", sampopt="", **kwargs):
        """
        APDL Command: DFSWAVE

        Specifies the incident planar waves with random phases for a diffuse
        sound field.

        Parameters
        ----------
        kcn
            Local coordinate system:

            N - Coordinate system number. Default = 0.

            DELETE - Delete defined incident diffused planar waves.

        radius
            Radius of the reference sphere on which the incident planar waves
            are distributed with equal energy. Defaults to 50 x the half-
            maximum dimension of the structural panel.

        psdref
            Reference power spectral density. Default = 1.

        dens
            Mass density of incident planar wave media. Default = 2041 kg/m3.

        sonic
            Sound speed in incident planar wave media. Default = 343.24 m/s)

        incang
            Maximum incident angle (0o <= degree <= 180o) against the positive
            z axis in the local coordinate system KCN. Default = 0o.

        npara
            Number of divisions on the reference sphere with cutting planes
            parallel to the x-y coordinate plane of the local coordinate
            system. Default = 20.

        sampopt
            Random sampling option:

            ALL - Initializes the random generator of incident planar wave phases and samples the
                  phases at each solving frequency.

            MULT - Initializes the random generator of incident planar wave phases at the first
                   frequency and samples the phases at each solving frequency.

            MONO - Initializes the random generator of incident planar wave phases and samples the
                   phases only once at first solving frequency so that the same
                   phases are used over the whole frequency range for each
                   incident planar wave.

        Notes
        -----
        Issue the DFSWAVE command to activate a diffuse sound field. (The AWAVE
        command does not activate a diffuse sound field.)

        The SURF154 surface element must be defined on the surface of the
        structural solid element for the excitation.

        The acoustic elements and the absorbing boundary condition must be
        defined in the open acoustic domain. Do not define the acoustic domain
        on the excitation side.

        The PLST command calculates the average transmission loss for multiple
        sampling phases at each frequency over the frequency range.

        The symmetry of a panel structure cannot be used to reduce the
        simulation size, as the incident plane waves have varying random phase
        angles. The z axis of the Cartesian coordinate system (KCN) must be
        consistent with the panel’s outward normal unit vector at the center of
        the panel’s sending side.

        """
        command = "DFSWAVE,%s,%s,%s,%s,%s,%s,%s,%s" % (str(kcn), str(radius), str(psdref), str(dens), str(sonic), str(incang), str(npara), str(sampopt))
        self.RunCommand(command, **kwargs)

    def Sucalc(self, rsetname="", lab1="", oper="", lab2="", fact1="",
               fact2="", const="", **kwargs):
        """
        APDL Command: SUCALC

        Create new result data by operating on two existing result data sets on
        a given surface.

        Parameters
        ----------
        rsetname
            Eight character name for new result data.

        lab1
            First  result data upon which to operate.

        oper
            Mathematical operation to perform.

            ADD - (lab1 + lab2 + const)

            SUB - (lab1 - lab2 + const)

            MULT - (lab1 * lab2 + const)

            DIV - (lab1 / lab2 + const)

            EXP - (lab1 ^ fact1 + lab2 ^ fact2 + const)

            COS - (cos (lab1) + const)

            SIN - (sin (lab1) + const)

            ACOS - (acos (lab1) + const)

            ASIN - (asin (lab1) + const)

            ATAN - (atan (lab1) + const)

            ATA2 - (atan2 (lab1 / lab2) + const)

            LOG - (log (lab1) + const)

            ABS - (abs (lab1) + const)

            ZERO - (0 + const)

        lab2
            Second result data upon which to operate.

        fact1
            First scaling factor (for EXP option only).

        fact2
            Second scaling factor (for EXP option only).

        const
            Constant added to the values in the resulting data.

        """
        command = "SUCALC,%s,%s,%s,%s,%s,%s,%s" % (str(rsetname), str(lab1), str(oper), str(lab2), str(fact1), str(fact2), str(const))
        self.RunCommand(command, **kwargs)

    def Cskp(self, kcn="", kcs="", porig="", pxaxs="", pxypl="", par1="",
             par2="", **kwargs):
        """
        APDL Command: CSKP

        Defines a local coordinate system by three keypoint locations.

        Parameters
        ----------
        kcn
            Arbitrary reference number assigned to this coordinate system.
            Must be greater than 10.  A coordinate system previously defined
            with this number will be redefined.

        kcs
            Coordinate system type:

            0 or CART - Cartesian

            1 or CYLIN - Cylindrical (circular or elliptical)

            2 or SPHE - Spherical (or spheroidal)

            3 or TORO - Toroidal

        porig
            Keypoint defining the origin of this coordinate system.  If PORIG =
            P, graphical picking is enabled and all remaining command fields
            are ignored (valid only in the GUI).

        pxaxs
            Keypoint defining the positive x-axis orientation of this
            coordinate system.

        pxypl
            Keypoint defining the x-y plane (with PORIG and PXAXS) in the first
            or second quadrant of this coordinate system.

        par1
            Used for elliptical, spheroidal, or toroidal systems.  If KCS = 1
            or 2, PAR1 is the ratio of the ellipse Y-axis radius to X-axis
            radius (defaults to 1.0 (circle)).  If KCS = 3, PAR1 is the major
            radius of the torus.

        par2
            Used for spheroidal systems.  If KCS = 2, PAR2 = ratio of ellipse
            Z-axis radius to X-axis radius (defaults to 1.0 (circle)).

        Notes
        -----
        Defines and activates a local right-handed coordinate system by
        specifying three existing keypoints: to locate the origin, to locate
        the positive x-axis, and to define the positive x-y plane.  This local
        system becomes the active coordinate system.  See the CLOCAL, CS,
        CSWPLA, and LOCAL commands for alternate definitions.  Local coordinate
        systems may be displayed with the /PSYMB command.

        This command is valid in any processor.

        """
        command = "CSKP,%s,%s,%s,%s,%s,%s,%s" % (str(kcn), str(kcs), str(porig), str(pxaxs), str(pxypl), str(par1), str(par2))
        self.RunCommand(command, **kwargs)

    def Expand(self, nrepeat="", hindex="", icsys="", sctang="", ph_ase="",
               **kwargs):
        """
        APDL Command: EXPAND

        Displays the results of a modal cyclic symmetry analysis.

        Parameters
        ----------
        nrepeat
            Number of sector repetitions for expansion. The default is 0 (no
            expansion).

        modal
            Specifies that the expansion is for a modal cyclic symmetry
            analysis.

        hindex
            The harmonic index ID for the results to expand.

        icsys
            The coordinate system number used in the modal cyclic symmetry
            solution. The default is the global cylindrical coordinate system
            (specified via the CSYS command where KCN = 1).

        sctang
            The sector angle in degrees, equal to 360 divided by the number of
            cyclic sectors.

        --
            This field is reserved for future use.

        phase
            The phase angle in degrees to use for the expansion. The default is
            0. Typically, the value is the peak displacement (or stress/strain)
            phase angle obtained via the CYCPHASE command.

        Notes
        -----
        Issue this command to display the results of a modal cyclic symmetry
        analysis.

        When you issue the EXPAND,Nrepeat command, subsequent SET commands read
        data from the results file and expand them to Nrepeat sectors. As long
        as no entities have been modified, this expansion can be negated (that
        is, reverted to single sector) by issuing EXPAND with no arguments. If
        you modify entities and wish to return to the partial model, use the
        Session Editor (see Restoring Database Contents in the Operations
        Guide).

        EXPAND displays the results and allows you to print them, as if for a
        full model. The harmonic index (automatically retrieved from the
        results file) appears in the legend column.

        When plotting or printing element strain energy (SENE), the EXPAND
        command works with brick or tet models only. Element kinetic energy
        (KENE) plotting or printing is not supported.

        EXPAND is a specification command valid only in POST1.  It is
        significantly different from the /CYCEXPAND command in several
        respects, (although you can use either command to display the results
        of a modal cyclic symmetry analysis):

        EXPAND has none of the limitations of the /CYCEXPAND command.

        EXPAND changes the database by modifying the geometry, the nodal
        displacements, and element stresses as they are read from the results
        file, whereas the /CYCEXPAND command does not change the database.

        Caution:: : The EXPAND command creates new nodes and elements;
        therefore, saving (or issuing the /EXIT, ALL command) after issuing the
        EXPAND command can result in large databases.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EXPAND,%s,%s,%s,%s,%s" % (str(nrepeat), str(hindex), str(icsys), str(sctang), str(ph_ase))
        self.RunCommand(command, **kwargs)

    def Quad(self, node1="", nintr="", node2="", nfill="", nstrt="", ninc="",
             pkfac="", **kwargs):
        """
        APDL Command: QUAD

        Generates a quadratic line of nodes from three nodes.

        Parameters
        ----------
        node1
            Begin fill-in from this node location.  If NODE1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).

        nintr
            Intermediate or guiding node.  Quadratic curve will pass through
            this location.  NINTR may have any node number and any location.
            If the quadratic line also generates a node with number NINTR, the
            generated location overrides the previous NINTR location.

        node2
            End quadratic fill-in at this node location.

        nfill
            Fill-in NFILL nodes between NODE1 and NODE2 (defaults to
            |NODE2-NODE1|-1).  NFILL must be positive.

        nstrt
            Node number assigned to first filled-in node (defaults to NODE1 +
            NINC).

        ninc
            Add this increment to each of the remaining filled-in node numbers
            (may be positive or negative).  Defaults to (NODE2-NODE1)/(NFILL +
            1), i.e., linear interpolation.

        pkfac
            Peak location factor.  If PKFAC=0.5, the peak of the quadratic
            shape occurs at the NINTR location.  If 0.0 < PKFAC < 0.5, the peak
            occurs to the NODE2 side of the NINTR location.  If 0.5 < PKFAC <
            1.0, the peak occurs to the NODE1 side of the NINTR location.
            Defaults to 0.5.

        Notes
        -----
        Generates a quadratic line of nodes (in the active coordinate system)
        from three nodes.  The three nodes determine the plane of the curve and
        may have been defined in any coordinate system.  Any number of nodes
        may be filled-in and any node number sequence may be assigned.

        The quadratic line feature uses three nodes (NODE1,NINTR,NODE2) to
        determine the plane of the curve.  The curve passes through the three
        points, beginning from NODE1, through the intermediate (or guiding)
        point NINTR, and toward NODE2.

        Generated nodes are also quadratically spaced.  If the guiding node
        number is within the set being generated, it will be relocated
        according to the quadratic spacing.

        The peak location factor is used to determine how the quadratic fits
        through the three points.  Various nodal progressions can be obtained
        by different combinations of PKFAC and the guiding node location.  If
        the guiding node is at mid-length between NODE1 and NODE2, 0.293:
        PKFAC< 0.707 will ensure that all generated nodes fall within the
        NODE1,NODE2 bounds.  In the limit, as PKFAC approaches 0.0, the peak
        approaches the line through NODE1 and NINTR at an infinite distance
        from NODE1.  The QUAD command generates quadratic lines of nodes, which
        in turn may be used as a base line for generating irregular surfaces of
        nodes (by repeating [*REPEAT], generating [NGEN, NSCALE], etc.).
        Irregular surfaces may also be generated with the meshing commands.

        """
        command = "QUAD,%s,%s,%s,%s,%s,%s,%s" % (str(node1), str(nintr), str(node2), str(nfill), str(nstrt), str(ninc), str(pkfac))
        self.RunCommand(command, **kwargs)

    def Vscfun(self, parr="", func="", par1="", **kwargs):
        """
        APDL Command: *VSCFUN

        Determines properties of an array parameter.

        Parameters
        ----------
        parr
            The name of the resulting scalar parameter.  See *SET for name
            restrictions.

        func
            Functions:

            Maximum: the maximum Par1 array element value. - Minimum: the minimum Par1 array element value.

            Index location of the maximum Par1 array element value.  Array Par1 is searched starting from its specified index. - Index location of the minimum Par1 array element value.  Array Par1 is searched
                              starting from its specified index.

            Index location of the first nonzero value in array Par1.  Array Par1 is searched starting from its specified index. - Index location of the last nonzero value in array Par1.  Array Par1 is searched
                              starting from its specified index.

            Sum:  Par1 (the summation of the Par1 array element values). - Median: value of Par1 at which there are an equal number of values above and
                              below.

            Mean: (σ Par1)/NUM, where NUM is the number of summed values. - Variance: (σ ((Par1-MEAN)**2))/NUM.

            Standard deviation: square root of VARI. - Root-mean-square: square root of (σ (Par1**2))/NUM.

        par1
            Array parameter vector in the operation.

        Notes
        -----
        Operates on one input array parameter vector and produces one output
        scalar parameter according to:

        ParR = f(Par1)

        where the functions (f) are described below. The starting array element
        number must be defined for the array parameter vector.  For example,
        *VSCFUN,MU,MEAN,A(1) finds the mean of the A vector values, starting
        from the first value and stores the result as parameter MU.  Operations
        use successive array elements [*VLEN, *VMASK] with the default being
        all successive array elements.  Absolute values and scale factors may
        be applied to all parameters [*VABS, *VFACT].  Results may be
        cumulative [*VCUM].  See the *VOPER command for details.

        This command is valid in any processor.

        """
        command = "*VSCFUN,%s,%s,%s" % (str(parr), str(func), str(par1))
        self.RunCommand(command, **kwargs)

    def Abbsav(self, lab="", fname="", ext="", **kwargs):
        """
        APDL Command: ABBSAV

        Writes the current abbreviation set to a coded file.

        Parameters
        ----------
        lab
            Label that specifies the write operation:

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        Existing abbreviations on this file, if any, will be overwritten.  The
        abbreviation file may be read with the ABBRES command.

        This command is valid in any processor.

        """
        command = "ABBSAV,%s,%s,%s" % (str(lab), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Polygon(self, nvert="", x1="", y1="", x2="", y2="", x3="", y3="",
                x4="", y4="", **kwargs):
        """
        APDL Command: /POLYGON

        Creates annotation polygons (GUI).

        Parameters
        ----------
        nvert
            Number of vertices of polygon (3  NVERT   8).  Use /PMORE for
            polygons with more than 4 vertices.

        x1
            X location for vertex 1 of polygon (-1.0 < X < 2.0).

        y1
            Y location for vertex 1 of polygon (-1.0 < Y < 1.0).

        x2
            X location for vertex 2 of polygon (-1.0 < X < 2.0).

        y2
            Y location for vertex 2 of polygon (-1.0 < Y < 1.0).

        x3
            X location for vertex 3 of polygon (-1.0 < X < 2.0).

        y3
            Y location for vertex 3 of polygon (-1.0 < Y < 1.0).

        x4
            X location for vertex 4 of polygon (-1.0 < X < 2.0).

        y4
            Y location for vertex 4 of polygon (-1.0 < Y < 1.0).

        Notes
        -----
        Creates annotation polygons to be written directly onto the display at
        a specified location.  This is a command generated by the Graphical
        User Interface (GUI) and will appear in the log file (Jobname.LOG) if
        annotation is used.  This command is not intended to be typed in
        directly in an ANSYS session (although it can be included in an input
        file for batch input or for use with the /INPUT command).

        All polygons are shown on subsequent displays unless the annotation is
        turned off or deleted.  Use the /LSPEC and the /PSPEC command to set
        the attributes of the polygon.  Use the /PMORE command to define the
        5th through 8th vertices of the polygon.

        This command is valid in any processor.

        """
        command = "/POLYGON,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nvert), str(x1), str(y1), str(x2), str(y2), str(x3), str(y3), str(x4), str(y4))
        self.RunCommand(command, **kwargs)

    def Pdcorr(self, name1="", name2="", corr="", **kwargs):
        """
        APDL Command: PDCORR

        Specifies the correlation between two random input variables.

        Parameters
        ----------
        name1
            Parameter name. The parameter must have been previously defined as
            a random input variable with the PDVAR command.

        name2
            Parameter name. The parameter must have been previously defined as
            a random input variable with the PDVAR command. Must be different
            from Name1.

        corr
            Specification of the correlation:

            Value - Sets the correlation coefficient between Name1 and Name2 to the specified
                    value. If this correlation coefficient was already defined
                    it will be changed to this new value. The correlation
                    coefficient must be between -1.0 and +1.0.

            DEL - Delete the previously defined correlation between Name1 and Name2.

        Notes
        -----
        Specifies the correlations between two random input variables. The PDS
        tool generates correlated samples if correlations exist. This applies
        to both probabilistic methods (Monte Carlo Simulation and Response
        Surface Methods).

        If there are correlated random input variables, the sampled locations
        of the random input variables reflect the correlation as illustrated
        below for the case of two standard normal distributed variables X1 and
        X2. The illustrations show no correlation (correlation coefficient
        0.0), a relatively moderate negative correlation (correlation
        coefficient -0.6), and a relative strong positive correlation
        (correlation coefficient 0.9).

        : : :

        """
        command = "PDCORR,%s,%s,%s" % (str(name1), str(name2), str(corr))
        self.RunCommand(command, **kwargs)

    def Psmesh(self, secid="", name="", p0="", egroup="", num="", kcn="",
               kdir="", value="", ndplane="", pstol="", pstype="", ecomp="",
               ncomp="", **kwargs):
        """
        APDL Command: PSMESH

        Create and mesh a pretension section

        Parameters
        ----------
        secid
            Unique section number. This number must not already be assigned to
            a section.

        name
            Unique eight character descriptive name, if desired.

        p0
            Pretension node number. The node will be defined if it doesn't
            exist and the number defaults to the highest node number plus one.

        egroup, num
            Element group on which PSMESH will operate. If Egroup = P,
            graphical picking is enabled and NUM is ignored (valid only in the
            GUI).

            L (or LINE) - PSMESH operates on all elements in the line specified by NUM. New pretension
                          nodes are associated with NUM or entities below it.
                          Any subsequent LCLEAR operation of NUM deletes the
                          pretension elements and nodes created by PSMESH.

            A (or AREA) - PSMESH operates on all elements in the area specified by NUM. New pretension
                          nodes are associated with NUM or entities below it.
                          Any subsequent ACLEAR of NUM deletes the pretension
                          elements and nodes created by PSMESH.

            V (or VOLU) - PSMESH operates on all elements in the volume specified by NUM. New pretension
                          nodes are associated with NUM or entities below it.
                          Any subsequent VCLEAR of NUM deletes the pretension
                          elements and nodes created by PSMESH.

            P - PSMESH operates on elements selected through the subsequent picking operations,
                and NUM is ignored

            ALL - The command operates on all selected elements, and NUM is ignored.

        kcn
            Coordinate system number for the separation surface and normal
            direction.

        kdir
            Direction (x, y, or z) normal to separation surface in the KCN
            coordinate system.

        value
            Point along the KDIR axis at which to locate the separation
            surface. Ignored if NDPLANE is supplied.

        ndplane
            Existing node that PSMESH will use to locate the separation
            surface. If NDPLANE is supplied, the location of the separation
            surface is defined by the KDIR coordinate of NDPLANE.

        pstol
            Optional tolerance below VALUE. Allows nodes occurring precisely at
            or slightly below the separation  to be identified properly as
            above the plane. Has the effect of shifting the plane down by
            PSTOL.  The following expression represents the default value:

        pstype
            If specified, this value is the type number for pretension
            elements. (If not specified, ANSYS defines this value.) If already
            defined, it must be of type PRETS179.

        ecomp
            If specified, the name of a component to be composed of new
            pretension elements and existing elements modified by the PSMESH
            command.

        ncomp
            Name of a component to be composed of nodes on new pretension
            elements.

        Notes
        -----
        The PSMESH command creates a pretension section normal to the
        pretension load direction by cutting the mesh along existing element
        boundaries at the point defined by VALUE or NDPLANE and inserting
        PRETS179 elements. The PSMESH command verifies that PSTYPE is PRETS179;
        if it is not, the command finds the lowest available ITYPE that is
        PRETS179, or if necessary will create a new one.

        When it is necessary to define the pretension node, ANSYS uses node
        NDPLANE. If the NDPLANE value is not specified, ANSYS defines the
        pretension node at:

        The centroid of geometric entity NUM, if Egroup = LINE, AREA or VOLU

        The centroid location of all selected elements, if Egroup = ALL or if
        graphical picking is used.

        If the elements to which the pretension load is to be applied have
        already been meshed in two groups, PSMESH cannot be used to insert the
        pretension elements. The EINTF command must be used to insert the
        PRETS179 elements between the two meshed groups.

        The PSMESH operation copies any nodal temperatures you have defined on
        the split surface of the original mesh from the original nodes to the
        newly created coincident duplicate nodes. However, displacements,
        forces, and other boundary conditions are not copied.

        By mathematical definition, the pretension surface must always be a
        flat plane. In a non-Cartesian coordinate system, the PSMESH command
        creates that plane at the indicated position, oriented with respect to
        the specified direction of the active system (in the same manner that
        the NROTAT command orients a nodal system with respect to a curved
        system). For example, assuming a X = 1 and Y = 45 in a cylindrical
        coordinate system with Z as the axis of rotation (KCN = 1), a
        pretension surface normal to X tilts 45 degrees away from the global X
        axis.

        The PSMESH command is valid for structural analyses only.

        """
        command = "PSMESH,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(secid), str(name), str(p0), str(egroup), str(num), str(kcn), str(kdir), str(value), str(ndplane), str(pstol), str(pstype), str(ecomp), str(ncomp))
        self.RunCommand(command, **kwargs)

    def Edgcale(self, nadv="", meth="", **kwargs):
        """
        APDL Command: EDGCALE

        Defines global ALE controls for an explicit dynamic analysis.

        Parameters
        ----------
        nadv
            Number of cycles between advection (default = 0).

        meth
            Advection method.

            0 - Donor cell + Half Index Shift (first order accurate) (default).

            1 - Van Leer + Half Index Shift (second order accurate).

        Notes
        -----
        This command sets global ALE controls in an explicit dynamic analysis.
        These ALE controls apply to all PLANE162 or SOLID164 elements in the
        model that are flagged to use the ALE formulation (KEYOPT(5) = 1). To
        activate the ALE formulation, you must specify the number of cycles
        between advection on this command and at least one smoothing weight
        factor on the EDALE command. See Arbitrary Lagrangian-Eulerian
        Formulation in the ANSYS LS-DYNA User's Guide for more information.

        To see the current EDGCALE settings, issue the command EDALE,LIST.

        The EDGCALE command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDGCALE,%s,%s" % (str(nadv), str(meth))
        self.RunCommand(command, **kwargs)

    def Jsol(self, nvar="", elem="", item="", comp="", name="", **kwargs):
        """
        APDL Command: JSOL

        Specifies result items to be stored for the joint element.

        Parameters
        ----------
        nvar
            Arbitrary reference number or name assigned to this variable.
            Variable numbers can be 2 to NV (NUMVAR) while the name can be an
            eight-byte character string. Overwrites any existing results for
            this variable.

        elem
            Element number for which to store results.

        item
            Label identifying the item.  Valid item labels are shown in Table
            202: JSOL - Valid Item and Component Labels below.

        comp
            Component of the Item (if required).  Valid component labels are
            shown in Table 202: JSOL - Valid Item and Component Labels below.

        name
            Thirty-two character name identifying the item on printouts and
            displays.  Defaults to a label formed by concatenating the first
            four characters of the Item and Comp labels.

        Notes
        -----
        This command is valid for the MPC184 joint elements. The values stored
        are for the free or unconstrained degrees of freedom of a joint
        element. Relative reaction forces and moments are available only if
        stiffness, damping, or friction is associated with the joint element.

        Table: 202:: : JSOL - Valid Item and Component Labels



        """
        command = "JSOL,%s,%s,%s,%s,%s" % (str(nvar), str(elem), str(item), str(comp), str(name))
        self.RunCommand(command, **kwargs)

    def Rezone(self, option="", ldstep="", sbstep="", **kwargs):
        """
        APDL Command: REZONE

        Initiates the rezoning process, sets rezoning options, and rebuilds the
        database.

        Parameters
        ----------
        option
            The rezoning method to employ:

            MANUAL - Manual rezoning. You decide when to use rezoning, what region(s) to rezone, and
                     what remeshing method to use on the selected region(s).
                     This method is currently the default and only option.

        ldstep
            The load step number at which rezoning should occur. The default
            value is the highest load step number found in the Jobname.Rnnn
            files (for the current jobname and in the current directory).

        sbstep
            The substep number of the specified load step (LDSTEP) at which
            rezoning should occur. The default value is the highest substep
            number found in the specified load step in the Jobname.Rnnn files
            (for the current jobname and in the current directory).

        Notes
        -----
        The REZONE command rebuilds the database (.db file) based on the
        specified load step and substep information, and updates nodes to their
        deformed position for remeshing.

        Before issuing this command, clear the database via the /CLEAR command.

        For more information, see Rezoning in the Advanced Analysis Guide.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "REZONE,%s,%s,%s" % (str(option), str(ldstep), str(sbstep))
        self.RunCommand(command, **kwargs)

    def Dflab(self, dof="", displab="", forcelab="", **kwargs):
        """
        APDL Command: /DFLAB

        Changes degree-of-freedom labels for user custom elements.

        Parameters
        ----------
        dof
            Number between 1 and 32 indicating which degree of freedom is to
            have its labels changed. For a list of these quantities, see the
            degree-of-freedom table in the echprm.inc file. The first few
            quantities follow:

        displab
            New label (four-character maximum) for the displacement label. The
            prior label is no longer valid.

        forcelab
            New label (four-character maximum) for the force label for this
            degree of freedom. The prior label is no longer valid.

        Notes
        -----
        The /DFLAB command is rarely used. Use it if you are writing a custom
        element and want to use degrees of freedom that are not part of the
        standard element set.

        """
        command = "/DFLAB,%s,%s,%s" % (str(dof), str(displab), str(forcelab))
        self.RunCommand(command, **kwargs)

    def Nslv(self, type="", nkey="", **kwargs):
        """
        APDL Command: NSLV

        Selects those nodes associated with the selected volumes.

        Parameters
        ----------
        type
            Label identifying the type of node select:

            S - Select a new set (default).

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        nkey
            Specifies whether only interior volume nodes are to be selected:

            0 - Select only nodes interior to selected volumes.

            1 - Select all nodes (interior to volume, interior to areas, interior to lines, and
                at keypoints) associated with the selected volumes.

        Notes
        -----
        Valid only if the nodes were generated by a volume meshing operation
        [VMESH] on a solid model that contains the selected volumes.

        This command is valid in any processor.

        """
        command = "NSLV,%s,%s" % (str(type), str(nkey))
        self.RunCommand(command, **kwargs)

    def Gline(self, wn="", style="", **kwargs):
        """
        APDL Command: /GLINE

        Specifies the element outline style.

        Parameters
        ----------
        wn
            Window number (or ALL) to which command applies (defaults to 1).

        style
            Outline key:

             0  - Solid element outlines (default)

             1  - Dashed element outlines

            -1  - No element outlines

        Notes
        -----
        Determines the element outline style.  Often used when node numbers are
        displayed to prevent element lines from overwriting node numbers.

        Unless you are using an OpenGL or Starbase driver, the dashed element
        outline option (/GLINE,WN,1) is not available in the following
        situations:

        Z-buffered displays (/TYPE,WN,6).

        Capped Z-buffered displays (/TYPE,WN,7).

        Qslice Z-buffered displays (/TYPE,WN,8).

        This command is valid in any processor.

        """
        command = "/GLINE,%s,%s" % (str(wn), str(style))
        self.RunCommand(command, **kwargs)

    def Satin(self, name="", extension="", path="", entity="", fmt="", nocl="",
              noan="", **kwargs):
        """
        APDL Command: ~SATIN

        Transfers a .SAT file into the ANSYS program.

        Parameters
        ----------
        name
            The name of a valid .SAT file, created with a supported version of
            ACIS. The first character of the file name must be an alphanumeric.
            Special characters such as & - and * and spaces are not permitted
            in the part name. See File Names in the Command Reference for more
            information about ANSYS file naming conventions.

        extension
            The extension for the file. The default extension is .sat.

        path
            The path name of the directory in which the file resides enclosed
            in single quotes. The default path name is the current working
            directory.

        entity
            Entity to be imported.

            SOLIDS - Solids only, imported as ANSYS volumes (Not implemented, imports All).

            SURFACES - Surfaces only, imported as ANSYS areas (Not implemented, imports All).

            WIREFRAME - Wireframe only, imported as ANSYS lines (Not implemented, imports All).

            ALL - All entities. Use this option when the file contains different types of
                  entities.

        fmt
            The format in which ANSYS will store the model.

            0 - Neutral format (default). Defeaturing after import is restricted.

            1 - Solid format; this allows defeaturing after import.

        nocl
            Remove tiny objects.

            0 - Remove tiny objects without checking model validity (default).

            1 - Do not remove tiny objects.

        noan
            Perform an ACIS analysis of the model.

            0 - Analyze the model (default).

            1 - Do not analyze the model.

        Notes
        -----
        More information on importing ACIS parts is available in ACIS in the
        Connection User's Guide.

        """
        command = "~SATIN,%s,%s,%s,%s,%s,%s,%s" % (str(name), str(extension), str(path), str(entity), str(fmt), str(nocl), str(noan))
        self.RunCommand(command, **kwargs)

    def Facet(self, lab="", **kwargs):
        """
        APDL Command: /FACET

        Specifies the facet representation used to form solid model displays.

        Parameters
        ----------
        lab
            Valid labels:

            FINE - Use finer tessellation to increase the number of facets for the display.
                   Provides the best representation (but decreases speed of
                   operation).

            NORML - Use the basic number of facets for the display (default).

            COAR - Use a limited number of facets for the display. This option will increase the
                   speed of the operations, but may produce poor
                   representations for some imported models.

            WIRE - Display model with a wireframe representation (fast, but surfaces will not be
                   shown).

        Notes
        -----
        Specifies the facet (or polygon) representation used to form solid
        model displays.  Used only with the APLOT, ASUM, VPLOT, and VSUM
        commands.

        This command is valid in any processor.

        """
        command = "/FACET,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Rbe3(self, m_aster="", dof="", slaves="", wtfact="", **kwargs):
        """
        APDL Command: RBE3

        Distributes the force/moment applied at the master node to a set  of
        slave nodes, taking into account the geometry of the slave nodes as
        well as weighting factors.

        Parameters
        ----------
        master
            Node at which the force/moment to be distributed will be applied.
            This node must be associated with an element for the master node to
            be included in the DOF solution.

        dof
            Refers to the master node degrees of freedom to be used in
            constraint equations. Valid labels are: UX, UY, UZ, ROTX, ROTY,
            ROTZ, UXYZ, RXYZ, ALL

        slaves
            The name of an array parameter that contains a list of slave nodes.
            Must specify the starting index number. ALL can be used for
            currently selected set of nodes. The slave nodes may not be
            colinear, that is, not be all located on the same straight line
            (see Notes below).

        wtfact
            The name of an array parameter that contains a list of weighting
            factors corresponding to each slave node above. Must have the
            starting index number. If not specified, the weighting factor for
            each slave node defaults to 1.

        Notes
        -----
        The force is distributed to the slave nodes proportional to the
        weighting factors. The moment is distributed as forces to the slaves;
        these forces are proportional to the distance from the center of
        gravity of the slave nodes times the weighting factors.  Only the
        translational degrees of freedom of the slave nodes are used for
        constructing the constraint equations. Constraint equations are
        converted to distributed forces/moments on the slave nodes during
        solution.

        RBE3 creates constraint equations such that the motion of the master is
        the average of the slaves. For the rotations, a least-squares approach
        is used to define the "average rotation" at the master from the
        translations of the slaves. If the slave nodes are colinear, then one
        of the master rotations that is parallel to the colinear direction can
        not be determined in terms of the translations of the slave nodes.
        Therefore, the associated moment component on the master node in that
        direction can not be transmitted. When this case occurs, a warning
        message is issued and the constraint equations created by RBE3 are
        ignored.

        Applying this command to a large number of slave nodes may result in
        constraint equations with a large number of coefficients. This may
        significantly increase the peak memory required during the process of
        element assembly. If real memory or virtual memory is not available,
        consider reducing the number of slave nodes.

        As an alternative to the RBE3 command, you can apply a similar type of
        constraint using contact elements and the internal multipoint
        constraint (MPC) algorithm. See Surface-based Constraints for more
        information.

        This command is also valid in SOLUTION.

        """
        command = "RBE3,%s,%s,%s,%s" % (str(m_aster), str(dof), str(slaves), str(wtfact))
        self.RunCommand(command, **kwargs)

    def Color(self, lab="", clab="", n1="", n2="", ninc="", **kwargs):
        """
        APDL Command: /COLOR

        Specifies the color mapping for various items.

        Parameters
        ----------
        lab
            Apply color to the items specified by the following labels:

            AXES - Determines the color (specified in next argument, Clab) that the axes of a
                   graph will be plotted in.

            AXNUM - Determines the color (specified in next argument, Clab) that the numbering on
                    the axes of a graph will be plotted in.

            NUM - Discretely numbered items (such as element types, element materials, etc., as
                  shown on the /PNUM command).  Also specify number (1 to 11)
                  in the N1 field.  For example, /COLOR,NUM,RED,3 will assign
                  the color red to all items having the discrete number 3
                  (material displays would show elements having material 3 as
                  red).

            OUTL - Outline of elements, areas, and volumes.  Ex: /COLOR,OUTL,BLUE.

            ELEM - Elements. Use N1, N2, NINC fields for element numbers.

            LINE - Solid model lines.  Use N1, N2, NINC fields for line numbers.

            AREA - Solid model areas.  Use N1, N2, NINC fields for area numbers.

            VOLU - Solid model volumes.  Use N1, N2, NINC fields for volume numbers.

            ISURF - Isosurfaces (surfaces of constant stress, etc.).   This option is particularly
                    useful when capturing frames for animating a single
                    isosurface value.

            WBAK - Window background.  Use N1, N2, NINC fields for window numbers. The options
                   that you select using Lab = PBAK will supersede those
                   applied using Lab = WBAK.

            b.c.label - Boundary condition label.  Enter U, ROT, TEMP, PRES, V, VOLT, MAG, A, EMF,
                        CURR, F, M, HEAT, FLOW, VF, AMPS, FLUX, CSG, CURT,
                        VLTG, MAST, CP, CE, NFOR, NMOM, RFOR, RMOM, PATH. See
                        the /PBC command for boundary condition label
                        definitions.

            GRBAK - Graph background.

            GRID - Graph grid lines.

            AXLAB - Graph X and Y axis labels.

            CURVE - Graph curves (identify curve numbers (1-10) in N1, N2, NINC fields).

            CM - Component group.  Use N1 field for component name, ignore N2 and NINC.

            CNTR - ANSYS contour stress colors.  The maximum number of contours available is 128.
                   The number of colors that can be specified interactively
                   (GUI) is 9. (/CONTOUR, , 9). Any other setting will yield
                   inconsistent results.

            SMAX - Specifies that all stress values above the maximum value entered in /CONTOUR
                   will be displayed in the color designated in the Clab field.
                   Defaults to dark grey.

            SMIN - Specifies that all stress values below the minimum value entered in /CONTOUR
                   will be displayed in the color designated in the Clab field.
                   Defaults to dark grey.

            PBAK - Activates background shading options (see command syntax at end of argument
                   descriptions below). The options that you select using Lab =
                   PBAK will supersede those applied using Lab = WBAK.

        clab
            Valid color labels are:

            BLAC (0) - Black

            MRED (1) - Magenta-Red

            MAGE (2) - Magenta

            BMAG (3) - Blue-Magenta

            BLUE (4) - Blue

            CBLU (5) - Cyan-Blue

            CYAN (6) - Cyan

            GCYA ((7) - Green-Cyan

            GREE (8) - Green

            YGRE (9) - Yellow-Green

            YELL (10) - Yellow

            ORAN (11) - Orange

            RED (12) - Red

            DGRA (13) - Dark Gray

            LGRA (14) - Light Gray

            WHIT (15) - White

        n1, n2, ninc
            Apply color to Lab items numbered N1 to N2 (defaults to N1) in
            steps of NINC (defaults to 1).  If N1 is blank, apply color to
            entire selected range.  If Lab is CM, use component name for N1 and
            ignore N2 and NINC.  If N1 = P, graphical picking of elements,
            lines, areas and volumes is enabled; your can assign colors to the
            entities via the picker. When picking is enabled, the Lab and Clab
            fields are ignored.

        Notes
        -----
        Issue /COLOR,STAT to display the current color mapping.  Issue
        /COLOR,DEFA to reset the default color mapping.

        Note:: : Color labels may also be reassigned any "color" with the /CMAP
        command.

        This command is valid anywhere.

        """
        command = "/COLOR,%s,%s,%s,%s,%s" % (str(lab), str(clab), str(n1), str(n2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Nsmooth(self, np_ass="", **kwargs):
        """
        APDL Command: NSMOOTH

        Smooths selected nodes among selected elements.

        Parameters
        ----------
        npass
            Number of smoothing passes. Defaults to 3.

        Notes
        -----
        Repositions each selected node at the average position of its immediate
        neighbors on the selected elements. The node positions converge after
        some number of smoothing passes. For some initial conditions, NPASS may
        need to be much larger than 3. If the boundary of a mesh is to be
        undisturbed (usually desirable), the boundary nodes should be
        unselected before issuing NSMOOTH.

        """
        command = "NSMOOTH,%s" % (str(np_ass))
        self.RunCommand(command, **kwargs)

    def Elseif(self, val1="", oper1="", val2="", conj="", val3="", oper2="",
               val4="", **kwargs):
        """
        APDL Command: *ELSEIF

        Separates an intermediate if-then-else block.

        Parameters
        ----------
        val1
            First numerical value (or parameter which evaluates to numerical
            value) in the conditional comparison operation. VAL1, VAL2, VAL3,
            and VAL4 can also be character strings (enclosed in quotes) or
            parameters for Oper = EQ and NE only.

        oper1
            Operation label. A tolerance of 1.0E-10 is used for comparisons
            between real numbers:

            Equal (for VAL1 = VAL2). - Not equal (for VAL1 ≠VAL2).

            Less than (for VAL1<VAL2). - Greater than (for VAL1>VAL2).

            Less than or equal (for VAL1 VAL2). - Greater than or equal (for VAL1 VAL2).

            Absolute values of VAL1 and VAL2 before < operation. - Absolute values of VAL1 and VAL2 before > operation.

        val2
            Second numerical value (or parameter which evaluates to numerical
            value) in the conditional comparison operation.

        conj
            (Optional) Connection between two logical clauses.

            True if both clauses (Oper1 and Oper2) are true.  - True if either clause is true.

        val3
            (Optional) Third numerical value (or parameter which evaluates to
            numerical value).

        oper2
            (Optional) Operation label.  This will have the same labels as
            Oper1, except it uses Val3 and Val4. A tolerance of 1.0E-10 is used
            for comparisons between real numbers.

        val4
            (Optional) Fourth numerical value (or parameter value which
            evaluates to a numerical value).

        Notes
        -----
        Optional intermediate block separator within an if-then-else construct.
        All seven characters of the command name (*ELSEIF) must be input. This
        command is similar to the *IF  command except that the Base field is
        not used. The *IF, *ELSEIF, *ELSE, and *ENDIF commands for each if-
        then-else construct must all be read from the same file (or keyboard).

        This command is valid in any processor.

        """
        command = "*ELSEIF,%s,%s,%s,%s,%s,%s,%s" % (str(val1), str(oper1), str(val2), str(conj), str(val3), str(oper2), str(val4))
        self.RunCommand(command, **kwargs)

    def Edhgls(self, hgco="", **kwargs):
        """
        APDL Command: EDHGLS

        Specifies the hourglass coefficient for an explicit dynamics analysis.

        Parameters
        ----------
        hgco
            Hourglass coefficient value (defaults to 0.1). Values greater than
            0.15 may cause instabilities.

        Notes
        -----
        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDHGLS,%s" % (str(hgco))
        self.RunCommand(command, **kwargs)

    def Cutcontrol(self, lab="", value="", option="", **kwargs):
        """
        APDL Command: CUTCONTROL

        Controls time-step cutback during a nonlinear solution.

        Parameters
        ----------
        lab
            Specifies the criteria for causing a cutback.  Valid labels are:

            PLSLIMIT  - Maximum equivalent plastic strain allowed within a time-step (substep).  If the
                        calculated value exceeds the VALUE, the program
                        performs a cutback (bisection).  VALUE defaults to 0.15
                        (15%).

            CRPLIMIT  - Set values for calculating the maximum equivalent creep ratio allowed within a
                        time step. If the calculated maximum creep ratio
                        exceeds the defined creep ratio limit, the program
                        performs a cutback.

            DSPLIMIT  - Maximum incremental displacement within the solution field in a time step
                        (substep).  If the maximum calculated value exceeds
                        VALUE, the program performs a cutback (bisection).
                        VALUE defaults to 1.0 x 107.

            NPOINT  - Number of points in a cycle for a second order dynamic equation, used to
                      control automatic time stepping.  If the number of
                      solution points per cycle is less than VALUE, the program
                      performs a cutback in time step size. VALUE defaults to
                      13 for linear analysis, 5 for nonlinear analysis. A
                      larger number of points yields a more accurate solution
                      but also increases the solution run time.

            This option works well for linear problems. For nonlinear analyses, other factors such as contact status changes and solution convergence rate can overwrite NPOINT. See Automatic Time Stepping in the Mechanical APDL Theory Reference for more information on automatic time stepping. - NOITERPREDICT

            If VALUE is 0 (default), an internal auto time step scheme will predict the number of iterations for nonlinear convergence and perform a cutback earlier than the number of iterations specified by the NEQIT command. This is the recommended option. If VALUE is 1, the solution will iterate (if nonconvergent) to NEQIT number of iterations before a cutback is invoked. It is sometimes useful for poorly-convergent problems, but rarely needed in general. - Bisection is also controlled by contact status change, plasticity or creep
                              strain limit, and other factors. If any of these
                              factors occur, bisection will still take place,
                              regardless of the NOITERPREDICT setting.

            CUTBACKFACTOR  - Changes the cutback value for bisection. Default is 0.5. VALUE must be greater
                             than 0.0 and less than 1.0. This option is active
                             only if AUTOTS,ON is set.

        value
            Numeric value for the specified cutback criterion. For Lab =
            CRPLIMIT, VALUE is the creep criteria for the creep ratio limit.

        option
            Type of creep analysis. Valid for Lab = CRPLIMIT only.

            IMPRATIO  - Set the maximum creep ratio value for implicit creep. The default is 0.0 (i.e.,
                        no creep limit control) and any positive value is
                        valid. (See Implicit Creep Procedure in the Structural
                        Analysis Guide for information on how to define
                        implicit creep.)

            EXPRATIO   - Set the maximum creep ratio value for explicit creep. The default value is 0.1
                         and any positive value up to 0.25 is allowed. (See
                         Explicit Creep Procedure in the Structural Analysis
                         Guide for information on how to define explicit
                         creep.)

            STSLIMIT   - Stress threshold for calculating the creep ratio. For integration points with
                         effective stress below this threshold, the creep ratio
                         does not cause cutback. The default value is 0.0 and
                         any positive value is valid.

            STNLIMIT   - Elastic strain threshold for calculating the creep ratio. For integration
                         points with effective elastic strain below this
                         threshold, the creep ratio does not cause cutback. The
                         default value is 0.0 and any positive value is valid.

        Notes
        -----
        A cutback is a method for automatically reducing the step size when
        either the solution error is too large or the solution encounters
        convergence difficulties during a nonlinear analysis.

        Should a convergence failure occur, the program reduces the time step
        interval to a fraction of its previous size and automatically continues
        the solution from the last successfully converged time step. If the
        reduced time step again fails to converge, the program again reduces
        the time step size and proceeds with the solution. This process
        continues until convergence is achieved or the minimum specified time
        step value is reached.

        For creep analysis, the cutback procedure is similar; the process
        continues until the minimum specified time step size is reached.
        However, if the creep ratio limit is exceeded, the program issues a
        warning but continues the substep until the analysis is complete. In
        this case, convergence is achieved but the creep ratio criteria is not
        satisfied.

        The CRPLIM command is functionally equivalent to Lab = CRPLIMIT with
        options IMPRATIO and EXPRATIO

        """
        command = "CUTCONTROL,%s,%s,%s" % (str(lab), str(value), str(option))
        self.RunCommand(command, **kwargs)

    def Mptemp(self, sloc="", t1="", t2="", t3="", t4="", t5="", t6="",
               **kwargs):
        """
        APDL Command: MPTEMP

        Defines a temperature table for material properties.

        Parameters
        ----------
        sloc
            Starting location in table for entering temperatures.  For example,
            if SLOC = 1, data input in the T1 field applies to the first
            constant in the table.  If SLOC = 7, data input in the T1 field
            applies to the seventh constant in the table, etc.  Defaults to the
            last location filled + 1.

        t1, t2, t3, . . . , t6
            Temperatures assigned to six locations starting with SLOC.  If a
            value is already in this location, it will be redefined.  A blank
            (or zero) value for T1 resets the previous value in SLOC to zero.
            A value of zero can only be assigned by T1.  Blank (or zero) values
            for T2 to T6 leave the corresponding previous values unchanged.

        Notes
        -----
        Defines a temperature table to be associated with the property data
        table [MPDATA].  These temperatures are also used for polynomial
        property evaluation, if defined [MP].  Temperatures must be defined in
        non-descending order.  Issue MATER $ STAT to list the current
        temperature table.  Repeat MPTEMP command for additional temperatures
        (100 maximum).  If all arguments are blank, the temperature table is
        erased.

        For clear definition, the temperature range you define with the MPTEMP
        command should include the entire range you'll use in subsequently
        defined materials.  To assist the user in this, the first (and only the
        first) excursion out of the temperature range defined by the MPTEMP
        commands is flagged with a warning message.  Similarly, the reference
        temperature (TREF or MP,reft commands) should also fall in this same
        temperature range.  If not and MP,alpx was used, a note will be output.
        If not, and MP,ctex or MP,thsx was used, an error message will be
        output.

        This command is also valid in SOLUTION.

        """
        command = "MPTEMP,%s,%s,%s,%s,%s,%s,%s" % (str(sloc), str(t1), str(t2), str(t3), str(t4), str(t5), str(t6))
        self.RunCommand(command, **kwargs)

    def Smbc(self, mode="", **kwargs):
        """
        APDL Command: /SMBC

        Controls the display of solid model boundary condition symbols and
        labels.

        Parameters
        ----------
        mode
            CENT

            CENT - Solid model boundary condition symbols and labels appear at the centroid of the
                   solid model entity (default).

            TESS - Solid model boundary condition symbols and labels appear inside each
                   constituent element of the tessellation.

        Notes
        -----
        Mode = CENT is designed to reduce the clutter of boundary condition
        symbols in solid model plots. For example, if you have assigned normal
        pressure loads to an area, you may choose to display the pressures as
        arrows with the /PSF command using /PSF,PRES,NORM,2. When Mode = CENT,
        the pressure arrow is displayed at the centroid of the area. When Mode
        = TESS, a pressure arrow is displayed at the centroid of each polygon
        of the area's tessellation.

        This command is valid in any processor.

        """
        command = "/SMBC,%s" % (str(mode))
        self.RunCommand(command, **kwargs)

    def Ssum(self, **kwargs):
        """
        APDL Command: SSUM

        Calculates and prints the sum of element table items.

        Notes
        -----
        Calculates and prints the tabular sum of each existing labeled result
        item [ETABLE] for the selected elements.  If absolute values are
        requested [SABS,1], absolute values are used.

        """
        command = "SSUM," % ()
        self.RunCommand(command, **kwargs)

    def Mpplot(self, lab="", mat="", tmin="", tmax="", pmin="", pmax="",
               **kwargs):
        """
        APDL Command: MPPLOT

        Plots linear material properties as a function of temperature.

        Parameters
        ----------
        lab
            Linear material property label (EX, EY, etc.) [MP].

        mat
            Material reference number. Defaults to 1.

        tmin
            Minimum abscissa value to be displayed.

        tmax
            Maximum abscissa value.

        pmin
            Minimum property (ordinate) value to be displayed.

        pmax
            Maximum property value.

        Notes
        -----
        When the property is from tables, the MPPLOT command will not be valid
        because the property could be a function of more than temperature.

        This command is valid in any processor.

        """
        command = "MPPLOT,%s,%s,%s,%s,%s,%s" % (str(lab), str(mat), str(tmin), str(tmax), str(pmin), str(pmax))
        self.RunCommand(command, **kwargs)

    def Vlen(self, nrow="", ninc="", **kwargs):
        """
        APDL Command: *VLEN

        Specifies the number of rows to be used in array parameter operations.

        Parameters
        ----------
        nrow
            Number of rows to be used with the *VXX or *MXX operations.
            Defaults to the number of rows needed to fill the result array.

        ninc
            Perform the operation on every NINC row (defaults to 1).

        Notes
        -----
        Specifies the number of rows to be used in array parameter operations.
        The size of the submatrix used is determined from the upper left
        starting array element (defined on the operation command) to the lower
        right array element (defined by the number of rows on this command and
        the number of columns on the *VCOL command).  NINC allows skipping row
        operations for some operation commands.  Skipped rows are included in
        the row count.  The starting row number must be defined on the
        operation command for each parameter read and for the result written.

        The default NROW is calculated from the maximum number of rows of the
        result array (the *DIM row dimension) minus the starting location + 1.
        For example, *DIM,R,,10 and a starting location of R(7) gives a default
        of 4 loops (filling R(7), R(8), R(9), and R(10)).  Repeat operations
        automatically terminate at the last row of the result array.  Existing
        values in the rows and columns of the results matrix remain unchanged
        where not overwritten by the requested input or operation values.

        The stride (NINC) allows operations to be performed at regular
        intervals.  It has no effect on the total number of row operations.
        Skipped operations retain the previous result.  For example, *DIM,R,,6,
        with a starting location of R(1), NROW = 10, and NINC = 2 calculates
        values for locations R(1), R(3), and R(5) and retains values for
        locations R(2), R(4), and R(6).  A more general skip control may be
        done by masking [*VMASK].  The row control settings are reset to the
        defaults after each *VXX or *MXX operation.  Use *VSTAT to list
        settings.

        This command is valid in any processor.

        """
        command = "*VLEN,%s,%s" % (str(nrow), str(ninc))
        self.RunCommand(command, **kwargs)

    def Ndele(self, node1="", node2="", ninc="", **kwargs):
        """
        APDL Command: NDELE

        Deletes nodes.

        Parameters
        ----------
        node1, node2, ninc
            Delete nodes from NODE1 to NODE2 (defaults to NODE1) in steps of
            NINC (defaults to 1).  If NODE1 = ALL, NODE2 and NINC are ignored
            and all selected nodes [NSEL] are deleted.  If NODE1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).  A component name may also be substituted
            for NODE1.

        Notes
        -----
        Deletes selected nodes that are not connected to elements.  Nodes may
        also be redefined instead of deleted, if desired.  Boundary conditions
        (displacements, forces, etc.) as well as any coupling or constraint
        equations containing the deleted nodes are also deleted.

        This command is also valid in the /MAP processor.

        """
        command = "NDELE,%s,%s,%s" % (str(node1), str(node2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Dv3d(self, lab="", key="", **kwargs):
        """
        APDL Command: /DV3D

        Sets 3-D device option modes.

        Parameters
        ----------
        lab
            Mode label:

            ACCU - Allows ANSYS to use the accumulation buffer for OpenGL graphics. Activating
                   this feature will provide faster model rotation when shaded
                   backgrounds are in use. This feature is off by default.

            ACTR - Label term to designate the cursor position as the center for automatic dynamic
                   rotational center capability.  The subsequent Key value (see
                   below) turns this capability on and off. This feature is on
                   by default. (Available for OpenGL displays only)

            ANIM - Animation mode.  The ANIM option allows you to create animation frames in
                   pixmap mode instead of display list mode.  This may improve
                   large model performance, but it eliminates local
                   manipulation while animation is in progress. This feature is
                   on by default.

            ANTI - Label term to control Anti-aliasing, a smoothing technique for your graph
                   plots. (see below) The subsequent Key value turns this
                   capability on and off. The default for this feature is off.
                   (Available for OpenGL displays only).

            CNTR - Switches banded contours on (1) or off (0) for your 3–D contour display. The
                   default is 1 (ON). Other contour parameters such as number
                   of contours or the increment and range are defined using the
                   /CONTOUR command. When either 9 or 128 contours are
                   specified via /CONTOUR, this command is ignored and a smooth
                   contour is always displayed.

            DGEN - Local manipulation degenerate mode.  You access the DGEN option to set wire-
                   frame local manipulation mode for 3-D devices (device
                   dependent). This feature is off by default.

            DLIST - With DLIST, you can specify whether screen updates and redraws will be
                    performed using the ANSYS Display List (off), or the 3-D
                    device's Display List (on). DLIST is on by default for
                    Windows systems, but off for Linux.

            DELS - You use DELS to suppress contour display screen overwrites when /NOERASE is
                   active. This prevents the bleed-through that occurs when you
                   overlay contour plots.

            TRIS - Triangle strip mode. Tri-stripping provides faster 3-D display capabilities and
                   is on by default. Some display enhancements, such as
                   texturing, are adversely affected by tri-stripping. You can
                   turn off tri-stripping in order to improve these display
                   functions. Be sure to turn tri-stripping on after the
                   desired output is obtained.

        key
            The following key options apply to Lab = ACCU:

            0 - (OFF)  The accumulation buffer is not accessed. (default)

            1 - (ON)  Access to the buffer is enabled.

        Notes
        -----
        ANSYS uses display list animation for its 3-D models. This memory
        resident array method interfaces with the OpenGL model information to
        allow the program to efficiently pan, zoom, rotate and dynamically
        manipulate your model during animation. The logo, legend, contour and
        other annotation items are produced in 2-D and will not appear when
        /DV3D, ANIM, 0 is in effect. To display these items, use /DV3D, ANIM,
        1. All screen data will be displayed, but manipulation of the model
        will not be possible.

        """
        command = "/DV3D,%s,%s" % (str(lab), str(key))
        self.RunCommand(command, **kwargs)

    def Ealive(self, elem="", **kwargs):
        """
        APDL Command: EALIVE

        Reactivates an element (for the birth and death capability).

        Parameters
        ----------
        elem
            Element to be reactivated:

            ALL  - Reactivates all selected elements (ESEL).

            P  - Enables graphical picking of elements. All remaining command fields are
                 ignored. (Valid only in the ANSYS GUI.)

            Comp - Specifies a component name.

        Notes
        -----
        Reactivates the specified element when the birth and death capability
        is being used. An element can be reactivated only after it has been
        deactivated (EKILL).

        Reactivated elements have a zero strain (or thermal heat storage, etc.)
        state.

        ANSYS, Inc. recommends using the element deactivation/reactivation
        procedure for analyses involving linear elastic materials only. Do not
        use element deactivation/reactivation in analyses involving time-
        dependent materials, such as viscoelasticity, viscoplasticity, and
        creep analysis.

        This command is also valid in PREP7.

        """
        command = "EALIVE,%s" % (str(elem))
        self.RunCommand(command, **kwargs)

    def Mpamod(self, mat="", deftemp="", **kwargs):
        """
        APDL Command: MPAMOD

        Modifies temperature-dependent secant coefficients of thermal
        expansion.

        Parameters
        ----------
        mat
            Material number for which the secant coefficients of thermal
            expansion (SCTE's) are to be modified.  Defaults to 1.

        deftemp
            Definition temperature at which the existing SCTE-versus-
            temperature tables were defined.  Defaults to zero.

        Notes
        -----
        This command converts temperature-dependent SCTE data (properties ALPX,
        ALPY, ALPZ)  from the definition temperature (DEFTEMP) to the reference
        temperature defined by MP,REFT or TREF.  If both the MP,REFT and TREF
        commands have been issued, the reference temperature defined by the
        MP,REFT command will be used.

        This command does not apply to the instantaneous coefficients of
        thermal expansion (properties CTEX, CTEY, CTEZ) or to the thermal
        strains (properties THSX, THSY, THSZ).

        See Linear Material Properties in the Mechanical APDL Material
        Reference and the Mechanical APDL Theory Reference for more details.

        This command is also valid in SOLUTION.

        """
        command = "MPAMOD,%s,%s" % (str(mat), str(deftemp))
        self.RunCommand(command, **kwargs)

    def Image(self, label="", fname="", ext="", **kwargs):
        """
        APDL Command: /IMAGE

        Allows graphics data to be captured and saved.

        Parameters
        ----------
        label
            Label specifying the operation to be performed:

            CAPTURE - Capture the image from the graphics window to a new window.

            RESTORE - Restore the image from a file to a new window.

            SAVE - Save the contents of the graphic window to a file.

            DELETE - Delete the window that contains the file.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        """
        command = "/IMAGE,%s,%s,%s" % (str(label), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Cdwrite(self, option="", fname="", ext="", fnamei="", exti="", fmat="",
                **kwargs):
        """
        APDL Command: CDWRITE

        Writes geometry and load database items to a file.

        Parameters
        ----------
        option
            Selects which data to write:

            ALL - Write all appropriate geometry, material property, load, and component data
                  (default). Two files will be produced. Fname.Ext will contain
                  all data items mentioned in "Notes", except the solid model
                  data. Fnamei.Exti will contain the solid model geometry and
                  solid model loads data in the form of IGES commands. This
                  option is not valid when CDOPT,ANF is active.

            COMB - Write all data mentioned, but to a single file, Fname.Ext. Solid model geometry
                   data will be written in either IGES or ANF format as
                   specified in the CDOPT command, followed by the remainder of
                   the data in the form of ANSYS commands. More information on
                   these (IGES/ANF) file formats is provided in "Notes".

            DB - Write all database information except the solid model and solid model loads to
                 Fname.Ext in the form of ANSYS commands. This option is not
                 valid when CDOPT,ANF is active.

            SOLID - Write only the solid model geometry and solid model load data. This output will
                    be in IGES or ANF format, as specified in the CDOPT
                    command. More information on these (IGES/ANF) file formats
                    is provided in "Notes".

            GEOM - Write only element and nodal geometry data. Neither solid model geometry nor
                   element attribute data will be written. One file, Fname.Ext,
                   will be produced. Use CDREAD,DB to read in a file written
                   with this option. Element types [ET] compatible with the
                   connectivity of the elements on the file must first be
                   defined before reading the file in with CDREAD,DB.

            CM - Write only node and element component and geometry data to Fname.Ext.

            MAT - Write only material property data (both linear and nonlinear) to Fname.Ext.

            LOAD - Write only loads for current load step to Fname.Ext.

            SECT - Write only section data to Fname.Ext. Pretension sections are not included.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        fnamei
            Name of the IGES file and its directory path (248 characters
            maximum, including directory). If you do not specify a directory
            path, it will default to your working directory and you can use all
            248 characters for the file name.

        exti
            Filename extension (eight-character maximum).

        fmat
            Format of the output file (defaults to BLOCKED).

            BLOCKED - Blocked format. This format allows faster reading of the output file. The time
                      savings is most significant when BLOCKED is used to read
                      .cdb files associated with very large models.

            UNBLOCKED - Unblocked format.

        Notes
        -----
        Load data includes the current load step only. Loads applied to the
        solid model (if any) are automatically transferred to the finite
        element model when this command is issued. CDWRITE writes out solid
        model loads for meshed models only. If the model is not meshed, the
        solid model loads cannot be saved. Component data include component
        definitions, but not assembly definitions. Appropriate NUMOFF commands
        are included at the beginning of the file; this is to avoid overlap of
        an existing database when the file is read in.

        Element order information (resulting from a WAVES command) is not
        written. The data in the database remain untouched.

        Solution control commands are typically not written to the file unless
        you specifically change a default solution setting.

        CDWRITE does not support the GSBDATA and GSGDATA commands, and these
        commands are not written to the file.

        The data may be reread (on a different machine, for example) with the
        CDREAD command. Caution: When the file is read in, the NUMOFF,MAT
        command may cause a mismatch between material definitions and material
        numbers referenced by certain loads and element real constants. See
        NUMOFF for details. Also, be aware that the files created by the
        CDWRITE command explicitly set the active coordinate system to
        Cartesian (CSYS,0).

        You should generally use the blocked format (Fmat = BLOCKED) when
        writing out model data with CDWRITE. This is a compressed data format
        that greatly reduces the time required to read large models through the
        CDREAD command. The blocked and unblocked formats are described in
        Chapter 3 of the Guide to Interfacing with ANSYS.

        If you use CDWRITE in any of the derived products (ANSYS Emag, ANSYS
        Professional), then before reading the file, you must edit the
        Jobname.cdb file to remove commands that are not available in the
        respective component product.

        The CDWRITE command writes PART information for any ANSYS LS-DYNA input
        file to the Jobname.cdb file via the EDPREAD command. (EDPREAD is not a
        documented command; it is written only when the CDWRITE command is
        issued.) The PART information can be automatically read in via the
        CDREAD command; however, if more than one Jobname.cdb file is read, the
        PART list from the last Jobname.cdb file overwrites the existing PART
        list of the total model. This behavior affects all PART-related
        commands contained in the Jobname.cdb file. You can join models, but
        not PART-related inputs, which you must modify using the newly-created
        PART numbers. In limited cases, an update of the PART list
        (EDWRITE,PUPDATE) is possible; doing so requires that no used
        combination of MAT/TYPE/REAL appears more than once in the list.

        The CDWRITE command does not support (for beam meshing) any line
        operation that relies on solid model associativity. For example,
        meshing the areas adjacent to the meshed line, plotting the line that
        contains the orientation nodes, or clearing the mesh from the line that
        contains orientation nodes may not work as expected. For more
        information about beam meshing, see Meshing Your Solid Model in the
        Modeling and Meshing Guide.

        IGES and ANF File Formats for Solid Model Geometry Information

        The format used for solid model geometry information is determined by
        the current CDOPT command setting. The default format is IGES.

        IGES option (default) to write solid model information (CDOPT, IGS):

        Before writing solid model entities, select all corresponding lower
        level entities (ALLSEL,BELOW,ALL).

        Section properties assigned to areas, lines and other solid model
        entities are not maintained when the model is exported.

        If you issue CDWRITE after generating a beam mesh with orientation
        nodes, the database file will contain all of the nodes for every beam
        element, including the orientation nodes; however, the orientation
        keypoints that were specified for the line (LATT) are no longer
        associated with the line and won't be written out to the geometry file.
        All associativity between the line and the orientation keypoints is
        lost.

        For beam meshing, this option does not support any line operation that
        relies on solid model associativity. For example, meshing the areas
        adjacent to the meshed line, plotting the line that contains the
        orientation nodes, or clearing the mesh from the line that contains
        orientation nodes may not work as expected.

        Concatenated lines are not written. The line segments that make up the
        concatenated lines are written; however, if the command encounters an
        area that contains a concatenated line, the write operation halts (that
        area cannot be recreated during the read operation). If your model has
        areas that contain concatenated lines, you must first list these and
        then unconcatenate them before issuing the CDWRITE command. Similarly,
        hardpoint information cannot be written.

        ANF option to write solid model information (CDOPT, ANF):

        Writes all model information in the database (regardless of select
        status) to the archive file; however, when you restore the database
        using this archived file, the select status of entities is also
        restored.

        Restores all line attributes, including orientation keypoints. It also
        writes out any components (not assemblies) comprised of solid model
        entities.

        Hhalts CDWRITE when a concatenated line or an area that contains a
        concatenated line is detected. You must delete the concatenated lines
        before issuing CDWRITE. Similarly, hardpoint information cannot be
        written.

        This command is also valid in SOLUTION.

        """
        command = "CDWRITE,%s,%s,%s,%s,%s,%s" % (str(option), str(fname), str(ext), str(fnamei), str(exti), str(fmat))
        self.RunCommand(command, **kwargs)

    def Resvec(self, key="", **kwargs):
        """
        APDL Command: RESVEC

        Calculates or includes residual vectors.

        Parameters
        ----------
        key
            Residual vector key:

            OFF - Do not calculate or include residual vectors. This option is the default.

            ON  - Calculate or include residual vectors.

        Notes
        -----
        In a modal analysis, the RESVEC command calculates residual vectors. In
        a mode-superposition transient dynamic, mode-superposition harmonic,
        PSD or spectrum analysis, the command includes residual vectors.

        In a component mode synthesis (CMS) generation pass, the RESVEC command
        calculates one residual vector which is included in the normal modes
        basis used in the transformation matrix. It is supported for the three
        available CMS methods. RESVEC,ON can only be specified in the first
        load step of a generation pass and is ignored if issued at another load
        step.

        If rigid-body modes exist, pseudo-constraints are required for the
        calculation. Issue the D,,,SUPPORT command to specify only the minimum
        number of pseudo-constraints necessary to prevent rigid-body motion.

        For more information about residual vector formulation, see Residual
        Vector Method in the Mechanical APDL Theory Reference.

        """
        command = "RESVEC,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Lczero(self, **kwargs):
        """
        APDL Command: LCZERO

        Zeroes the results portion of the database.

        Notes
        -----
        Often used before the LCOPER command.  Same as LCOPER,ZERO.

        """
        command = "LCZERO," % ()
        self.RunCommand(command, **kwargs)

    def Trtime(self, time="", spacing="", offset="", size="", length="",
               **kwargs):
        """
        APDL Command: TRTIME

        Defines the options used for the PLTRAC (particle flow or charged
        particle trace) command.

        Parameters
        ----------
        time
            Total Trace Time (seconds) (defaults to 0, which is the full flow
            trace).

        spacing
            Particle spacing in seconds (defaults to 0).

        offset
            Particle offset in seconds (defaults to 0).  It is used internally
            in the ANFLOW macro to produce an animation of particle flow in a
            flowing fluid or charged particle motion in an electric or magnetic
            field.

        size
            Particle size (defaults to 0, which is a line).

        length
            Particle length fraction (defaults to .1).

        Notes
        -----
        The TRTIME command varies the type of   PLTRAC display produced.
        Particle flow or charged particle traces follow a particle's path in
        the forward and backward direction of travel. The DOF selected
        determines the color of the particle trace.  SPACING defines the
        particle spacing in seconds from adjacent particles in the stream line.
        OFFSET defines the offset in seconds from the spacing set by the
        SPACING argument.

        LENGTH defines the particle length fraction.  The default value (.1),
        means the particle occupies 10% of the flow region, and the other 90%
        is a color-coded line.

        SIZE sets the radius of the particle. Use SPACING, OFFSET and LENGTH
        only when SIZE is nonzero (i.e., the particle is bigger than the line).

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "TRTIME,%s,%s,%s,%s,%s" % (str(time), str(spacing), str(offset), str(size), str(length))
        self.RunCommand(command, **kwargs)

    def Staopt(self, method="", **kwargs):
        """
        APDL Command: STAOPT

        Specifies static analysis options.

        Parameters
        ----------
        method
            Solution method for the static analysis:

            DEFA - Standard ANSYS solve (default).

            VT - Solve with Variational Technology.

        Notes
        -----
        Specifies the method of solution for a static analysis (ANTYPE,STATIC).
        If used in SOLUTION, this command is valid only within the first load
        step.

        The VT option is valid for either thermal or structural nonlinear
        analyses, where it attempts to reduce the total number of iterations.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: The VT static solution method is not
        supported in Distributed ANSYS.

        """
        command = "STAOPT,%s" % (str(method))
        self.RunCommand(command, **kwargs)

    def Asum(self, lab="", **kwargs):
        """
        APDL Command: ASUM

        Calculates and prints geometry statistics of the selected areas.

        Parameters
        ----------
        lab
            Controls the degree of tessellation used in the calculation of area
            properties.  If LAB = DEFAULT, area calculations will use the
            degree of tessellation set through the /FACET command.  If LAB =
            FINE, area calculations are based on a finer tessellation.

        Notes
        -----
        Calculates and prints geometry statistics (area, centroid location,
        moments of inertia, volume, etc.) associated with the selected areas.
        ASUM should only be used on perfectly flat areas.

        Geometry items are reported in the global Cartesian coordinate system.
        A unit thickness is assumed unless the areas have a non-zero total
        thickness defined by real constant or section data.

        For layered areas, a unit density is always assumed. For single-layer
        areas, a unit density is assumed unless the areas have a valid material
        (density).

        The thickness and density are associated to the areas via the AATT
        command.

        Items calculated via ASUM and later retrieved via a *GET or *VGET
        command are valid only if the model is not modified after issuing the
        ASUM command.

        Setting a finer degree of tessellation will provide area calculations
        with greater accuracy, especially for thin, hollow models.  However,
        using a finer degree of tessellation requires longer processing.

        For very narrow (sliver) areas, such that the ratio of the minimum to
        the maximum dimension is less than 0.01, the ASUM command can provide
        erroneous area information.  To ensure that the calculations are
        accurate, subdivide such areas so that the ratio of the minimum to the
        maximum is at least 0.05.

        """
        command = "ASUM,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Czmesh(self, ecomps1="", ecomps2="", kcn="", kdir="", value="",
               cztol="", **kwargs):
        """
        APDL Command: CZMESH

        Create and mesh an interface area composed of cohesive zone elements.

        Parameters
        ----------
        ecomps1
            Component name or number for the group of plane or solid structural
            elements adjacent to the interface being meshed.

        ecomps2
            Component name or number for the opposing (from ecomps1) group of
            plane or solid structural elements adjacent to the interface being
            meshed.

        kcn
            Coordinate system number for the separation surface and normal
            direction. (if ecomps1 and ecomps2 not specified)

        kdir
            Direction (x, y, or z) normal to separation surface in the KCN
            coordinate system (if ecomps1 and ecomps2 not specified).

        value
            Coordinate value along the KDIR axis at which to locate the
            interface (if ecomps1 and ecomps2 not specified).

        cztol
            Optional absolute tolerance about VALUE (if ecomps1 and ecomps2 not
            specified). Allows nodes occurring slightly above or below the
            separation to be grouped properly. The following expression
            represents the default value:

        Notes
        -----
        CZMESH is used on a mesh with shared nodes at the interface.

        If ecomps1 and ecomps2 are specified, the CZMESH command creates/meshes
        interface elements (INTER202, INTER203, INTER204, INTER205) along the
        boundary between the two components or groups of elements.

        The elements in each of the components or groups of elements will share
        nodes with each other and also with the interface elements. This one-
        element thick boundary of interface elements will split the body
        between the two components or groups of elements.

        Subsequent separation (delamination and failure) of the interface zone
        will result in an increasing displacement between the nodes (within the
        interface element) along the cohesive zone elements. Unless otherwise
        specified, the CZMESH command analyzes the configuration and geometry
        of the adjacent structural elements and provides the appropriate
        interface element.

        The CZMESH operation copies any nodal temperatures you have defined on
        the split surface of the original mesh from the original nodes to the
        newly created coincident duplicate nodes. However, displacements,
        forces, and other boundary conditions are not copied.

        The CZMESH command is valid for structural analyses only.

        """
        command = "CZMESH,%s,%s,%s,%s,%s,%s" % (str(ecomps1), str(ecomps2), str(kcn), str(kdir), str(value), str(cztol))
        self.RunCommand(command, **kwargs)

    def Expsol(self, lstep="", sbstep="", timfrq="", elcalc="", **kwargs):
        """
        APDL Command: EXPSOL

        Specifies the solution to be expanded for mode-superposition analyses
        or substructure analyses.

        Parameters
        ----------
        lstep, sbstep
            Expand the solution identified as load step LSTEP and substep
            SBSTEP.

        timfrq
            As an alternative to LSTEP and SBSTEP, expand the solution at, or
            nearest to, the time value TIMFRQ (for ANTYPE,TRANS or
            ANTYPE,SUBSTR) or frequency value TIMFRQ (for ANTYPE,HARMIC).
            LSTEP and SBSTEP should be blank.

        elcalc
            Element calculation key:

            YES - Calculate element results, nodal loads, and reaction loads.

            NO - Do not calculate these items.

        Notes
        -----
        Specifies the solution to be expanded from analyses that use the mode-
        superposition method (ANTYPE,HARMIC or TRANS) or substructuring
        (ANTYPE,SUBSTR). Use the NUMEXP command to expand a group of solutions.

        The resulting results file will maintain the same load step, substep,
        and time (or frequency) values as the requested solution to be
        expanded.

        This command is also valid in PREP7.

        """
        command = "EXPSOL,%s,%s,%s,%s" % (str(lstep), str(sbstep), str(timfrq), str(elcalc))
        self.RunCommand(command, **kwargs)

    def Units(self, label="", lenfact="", m_assfact="", timefact="",
              tempfact="", toffset="", chargefact="", forcefact="",
              heatfact="", **kwargs):
        """
        APDL Command: /UNITS

        Annotates the database with the system of units used.

        Parameters
        ----------
        label
            Label to denote the system of units used in this job:

            USER - User-defined system (default).

            SI - International system (m, kg, s, K).

            MKS - MKS system (m, kg, s, °C).

            uMKS - μMKS system (μm, kg, s, °C).

            CGS - CGS system (cm, g, s, °C).

            MPA - MPA system (mm, Mg, s, °C).

            BFT - U. S. Customary system using feet (ft, slug, s, °F).

            BIN - U. S. Customary system using inches (in, lbf*s2/in, s, °F).

        Notes
        -----
        Allows the user to set a marker in the database indicating the system
        of units used.  The setting may be reviewed with the /STATUS command at
        the Begin level.  The units label and conversion factors on this
        command are for user convenience only and have no effect on the
        analysis or data.  That is, /UNITS will not convert database items from
        one system to another (e.g., from U. S. Customary to SI, etc.).  The
        units setting will be written to the file of IGES data [IGESOUT or
        CDWRITE], which can then be read by many programs that read IGES files.
        The user must still use consistent units for the results to be valid.

        If you choose the MKS system of units, the EPZRO option for the EMUNIT
        command is set to 8.85 e-12 F/m.  (EPZRO specifies alternate free-space
        permittivity.)

        For micro-electromechanical systems (MEMS), where dimensions are on the
        order of microns, see the conversion factors in System of Units in the
        Coupled-Field Analysis Guide.

        If you use the ANSYS ADAMS Interface to export model information to the
        ADAMS program, the /UNITS command is required to ensure the correct
        transfer of data between ANSYS and ADAMS. You may choose a predefined
        unit system label (Label = SI, CGS, etc.) or you can select the user-
        defined system option (Label = USER) and input the appropriate
        conversion factors (LENFACT, MASSFACT, TIMEFACT, and FORCEFACT). The
        conversion factors will be written to the ADAMS input file Jobname.MNF
        in order to correctly generate the load. For more information, see
        Export to ADAMS in the Substructuring Analysis Guide.

        All differences between the base solution units used by the ANSYS and
        CFX solvers will be noted in the ANSYS output file.   Unit conversions
        are automatically applied to all loads transferred unless Label = USER.
        Unit conversions are not applied to any of the loads transferred
        between the ANSYS and CFX solvers if they use a user-defined unit
        system.

        This command is valid in any processor.

        """
        command = "/UNITS,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(label), str(lenfact), str(m_assfact), str(timefact), str(tempfact), str(toffset), str(chargefact), str(forcefact), str(heatfact))
        self.RunCommand(command, **kwargs)

    def Prrfor(self, lab="", **kwargs):
        """
        APDL Command: PRRFOR

        Prints the constrained node reaction solution. Used with the FORCE
        command.

        Parameters
        ----------
        lab
            Nodal reaction load type.  If blank, use the first ten of all
            available labels. Valid labels are:

        Notes
        -----
        PRRFOR has the same functionality as the PRRSOL command; use PRRFOR
        instead of PRRSOL when a FORCE command has been issued.

        In a non-spectrum analysis, if either contact or pretension elements
        exist in the model, PRRFOR uses the PRRSOL command internally and the
        FORCE setting is ignored.

        Because modal displacements cannot be used to calculate contact element
        nodal forces,: those forces are not included in the spectrum and PSD
        analyses reaction solution. As a consequence, the: PRRFOR: command is
        not supported when constraints on contact element pilot nodes are
        present.

        """
        command = "PRRFOR,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Bfklist(self, kpoi="", lab="", **kwargs):
        """
        APDL Command: BFKLIST

        Lists the body force loads at keypoints.

        Parameters
        ----------
        kpoi
            Keypoint at which body load is to be listed.  If ALL (or blank),
            list for all selected keypoints [KSEL].  If KPOI = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).  A component name may also be substituted
            for KPOI

        lab
            Valid body load label. If ALL, use all appropriate labels. Load
            labels are listed under "Body Loads" in the input table for each
            element type in the Element Reference. See the BFK command for
            labels.

        Notes
        -----
        Lists the body force loads for the specified keypoint and label.
        Keypoint body loads may be defined with the BFK command.

        This command is valid in any processor.

        """
        command = "BFKLIST,%s,%s" % (str(kpoi), str(lab))
        self.RunCommand(command, **kwargs)

    def Sucr(self, surfname="", surftype="", nrefine="", radius="", tolout="",
             **kwargs):
        """
        APDL Command: SUCR

        Create a surface.

        Parameters
        ----------
        surfname
            Eight character surface name.

        surftype
            Surface type.

            CPLANE - Surface is defined by the cutting plane in window one (controlled by the
                     working plane (/CPLANE,1), NOT the view settings
                     (/CPLANE,0)).

            SPHERE - Surface is defined by a spherical surface centered about the working plane
                     origin.

            INFC - Surface is defined by a cylindrical surface centered about the working plane
                   origin and extending indefinitely in the positive and
                   negative Z directions.

        nrefine
            Refinement level.

            For SurfType = CPLANE -  The refinement level of the surface "mesh". This will be an integer between 0
                              and 3 (default = 0). See Notes below.

            For SurfType = SPHERE - The number of divisions along a 90° arc (minimum = 9). The default is 9.

            For SurfType = INFC - The number of divisions along a 90° arc (minimum = 9). The default is 9.

        radius
            Appropriate radius value (for INFC or SPHERE).

        tolout
            Tolerance value for inclusion of element facets within a prescribed
            volume. (for INFC)

        Notes
        -----
        This command creates a new surface and stores the following data for
        that surface:

        For SurfType = CPLANE, nRefine refers to the number of points that
        define the surface. An nRefine value of zero is used for points where
        the element face intersects the cutting plane.

        If SurfType = CPLANE and nRefine = 0, the points reside at the section
        cuts where the element intersects the cutting plane. Increasing nRefine
        from 0 to 1 will subdivide each surface facet into 4 subfacets, and
        increase the number of points at which results can be interpolated.

        For SurfType = CPLANE , the setting from the /EFACET command will
        affect the creation of surface facets and the quality of the fit of the
        surface in the model.  SUCR employs geometry data from PowerGraphics to
        aid in determining where the surface intersects the model.  If
        /EFACET,1 is in effect when the SUCR command is issued, then the
        curvature of high order elements (that is, elements with midside nodes)
        will be ignored.  If your model contains high order elements, you can
        see a better fit for your surface if /EFACET,2 is in effect when the
        SUCR command is issued.  Currently, the SUCR command interprets
        /EFACET,4 to mean /EFACET,2.

        For SurfType = INFC, a default tolerance of 0.01 will be applied to
        include the vertices of any facets that fall out of the cylinder
        definition. This tolerance increases the facet size by one percent to
        check for inclusion. Excluding facets under such a small tolerance may
        yield unacceptable (aesthetically) results. Increasing the tolerance by
        a larger amount (0.1 or 10%) will sometimes yield smother edges along
        the surface you create.

        """
        command = "SUCR,%s,%s,%s,%s,%s" % (str(surfname), str(surftype), str(nrefine), str(radius), str(tolout))
        self.RunCommand(command, **kwargs)

    def Pause(self, **kwargs):
        """
        APDL Command: PAUSE

        Temporarily releases the current product license.

        Notes
        -----
        The PAUSE command temporarily releases (or pauses) the current product
        license so that another application can use it.

        This application consumes a license as soon as you launch it, and
        retains that license until it is finished. If you launch the product
        interactively, the license is retained until you either close the
        application or issue a PAUSE command via the command line.

        No other operation (other than SAVE or /EXIT) is possible in the
        current application while use of the product license is paused.

        When the second application has finished and releases the license,
        issue an UNPAUSE command via the command line to restore use of the
        license to the current application.

        For more information, see the ANSYS, Inc. Licensing Guide.

        """
        command = "PAUSE," % ()
        self.RunCommand(command, **kwargs)

    def Rmalist(self, **kwargs):
        """
        APDL Command: RMALIST

        Lists all defined master nodes for a ROM method.

        Notes
        -----
        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "RMALIST," % ()
        self.RunCommand(command, **kwargs)

    def Linl(self, nl1="", nl2="", nl3="", nl4="", nl5="", nl6="", nl7="",
             nl8="", nl9="", **kwargs):
        """
        APDL Command: LINL

        Finds the common intersection of lines.

        Parameters
        ----------
        nl1, nl2, nl3, . . . , nl9
            Numbers of lines to be intersected.  If NL1 = ALL, find the
            intersection of all selected lines and NL2 to NL9 are ignored.  If
            NL1 = P, graphical picking is enabled and all remaining command
            fields are ignored (valid only in the GUI).  A component name may
            also be substituted for NL1.

        Notes
        -----
        Finds the common (not pairwise) intersection of lines.  The common
        intersection is defined as the regions shared (in common) by all lines
        listed on this command.  New lines will be generated where the original
        lines intersect.  If the regions of intersection are only points, new
        keypoints will be generated instead.  See the Modeling and Meshing
        Guide for an illustration.  See the BOPTN command for the options
        available to Boolean operations.  Element attributes and solid model
        boundary conditions assigned to the original entities will not be
        transferred to the new entities generated.

        """
        command = "LINL,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nl1), str(nl2), str(nl3), str(nl4), str(nl5), str(nl6), str(nl7), str(nl8), str(nl9))
        self.RunCommand(command, **kwargs)

    def Csys(self, kcn="", **kwargs):
        """
        APDL Command: CSYS

        Activates a previously defined coordinate system.

        Parameters
        ----------
        kcn
            Specifies the active coordinate system, as follows:

            0 (default) - Cartesian

            1 - Cylindrical with global Cartesian Z as the axis of rotation

            2 - Spherical

            4 or WP - Working Plane

            5 - Cylindrical with global Cartesian Y as the axis of rotation

            11 or greater - Any previously defined local coordinate system

        Notes
        -----
        The CSYS command activates a previously defined coordinate system for
        geometry input and generation.  The LOCAL, CLOCAL, CS, CSKP, and CSWPLA
        commands also activate coordinate systems as they are defined. To set
        the active element coordinate system attribute pointer, issue the ESYS
        command.

        The active coordinate system for files created via the CDWRITE command
        is Cartesian (CSYS,0).

        This command is valid in any processor.

        CSYS,4 (or CSYS,WP) activates working plane tracking, which updates the
        coordinate system to follow working plane changes. To deactivate
        working plane tracking, activate any other coordinate system (for
        example, CSYS,0 or CSYS,11).

        CSYS,5 is a cylindrical coordinate system with global Cartesian Y as
        the axis. The local x, y and z axes are radial, θ, and axial
        (respectively). The R-Theta plane is the global X-Z plane, as it is for
        an axisymmetric model. Thus, at θ = 0.0, CSYS,5 has a specific
        orientation: the local x is in the global +X direction, local y is in
        the global -Z direction, and local z (the cylindrical axis) is in the
        global +Y direction.

        """
        command = "CSYS,%s" % (str(kcn))
        self.RunCommand(command, **kwargs)

    def Asub(self, na1="", p1="", p2="", p3="", p4="", **kwargs):
        """
        APDL Command: ASUB

        Generates an area using the shape of an existing area.

        Parameters
        ----------
        na1
            Existing area number whose shape is to be used.  If P1 = P,
            graphical picking is enabled and all remaining arguments are
            ignored (valid only in the GUI).

        p1
            Keypoint defining starting corner of area.

        p2
            Keypoint defining second corner of area.

        p3
            Keypoint defining third corner of area.

        p4
            Keypoint defining fourth corner of area (defaults to P3).

        Notes
        -----
        The new area will overlay the old area.  Often used when the area to be
        subdivided consists of a complex shape that was not generated in a
        single coordinate system.  Keypoints and any corresponding lines must
        lie on the existing area.  Missing lines are generated to lie on the
        given area.  The active coordinate system is ignored.

        """
        command = "ASUB,%s,%s,%s,%s,%s" % (str(na1), str(p1), str(p2), str(p3), str(p4))
        self.RunCommand(command, **kwargs)

    def Mfoutput(self, freq="", **kwargs):
        """
        APDL Command: MFOUTPUT

        Specifies results file output frequency for an ANSYS Multi-field solver
        analysis.

        Parameters
        ----------
        freq
               N

             N - Write solution every Nth (and the last) time step. Defaults to 1, for every
                 time step.

            -N - Writes up to -N equally spaced results (for multifield auto time stepping).

             NONE - Suppresses writing of results for all multifield time steps.

             ALL - Writes results for every multifield time step (default).

             LAST - Writes results for only the last multifield time step.

            %array% - Where %array% is the name of an n X 1 X 1 dimensional array parameter defining
                      n key times, the data for the specified solution results
                      item is written at those key times. Key times in the
                      array parameter must appear in ascending order. Value
                      must be greater than or equal to the ending time values
                      for the load step.

            For restart runs (see MFRSTART command), either change the parameter values to fall between the beginning and ending time values of the load step, or erase the current settings and reissue the command with a new array parameter.  - For more information about defining array parameters, see the *DIM command
                              documentation.

        Notes
        -----
         A MFOUTPUT setting overrides any other output frequency setting
        (OUTRES). To select the solution items, use the OUTRES command.

        For the case of Freq = -n and Freq = %array%, the results at the time
        points which first time reaches or exceeds the targeting ouptupt time
        points will be written.

        This command is also valid in PREP7.

        See Multi-field Commands in the Coupled-Field Analysis Guide for a list
        of all ANSYS Multi-field solver commands and their availability for MFS
        and MFX analyses.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MFOUTPUT,%s" % (str(freq))
        self.RunCommand(command, **kwargs)

    def Sspb(self, b11="", b21="", b31="", b22="", b32="", b33="", t="",
             b12="", b13="", b23="", **kwargs):
        """
        APDL Command: SSPB

        Specifies a preintegrated coupling stiffness for shell sections.

        Parameters
        ----------
        b11, b21, b31, b22, b32, b33
            Coupling stiffness component (symmetric lower part of submatrix
            [B]).

        t
            Temperature.

        b12, b13, b23
            Upper part of submatrix [B]

        Notes
        -----
        The behavior of shell elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        If the coefficients B12, B13, B23 are undefined, ANSYS uses a symmetric
        form of submatrix [B].  If any one of the coefficients B12, B13, B23 is
        nonzero, ANSYS considers submatrix [B] to be unsymmetric.

        The SSPB command, one of several preintegrated shell section commands,
        specifies the coupling stiffness quantity (submatrix [B] data) for a
        preintegrated shell section. The section data defined is associated
        with the section most recently defined (via the SECTYPE command).

        Unspecified values default to zero.

        Related commands are SSPA, SSPD, SSPE, SSMT, SSBT, and SSPM.

        If you are using the SHELL181 or SHELL281 element's Membrane option
        (KEYOPT(1) = 1), it is not necessary to issue this command.

        For complete information, see Using Preintegrated General Shell
        Sections.

        """
        command = "SSPB,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(b11), str(b21), str(b31), str(b22), str(b32), str(b33), str(t), str(b12), str(b13), str(b23))
        self.RunCommand(command, **kwargs)

    def Parain(self, name="", extension="", path="", entity="", fmt="",
               scale="", **kwargs):
        """
        APDL Command: ~PARAIN

        Transfers a Parasolid file into the ANSYS program.

        Parameters
        ----------
        name
            The name of a valid Parasolid file. The first character of the file
            name must be an alphanumeric.

        extension
            The extension for the file. The default extension is .x_t on a PC
            or .xmt_txt on a Linux system. Parasolid files are compatible
            across systems, and do not need to be renamed to be used on another
            platform.

        path
            The path name of the directory in which the file resides, enclosed
            in single quotes. The default path name is the current working
            directory.

        entity
            Entity to be imported:

            SOLIDS - Solids only, imported as ANSYS volumes (default)

            SURFACES - Surfaces only, imported as ANSYS areas.

            WIREFRAME - Wireframe only, imported as ANSYS lines.

            ALL - All entities. Use this option when the file contains more than one type of
                  entity.

        fmt
            Sets the format in which ANSYS will store the model

            0 - Neutral format (default). Defeaturing after import is restricted. Use this
                option if you need to scale a model to a specific unit of
                measure (other than meters).

            1 - Solid format; this allows defeaturing after import.

        scale
            Allows scaling for the model

            0 - Do not rescale the model; retain the default Parasolid setting of meters
                (default).

            1 - Scale the model if warranted by the model size.

        Notes
        -----
        More information on importing Parasolid parts is available in Parasolid
        in the Connection User's Guide.

        """
        command = "~PARAIN,%s,%s,%s,%s,%s,%s" % (str(name), str(extension), str(path), str(entity), str(fmt), str(scale))
        self.RunCommand(command, **kwargs)

    def Cncheck(self, option="", rid1="", rid2="", rinc="", intertype="",
                trlevel="", cgap="", cpen="", ioff="", **kwargs):
        """
        APDL Command: CNCHECK

        Provides and/or adjusts the initial status of contact pairs.

        Parameters
        ----------
        option
            Option to be performed:

            DETAIL - List all contact pair properties (default).

            SUMMARY - List only the open/closed status for each contact pair.

            POST - Execute a partial solution to write the initial contact configuration to the
                   Jobname.RCN file.

            ADJUST - Physically move contact nodes to the target in order to close a gap or reduce
                     penetration. The initial adjustment is converted to
                     structural displacement values (UX, UY, UZ) and stored in
                     the Jobname.RCN file.

            RESET - Reset target element and contact element key options and real constants to
                    their default values. This option is not valid for general
                    contact.

            AUTO - Automatically sets certain real constants and key options to recommended values
                   or settings in order to achieve better convergence based on
                   overall contact pair behaviors. This option is not valid for
                   general contact.

            TRIM - Trim contact pair (remove certain contact and target elements).

            UNSE - Unselect certain contact and target elements.

        rid1, rid2, rinc
            For pair-based contact, the range of real constant pair ID's for
            which Option will be performed. If RID2 is not specified, it
            defaults to RID1. If no value is specified, all contact pairs in
            the selected set of elements are considered.

        intertype
            The type of contact interface (pair-based versus general contact)
            to be considered; or the type of contact pair to be
            trimmed/unselected/auto-set.

            (blank) - Include all contact definitions (pair-based and general contact).

            GCN - Include general contact definitions only (not valid when Option = RESET or
                  AUTO).

        trlevel
            Trimming level (used only when Option = TRIM or UNSE):

            (blank) - Normal trimming (default): remove/unselect contact and target elements which
                      are in far-field.

            AGGRE - Aggressive trimming: remove/unselect contact and target elements which are in
                    far-field, and certain elements in near-field.

        Notes
        -----
        The CNCHECK command provides information for surface-to-surface, node-
        to-surface, and line-to-line contact pairs (element types TARGE169,
        TARGE170, CONTA171, CONTA172, CONTA173, CONTA174, CONTA175, CONTA176,
        CONTA177). All contact and target elements of interest, along with the
        solid elements and nodes attached to them, must be selected for the
        command to function properly. For performance reasons, the program uses
        a subset of nodes and elements based on the specified contact regions
        (RID1, RID2, RINC) when executing the CNCHECK command.

        CNCHECK is available in both the PREP7 and SOLUTION processors, but
        only before the first solve operation (that is, only before the first
        load step or the first substep).

        If the contact and target elements were generated through mesh commands
        (AMESH, LMESH, etc.) instead of the ESURF command, you must issue
        MODMSH,DETACH before CNCHECK. Otherwise, CNCHECK will not work
        correctly.

        The following additional notes are available:

        The command CNCHECK,POST solves the initial contact configuration in
        one substep. After issuing this command, you can postprocess the
        contact result items as you would for any other converged load step;
        however, only the contact status, contact penetration or gap, and
        contact pressure will have meaningful values. Other contact quantities
        (friction stress, sliding distance, chattering) will be available but
        are not useful.

        Because Option = POST forces a solve operation, the PrepPost (PP)
        license does not work with CNCHECK,POST.

        If CNCHECK,POST is issued within the solution processor, the SOLVE
        command that solves the first load step of your analysis should appear
        in a different step, as shown in the following example:

        CNCHECK,POST writes initial contact results to a file named
        Jobname.RCN.  When postprocessing the initial contact state, you need
        to explicitly read results from this file using the FILE and SET,FIRST
        commands in POST1 to properly read the corresponding contact data.
        Otherwise, the results may be read improperly. The following example
        shows a valid command sequence for plotting the initial contact gap:

        You can issue CNCHECK,ADJUST to physically move contact nodes to the
        target surface. See Physically Moving Contact Nodes Towards the Target
        Surface in the Contact Technology Guide for more information. Similar
        to the POST option, if CNCHECK,ADJUST is issued within the solution
        processor, the SOLVE command that solves the first load step of your
        analysis should appear in a different step:

        After issuing the CNCHECK,ADJUST command, the initial adjustment is
        converted to structural displacement values (UX, UY, UZ) and stored in
        a file named Jobname.RCN. You can use this file to plot or list nodal
        adjustment vectors or create a contour plot of the adjustment
        magnitudes via the displacements. When postprocessing the nodal
        adjustment values, you need to explicitly read results from this file
        using the FILE and SET,FIRST commands in POST1 to properly read the
        corresponding contact data. Otherwise, the results may be read
        improperly.

        Note:: : The Jobname.RCN file contains information generated from the
        CNCHECK,POST or CNCHECK,ADJUST command. If both commands are issued in
        the same analysis, the file is overwritten by the last CNCHECK command.

        The command CNCHECK,RESET allows you to reset all but a few key options
        and real constants associated with the specified contact pairs (RID1,
        RID2, RINC) to their default values. This option is only valid for
        pair-based contact definitions.

        The following key options and real constants remain unchanged when this
        command is issued:

        The command CNCHECK,AUTO automatically changes certain default or
        undefined key options and real constants to optimized settings or
        values. The changes are based on overall contact pair behaviors. In
        general, this command improves convergence for nonlinear contact
        analysis. This option is only valid for pair-based contact definitions.

        The tables below list typical KEYOPT and real constant settings
        implemented by CNCHECK,AUTO. The actual settings implemented for your
        specific model may vary from what is described here. You should always
        verify the modified settings by issuing CNCHECK,DETAIL to list current
        contact pair properties.

        Set to 0 if KEYOPT(2) > 1 for debonding.

        Set to 0 if underlying elements are superelements, or if KEYOPT(9) = 2
        was previously specified.

        PINB default depends on contact behavior (rigid vs. flexible target),
        NLGEOM,ON or OFF, KEYOPT(9) setting, KEYOPT(12) setting, and the value
        of real constant CNOF (see Using PINB).

        CNCHECK,AUTO also sets PRED,OFF for the case of a force-distributed
        constraint defined via MPC contact.

        You can issue CNCHECK,TRIM or CNCHECK,UNSEL to remove or unselect
        contact and target elements which are in far-field (that is, open and
        not near contact), or even in near-field if aggressive trimming logic
        is used (TRlevel = AGGRE).

        """
        command = "CNCHECK,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(option), str(rid1), str(rid2), str(rinc), str(intertype), str(trlevel), str(cgap), str(cpen), str(ioff))
        self.RunCommand(command, **kwargs)

    def Cpdele(self, nset1="", nset2="", ninc="", nsel="", **kwargs):
        """
        APDL Command: CPDELE

        Deletes coupled degree of freedom sets.

        Parameters
        ----------
        nset1, nset2, ninc
            Delete coupled sets from NSET1 to NSET2 (defaults to NSET1) in
            steps of NINC (defaults to 1).  If NSET1 = ALL, NSET2 and NINC are
            ignored and all coupled sets are deleted.

        nsel
            Additional node selection control:

            ANY - Delete coupled set if any of the selected nodes are in the set (default).

            ALL - Delete coupled set only if all of the selected nodes are in the set.

        Notes
        -----
        See the CP command for a method to delete individual nodes from a set.

        """
        command = "CPDELE,%s,%s,%s,%s" % (str(nset1), str(nset2), str(ninc), str(nsel))
        self.RunCommand(command, **kwargs)

    def Batch(self, lab="", **kwargs):
        """
        APDL Command: /BATCH

        Sets the program mode to "batch."

        Parameters
        ----------
        lab
            Specifies listing mode during a batch run:

            LIST - The batch output will include a listing of the input file.

            (blank) - Suppress input data listing.

        Notes
        -----
        Sets the program mode to "batch" when included as the first line on an
        input file of ANSYS commands.  For convenience, this command is
        automatically recorded on the log file (Jobname.LOG) at the beginning
        of an interactive ANSYS session so that the log file can be reused
        later for batch input.

        Caution:: : This command  should not be entered directly in an
        interactive ANSYS session since all subsequent keyboard input is simply
        copied to a file, without further processing or prompts from the
        program (use the "system break" to exit the ANSYS program if this
        occurs).

        The recommended method for choosing batch mode, rather than using the
        /BATCH command, is to select the Batch simulation environment from the
        ANSYS Product Launcher  task in the ANSYS launcher, or the batch mode
        entry option on the ANSYS execution command when entering the program.

        This command is valid only at the Begin Level.

        """
        command = "/BATCH,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Ssbt(self, bt11="", bt22="", bt12="", t="", **kwargs):
        """
        APDL Command: SSBT

        Specifies preintegrated bending thermal effects for shell sections.

        Parameters
        ----------
        bt11, bt22, bt12
            Bending thermal effects component [BT].

        t
            Temperature.

        Notes
        -----
        The behavior of shell elements is governed by the generalized-
        stress/generalized-strain relationship of the form:

        The SSBT command, one of several preintegrated shell section commands,
        specifies the bending thermal effects quantity (submatrix [BT] data)
        for a preintegrated shell section. The section data defined is
        associated with the section most recently defined (via the SECTYPE
        command).

        The [BT] quantity represents bending stress resultants caused by a unit
        raise in temperature on a fully constrained model. For a layered
        composite shell, it is usually necessary to specify both the [BT] and
        [MT] quantities (by issuing the SSBT and SSMT commands, respectively).

        Unspecified values default to zero.

        Related commands are SSPA, SSPB, SSPD, SSPE, SSMT, and SSPM.

        If you are using the SHELL181 or SHELL281 element's Membrane option
        (KEYOPT(1) = 1), it is not necessary to issue this command.

        For complete information, see Using Preintegrated General Shell
        Sections.

        """
        command = "SSBT,%s,%s,%s,%s" % (str(bt11), str(bt22), str(bt12), str(t))
        self.RunCommand(command, **kwargs)

    def Edpart(self, option="", partid="", cname="", **kwargs):
        """
        APDL Command: EDPART

        Configures parts for an explicit dynamics analysis.

        Parameters
        ----------
        option
            Option used to organize parts. (No default; Option must be
            specified.)

            CREATE - Creates new PART IDs assigned to groups of elements with unique combinations of
                     MAT, TYPE, and REAL set numbers. If this option is issued
                     repeatedly, the part list is overwritten, except for PART
                     IDs created with the ADD option. Remaining fields are
                     ignored for this option.

            UPDATE - Updates the PART IDs for the element groups without changing the order of the
                     existing part list. If elements are redefined (or new
                     elements are created) with different MAT, TYPE, or REAL
                     set numbers, then use this option to create an updated
                     list of PART IDs. Remaining fields are ignored for this
                     option.

            ADD - Assigns a user-specified PART ID (PARTID) to the elements contained in the
                  element component Cname, or to the currently selected set of
                  elements if Cname = ALL. Use this option to assign a specific
                  PART ID to an element group that has the same combination of
                  MAT, TYPE, and REAL set numbers. An UPDATE operation is
                  automatically performed on the currently selected set of
                  elements immediately following the ADD operation.

            DELE - Deletes a PART ID assigned by the ADD option. PARTID is also required. An
                   UPDATE operation is automatically performed on the currently
                   selected set of elements immediately following the DELE
                   operation.

            LIST - Lists the PART IDs for the element groups. The part list consists of five
                   columns of numbers, one each for PART, MAT, TYPE, and REAL
                   numbers, and one to indicate if the PART ID is used
                   (including how many elements use it). The part list is based
                   on the last CREATE or UPDATE operation. Remaining fields are
                   ignored for this option.

        partid
            A positive integer to be used as PART ID for the elements specified
            by Cname (no default). The number input must not be currently used
            for an existing part (except when Option = DELE). Any previously
            defined PART IDs for the elements, whether assigned by the user or
            created by ANSYS LS-DYNA, will be overwritten. The user-specified
            PART ID will not be changed by subsequent EDPART,CREATE or
            EDPART,UPDATE commands.

        cname
            Element component name for user-specified PART ID definition
            (Option = ADD). If Cname = ALL (default), all currently selected
            elements are considered for the part. The elements in the element
            component (or the currently selected set of elements if Cname = ALL
            or blank) must have the same combination of MAT, TYPE, and REAL set
            numbers, or the ADD option will be ignored.

        Notes
        -----
        Certain ANSYS LS-DYNA commands (such as EDCGEN, EDLOAD, EDREAD, etc.)
        refer to PART IDs. You must define PART IDs (EDPART,CREATE or
        EDPART,ADD) before using these commands.

        If parts are repeatedly created using Option = CREATE, the part list is
        continuously overwritten. This may cause problems for previously
        defined commands that reference a part number that has changed. To
        avoid this problem, the part list should be updated (Option = UPDATE)
        rather than recreated to obtain the current part list.

        EDPART,ADD allows you to assign a specific part number to a group of
        elements instead of a number generated by the ANSYS LS-DYNA program.
        The user-specified PART IDs will not be changed by subsequent
        EDPART,CREATE or EDPART,UPDATE commands. Thus, you can use EDPART,ADD
        to specify PART IDs for some element groups, and use EDPART,CREATE or
        EDPART,UPDATE to assign PART IDs for the remaining element groups. Use
        EDPART,DELE to delete a PART ID generated by the ADD option. In this
        case, ANSYS LS-DYNA will generate a new PART ID for those elements
        associated with the deleted PART ID.

        After creating or updating the part list, use EDPART,LIST to list the
        PART IDs and choose the correct one for use with other ANSYS LS-DYNA
        commands. For a detailed discussion on PART IDs, see The Definition of
        Part in the ANSYS LS-DYNA User's Guide.

        This command is also valid in SOLUTION.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDPART,%s,%s,%s" % (str(option), str(partid), str(cname))
        self.RunCommand(command, **kwargs)

    def Vddam(self, vf="", va="", vb="", vc="", **kwargs):
        """
        APDL Command: VDDAM

        Specifies the velocity spectrum computation constants for the analysis
        of shock resistance of shipboard structures.

        Parameters
        ----------
        vf
            Direction-dependent velocity coefficient for elastic or elastic-
            plastic analysis option (Default = 0).

        va, vb, vc
            Coefficients for the DDAM velocity spectrum equations.  See Dynamic
            Design Analysis Method in the Mechanical APDL Theory Reference.
            Default for these coefficients is zero.

        Notes
        -----
        This command specifies velocity coefficients to analyze shock
        resistance of shipboard equipment.  These coefficients are used to
        compute mode coefficients according to the equations given in Dynamic
        Design Analysis Method in the Mechanical APDL Theory Reference.  The
        form of these equations is based on the Naval NRL Dynamic Design
        Analysis Method.  This command, along with the ADDAM and SED commands,
        is used with the spectrum (ANTYPE,SPECTR) analysis as a special purpose
        alternative to the SV, FREQ, and SVTYP commands.  The mass and length
        units of the model must be in pounds and inches, respectively.

        DDASPEC may alternatively be used to calculate spectrum coefficients.

        This command is also valid in PREP7.

        """
        command = "VDDAM,%s,%s,%s,%s" % (str(vf), str(va), str(vb), str(vc))
        self.RunCommand(command, **kwargs)

    def Dmat(self, matrix="", type="", method="", val1="", val2="", val3="",
             val4="", val5="", **kwargs):
        """
        APDL Command: *DMAT

        Creates a dense matrix.

        Parameters
        ----------
        matrix
            Name used to identify the matrix. Must be specified.

        type
            Matrix type:

            Double precision real values (default). - Complex double precision values.

        method
            Method used to create the matrix:

            Allocate space for a matrix (default). - Resize an existing matrix to new row and column dimensions. Values are kept
                              from the original matrix. If the dimensions
                              specified by Val1 (rows) and Val2 (columns) are
                              greater than the original matrix size, the
                              additional entries are assigned a value of zero.

            Copy an existing matrix. - Link to an existing matrix. The memory will be shared between the original
                              matrix and the new matrix. This is useful for
                              manipulating a submatrix of a larger matrix. The
                              Val1 through Val5 arguments will be used to
                              specify the lower and upper bounds of row and
                              column numbers from the original matrix.

        val1, val2, val3, val4, val5
            Additional input. The meaning of Val1 through Val5 will vary
            depending on the specified Method. See details below.

        Notes
        -----
        This command allows you to create a dense matrix. To create a sparse
        matrix, use the *SMAT command. *SMAT is recommended for large matrices
        obtained from the .FULL or .HBMAT file. Refer to the HBMAT command
        documentation for more information about .FULL file contents.

        Use the *VEC command to create a vector.

        For very large matrices, use the OUTOFCORE option (Method = ALLOC or
        COPY) to keep some of the matrix on disk if there is insufficient
        memory.

        When importing a dense matrix from a DMIG file, you can define the
        formatting of the file using the Val3 and Val4 fields. Here are a few
        different example of formats:

        A formatted file (using Val3=’F’, and Val4=8):

        A non-formatted file with blank separators:

        A non-formatted file with a comma separator (using Val3=’,’):

        """
        command = "*DMAT,%s,%s,%s,%s,%s,%s,%s,%s" % (str(matrix), str(type), str(method), str(val1), str(val2), str(val3), str(val4), str(val5))
        self.RunCommand(command, **kwargs)

    def Ui(self, func="", type="", format="", screen="", color="", krev="",
           orient="", compress="", quality="", **kwargs):
        """
        APDL Command: /UI

        Activates specified GUI dialog boxes.

        Parameters
        ----------
        func
            Label identifying the dialog box to be activated:

            HELP - Activates the online help system. Valid only in non-UI graphics mode
                   (/MENU,GRPH).

            VIEW - Activates the Pan, Zoom, Rotate dialog box

            WPSE - Activates the Working Plane Settings dialog box.

            WPVI - Activates the Offset Working Plane dialog box.

            RESULT - Activates the Query Picking Menu for reviewing results.

            QUERY - Activates the Query Picked Entities (preprocess) dialog box.

            COPY - Activates the Hard Copy dialog box.

            ANNO - Activates the 2D Annotation dialog box.

            AN3D - Activates the 3D Annotation dialog box.

            SELECT - Activates the Select Entities dialog box.

            NSEL - Activates a picking menu to select nodes.

            ESEL - Activates a picking menu to select elements.

            KSEL - Activates a picking menu to select keypoints.

            LSEL - Activates a picking menu to select lines.

            ASEL - Activates a picking menu to select areas.

            VSEL - Activates a picking menu to select volumes.

            REFRESH - Refreshes the graphics window (non-UI mode only).

            COLL - Controls the collapse of the ANSYS Main Menu when a FINISH command is issued.
                   See Type below for a discussion of the arguments.

        type
            Label identifying the type of select operation.  Valid only for the
            following Func labels; NSEL, ESEL, KSEL, LSEL, ASEL, and VSEL:

            S - Select a new set.

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

        Notes
        -----
        Allows you to activate specified GUI dialog boxes directly in either
        GUI or non-GUI mode.

        The /UI command itself is valid in any processor, however certain
        dialog boxes are accessible only in a particular processor (e.g.,
        /UI,RESULT,... is valid only in the General Postprocessor).

        ANSYS JPEG software is based in part on the work of the Independent
        JPEG Group, Copyright 1998, Thomas G. Lane.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "/UI,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(func), str(type), str(format), str(screen), str(color), str(krev), str(orient), str(compress), str(quality))
        self.RunCommand(command, **kwargs)

    def Eqslv(self, lab="", toler="", mult="", keepfile="", **kwargs):
        """
        APDL Command: EQSLV

        Specifies the type of equation solver.

        Parameters
        ----------
        lab
            Equation solver type:

            SPARSE - Sparse direct equation solver.  Applicable to real-value or complex-value
                     symmetric and unsymmetric matrices. Available only for
                     STATIC, HARMIC (full method only), TRANS (full method
                     only), SUBSTR, and PSD spectrum analysis types [ANTYPE].
                     Can be used for nonlinear and linear analyses, especially
                     nonlinear analysis where indefinite matrices are
                     frequently encountered. Well suited for contact analysis
                     where contact status alters the mesh topology. Other
                     typical well-suited applications are: (a) models
                     consisting of shell/beam or shell/beam and solid elements
                     (b) models with a multi-branch structure, such as an
                     automobile exhaust or a turbine fan. This is an
                     alternative to iterative solvers since it combines both
                     speed and robustness. Generally, it requires considerably
                     more memory (~10x) than the PCG solver to obtain optimal
                     performance (running totally in-core). When memory is
                     limited, the solver works partly in-core and out-of-core,
                     which can noticeably slow down the performance of the
                     solver. See the BCSOPTION command for more details on the
                     various modes of operation for this solver.

            This solver can be run in shared memory parallel or distributed memory parallel (Distributed ANSYS) mode. When used in Distributed ANSYS, this solver preserves all of the merits of the classic or shared memory sparse solver. The total sum of memory (summed for all processes) is usually higher than the shared memory sparse solver. System configuration also affects the performance of the distributed memory parallel solver. If enough physical memory is available, running this solver in the in-core memory mode achieves optimal performance. The ideal configuration when using the out-of-core memory mode is to use one processor per machine on multiple machines (a cluster), spreading the I/O across the hard drives of each machine, assuming that you are using a high-speed network such as Infiniband to efficiently support all communication across the multiple machines.  - This solver supports use of the GPU accelerator capability.

            JCG - Jacobi Conjugate Gradient iterative equation solver. Available only for STATIC,
                  HARMIC (full method only), and TRANS (full method only)
                  analysis types [ANTYPE]. Can be used for structural, thermal,
                  and multiphysics applications. Applicable for symmetric,
                  unsymmetric, complex, definite, and indefinite matrices.
                  Recommended for 3-D harmonic analyses in structural and
                  multiphysics applications. Efficient for heat transfer,
                  electromagnetics, piezoelectrics, and acoustic field
                  problems.

            This solver can be run in shared memory parallel or distributed memory parallel (Distributed ANSYS) mode. When used in Distributed ANSYS, in addition to the limitations listed above, this solver only runs in a distributed parallel fashion for STATIC and TRANS (full method) analyses in which the stiffness is symmetric and only when not using the fast thermal option (THOPT). Otherwise, this solver runs in shared memory parallel mode inside Distributed ANSYS. - This solver supports use of the GPU accelerator capability. When using the GPU
                              accelerator capability, in addition to the
                              limitations listed above, this solver is
                              available only for STATIC and TRANS (full method)
                              analyses where the stiffness is symmetric and
                              does not support the fast thermal option (THOPT).

            ICCG - Incomplete Cholesky Conjugate Gradient iterative equation solver. Available for
                   STATIC, HARMIC (full method only), and TRANS (full method
                   only) analysis types [ANTYPE].  Can be used for structural,
                   thermal, and multiphysics applications, and for symmetric,
                   unsymmetric, complex, definite, and indefinite matrices. The
                   ICCG solver requires more memory than the JCG solver, but is
                   more robust than the JCG solver for ill-conditioned
                   matrices.

            This solver can only be run in shared memory parallel mode. This is also true when the solver is used inside Distributed ANSYS. - This solver does not support use of the GPU accelerator capability.

            QMR - Quasi-Minimal Residual iterative equation solver. Available for the HARMIC
                  (full method only) analysis type [ANTYPE]. Can be used for
                  high-frequency electromagnetic applications, and for
                  symmetric, complex, definite, and indefinite matrices. The
                  QMR solver is more stable than the ICCG solver.

            This solver can only be run in shared memory parallel mode. This is also true when the solver is used inside Distributed ANSYS. - This solver does not support use of the GPU accelerator capability.

            PCG - Preconditioned Conjugate Gradient iterative equation solver (licensed from
                  Computational Applications and Systems Integration, Inc.).
                  Requires less disk file space than SPARSE and is faster for
                  large models. Useful for plates, shells, 3-D models, large
                  2-D models, and other problems having symmetric, sparse,
                  definite or indefinite matrices for nonlinear analysis.
                  Requires twice as much memory as JCG. Available only for
                  analysis types [ANTYPE] STATIC, TRANS (full method only), or
                  MODAL (with PCG Lanczos option only). Also available for the
                  use pass of substructure analyses (MATRIX50). The PCG solver
                  can robustly solve equations with constraint equations (CE,
                  CEINTF, CPINTF, and CERIG).  With this solver, you can use
                  the MSAVE command to obtain a considerable memory savings.

            The PCG solver can handle ill-conditioned problems by using a higher level of difficulty (see PCGOPT). Ill-conditioning arises from elements with high aspect ratios, contact, and plasticity. - This solver can be run in shared memory parallel or distributed memory parallel
                              (Distributed ANSYS) mode. When used in
                              Distributed ANSYS, this solver preserves all of
                              the merits of the classic or shared memory PCG
                              solver. The total sum of memory (summed for all
                              processes) is about 30% more than the shared
                              memory PCG solver.

        toler
            Iterative solver tolerance value. Used only with the Jacobi
            Conjugate Gradient, Incomplete Cholesky Conjugate Gradient, Pre-
            conditioned Conjugate Gradient, and Quasi-Minimal Residual equation
            solvers. For the PCG solver, the default is 1.0E-8. The value
            1.0E-5 may be acceptable in many situations. When using the PCG
            Lanczos mode extraction method, the default solver tolerance value
            is 1.0E-4. For the JCG and ICCG solvers with symmetric matrices,
            the default is 1.0E-8. For the JCG and ICCG solvers with
            unsymmetric matrices, and for the QMR solver, the default is
            1.0E-6. Iterations continue until the SRSS norm of the residual is
            less than TOLER times the norm of the applied load vector. For the
            PCG solver in the linear static analysis case, 3 error norms are
            used. If one of the error norms is smaller than TOLER, and the SRSS
            norm of the residual is smaller than 1.0E-2, convergence is assumed
            to have been reached. See Iterative Solver in the Mechanical APDL
            Theory Reference for details.

        mult
            Multiplier (defaults to 2.5 for nonlinear analyses; 1.0 for linear
            analyses) used to control the maximum number of iterations
            performed during convergence calculations. Used only with the Pre-
            conditioned Conjugate Gradient equation solver (PCG). The maximum
            number of iterations is equal to the multiplier (MULT) times the
            number of degrees of freedom (DOF). If MULT is input as a negative
            value, then the maximum number of iterations is equal to abs(MULT).
            Iterations continue until either the maximum number of iterations
            or solution convergence has been reached. In general, the default
            value for MULT is adequate for reaching convergence.  However, for
            ill-conditioned matrices (that is, models containing elements with
            high aspect ratios or material type discontinuities) the multiplier
            may be used to increase the maximum number of iterations used to
            achieve convergence.  The recommended range for the multiplier is
            1.0 MULT 3.0.  Normally, a value greater than 3.0 adds no further
            benefit toward convergence, and merely increases time requirements.
            If the solution does not converge with 1.0 MULT 3.0, or in less
            than 10,000 iterations, then convergence is highly unlikely and
            further examination of the model is recommended. Rather than
            increasing the default value of MULT, consider increasing the level
            of difficulty (Lev_Diff) on the PCGOPT command.

        --
            Unused field.

        keepfile
            Determines whether files from a SPARSE solver run should be deleted
            or retained. Applies only to Lab = SPARSE for static and full
            transient analyses.

            DELE  - Deletes all files from the SPARSE solver run, including the factorized file,
                    .LN22, upon FINISH or /EXIT (default).

            KEEP  - Retains all necessary files from the SPARSE solver run, including the .LN22
                    file, in the working directory.

        Notes
        -----
        The selection of a solver can affect the speed and accuracy of a
        solution. For a more detailed discussion of the merits of each solver,
        see Solution in the Basic Analysis Guide.

        You may only specify the solver type in the first load step. You may,
        however, modify the solver tolerance in subsequent load steps for the
        iterative solvers.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: All equation solvers are supported in
        Distributed ANSYS. However, the SPARSE and PCG solvers are the only
        distributed solvers that always run a fully distributed solution. The
        JCG solver runs in a fully distributed mode in some cases; in other
        cases, it does not. The ICCG and QMR solvers are not distributed
        solvers; therefore, you will not see the full performance improvements
        with these solvers that you would with a fully distributed solution.

        The ANSYS DesignSpace (DS) product does not support distributed
        solutions (Distributed ANSYS).

        """
        command = "EQSLV,%s,%s,%s,%s" % (str(lab), str(toler), str(mult), str(keepfile))
        self.RunCommand(command, **kwargs)

    def Elist(self, iel1="", iel2="", inc="", nnkey="", rkey="", ptkey="",
              **kwargs):
        """
        APDL Command: ELIST

        Lists the elements and their attributes.

        Parameters
        ----------
        iel1, iel2, inc
            Lists elements from IEL1 to IEL2 (defaults to IEL1) in steps of INC
            (defaults to 1). If IEL1 = ALL (default), IEL2 and INC are ignored
            and all selected elements [ESEL] are listed. If IEL1 = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI). A component name may also be substituted
            for IEL1 (IEL2 and INC are ignored).

        nnkey
            Node listing key:

            0 - List attribute references and nodes.

            1 - List attribute references but not nodes.

        rkey
            Real constant listing key:

            0 - Do not show real constants for each element.

            1 - Show real constants for each element. This includes default values chosen for
                the element.

        ptkey
            LS-DYNA part number listing key (applicable to ANSYS LS-DYNA only):

            0 - Do not show part ID number for each element.

            1 - Show part ID number for each element.

        Notes
        -----
        Lists the elements with their nodes and attributes (MAT, TYPE, REAL,
        ESYS, SECNUM, PART). See also the LAYLIST command for listing layered
        elements.

        This command is valid in any processor.

        """
        command = "ELIST,%s,%s,%s,%s,%s,%s" % (str(iel1), str(iel2), str(inc), str(nnkey), str(rkey), str(ptkey))
        self.RunCommand(command, **kwargs)

    def Eshape(self, scale="", key="", **kwargs):
        """
        APDL Command: /ESHAPE

        Displays elements with shapes determined from the real constants or
        section definition.

        Parameters
        ----------
        scale
            Scaling factor:

            0 - Use simple display of line and area elements. This value is the default.

            1 - Use real constants or section definition to form a solid shape display of the
                applicable elements.

            FAC - Multiply certain real constants, such as thickness, by FAC (where FAC > 0.01)
                  and use them to form a solid shape display of elements.

        key
            Current shell thickness key:

            0 - Use current thickness in the displaced solid shape display of shell elements
                (valid for SHELL181, SHELL208, SHELL209, and SHELL281). This
                value is the default.

            1 - Use initial thickness in the displaced solid shape display of shell elements.

        Notes
        -----
        The /ESHAPE command allows beams, shells, current sources, and certain
        special-purpose elements to be displayed as solids with the shape
        determined from the real constants or section types. Elements are
        displayed via the EPLOT command. No checks for valid or complete input
        are made for the display.

        Following are details about using this command with various element
        types:

        SOLID65 elements are displayed with internal lines that represent rebar
        sizes and orientations (requires vector mode [/DEVICE] with a basic
        type of display [/TYPE,,BASIC]). The rebar with the largest volume
        ratio in each element plots as a red line, the next largest as green,
        and the smallest as blue.

        COMBIN14, COMBIN39, and MASS21 are displayed with a graphics icon, with
        the offset determined by the real constants and KEYOPT settings.

        BEAM188, BEAM189, PIPE288, PIPE289 and ELBOW290 are displayed as solids
        with the shape determined via the section-definition commands (SECTYPE
        and SECDATA). The arbitrary section option (Subtype = ASEC) has no
        definite shape and appears as a thin rectangle to show orientation. The
        elements are displayed with internal lines representing the cross-
        section mesh.

        SOLID272 and SOLID273 are displayed as solids with the shape determined
        via the section-definition commands (SECTYPE and SECDATA).  The 2-D
        master plane is revolved around the prescribed axis of symmetry.

        Contour plots are available for these elements in postprocessing for
        PowerGraphics only (/GRAPHICS,POWER). To view 3-D deformed shapes for
        the elements, issue OUTRES,MISC or OUTRES,ALL for static or transient
        analyses. To view 3-D mode shapes for a modal or eigenvalue buckling
        analysis, expand the modes with element results calculation ON (Elcalc
        = YES for MXPAND).

        SOURC36, CIRCU124, and TRANS126 elements always plot using /ESHAPE when
        PowerGraphics is activated (/GRAPHICS,POWER).

        In most cases, /ESHAPE renders a thickness representation of your
        shell, plane and layered elements more readily in PowerGraphics
        (/GRAPHICS,POWER). This type of representation employs PowerGraphics to
        generate the enhanced representation, and will often provide no
        enhancement in Full Graphics (/GRAPHICS,FULL). This is especially true
        for POST1 results displays, where /ESHAPE is not supported for most
        element types with FULL graphics.

        When PowerGraphics is active, /ESHAPE may degrade the image if adjacent
        elements have overlapping material, such as shell elements which are
        not co-planar. Additionally, if adjacent elements have different
        thicknesses, the polygons depicting the connectivity between the
        “thicker” and “thinner” elements along the shared element edges may not
        always be displayed.

        For POST1 results displays (such as PLNSOL), the following limitations
        apply:

        Rotational displacements for beam elements are used to create a more
        realistic displacement display. When /ESHAPE is active, displacement
        plots (via PLNSOL,U,X and PLDISP, for example) may disagree with your
        PRNSOL listings. This discrepancy will become more noticeable when the
        SCALE value is not equal to one.

        When shell elements are not co-planar, the resulting PLNSOL display
        with /ESHAPE will actually be a PLESOL display as the non-coincident
        pseudo-nodes are not averaged. Additionally, /ESHAPE should not be used
        with coincident elements because the plot may incorrectly average the
        displacements of the coincident elements.

        When nodes are initially coincident and PowerGraphics is active,
        duplicate polygons are eliminated to conserve display time and disk
        space. The command may degrade the image if initially coincident nodes
        have different displacements. The tolerance for determining coincidence
        is 1E-9 times the model’s bounding box diagonal.

        If you want to view solution results (PLNSOL, etc.) on layered elements
        (such as SHELL181, SOLSH190, SOLID185 Layered Solid, SOLID186 Layered
        Solid, SHELL208, SHELL209, SHELL281, and ELBOW290), set KEYOPT(8) = 1
        for the layer elements so that the data for all layers is stored in the
        results file.

        You can plot the through-thickness temperatures of elements SHELL131
        and SHELL132 regardless of the thermal DOFs in use by issuing the
        PLNSOL,TEMP command (with PowerGraphics and /ESHAPE active).

        The /ESHAPE,1 and /ESHAPE,FAC commands are incompatible with the
        /CYCEXPAND command used in cyclic symmetry analyses.

        This command is valid in any processor.

        """
        command = "/ESHAPE,%s,%s" % (str(scale), str(key))
        self.RunCommand(command, **kwargs)

    def Write(self, fname="", **kwargs):
        """
        APDL Command: WRITE

        Writes the radiation matrix file.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        Notes
        -----
        Writes radiation matrix file (File.SUB) for input to the substructure
        thermal "use" pass.  Subsequent WRITE operations to the same file
        overwrite the file.

        """
        command = "WRITE,%s" % (str(fname))
        self.RunCommand(command, **kwargs)

    def Bfa(self, area="", lab="", val1="", val2="", val3="", val4="",
            **kwargs):
        """
        APDL Command: BFA

        Defines a body force load on an area.

        Parameters
        ----------
        area
            Area to which body load applies.  If ALL, apply to all selected
            areas [ASEL]. A component name may also be substituted for Area.

        lab
            Valid body load label. Load labels are listed under "Body Loads" in
            the input table for each element type in the Element Reference.

        val1, val2, val3
            Value associated with the Lab item or a table name for specifying
            tabular boundary conditions. Use only VAL1 for TEMP, FLUE, HGEN,
            CHRGD. For Lab = JS in magnetics, use VAL1, VAL2, and VAL3 for the
            X, Y, and Z components. For acoustics, if Lab = JS, use VAL1 for
            mass source in a harmonic analysis or mass source rate in a
            transient analysis, and ignore VAL2 and VAL3. For Lab = VLTG, VAL1
            is the voltage drop and VAL2 is the phase angle. If Lab = IMPD,
            VAL1 is the resistance and VAL2 is the reactance in ohms/square.
            When specifying a table name, you must enclose the table name in
            percent signs (%), e.g., BFA,Area,Lab,%tabname%. Use the *DIM
            command to define a table.

        val4
            If Lab = JS, VAL4 is the phase angle in degrees.

        Notes
        -----
        Defines a body force load (such as temperature in a structural
        analysis, heat generation rate in a thermal analysis, etc.) on an area.
        Body loads may be transferred from areas to area elements (or to nodes
        if area elements do not exist) with the BFTRAN or SBCTRAN commands.
        Body loads default to the value specified on the BFUNIF command, if it
        was previously specified.

        You can specify a table name only when using temperature (TEMP) and
        heat generation rate (HGEN) body load labels.

        Body loads specified by the BFA command can conflict with other
        specified body loads.  See Resolution of Conflicting Body Load
        Specifications in the Basic Analysis Guide for details.

        Graphical picking is available only via the listed menu paths.

        This command is also valid in PREP7.

        """
        command = "BFA,%s,%s,%s,%s,%s,%s" % (str(area), str(lab), str(val1), str(val2), str(val3), str(val4))
        self.RunCommand(command, **kwargs)

    def Plzz(self, rotvel="", deltarotvel="", **kwargs):
        """
        APDL Command: PLZZ

        Plots the interference diagram from a cyclic modal analysis.

        Parameters
        ----------
        rotvel
            Rotational speed in revolutions per minute (RPM) used to define the
            speed line. If blank, use the rotational speed (from OMEGA)
            specified in the prestressing step of the linear perturbation
            analysis. If explicitly input as 0, or if the linear perturbation
            was not used, no speed lines are plotted.

        deltarotvel
            Adds speed lines about the RotVel speed line corresponding to
            RotVel ± DeltaRotVel. Only plotted if RotVel is known.

        Notes
        -----
        PLZZ plots the cyclic modal frequencies as points on a frequency vs.
        harmonic index (nodal diameter) graph. If rotational speed (RotVel) is
        provided, the speed line is also plotted, leading to the interference
        diagram (also known as the SAFE or ZZENF diagram). If DeltaRotVel is
        also provided, two additional speed lines are plotted, enveloping the
        safe speed line itself.

        For more information, see Postprocessing a Modal Cyclic Symmetry
        Analysis in the Cyclic Symmetry Analysis Guide.

        """
        command = "PLZZ,%s,%s" % (str(rotvel), str(deltarotvel))
        self.RunCommand(command, **kwargs)

    def Vsymm(self, ncomp="", nv1="", nv2="", ninc="", kinc="", noelem="",
              imove="", **kwargs):
        """
        APDL Command: VSYMM

        Generates volumes from a volume pattern by symmetry reflection.

        Parameters
        ----------
        ncomp
            Symmetry key:

            X - X symmetry (default).

            Y - Y symmetry.

            Z - Z symmetry.

        nv1, nv2, ninc
            Reflect volumes from pattern beginning with NV1 to NV2 (defaults to
            NV1) in steps of NINC (defaults to 1).  If NV1 = ALL, NV2 and NINC
            are ignored and the pattern is all selected volumes [VSEL].  If NV1
            = P, graphical picking is enabled and all remaining command fields
            are ignored (valid only in the GUI).  A component name may also be
            substituted for NV1 (NV2 and NINC are ignored).

        kinc
            Keypoint increment between sets.  If zero, the lowest available
            keypoint numbers are assigned [NUMSTR].

        noelem
            Specifies whether nodes and elements are also to be generated:

            0 - Generate nodes and elements associated with the original volumes, if they
                exist.

            1 - Do not generate nodes and elements.

        imove
            Specifies whether volumes will be moved or newly defined:

            0 - Generate additional volumes.

            1 - Move original volumes to new position retaining the same keypoint numbers (KINC
                and NOELEM are ignored).  Corresponding meshed items are also
                moved if not needed at their original position.

        Notes
        -----
        Generates a reflected set of volumes (and their corresponding
        keypoints, lines, areas and mesh) from a given volume pattern by a
        symmetry reflection (see analogous node symmetry command, NSYM).  The
        MAT, TYPE, REAL, and ESYS attributes are based upon the volumes in the
        pattern and not upon the current settings.  Reflection is done in the
        active coordinate system by changing a particular coordinate sign.  The
        active coordinate system must be a Cartesian system.  Volumes in the
        pattern may have been generated in any coordinate system.   However,
        solid modeling in a toroidal coordinate system is not recommended.
        Volumes are generated as described in the VGEN command.

        See the ESYM command for additional information about symmetry
        elements.

        """
        command = "VSYMM,%s,%s,%s,%s,%s,%s,%s" % (str(ncomp), str(nv1), str(nv2), str(ninc), str(kinc), str(noelem), str(imove))
        self.RunCommand(command, **kwargs)

    def Eorient(self, etype="", dir="", toler="", **kwargs):
        """
        APDL Command: EORIENT

        Reorients solid element normals.

        Parameters
        ----------
        etype
            Specifies which elements to orient.

            LYSL - Specifies that certain solid elements (such as SOLID185 with KEYOPT(3) = 1,
                   SOLID186 with KEYOPT(3) = 1, and SOLSH190) will be oriented.
                   This value is the default.

        Notes
        -----
        EORIENT renumbers the element faces, designating the face  most
        parallel to the XY plane of the element coordinate system (set with
        ESYS) as face 1 (nodes I-J-K-L, parallel to the layers in layered
        elements). It calculates the outward normal of each face and changes
        the node designation  of the elements so the face with a normal most
        nearly parallel with and in the same general direction as the target
        axis becomes face 1.

        The target axis, defined by Dir, is either the negative or positive
        indicated axis or the outward normal of face 1 of that element.

        All SOLID185 Layered Structural Solid, SOLID186 Layered Structural
        Solid, and SOLSH190 solid shell elements in the selected set are
        considered for reorientation.

        After reorienting elements, you should always display and graphically
        review results using the /ESHAPE command. When plotting models with
        many or symmetric layers, it may be useful to temporarily reduce the
        number of layers to two, with one layer being much thicker than the
        other.

        You cannot use EORIENT to change the normal direction of any element
        that has a body or surface load.  We recommend that you apply all of
        your loads only after ensuring that the element normal directions are
        acceptable.

        Prisms and tetrahedrals are also supported, within the current
        limitations of the SOLID185, SOLID186, and SOLSH190 elements. (Layers
        parallel to the four-node face of the prism are not supported.)

        """
        command = "EORIENT,%s,%s,%s" % (str(etype), str(dir), str(toler))
        self.RunCommand(command, **kwargs)

    def Lcsel(self, type="", lcmin="", lcmax="", lcinc="", **kwargs):
        """
        APDL Command: LCSEL

        Selects a subset of load cases.

        Parameters
        ----------
        type
            Label identifying the type of select:

            S - Select a new set.

            R - Reselect a set from the current set.

            A - Additionally select a set and extend the current set.

            U - Unselect a set from the current set.

            ALL - Restore the full set.

            NONE - Unselect the full set.

            INVE - Invert the current set (selected becomes unselected and vice versa).

            STAT - Display the current select status.

        lcmin
            Minimum value of load case pointer range.

        lcmax
            Maximum value of load case pointer range.  LCMAX defaults to LCMIN.

        lcinc
            Value increment within range.  Defaults to 1.  LCINC cannot be
            negative.

        Notes
        -----
        Selects a subset of load cases for other operations.  For example, to
        select a new set of load cases based on load cases 1 through 7, use
        LCSEL,S,1,7.  The subset is used when the ALL label is entered (or
        implied) on other commands, such as LCFACT, LCABS, LCOPER, etc.  Load
        cases are flagged as selected and unselected; no load case pointers
        [LCDEF, LCWRITE, LCFILE] are actually deleted from the database.

        """
        command = "LCSEL,%s,%s,%s,%s" % (str(type), str(lcmin), str(lcmax), str(lcinc))
        self.RunCommand(command, **kwargs)

    def Sedlist(self, sename="", kopt="", **kwargs):
        """
        APDL Command: SEDLIST

        Lists the DOF solution of a superelement after the use pass.

        Parameters
        ----------
        sename
            Name of the superelement in Jobname.DSUB to be listed.  If a
            number, it is the element number of the superelement as used in the
            use pass.  If ALL, list results for all superelements.

        kopt
            List key:

            0 - List summary data only.

            1 - List full contents.  Be aware that the listing may be extensive.

        Notes
        -----
        Lists the degree of freedom solution of a superelement after the
        substructure use pass.  Results may be listed for any superelement on
        File.DSUB.

        This command is valid in any processor.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "SEDLIST,%s,%s" % (str(sename), str(kopt))
        self.RunCommand(command, **kwargs)

    def Lclear(self, nl1="", nl2="", ninc="", **kwargs):
        """
        APDL Command: LCLEAR

        Deletes nodes and line elements associated with selected lines.

        Parameters
        ----------
        nl1, nl2, ninc
            Delete mesh for lines NL1 to NL2 (defaults to NL1) in steps of NINC
            (defaults to 1).  If NL1 = ALL, NL2 and NINC are ignored and the
            mesh for all selected lines [LSEL] is deleted.  If NL1 = P,
            graphical picking is enabled and all remaining command fields are
            ignored (valid only in the GUI).  A component name may also be
            substituted for NL1 (NL2 and NINC are ignored).

        Notes
        -----
        Deletes all nodes and line elements associated with selected lines
        (regardless of whether the nodes or elements are selected).  Nodes
        shared by adjacent meshed lines and nodes associated with non-line
        elements will not be deleted.  Attributes assigned as a result of LATT
        are maintained.  In the program's response to the command, if a line or
        keypoint is tallied as "cleared," it means either its node or element
        reference was deleted.

        """
        command = "LCLEAR,%s,%s,%s" % (str(nl1), str(nl2), str(ninc))
        self.RunCommand(command, **kwargs)

    def Kl(self, nl1="", ratio="", nk1="", **kwargs):
        """
        APDL Command: KL

        Generates a keypoint at a specified location on an existing line.

        Parameters
        ----------
        nl1
            Number of the line.  If negative, the direction of line (as
            interpreted for RATIO) is reversed.  If NL1 = P, graphical picking
            is enabled and all remaining command fields are ignored (valid only
            in the GUI).

        ratio
            Ratio of line length to locate keypoint.  Must be between 0.0 and
            1.0.  Defaults to 0.5 (divide the line in half).

        nk1
            Number to be assigned to keypoint generated at division location
            (defaults to lowest available keypoint number [NUMSTR]).

        """
        command = "KL,%s,%s,%s" % (str(nl1), str(ratio), str(nk1))
        self.RunCommand(command, **kwargs)

    def Outopt(self, **kwargs):
        """
        APDL Command: OUTOPT

        Specifies "Output options" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "OUTOPT," % ()
        self.RunCommand(command, **kwargs)

    def Bfint(self, fname1="", ext1="", fname2="", ext2="", kpos="", clab="",
              kshs="", tolout="", tolhgt="", **kwargs):
        """
        APDL Command: BFINT

        Activates the body force interpolation operation.

        Parameters
        ----------
        fname1
            File name and directory path (248 characters maximum, including
            directory) from which to read data for interpolation. If you do not
            specify a directory path, it will default to your working directory
            and you can use all 248 characters for the file name.

        ext1
            Filename extension (eight-character maximum).

        --
            Unused field.

        fname2
            File name and directory path (248 characters maximum, including
            directory) to which BF commands are written. If you do not specify
            a directory path, it will default to your working directory and you
            can use all 248 characters for the file name.

        ext2
            Filename extension (eight-character maximum).

        --
            Unused field.

        kpos
            Position on Fname2 to write block of BF commands:

            0 - Beginning of file (overwrite existing file).

            1 - End of file (append to existing file).

        clab
            Label (8 characters maximum, including the colon) for this block of
            BF commands in Fname2.  This label is appended to the colon (:).
            Defaults to BFn, where n is the cumulative iteration number for the
            data set currently in the database.

        kshs
            Shell-to-solid submodeling key:

            0 - Solid-to-solid or shell-to-shell submodel.

            1 - Shell-to-solid submodel.

        tolout
            Extrapolation tolerance about elements, based on a fraction of the
            element dimension. Submodel nodes outside the element by more than
            TOLOUT are not accepted as candidates for DOF extrapolation.
            Defaults to 0.5 (50%).

        tolhgt
            Height tolerance above or below shell elements, in units of length.
            Used only for shell-to-shell submodeling (KSHS = 0). Submodel nodes
            off the element surface by more than TOLHGT are not accepted as
            candidates for DOF interpolation or extrapolation. Defaults to
            0.0001 times the maximum element dimension.

        Notes
        -----
        File Fname1 should contain a node list for which body forces are to be
        interpolated [NWRITE].  File Fname2 is created, and contains
        interpolated body forces written as a block of nodal BF commands.

        Body forces are interpolated from elements having TEMP as a valid body
        force or degree of freedom, and only the label TEMP is written on the
        nodal BF commands.  Interpolation is performed for all nodes on file
        Fname1 using the results data currently in the database. For layered
        elements, use the LAYER command to select the locations of the
        temperatures to be used for interpolation. Default locations are the
        bottom of the bottom layer and the top of the top layer.

        The block of BF commands begins with an identifying colon label command
        and ends with a /EOF command.  The colon label command is of the form
        :Clab, where Clab is described above.  Interpolation from multiple
        results sets can be performed by looping through the results file in a
        user-defined macro.  Additional blocks can be appended to Fname2 by
        using KPOS and unique colon labels.  A /INPUT command, with the
        appropriate colon label, may be used to read the block of commands.

        If the model has coincident (or very close) nodes, BFINT must be
        applied to each part of the model separately to ensure that the mapping
        of the nodes is correct.  For example, if nodes belonging to two
        adjacent parts linked by springs are coincident, the operation should
        be performed on each part of the model separately.

        """
        command = "BFINT,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(fname1), str(ext1), str(fname2), str(ext2), str(kpos), str(clab), str(kshs), str(tolout), str(tolhgt))
        self.RunCommand(command, **kwargs)

    def Hfang(self, lab="", phi1="", phi2="", theta1="", theta2="", **kwargs):
        """
        APDL Command: HFANG

        Defines or displays spatial angles of a spherical radiation surface for
        sound radiation parameter calculations.

        Parameters
        ----------
        lab
            Spatial angle label.

            ANGLE - Define spatial angles (default).

            STATE - Display spatial angles. PHI1, PHI2, THETA1, and THETA2 are ignored.

        phi1, phi2
            Starting and ending ϕ angles (degrees) in the spherical coordinate
            system. Defaults to 0.

        theta1, theta2
            Starting and ending θ angles (degrees) in the spherical coordinate
            system. Defaults to 0.

        Notes
        -----
        Defines or displays spatial angles of a spherical radiation surface.

        Use this command only with PLFAR,Lab = PRES, or PRFAR,Lab = PRES.

        """
        command = "HFANG,%s,%s,%s,%s,%s" % (str(lab), str(phi1), str(phi2), str(theta1), str(theta2))
        self.RunCommand(command, **kwargs)

    def Prim(self, **kwargs):
        """
        APDL Command: PRIM

        Specifies "Solid model primitives" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command.  Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        """
        command = "PRIM," % ()
        self.RunCommand(command, **kwargs)

    def Pspec(self, pcolor="", kfill="", kbordr="", **kwargs):
        """
        APDL Command: /PSPEC

        Creates annotation polygon attributes (GUI).

        Parameters
        ----------
        pcolor
            Polygon color (0  PCOLOR   15):

            0 - Black.

            1 - Red-Magenta.

            2 - Magenta.

            3 - Blue-Magenta.

            4 - Blue.

            5 - Cyan-Blue.

            6 - Cyan.

            7 - Green-Cyan.

            8 - Green.

            9 - Yellow-Green.

            10 - Yellow.

            11 - Orange.

            12 - Red.

            13 - Dark Gray.

            14 - Light Gray.

            15 - White.

        kfill
            Polygon fill key:

            0 - Hollow polygon.

            1 - Filled polygon.

        kbordr
            Polygon border key:

            0 - No border.

            1 - Border.

        Notes
        -----
        Creates annotation polygon attributes to control certain
        characteristics of the polygons created via the /POLYGON, /PMORE,
        /PCIRCLE and /PWEDGE commands.  This is a command generated by the
        Graphical User Interface (GUI) and will appear in the log file
        (Jobname.LOG) if annotation is used.  This command is not intended to
        be typed in directly in an ANSYS session (although it can be included
        in an input file for batch input or for use with the /INPUT command).

        This command is valid in any processor.

        """
        command = "/PSPEC,%s,%s,%s" % (str(pcolor), str(kfill), str(kbordr))
        self.RunCommand(command, **kwargs)

    def Andyna(self, delay="", ncycl="", start="", end="", inc="",
               autocontourkey="", **kwargs):
        """
        APDL Command: ANDYNA

        Produces an animated sequence of contour values through substeps.

        Parameters
        ----------
        delay
            Time delay during animation (defaults to 0.1 seconds).

        ncycl
            Number of animation cycles (defaults to 5).  Available in non-UI
            mode only.

        start
            Number of the starting substep (defaults to 1).

        end
            Number of the ending substep (defaults to the maximum substep).

        inc
            Increment between substeps (defaults to 1).

        autocontourkey
            Auto-scales contour values, based on the overall subset range of
            values  (defaults to 0, no auto-scaling).

        Notes
        -----
        ANDYNA involves an ANSYS macro which produces an animation of contour
        values through all the substeps of the last plot action command.  This
        command operates only on graphic display platforms supporting the /SEG
        command.  After executing ANDYNA, you can replay the animated sequence
        by issuing the ANIM command.

        The command functions only in the postprocessor.

        """
        command = "ANDYNA,%s,%s,%s,%s,%s,%s" % (str(delay), str(ncycl), str(start), str(end), str(inc), str(autocontourkey))
        self.RunCommand(command, **kwargs)

    def Ascres(self, opt="", **kwargs):
        """
        APDL Command: ASCRES

        Specifies the output type for an acoustic scattering analysis.

        Parameters
        ----------
        opt
            Output option:

            TOTAL - Output the total pressure field (default).

            SCAT - Output the scattered pressure field.

        Notes
        -----
        Use the ASCRES command to specify the output type for an acoustic
        scattering analysis.

        The scattered option (Opt = SCAT) provides a scattered pressure output,
        psc, required for calculating target strength (TS).

        The default behavior (Opt = TOTAL) provides a sum of the incident and
        scattering fields, ptotal = pinc + psc.

        Issue the AWAVE command to define the incident pressure pinc. If the
        AWAVE command is defined with Opt2 = INT, only the total pressure field
        is output regardless of the ASCRES,Opt command.

        """
        command = "ASCRES,%s" % (str(opt))
        self.RunCommand(command, **kwargs)

    def Prvect(self, item="", lab2="", lab3="", labp="", **kwargs):
        """
        APDL Command: PRVECT

        Prints results as vector magnitude and direction cosines.

        Parameters
        ----------
        item
            Predefined vector item (from Table 226: PRVECT - Valid Item and
            Component Labels below) or a label identifying the i-component of a
            user-defined vector.

        lab2
            Label identifying the j-component of a user-defined vector. In most
            cases, this value must be blank if Item is selected from Table 226:
            PRVECT - Valid Item and Component Labels. Individual principal
            stresses (Item = S) or principal strains (Item = EPxx) may be
            printed by specifying the value as 1, 2, or 3.

        lab3
            Label identifying the k-component of a user-defined vector.  Must
            be blank if Item is selected from list below or for 2-D user
            defined vector.

        labp
            Label assigned to resultant vector for printout labeling (defaults
            to Item).

        Notes
        -----
        Prints various solution results as vector magnitude and direction
        cosines for the selected nodes and/or elements.  For example, PRVECT,U
        prints the displacement magnitude and its direction cosines for all
        selected nodes.  For nodal degree of freedom vector results, direction
        cosines are with respect to the results coordinate system RSYS. For
        element results, direction cosines are with respect to the global
        Cartesian system. Item components may be printed with the PRNSOL
        command.  Various results also depend upon the recalculation method and
        the selected results location [LAYER, SHELL, NSEL, and ESEL].  Items
        may be selected from a set of recognized vector labels (Item) or a
        vector may be defined from up to three scalar labels (Item,Lab2,Lab3).
        Scalar labels may be user-defined with the ETABLE command.

        Portions of this command are not supported by PowerGraphics
        [/GRAPHICS,POWER].

        Table: 226:: : PRVECT - Valid Item and Component Labels

        """
        command = "PRVECT,%s,%s,%s,%s" % (str(item), str(lab2), str(lab3), str(labp))
        self.RunCommand(command, **kwargs)

    def Hmagsolv(self, freq="", nramp="", cnva="", cnvv="", cnvc="", cnve="",
                 neqit="", **kwargs):
        """
        APDL Command: HMAGSOLV

        Specifies 2-D or axisymmetric harmonic magnetic solution options and
        initiates the solution.

        Parameters
        ----------
        freq
            Analysis frequency (Hz).

        nramp
            Number of ramped substeps for the first load step of a nonlinear
            2-D harmonic electromagnetic solution.  Defaults to 3.  If NRAMP =
            -1, ignore the ramped load step entirely.

        cnva
            Convergence tolerance on the program calculated reference value for
            the magnetic vector potential degree of freedom.  Defaults to
            0.001.

        cnvv
            Convergence tolerance on the program calculated reference value for
            the time-integrated electric potential VOLT.    Defaults to 0.001.

        cnvc
            Convergence tolerance on the program calculated reference value for
            the current degree of freedom CURR.  Defaults to 0.001.

        cnve
            Convergence tolerance on the program calculated reference value for
            the voltage drop degree of freedom EMF.  Defaults to 0.001.

        neqit
            Maximum number of equilibrium iterations per load step.  Defaults
            to 50.

        Notes
        -----
        HMAGSOLV invokes an ANSYS macro which specifies harmonic
        electromagnetic solution options and initiates the solution.  The macro
        is applicable to any ANSYS 2-D or axisymmetric linear or nonlinear
        harmonic analysis.  Results are only stored for the final converged
        solution.  (In POST1, issue *SET,LIST to identify the load step of
        solution results.)  The macro internally determines if a nonlinear
        analysis is required based on magnetic material properties defined in
        the database.

        The macro performs a two-load-step solution sequence.  The first load
        step ramps the applied loads over a prescribed number of substeps
        (NRAMP), and the second load step calculates the converged solution.
        For linear problems, only a single load step solution is performed.
        The ramped load step can be bypassed by setting NRAMP to -1.

        A 3-D harmonic electromagnetic analysis is available for linear
        solutions only and does not require this solution macro.

        The following analysis options and nonlinear options are controlled by
        this macro:  KBC, NEQIT, NSUBST, CNVTOL, OUTRES.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "HMAGSOLV,%s,%s,%s,%s,%s,%s,%s" % (str(freq), str(nramp), str(cnva), str(cnvv), str(cnvc), str(cnve), str(neqit))
        self.RunCommand(command, **kwargs)

    def Ndist(self, nd1="", nd2="", **kwargs):
        """
        APDL Command: NDIST

        Calculates and lists the distance between two nodes.

        Parameters
        ----------
        nd1
            First node in distance calculation.  If ND1 = P, graphical picking
            is enabled and all remaining command fields are ignored (valid only
            in the GUI).

        nd2
            Second node in distance calculation.

        Notes
        -----
        NDIST lists the distance between nodes ND1 and ND2, as well as the
        current coordinate system offsets from ND1 to ND2, where the X, Y, and
        Z locations of ND1 are subtracted from the X, Y, and Z locations of ND2
        (respectively) to determine the offsets.  NDIST is valid in any
        coordinate system except toroidal [CSYS,3].

        NDIST returns a variable, called "_RETURN," which contains the distance
        value. You can use this value for various purposes, such as the
        calculation of distributed loads. In interactive mode, you can access
        this command by using the Model Query Picker (Utility Menu> List>
        Picked Entities), where you can also access automatic annotation
        functions and display the value on your model.

        This command is valid in any processor.

        """
        command = "NDIST,%s,%s" % (str(nd1), str(nd2))
        self.RunCommand(command, **kwargs)

    def Output(self, fname="", ext="", loc="", **kwargs):
        """
        APDL Command: /OUTPUT

        Redirects text output to a file or to the screen.

        Parameters
        ----------
        fname
            Filename and directory path (248 character maximum, including
            directory) to which text output will be redirected (defaults to
            Jobname if Ext is specified).  For interactive runs, Fname = TERM
            (or blank) redirects output to the screen.  For batch runs, Fname =
            blank (with all remaining command arguments blank) redirects output
            to the  default system output file.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        loc
            Location within a file to which output will be written:

            (blank) - Output is written starting at the top of the file (default).

            APPEND - Output is appended to the existing file.

        Notes
        -----
        Text output includes responses to every command and GUI function,
        notes, warnings, errors, and other informational messages.  Upon
        execution of /OUTPUT,Fname, Ext, ..., all subsequent text output is
        redirected to the file Fname.Ext.  To redirect output back to the
        default location, issue /OUTPUT (no arguments).

        Note:: : When using the GUI, output from list operations [NLIST, DLIST,
        etc.] is always sent to a list window regardless of the /OUTPUT
        setting.  The output can then be saved on a file or copied to the
        /OUTPUT location using the File menu in the list window.

        This command is valid in any processor.

        """
        command = "/OUTPUT,%s,%s,%s" % (str(fname), str(ext), str(loc))
        self.RunCommand(command, **kwargs)

    def Egen(self, itime="", ninc="", iel1="", iel2="", ieinc="", minc="",
             tinc="", rinc="", cinc="", sinc="", dx="", dy="", dz="",
             **kwargs):
        """
        APDL Command: EGEN

        Generates elements from an existing pattern.

        Parameters
        ----------
        itime, ninc
            Do this generation operation a total of ITIMEs, incrementing all
            nodes in the given pattern by NINC each time after the first. ITIME
            must be >1 if generation is to occur. NINC may be positive, zero,
            or negative. If DX, DY, and/or DZ is specified, NINC should be set
            so any existing nodes (as on NGEN) are not overwritten.

        iel1, iel2, ieinc
            Generate elements from selected pattern beginning with IEL1 to IEL2
            (defaults to IEL1) in steps of IEINC (defaults to 1). If IEL1 is
            negative, IEL2 and IEINC are ignored and the last |IEL1| elements
            (in sequence backward from the maximum element number) are used as
            the pattern to be repeated.  If IEL1 = ALL, IEL2 and IEINC are
            ignored and use all selected elements [ESEL] as pattern to be
            repeated. If P1 = P, graphical picking is enabled and all remaining
            command fields are ignored (valid only in the GUI).  A component
            name may also be substituted for IEL1 (IEL2 and INC are ignored).

        minc
            Increment material number of all elements in the given pattern by
            MINC each time after the first.

        tinc
            Increment type number by TINC.

        rinc
            Increment real constant table number by RINC.

        cinc
            Increment element coordinate system number by CINC.

        sinc
            Increment section ID number by SINC.

        dx, dy, dz
            Define nodes that do not already exist but are needed by generated
            elements (as though the NGEN,ITIME,INC,NODE1,,,DX,DY,DZ were issued
            before EGEN). Zero is a valid value. If blank, DX, DY, and DZ are
            ignored.

        Notes
        -----
        A pattern may consist of any number of previously defined elements. The
        MAT, TYPE, REAL, ESYS, and SECNUM numbers of the new elements are based
        upon the elements in the pattern and not upon the current specification
        settings.

        You can use the EGEN command to generate interface elements (INTER192,
        INTER193, INTER194, and INTER195) directly. However, because interface
        elements require that the element connectivity be started from the
        bottom surface, you must make sure that you use the correct element
        node connectivity. See the element descriptions for INTER192, INTER193,
        INTER194, and INTER195 for the correct element node definition.

        """
        command = "EGEN,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(itime), str(ninc), str(iel1), str(iel2), str(ieinc), str(minc), str(tinc), str(rinc), str(cinc), str(sinc), str(dx), str(dy), str(dz))
        self.RunCommand(command, **kwargs)

    def Lsclear(self, lab="", **kwargs):
        """
        APDL Command: LSCLEAR

        Clears loads and load step options from the database.

        Parameters
        ----------
        lab
            Label identifying the data to be cleared:

            SOLID - Delete only solid model loads.

            FE - Delete only finite element loads.

            INER - Delete only inertia loads (ACEL, etc.).

            LFACT - Initialize only load factors (on DCUM, FCUM, SFCUM, etc.).

            LSOPT - Initialize only load step options.

            ALL - Delete all loads and initialize all load step options and load factors.

        Notes
        -----
        Loads are deleted, and load step options are initialized to their
        default values.

        This command is also valid in PREP7.

        """
        command = "LSCLEAR,%s" % (str(lab))
        self.RunCommand(command, **kwargs)

    def Aesize(self, anum="", size="", **kwargs):
        """
        APDL Command: AESIZE

        Specifies the element size to be meshed onto areas.

        Parameters
        ----------
        anum
            Area number of the area to which this element size specification
            applies. If ANUM = ALL, size applies to all selected areas. If ANUM
            = P, graphical picking is enabled. A component name may also be
            substituted for ANUM.

        size
             Desired element size.

        Notes
        -----
        AESIZE allows control over the element sizing inside any area or on the
        face(s) of a volume.

        SIZE controls element size on the interior of the area. For any line on
        the area not having its own size assignment and not controlled by
        keypoint size assignments, it specifies the element size along the line
        as well, so long as no adjacent area has a smaller size, which would
        take precedence. If the AESIZE governs the boundary and SmartSizing is
        on, the boundary size can be refined for curvature or proximity.

        This command is also valid for rezoning.

        """
        command = "AESIZE,%s,%s" % (str(anum), str(size))
        self.RunCommand(command, **kwargs)

    def Pldisp(self, kund="", **kwargs):
        """
        APDL Command: PLDISP

        Displays the displaced structure.

        Parameters
        ----------
        kund
            Undisplaced shape key:

            0 - Display only displaced structure.

            1 - Overlay displaced display with similar undisplaced display (appearance is
                system-dependent).

            2 - Same as 1 except overlay with undisplaced edge display (appearance is system-
                dependent).

        Notes
        -----
        Displays the displaced structure for the selected elements.

        For information on true scale plots, refer to the description of the
        /DSCALE command [/DSCALE,,1.0].

        """
        command = "PLDISP,%s" % (str(kund))
        self.RunCommand(command, **kwargs)

    def Einfin(self, compname="", pnode="", **kwargs):
        """
        APDL Command: EINFIN

        Generates structural infinite elements from selected nodes.

        Parameters
        ----------
        compname
            Component name containing one node to be used as the pole node for
            generating INFIN257 structural infinite elements. The pole node is
            generally located at or near the geometric center of the finite
            element domain.

        pnode
            Node number for the direct input of the pole node. A parameter or
            parametric expression is also valid. Specify this value when no
            CompName has been specified. If CompName is specified, this value
            is ignored.

        Notes
        -----
        The EINFIN command generates structural infinite elements (INFIN257)
        directly from the selected face of valid base elements (existing
        standard elements in your model). The command scans all base elements
        for the selected nodes and generates a compatible infinite element type
        for each base element. A combination of different base element types is
        allowed if the types are all compatible with the infinite elements.

        The infinite element type requires no predefinition (ET).

        The faces of base elements are determined from the selected node set
        (NSEL), and the geometry of the infinite element is determined based on
        the shape of the face. Element characteristics and options are
        determined according to the base element. For the face to be used, all
        nodes on the face of a base element must be selected

        Use base elements to model the near-field domain that interacts with
        the solid structures or applied loads. To apply the truncated far-field
        effect, a single layer of infinite elements must be attached to the
        near-field domain. The outer surface of the near-field domain must be
        convex.

        After the EINFIN command executes, you can verify the newly created
        infinite element types and elements (ETLIST, ELIST, EPLOT).

        Infinite elements do not account for any subsequent modifications made
        to the base elements. It is good practice to issue the EINFIN command
        only after the base elements are finalized. If you delete or modify
        base elements, remove all affected infinite elements and reissue the
        EINFIN command; doing so prevents inconsistencies.

        """
        command = "EINFIN,%s,%s" % (str(compname), str(pnode))
        self.RunCommand(command, **kwargs)

    def Reswrite(self, fname="", cflag="", **kwargs):
        """
        APDL Command: RESWRITE

        Appends results data from the database to a results file.

        Parameters
        ----------
        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path). An unspecified directory
            path defaults to the working directory; in this case, you can use
            all 248 characters for the file name.

        --, --, --
            Unused fields.

        cflag
            0

            0 - The complex results flag is set to 0 in the results file header. This is the
                default option.

            1 - The complex results flag is set to 1 in the results file header.

        Notes
        -----
        The RESWRITE command appends a data set to the specified file by
        writing the results data currently in the database. If the intended
        results file does not exist, it will be created and will include the
        geometry records. The current load step, substep, and time (or
        frequency) value are maintained. All data (summable and nonsummable)
        are written.

        When complex results are appended, cFlag must be set to 1 so that the
        header is consistent with the results written on the file.

        The command is primarily intended for use in a top-down substructuring
        analysis, where the full model is resumed and the results data read
        from the use pass results file (SET), and subsequently from all
        substructure expansion pass results files (APPEND). The full set of
        data in memory can then be written out via the RESWRITE command to
        create a complete results file (as though you had run a
        nonsubstructured analysis).

        The RESWRITE command can also be used to write a global results file
        for a distributed parallel (Distributed ANSYS) solution. This should
        only be necessary if the RESCOMBINE command was used to combine results
        from local results files into the database. The RESWRITE command can
        then be used to write the combined results into a new results file.
        This new results file will essentially contain the current set of
        results data for the entire (i.e., global) model.

        """
        command = "RESWRITE,%s,%s" % (str(fname), str(cflag))
        self.RunCommand(command, **kwargs)

    def En(self, iel="", i="", j="", k="", l="", m="", n="", o="", p="",
           **kwargs):
        """
        APDL Command: EN

        Defines an element by its number and node connectivity.

        Parameters
        ----------
        iel
            Number assigned to element being defined. If IEL = P, graphical
            picking is enabled and all remaining command fields are ignored
            (valid only in the GUI).

        i
            Number of node assigned to first nodal position (node I).

        j, k, l, m, n, o, p
            Number assigned to second (node J) through eighth (node P) nodal
            position, if any.

        Notes
        -----
        Defines an element by its nodes and attribute values. Similar to the E
        command except it allows the element number (IEL) to be defined
        explicitly.  Element numbers need not be consecutive. Any existing
        element already having this number will be redefined.

        Up to 8 nodes may be specified with the EN command. If more nodes are
        needed for the element, use the EMORE command. The number of nodes
        required and the order in which they should be specified are described
        in the Element Reference for each element type.  The current (or
        default) MAT, TYPE, REAL, SECNUM, and ESYS attribute values are also
        assigned to the element.

        When creating elements with more than 8 nodes using this command and
        the EMORE command, it may be necessary to turn off shape checking using
        the SHPP command before issuing this command. If a valid element type
        can be created without using the additional nodes on the EMORE command,
        this command will create that element. The EMORE command will then
        modify the element to include the additional nodes. If shape checking
        is active, it will be performed before the EMORE command is issued.
        Therefore, if the shape checking limits are exceeded, element creation
        may fail before the EMORE command modifies the element into an
        acceptable shape.

        """
        command = "EN,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(iel), str(i), str(j), str(k), str(l), str(m), str(n), str(o), str(p))
        self.RunCommand(command, **kwargs)

    def Vadd(self, nv1="", nv2="", nv3="", nv4="", nv5="", nv6="", nv7="",
             nv8="", nv9="", **kwargs):
        """
        APDL Command: VADD

        Adds separate volumes to create a single volume.

        Parameters
        ----------
        nv1, nv2, nv3, . . . , nv9
            Numbers of volumes to be added.  If NV1 = ALL, add all selected
            volumes and ignore NV2 to NV9.  If NV1 = P, graphical picking is
            enabled and all remaining command fields are ignored (valid only in
            the GUI).  A component name may also be substituted for NV1.

        Notes
        -----
        Adds separate volumes to create a single volume.  The original volumes
        (and their corresponding areas, lines and keypoints) will be deleted by
        default [BOPTN].  See the BOPTN command for the options available to
        Boolean operations.  Element attributes and solid model boundary
        conditions assigned to the original entities will not be transferred to
        the new entities generated.  Concatenated entities are not valid with
        this command.

        """
        command = "VADD,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(nv1), str(nv2), str(nv3), str(nv4), str(nv5), str(nv6), str(nv7), str(nv8), str(nv9))
        self.RunCommand(command, **kwargs)

    def Anfile(self, lab="", fname="", ext="", **kwargs):
        """
        APDL Command: /ANFILE

        Saves or resumes an animation sequence to or from a file.

        Parameters
        ----------
        lab
            Label type.

            SAVE - Save the current animation to a file.

            RESUME - Resume an animation from a file.

        fname
            File name and directory path (248 characters maximum, including the
            characters needed for the directory path).  An unspecified
            directory path defaults to the working directory; in this case, you
            can use all 248 characters for the file name.

        ext
            Filename extension (eight-character maximum).

        --
            Unused field.

        Notes
        -----
        This command saves an animation to a file from local terminal segments
        or resumes an animation from a file to local terminal segments.  See
        the /SEG command for details on segment storage.  See the ANCNTR macro
        for a convenient method of storing graphics frames in terminal memory
        segments.  This command is device dependent and is valid in any
        processor.

        """
        command = "/ANFILE,%s,%s,%s" % (str(lab), str(fname), str(ext))
        self.RunCommand(command, **kwargs)

    def Ocdelete(self, datatype="", zonename="", **kwargs):
        """
        APDL Command: OCDELETE

        Deletes a previously defined ocean load.

        Parameters
        ----------
        datatype
            Ocean data type to delete. Valid values are BASIC, CURRENT, WAVE,
            ZONE, and ALL.

        zonename
            The name of the ocean zone to delete. If no name is specified, all
            defined ocean zones are deleted. Valid only when DataType = ZONE.

        Notes
        -----
        The OCDELETE command deletes previously specified ocean data from the
        database.

        This command is also valid in PREP7.

        """
        command = "OCDELETE,%s,%s" % (str(datatype), str(zonename))
        self.RunCommand(command, **kwargs)

    def Mdplot(self, function="", dmpname="", scale="", **kwargs):
        """
        APDL Command: MDPLOT

        Plots frequency-dependent modal damping coefficients calculated by
        DMPEXT.

        Parameters
        ----------
        function
            Function to display.

            d_coeff - Damping coefficient

            s_coeff - Squeeze coefficient

            d_ratio - Damping ratio

            s_ratio - Squeeze stiffness ratio

        dmpname
            Array parameter name where damping information is stored. Defaults
            to d_damp.

        scale
            Indicates whether to perform a linear or a double logarithmic plot.

            LIN - Perform a linear plot. Default

            LOG - Perform a double logarithmic plot.

        Notes
        -----
        See Thin Film Analysis for more information on thin film analyses.

        """
        command = "MDPLOT,%s,%s,%s" % (str(function), str(dmpname), str(scale))
        self.RunCommand(command, **kwargs)

    def Conjug(self, ir="", ia="", name="", facta="", **kwargs):
        """
        APDL Command: CONJUG

        Forms the complex conjugate of a variable.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]).  If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.

        ia
            Reference number of the variable to be operated on.

        --, --
            Unused fields.

        name
            Thirty-two character name for identifying the variable on printouts
            and displays.  Embedded blanks are compressed for output.

        --, --
            Unused fields.

        facta
            Scaling factor (positive or negative) applied to variable (default
            to 1.0).

        Notes
        -----
        Used only with harmonic analyses (ANTYPE,HARMIC).

        """
        command = "CONJUG,%s,%s,%s,%s" % (str(ir), str(ia), str(name), str(facta))
        self.RunCommand(command, **kwargs)

    def Plfar(self, lab="", option="", phi1="", phi2="", nph1="", theta1="",
              theta2="", ntheta="", val1="", val2="", val3="", **kwargs):
        """
        APDL Command: PLFAR

        Plots pressure far fields and far field parameters.

        Parameters
        ----------
        lab
            Parameters to plot :

            PRES - Acoustic parameters

            PROT - Acoustic parameters with the y-axis rotated extrusion

        option
            Plot option, based on the specified plot parameter type:

        phi1, phi2
            Starting and ending φ angles (degrees) in the spherical coordinate
            system. Defaults to 0.

        nphi
            Number of divisions between the starting and ending φ angles for
            data computations. Defaults to 0.

        theta1, theta2
            Starting and ending θ angles (degrees) in the spherical coordinate
            system. Defaults to 0 in 3-D and 90 in 2-D.

        ntheta
            Number of divisions between the starting and ending θ angles for
            data computations. Defaults to 0.

        val1
            Radius of the sphere surface. Used only when Option = SUMC, SUMP,
            PHSC, PHSP, SPLC, SPLP, SPAC, SPAP, PSCT, PSPL, TSCT, or TSPL.

        val2
            When Option = SPLC, SPLP, SPAC, or SPAP: Reference rms sound
            pressure. Defaults to 2x10-5 Pa.

        val3
            When Lab = PRES: Thickness of 2-D model extrusion in the z
            direction (no default).

        Notes
        -----
        The PLFAR command plots pressure far fields and far field parameters as
        determined by the equivalent source principle. Use this command to plot
        pressure and acoustic    parameters. See the HFSYM command for the
        model symmetry and the     HFANG command for spatial radiation angles.

        When Option = PWL no plot is generated, but the sound power level
        appears in the output.

        To retrieve saved equivalent source data, issue the
        SET,Lstep,Sbstep,,REAL command.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "PLFAR,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(lab), str(option), str(phi1), str(phi2), str(nph1), str(theta1), str(theta2), str(ntheta), str(val1), str(val2), str(val3))
        self.RunCommand(command, **kwargs)

    def Lumpm(self, key="", **kwargs):
        """
        APDL Command: LUMPM

        Specifies a lumped mass matrix formulation.

        Parameters
        ----------
        key
            Formulation key:

            OFF - Use the element-dependent default mass matrix formulation (default).

            ON - Use a lumped mass approximation.

        Notes
        -----
        This command is also valid in PREP7.  If used in SOLUTION, this command
        is valid only within the first load step.

        """
        command = "LUMPM,%s" % (str(key))
        self.RunCommand(command, **kwargs)

    def Pdprob(self, rlab="", name="", relation="", limit="", conf="",
               **kwargs):
        """
        APDL Command: PDPROB

        Prints a probability result.

        Parameters
        ----------
        rlab
            Result set label. Identifies the result set to be used for
            postprocessing. A result set label can be the solution set label
            you defined in a PDEXE command (if you are directly postprocessing
            Monte Carlo Simulation results), or the response surface set label
            defined in an RSFIT command (for Response Surface Analyses).

        name
            Parameter name. The parameter must have been previously defined as
            a random input variable or a random output parameter using the
            PDVAR command.

        relation
            Relation label for the relation between probabilistic design
            parameter Name and the limit value LIMIT:

            LT - Less than (default).

            GT - Greater than.

        limit
            Limit value.

        --
            Unused field.

        conf
            Confidence level. The confidence level is used to print the
            confidence bounds on the probability. The value for the confidence
            level must be between 0.0 and 1.0 and it defaults to 0.95 (95%).
            Printing of confidence bound is suppressed for CONF  0.5. This
            parameter is ignored for response surface methods results
            postprocessing.

        Notes
        -----
        Prints the probability that the probabilistic design input or output
        variable denoted with Name is smaller or larger than a certain limit
        value.

        If Rlab is left blank, then the result set label is inherited from the
        last PDEXE command (Slab), RSFIT command (RSlab), or the most recently
        used PDS postprocessing command where a result set label was explicitly
        specified.

        Use the relation label Relation to specify if you want to print a
        traditional probability value (LT) or the exceedence probability (GT).
        The LIMIT directly specifies at which value of Name (the design
        parameter) the probability should be evaluated. If LIMIT is between two
        sample values of Name the resulting probability is linearly
        interpolated between the sample values. If LIMIT is smaller than all
        sample values of Name the probability is 0.0 for Relation = LT and 1.0
        for Relation = GT. If LIMIT is greater than all sample values for Name
        the probability is 1.0 for Relation = LT and 0.0 for Relation = GT.

        The confidence level is a probability expressing the confidence that
        the value for the requested probability is in fact between the
        confidence bounds. The larger the confidence level, the wider the
        confidence bounds. Printing the confidence bounds only makes sense for
        postprocessing Monte Carlo simulation results, where the confidence
        bounds represent the accuracy of the results. With increasing sample
        sizes, the width of the confidence bounds gets smaller for the same
        confidence level. For response surface analysis methods, the number of
        simulations done on the response surface is usually very large;
        therefore, the accuracy of the results is determined by the response
        surface fit and not by the confidence level.

        The PDPROB command cannot be used to postprocess the results in a
        solution set that is based on Response Surface Methods, only Monte
        Carlo Simulations.

        """
        command = "PDPROB,%s,%s,%s,%s,%s" % (str(rlab), str(name), str(relation), str(limit), str(conf))
        self.RunCommand(command, **kwargs)

    def Mapsolve(self, maxsbstep="", **kwargs):
        """
        APDL Command: MAPSOLVE

        Maps solved node and element solutions from an original mesh to a new
        mesh.

        Parameters
        ----------
        maxsbstep
            The maximum number of substeps for rebalancing the residuals. The
            default value is 5.

        Notes
        -----
        Used during the rezoning process, the MAPSOLVE command maps solved node
        and element solutions from the original mesh to the new mesh and
        achieves equilibrium based on the new mesh.

        Additional substeps are necessary to reduce the residuals to zero.

        During the rebalancing stage, the external loads and time remain
        unchanged.

        The MAPSOLVE command is valid only for rezoning (REZONE).

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "MAPSOLVE,%s" % (str(maxsbstep))
        self.RunCommand(command, **kwargs)

    def Vtype(self, nohid="", nzone="", **kwargs):
        """
        APDL Command: VTYPE

        Specifies the viewing procedure used to determine the form factors for
        the Radiation Matrix method.

        Parameters
        ----------
        nohid
            Type of viewing procedure:

            0 - Hidden procedure.

            1 - Non-hidden (faster, but less general) procedure.

        nzone
            Number of sampling zones for the hidden procedure (100 maximum for
            3-D, 1000 maximum for 2-D).  Defaults to 20 for 3-D, 200 for 2-D.
            Number of points is 2*NZONE for 2-D and 2*NZONE*(NZONE+1) for 3-D.

        """
        command = "VTYPE,%s,%s" % (str(nohid), str(nzone))
        self.RunCommand(command, **kwargs)

    def Edstart(self, restart="", memory="", fsize="", dumpfile="", **kwargs):
        """
        APDL Command: EDSTART

        Specifies status (new or restart) of an explicit dynamics analysis.

        Parameters
        ----------
        restart
            Status of the analysis (new or restart).

            0 - New analysis (default).

            1 - Simple restart.

            2 - Small restart.

            3 - Full restart.

        memory
            Memory to be used (in words). If blank, LS-DYNA assigns a value
            (default). If more or less memory is needed, specify the number of
            words (a word is usually 32 bits on a workstation).

        fsize
            Scale factor for binary file sizes. Defaults to 7, which is
            (7x262144) = 1835008 words.

        dumpfile
            Name of dump file to use during a restart (for example, d3dumpnn,
            where nn = 01, 02, 03,...,99 and defaults to 01). Leave this field
            blank when running a new analysis (RESTART = 0) so that the default
            dump file d3dump01 will be created.

        Notes
        -----
        EDSTART can be issued before the SOLVE command to specify a new
        analysis, a simple restart, a small restart, or a full restart as
        described below.

        New analysis: For a new analysis, you do not need to issue EDSTART
        unless you want to change the MEMORY or FSIZE option. If you do not
        specify the dump file name, d3dump01 will be created by default.

        Simple restart: This option assumes that the database has not been
        altered. Upon restarting, results will be appended to the existing
        results files. Issue EDSTART,1,,,d3dumpnn to indicate which restart
        file to use as a starting point. The dump file to be used must have
        been created in an earlier run and must be available at the time this
        command is issued.  You would typically use a simple restart when you
        interrupt the LS-DYNA run via Ctrl+C and terminate the run prematurely
        by issuing the "sense switch control" key SW1 (see Solution Control and
        Monitoring in the ANSYS LS-DYNA User's Guide). At this point you should
        be able to view the partial solution using ANSYS postprocessors. After
        you are done viewing the partial solution, you can reenter the solution
        processor and issue EDSTART,1,,,d3dumpnn, followed by SOLVE to continue
        with the analysis. The results will be appended to the results files
        Jobname.RST and Jobname.HIS.  You can perform multiple simple restarts
        by issuing EDSTART,1,,,d3dumpnn multiple times, as needed. The
        solutions in the Jobname.RST file will all be in load step number 1.

        Small restart: This option can be used when minor changes in the
        database are necessary. For example, you can reset the termination
        time, reset the output interval, add displacement constraints, change
        initial velocities, switch parts from a deformable to rigid state, etc.
        (See A Small Restart in theANSYS LS-DYNA User's Guide for a complete
        description of database items that can be changed.) Issue
        EDSTART,2,,,d3dumpnn followed by the commands required to change the
        database, then issue SOLVE. The results will be appended to the results
        files Jobname.RST and Jobname.HIS. You can perform multiple restarts by
        issuing EDSTART,2,,,d3dumpnn multiple times, as needed. The additional
        restart solutions will be stored in Jobname.RST as load step numbers 2,
        3, etc.

        Full restart: A full restart is appropriate when many modifications to
        the database are required. For example, you can change the model
        geometry, apply different loading conditions, etc. Issue
        EDSTART,3,,,d3dumpnn to denote a full restart analysis. The Jobname
        will automatically be changed to Jobname_nn, (nn = 01 initially, and
        will be incremented each time EDSTART,3 is issued for subsequent full
        restarts). After the EDSTART command, you can input any commands needed
        to change the database. (Most commands which are applicable to an ANSYS
        LS-DYNA new analysis are also applicable to full restart analysis. A
        few commands related to contact specifications, initial velocity
        definitions, and adaptive meshing are not supported.) Then issue the
        EDIS command to specify which portions of the model should be
        initialized in the full restart using results data from the d3dumpnn
        file. Finally, issue the SOLVE command. At this point, new results
        files, Jobname_nn.RST and Jobname_nn.HIS, will be created. Time and
        output intervals in the new results files are continuous from the
        previous results files; the time is not reset to zero. (See A Full
        Restart in the ANSYS LS-DYNA User's Guide for a complete description of
        a full restart analysis.)

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDSTART,%s,%s,%s,%s" % (str(restart), str(memory), str(fsize), str(dumpfile))
        self.RunCommand(command, **kwargs)

    def Shpp(self, lab="", value1="", value2="", **kwargs):
        """
        APDL Command: SHPP

        Controls element shape checking.

        Parameters
        ----------
        lab
            Shape checking option.  (When Lab = WARN, STATUS, SUMMARY, or
            DEFAULT, the remaining arguments are ignored.)

            ON - Activates element shape checking.  New elements, regardless of how they are
                 created, are tested against existing warning and error limits.
                 (The existing limits may be the default limits, or previously
                 modified limits.)  Elements that violate error limits produce
                 error messages and either (a) cause a meshing failure, or (b)
                 for element creation or storage other than AMESH or VMESH, are
                 not stored.  Elements that violate warning limits produce
                 warning messages.  If shape checking was previously turned off
                 [SHPP,OFF] and you turn it on, existing elements are marked as
                 untested; use the CHECK command to retest them.  With this
                 option, you may also specify a value for VALUE1 to turn
                 individual shape tests on.  If you do not specify a value for
                 VALUE1, all shape tests are turned on.

            WARN - Activates element shape checking; however, in contrast to SHPP,ON, elements
                   that violate error limits do not cause either a meshing or
                   element storage failure.  Instead, they produce warning
                   messages to notify you that error limits have been violated.
                   This option does not alter current shape parameter limits.
                   Since the default shape parameter error limits are set to
                   allow almost any usable element, the elements this option
                   allows, which would otherwise be forbidden, are likely to be
                   very poorly shaped.

            OFF - Deactivates element shape checking.  This setting does not alter current shape
                  parameter limits.  Use of this option is risky, since poorly
                  shaped elements can lead to analysis results that are less
                  accurate than would otherwise be expected for a given mesh
                  density.  With this option, you may also specify a value for
                  VALUE1 to turn individual shape tests off.  If you do not
                  specify a value for VALUE1, all element shape tests are
                  turned off.

            SILENT - Determines whether element shape checking runs in silent mode.  In silent mode,
                     ANSYS checks elements without issuing warnings, with the
                     exception of the generic warnings that it issues at
                     solution.  With this option, you must also specify a value
                     for VALUE1  (During the execution of certain commands,
                     ANSYS automatically runs element shape checking in silent
                     mode, then internally summarizes the shape test results
                     for all of the new or modified elements.  ANSYS does this
                     when it executes any of the following commands:  AGEN,
                     AMESH, AREFINE, ARSYM, ATRAN, CDREAD, EGEN, ENGEN, ENSYM,
                     EREAD, EREFINE, ESYM, ET, FVMESH, KREFINE, LREFINE,
                     NREFINE, TIMP, VEXT, VGEN, VIMP, VMESH, VOFFST, VROTAT,
                     VSWEEP, VSYMM, and VTRAN.)

            STATUS - Lists the shape parameter limits currently in effect, along with status
                     information about element shape checking (for example,
                     whether any individual shape tests are off, whether any of
                     the shape parameter limits have been modified, and so on).

            SUMMARY - Lists a summary of element shape test results for all selected elements.

            DEFAULT - Resets element shape parameter limits to their default values.  Also, if any
                      individual tests were turned off, turns them back on.
                      (The SHPP,DEFAULT command may be useful if any parameter
                      limits were previously altered by using the MODIFY
                      option.)

            OBJECT - Determines whether element shape test results data is stored in memory.  When
                     this option is turned on, an "object" is created for
                     storing test results in memory.  When this option is
                     turned off, no object is created and no data is stored;
                     thus, any operation that requires shape parameters for an
                     existing element (such as use of the CHECK command) causes
                     the shape parameters to be recomputed.  (Note the
                     distinction between storing the data in memory and storing
                     it in the database; regardless of whether this option is
                     turned on or off, no element shape test results data will
                     be stored in the database.  The element shape parameter
                     object is deleted automatically before any solution.)
                     This setting is independent of shape checking status, with
                     one exception--if shape checking is turned off [SHPP,OFF],
                     the object is not created.  Keep in mind that recomputing
                     shape parameters is more computationally expensive than
                     retrieving them from the object.  With this option, you
                     must also specify a value for the VALUE1 argument; the
                     VALUE2 argument is ignored.

            LSTET - Determines, for Jacobian ratio tests, whether sampling is done at integration
                    points (DesignSpace product method), or at corner nodes.
                    When this option is turned on, sampling is done at
                    integration points, and the default limits for h-element
                    Jacobian ratios are a warning tolerance of 10 and an error
                    tolerance of 40. When this option is turned off, sampling
                    is done at corner nodes, and the corresponding default
                    limits are a warning tolerance of 30 and an error tolerance
                    of 1000. Sampling at the integration points (option on)
                    results in a lower Jacobian ratio, but that ratio is also
                    subjected to a more restrictive error limit. Some elements
                    that have passed the integration point sampling criterion,
                    have failed the corner mode sampling criterion. Because of
                    this, use integration point sampling only for simple linear
                    analyses. For other types of analyses (e.g., nonlinear,
                    electromagnetic), use sampling at corner nodes, which is
                    the more conservative approach. With this option, you must
                    also specify a value for the VALUE1 argument; the VALUE2
                    argument is ignored.

            MODIFY - Indicates that you want to respecify a shape parameter limit.  With this
                     option, you must also specify values for the VALUE1 and
                     VALUE2 arguments.

            FLAT - Determines the warning and error limits used to test elements that may exhibit
                   nonzero/nonconstant Z coordinates. With this option, you
                   must also specify values for the VALUE1 and/or VALUE2
                   arguments.

        value1
            Valid for the ON, OFF, SILENT, OBJECT, LSTET, MODIFY, and FLAT
            options only.  When Lab = ON or OFF, use VALUE1 to individually
            control (that is, turn off or turn on) specific element shape
            tests.  Thus, VALUE1 can be ANGD (SHELL28 corner angle deviation
            tests), ASPECT (aspect ratio tests), PARAL (deviation from
            parallelism of opposite edges tests), MAXANG (maximum corner angle
            tests), JACRAT (Jacobian ratio tests), WARP (warping factor tests),
            or ALL (all tests). When Lab = SILENT, VALUE1 can be ON (to turn
            silent mode on) or OFF (to turn silent mode off).  When Lab =
            OBJECT, VALUE1 can be either 1, YES, or ON to turn on storage of
            element shape test data (the default); or it can be 0, NO, or OFF
            to turn off storage of element shape test data (delete the data and
            recompute as necessary). When Lab = LSTET, VALUE1 can be either 1,
            YES, or ON to choose Jacobian sampling at integration points; or it
            can be 0, NO, or OFF to choose Jacobian sampling at nodes (the
            default). When Lab = MODIFY, VALUE1 is the numeric location (within
            the shape parameter limit array) of the shape parameter limit to be
            modified.  Locations are identified in the element shape checking
            status listing [SHPP,STATUS].  For more information, see the
            examples in the Notes section. When Lab = FLAT, VALUE1 is the
            warning limit for XY element constant Z sets performed at CHECK or
            SOLVE. The default is 1.0e-8.

        value2
            Valid for the MODIFY and FLAT options only.  When Lab = MODIFY,
            specifies the new limit for the shape parameter that is in the
            location indicated by the VALUE1 argument.  See the examples in the
            Notes section. When Lab = FLAT, VALUE2 is the error limit. The
            default is 1.0e-2.

        Notes
        -----
        The following examples illustrate how to use the
        SHPP,MODIFY,VALUE1,VALUE2 command to respecify shape parameter limits.
        Assume that you issued the SHPP,STATUS command, and you received the
        output below:

        Notice that in the sample output, the warning tolerance for aspect
        ratios is set to 20.  Now assume that you want to "loosen" this shape
        parameter limit so that it is less restrictive.  To allow elements with
        aspect ratios of up to 500 without causing warning messages, you would
        issue this command:

        Also notice that each shape parameter's numeric location within the
        shape parameter limit array appears in the sample output within
        parentheses.  For example, the numeric location of the aspect ratio
        shape parameter (for warning tolerance) is 1, which is why "1" is
        specified for the VALUE1 argument in the example command above.

        Now notice that the sample output indicates that any triangle element
        with an internal angle that is greater than 179.9 degrees will produce
        an error message.  Suppose that you want to "tighten" this shape
        parameter limit, so that it is more restrictive.  To cause any triangle
        or tetrahedron with an internal angle greater than 170 degrees to
        produce an error message, you would issue this command:

        The existence of badly shaped elements in a model may lead to certain
        computational errors that can cause your system to abort during ANSYS
        solution. Therefore, you run the risk of a system abort during solution
        any time that you turn element shape checking off entirely, run shape
        checking in warning-only mode, turn off individual shape checks, or
        loosen shape parameter limits.

        Changing any shape parameter limit marks all existing elements as
        untested; use the CHECK command to retest them.

        Since the shape parameter limit array was completely reorganized at
        ANSYS 5.4, you should revise any input files created prior to 5.4 that
        contain limit changes so that they reflect the reorganized data
        structure.

        For more information about element shape checking, see Meshing Your
        Solid Model in the Modeling and Meshing Guide.

        This command is also valid for rezoning.

        """
        command = "SHPP,%s,%s,%s" % (str(lab), str(value1), str(value2))
        self.RunCommand(command, **kwargs)

    def Clocal(self, kcn="", kcs="", xl="", yl="", zl="", thxy="", thyz="",
               thzx="", par1="", par2="", **kwargs):
        """
        APDL Command: CLOCAL

        Defines a local coordinate system relative to the active coordinate
        system.

        Parameters
        ----------
        kcn
            Arbitrary reference number assigned to this coordinate system.
            Must be greater than 10.  A coordinate system previously defined
            with this number will be redefined.

        kcs
            Coordinate system type:

            0 or CART - Cartesian

            1 or CYLIN - Cylindrical (circular or elliptical)

            2 or SPHE - Spherical (or spheroidal)

            3 or TORO - Toroidal

        xl, yl, zl
            Location (in the active coordinate system) of the origin of the new
            coordinate system (R, θ, Z for cylindrical, R, θ,Φ for spherical or
            toroidal).

        thxy
            First rotation about local Z (positive X toward Y).

        thyz
            Second rotation about local X (positive Y toward Z).

        thzx
            Third rotation about local Y (positive Z toward X).

        par1
            Used for elliptical, spheroidal, or toroidal systems.  If KCS = 1
            or 2, PAR1 is the ratio of the ellipse Y-axis radius to X-axis
            radius (defaults to 1.0 (circle)).  If KCS = 3, PAR1 is the major
            radius of the torus.

        par2
            Used for spheroidal systems.  If KCS = 2, PAR2 = ratio of ellipse
            Z-axis radius to X-axis radius (defaults to 1.0 (circle)).

        Notes
        -----
        Defines and activates a local coordinate system by origin location and
        orientation angles relative to the active coordinate system.  This
        local system becomes the active coordinate system, and is automatically
        aligned with the active system (i.e., x is radial if a cylindrical
        system is active, etc.).  Nonzero rotation angles (degrees) are
        relative to this automatic rotation.  See the CS, CSKP, CSWPLA, and
        LOCAL commands for alternate definitions.  Local coordinate systems may
        be displayed with the /PSYMB command.

        This command is valid in any processor.

        """
        command = "CLOCAL,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s" % (str(kcn), str(kcs), str(xl), str(yl), str(zl), str(thxy), str(thyz), str(thzx), str(par1), str(par2))
        self.RunCommand(command, **kwargs)

    def Xvar(self, n="", **kwargs):
        """
        APDL Command: XVAR

        Specifies the X variable to be displayed.

        Parameters
        ----------
        n
            X variable number:

            0 or 1 - Display PLVAR values vs. time (or frequency).

            n - Display PLVAR values vs. variable n (2 to NV [NUMVAR]).

            1 - Interchange time and PLVAR variable numbers with time as the curve parameter.
                PLVAR variable numbers are displayed uniformly spaced along
                X-axis from position 1 to 10.

        Notes
        -----
        Defines the X variable (displayed along the abscissa) against which the
        Y variable(s) [PLVAR] are to be displayed.

        """
        command = "XVAR,%s" % (str(n))
        self.RunCommand(command, **kwargs)

    def Edsolv(self, **kwargs):
        """
        APDL Command: EDSOLV

        Specifies "explicit dynamics solution" as the subsequent status topic.

        Notes
        -----
        This is a status [STAT] topic command. Status topic commands are
        generated by the GUI and will appear in the log file (Jobname.LOG) if
        status is requested for some items under Utility Menu> List> Status.
        This command will be immediately followed by a STAT command, which will
        report the status for the specified topic.

        If entered directly into the program, the STAT command should
        immediately follow this command.

        This command is also valid in PREP7.

        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "EDSOLV," % ()
        self.RunCommand(command, **kwargs)

    def Xrange(self, xmin="", xmax="", **kwargs):
        """
        APDL Command: /XRANGE

        Specifies a linear abscissa (X) scale range.

        Parameters
        ----------
        xmin
            Minimum abscissa scale value.

        xmax
            Maximum abscissa scale value.

        Notes
        -----
        Specifies a linear abscissa (X) scale range for the line graph display.
        Use /XRANGE,DEFAULT to return to automatic scaling.

        Automatic scaling will often yield inappropriate range values for
        logarithmic scales (/GROPT, LOGX).

        This command is valid in any processor.

        """
        command = "/XRANGE,%s,%s" % (str(xmin), str(xmax))
        self.RunCommand(command, **kwargs)

    def Rmrstatus(self, refname="", **kwargs):
        """
        APDL Command: RMRSTATUS

        Prints status of response surface for ROM function.

        Parameters
        ----------
        refname
            Reference name of ROM function. Valid reference names are "SENE"
            for the strain energy of the mechanical domain and any capacitance
            reference names [RMCAP], for the electrostatic domain.

        Notes
        -----
        Distributed ANSYS Restriction: This command is not supported in
        Distributed ANSYS.

        """
        command = "RMRSTATUS,%s" % (str(refname))
        self.RunCommand(command, **kwargs)

    def Quot(self, ir="", ia="", ib="", name="", facta="", factb="", **kwargs):
        """
        APDL Command: QUOT

        Divides two variables.

        Parameters
        ----------
        ir
            Arbitrary reference number assigned to the resulting variable (2 to
            NV [NUMVAR]).  If this number is the same as for a previously
            defined variable, the previously defined variable will be
            overwritten with this result.

        ia, ib
            Reference numbers of the two variables to be operated on.

        --
            Unused field.

        name
            Thirty-two character name identifying the variable on printouts and
            displays.  Embedded blanks are compressed for output.

        --, --
            Unused fields.

        facta, factb
            Scaling factors (positive or negative) applied to the corresponding
            variables (default to 1.0).

        Notes
        -----
        Divides two variables according to the operation:

        IR = (FACTA x IA)/(FACTB x IB)

        """
        command = "QUOT,%s,%s,%s,%s,%s,%s" % (str(ir), str(ia), str(ib), str(name), str(facta), str(factb))
        self.RunCommand(command, **kwargs)

    def Kesize(self, npt="", size="", fact1="", fact2="", **kwargs):
        """
        APDL Command: KESIZE

        Specifies the edge lengths of the elements nearest a keypoint.

        Parameters
        ----------
        npt
            Number of the keypoint whose lines will