# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import re
import json

from .common import InfoExtractor
from ..utils import (
    compat_str,
    qualities,
    determine_ext,
)


class AllocineIE(InfoExtractor):
    _VALID_URL = r'https?://(?:www\.)?allocine\.fr/(?P<typ>article|video|film)/(fichearticle_gen_carticle=|player_gen_cmedia=|fichefilm_gen_cfilm=)(?P<id>[0-9]+)(?:\.html)?'

    _TESTS = [{
        'url': 'http://www.allocine.fr/article/fichearticle_gen_carticle=18635087.html',
        'md5': '0c9fcf59a841f65635fa300ac43d8269',
        'info_dict': {
            'id': '19546517',
            'ext': 'mp4',
            'title': 'Astérix - Le Domaine des Dieux Teaser VF',
            'description': 'md5:abcd09ce503c6560512c14ebfdb720d2',
            'thumbnail': 're:http://.*\.jpg',
        },
    }, {
        'url': 'http://www.allocine.fr/video/player_gen_cmedia=19540403&cfilm=222257.html',
        'md5': 'd0cdce5d2b9522ce279fdfec07ff16e0',
        'info_dict': {
            'id': '19540403',
            'ext': 'mp4',
            'title': 'Planes 2 Bande-annonce VF',
            'description': 'md5:eeaffe7c2d634525e21159b93acf3b1e',
            'thumbnail': 're:http://.*\.jpg',
        },
    }, {
        'url': 'http://www.allocine.fr/film/fichefilm_gen_cfilm=181290.html',
        'md5': '101250fb127ef9ca3d73186ff22a47ce',
        'info_dict': {
            'id': '19544709',
            'ext': 'mp4',
            'title': 'Dragons 2 - Bande annonce finale VF',
            'description': 'md5:71742e3a74b0d692c7fce0dd2017a4ac',
            'thumbnail': 're:http://.*\.jpg',
        },
    }]

    def _real_extract(self, url):
        mobj = re.match(self._VALID_URL, url)
        typ = mobj.group('typ')
        display_id = mobj.group('id')

        webpage = self._download_webpage(url, display_id)

        if typ == 'film':
            video_id = self._search_regex(r'href="/video/player_gen_cmedia=([0-9]+).+"', webpage, 'video id')
        else:
            player = self._search_regex(r'data-player=\'([^\']+)\'>', webpage, 'data player')

            player_data = json.loads(player)
            video_id = compat_str(player_data['refMedia'])

        xml = self._download_xml('http://www.allocine.fr/ws/AcVisiondataV4.ashx?media=%s' % video_id, display_id)

        video = xml.find('.//AcVisionVideo').attrib
        quality = qualities(['ld', 'md', 'hd'])

        formats = []
        for k, v in video.items():
            if re.match(r'.+_path', k):
                format_id = k.split('_')[0]
                formats.append({
                    'format_id': format_id,
                    'quality': quality(format_id),
                    'url': v,
                    'ext': determine_ext(v),
                })

        self._sort_formats(formats)

        return {
            'id': video_id,
            'title': video['videoTitle'],
            'thumbnail': self._og_search_thumbnail(webpage),
            'formats': formats,
            'description': self._og_search_description(webpage),
        }
