import time
from contextlib import contextmanager

from win32evtlog import OpenEventLog, CloseEventLog, ClearEventLog, ReadEventLog
from win32evtlog import EVENTLOG_BACKWARDS_READ, EVENTLOG_SEQUENTIAL_READ
from win32evtlogutil import SafeFormatMessage


@contextmanager
def eventlog(name="Application"):
   "provide a clean event log via context manager"

   # first clear it
   log1 = OpenEventLog(None, name)
   ClearEventLog(log1, None)
   CloseEventLog(log1)

   # then re-open and return afresh
   log2 = OpenEventLog(None, name)
   yield log2

   # finally, close
   CloseEventLog(log2)


def wait_for_logmsg(log, msg, timeout=4):
   "return True if a string is found in a given log"

   for p in range(timeout * 5): # 0.2 second check rate
      entries = ReadEventLog(log, EVENTLOG_BACKWARDS_READ | EVENTLOG_SEQUENTIAL_READ, 0)
      logmsgs = [SafeFormatMessage(entry, "Application") for entry in entries]
      for logmsg in logmsgs:
         if msg in logmsg:
            return True
      time.sleep(0.2) # 0.2 second check rate
   raise Exception("timeout waiting for %s to be found in a log message" % msg)
