use std::mem;
use std::slice;
use std::os::raw::c_char;
use std::ffi::CStr;

use symbolic::common::{byteview::ByteView, types::Arch};
use symbolic::debuginfo::Object;
use symbolic::symcache::{InstructionInfo, SymCache, SYMCACHE_LATEST_VERSION};

use core::SymbolicStr;
use debuginfo::SymbolicObject;

/// Represents a symbolic sym cache.
pub struct SymbolicSymCache;

/// Represents a single symbol after lookup.
#[repr(C)]
pub struct SymbolicLineInfo {
    pub sym_addr: u64,
    pub line_addr: u64,
    pub instr_addr: u64,
    pub line: u32,
    pub lang: SymbolicStr,
    pub symbol: SymbolicStr,
    pub filename: SymbolicStr,
    pub base_dir: SymbolicStr,
    pub comp_dir: SymbolicStr,
}

/// Represents a lookup result of one or more items.
#[repr(C)]
pub struct SymbolicLookupResult {
    pub items: *mut SymbolicLineInfo,
    pub len: usize,
}

/// Represents an instruction info.
#[repr(C)]
pub struct SymbolicInstructionInfo {
    /// The address of the instruction we want to use as a base.
    pub addr: u64,
    /// The architecture we are dealing with.
    pub arch: *const SymbolicStr,
    /// This is true if the frame is the cause of the crash.
    pub crashing_frame: bool,
    /// If a signal is know that triggers the crash, it can be stored here (0 if unknown)
    pub signal: u32,
    /// The optional value of the IP register (0 if unknown).
    pub ip_reg: u64,
}

ffi_fn! {
    /// Creates a symcache from a given path.
    unsafe fn symbolic_symcache_from_path(path: *const c_char) -> Result<*mut SymbolicSymCache> {
        let byteview = ByteView::from_path(CStr::from_ptr(path).to_str()?)?;
        let cache = SymCache::new(byteview)?;
        Ok(Box::into_raw(Box::new(cache)) as *mut SymbolicSymCache)
    }
}

ffi_fn! {
    /// Creates a symcache from a byte buffer.
    unsafe fn symbolic_symcache_from_bytes(
        bytes: *const u8,
        len: usize,
    ) -> Result<*mut SymbolicSymCache> {
        let vec = slice::from_raw_parts(bytes, len).to_owned();
        let byteview = ByteView::from_vec(vec);
        let cache = SymCache::new(byteview)?;
        Ok(Box::into_raw(Box::new(cache)) as *mut SymbolicSymCache)
    }
}

ffi_fn! {
    /// Creates a symcache from a given object.
    unsafe fn symbolic_symcache_from_object(
        sobj: *const SymbolicObject,
    ) -> Result<*mut SymbolicSymCache> {
        let cache = SymCache::from_object(&*(sobj as *const Object))?;
        Ok(Box::into_raw(Box::new(cache)) as *mut SymbolicSymCache)
    }
}

ffi_fn! {
    /// Frees a symcache object.
    unsafe fn symbolic_symcache_free(scache: *mut SymbolicSymCache) {
        if !scache.is_null() {
            let cache = scache as *mut SymCache<'static>;
            Box::from_raw(cache);
        }
    }
}

ffi_fn! {
    /// Returns the internal buffer of the symcache.
    ///
    /// The internal buffer is exactly `symbolic_symcache_get_size` bytes long.
    unsafe fn symbolic_symcache_get_bytes(scache: *const SymbolicSymCache) -> Result<*const u8> {
        let cache = scache as *mut SymCache<'static>;
        Ok((*cache).as_bytes().as_ptr())
    }
}

ffi_fn! {
    /// Returns the size in bytes of the symcache.
    unsafe fn symbolic_symcache_get_size(scache: *const SymbolicSymCache) -> Result<usize> {
        let cache = scache as *mut SymCache<'static>;
        Ok((*cache).size())
    }
}

ffi_fn! {
    /// Returns the architecture of the symcache.
    unsafe fn symbolic_symcache_get_arch(scache: *const SymbolicSymCache) -> Result<SymbolicStr> {
        let cache = scache as *mut SymCache<'static>;
        Ok(SymbolicStr::new((*cache).arch()?.name()))
    }
}

ffi_fn! {
    /// Returns the architecture of the symcache.
    unsafe fn symbolic_symcache_get_id(scache: *const SymbolicSymCache) -> Result<SymbolicStr> {
        let cache = scache as *mut SymCache<'static>;
        Ok((*cache).id().map(|id| id.to_string()).unwrap_or_default().into())
    }
}

ffi_fn! {
    /// Returns true if the symcache has line infos.
    unsafe fn symbolic_symcache_has_line_info(scache: *const SymbolicSymCache) -> Result<bool> {
        let cache = scache as *mut SymCache<'static>;
        Ok((*cache).has_line_info()?)
    }
}

ffi_fn! {
    /// Returns true if the symcache has file infos.
    unsafe fn symbolic_symcache_has_file_info(scache: *const SymbolicSymCache) -> Result<bool> {
        let cache = scache as *mut SymCache<'static>;
        Ok((*cache).has_file_info()?)
    }
}

ffi_fn! {
    /// Returns the version of the cache file.
    unsafe fn symbolic_symcache_file_format_version(
        scache: *const SymbolicSymCache,
    ) -> Result<u32> {
        let cache = scache as *mut SymCache<'static>;
        Ok((*cache).file_format_version()?)
    }
}

ffi_fn! {
    /// Looks up a single symbol.
    unsafe fn symbolic_symcache_lookup(
        scache: *const SymbolicSymCache,
        addr: u64,
    ) -> Result<SymbolicLookupResult> {
        let cache = scache as *const SymCache<'static>;
        let vec = (*cache).lookup(addr)?;

        let mut items = vec![];
        for line_info in vec {
            items.push(SymbolicLineInfo {
                sym_addr: line_info.sym_addr(),
                line_addr: line_info.line_addr(),
                instr_addr: line_info.instr_addr(),
                line: line_info.line(),
                lang: SymbolicStr::new(line_info.lang().name()),
                symbol: SymbolicStr::new(line_info.symbol()),
                filename: SymbolicStr::new(line_info.filename()),
                base_dir: SymbolicStr::new(line_info.base_dir()),
                comp_dir: SymbolicStr::new(line_info.comp_dir()),
            });
        }

        items.shrink_to_fit();
        let rv = SymbolicLookupResult {
            items: items.as_mut_ptr(),
            len: items.len(),
        };
        mem::forget(items);
        Ok(rv)
    }
}

ffi_fn! {
    /// Frees a lookup result.
    unsafe fn symbolic_lookup_result_free(slr: *mut SymbolicLookupResult) {
        if !slr.is_null() {
            Vec::from_raw_parts((*slr).items, (*slr).len, (*slr).len);
        }
    }
}

ffi_fn! {
    /// Return the best instruction for an isntruction info.
    unsafe fn symbolic_find_best_instruction(ii: *const SymbolicInstructionInfo) -> Result<u64> {
        let real_ii = InstructionInfo {
            addr: (*ii).addr,
            arch: (*(*ii).arch).as_str().parse::<Arch>()?,
            crashing_frame: (*ii).crashing_frame,
            signal: if (*ii).signal == 0 { None } else { Some((*ii).signal) },
            ip_reg: if (*ii).ip_reg == 0 { None } else { Some((*ii).ip_reg) },
        };
        Ok(real_ii.caller_address())
    }
}

ffi_fn! {
    /// Returns the version of the cache file.
    unsafe fn symbolic_symcache_latest_file_format_version() -> Result<u32> {
        Ok(SYMCACHE_LATEST_VERSION)
    }
}
