"""
Logic function for AES cipher
"""


# Disable 'invalid-name', b/c 'logic' package uses unconventional names
# pylint: disable=C0103


from pyeda.boolalg.bfarray import exprzeros, uint2exprs, fcat


_SBOX = [
    0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5,
    0x30, 0x01, 0x67, 0x2b, 0xfe, 0xd7, 0xab, 0x76,
    0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0,
    0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0,
    0xb7, 0xfd, 0x93, 0x26, 0x36, 0x3f, 0xf7, 0xcc,
    0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
    0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a,
    0x07, 0x12, 0x80, 0xe2, 0xeb, 0x27, 0xb2, 0x75,
    0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0,
    0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84,
    0x53, 0xd1, 0x00, 0xed, 0x20, 0xfc, 0xb1, 0x5b,
    0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
    0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85,
    0x45, 0xf9, 0x02, 0x7f, 0x50, 0x3c, 0x9f, 0xa8,
    0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5,
    0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2,
    0xcd, 0x0c, 0x13, 0xec, 0x5f, 0x97, 0x44, 0x17,
    0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
    0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88,
    0x46, 0xee, 0xb8, 0x14, 0xde, 0x5e, 0x0b, 0xdb,
    0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c,
    0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79,
    0xe7, 0xc8, 0x37, 0x6d, 0x8d, 0xd5, 0x4e, 0xa9,
    0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
    0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6,
    0xe8, 0xdd, 0x74, 0x1f, 0x4b, 0xbd, 0x8b, 0x8a,
    0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e,
    0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e,
    0xe1, 0xf8, 0x98, 0x11, 0x69, 0xd9, 0x8e, 0x94,
    0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
    0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68,
    0x41, 0x99, 0x2d, 0x0f, 0xb0, 0x54, 0xbb, 0x16,
]

_ISBOX = [
    0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38,
    0xbf, 0x40, 0xa3, 0x9e, 0x81, 0xf3, 0xd7, 0xfb,
    0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87,
    0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb,
    0x54, 0x7b, 0x94, 0x32, 0xa6, 0xc2, 0x23, 0x3d,
    0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
    0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2,
    0x76, 0x5b, 0xa2, 0x49, 0x6d, 0x8b, 0xd1, 0x25,
    0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16,
    0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92,
    0x6c, 0x70, 0x48, 0x50, 0xfd, 0xed, 0xb9, 0xda,
    0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
    0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a,
    0xf7, 0xe4, 0x58, 0x05, 0xb8, 0xb3, 0x45, 0x06,
    0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02,
    0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b,
    0x3a, 0x91, 0x11, 0x41, 0x4f, 0x67, 0xdc, 0xea,
    0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
    0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85,
    0xe2, 0xf9, 0x37, 0xe8, 0x1c, 0x75, 0xdf, 0x6e,
    0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89,
    0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b,
    0xfc, 0x56, 0x3e, 0x4b, 0xc6, 0xd2, 0x79, 0x20,
    0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
    0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31,
    0xb1, 0x12, 0x10, 0x59, 0x27, 0x80, 0xec, 0x5f,
    0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d,
    0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef,
    0xa0, 0xe0, 0x3b, 0x4d, 0xae, 0x2a, 0xf5, 0xb0,
    0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
    0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26,
    0xe1, 0x69, 0x14, 0x63, 0x55, 0x21, 0x0c, 0x7d,
]

_RCON = [
    0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40,
    0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a,
    0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a,
    0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39,
    0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25,
    0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a,
    0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08,
    0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8,
    0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6,
    0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef,
    0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61,
    0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc,
    0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01,
    0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b,
    0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e,
    0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3,
    0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4,
    0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94,
    0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8,
    0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20,
    0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d,
    0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35,
    0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91,
    0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f,
    0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d,
    0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04,
    0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c,
    0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63,
    0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa,
    0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd,
    0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66,
    0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb,
]

_MA = [
    0x2311,
    0x1231,
    0x1123,
    0x3112,
]

_IMA = [
    0xebd9,
    0x9ebd,
    0xd9eb,
    0xbd9e,
]

# 16x16x8
SBOX = fcat(*[uint2exprs(x, 8) for x in _SBOX]).reshape(256, 8)
ISBOX = fcat(*[uint2exprs(x, 8) for x in _ISBOX]).reshape(256, 8)

# 255
RCON = fcat(*[uint2exprs(x, 8) for x in _RCON]).reshape(255, 8)

# 4x16
MA = fcat(*[uint2exprs(x, 16) for x in _MA]).reshape(4, 16)
IMA = fcat(*[uint2exprs(x, 16) for x in _IMA]).reshape(4, 16)


def subword(w):
    """
    Function used in the Key Expansion routine that takes a four-byte input word
    and applies an S-box to each of the four bytes to produce an output word.
    """
    w = w.reshape(4, 8)
    return SBOX[w[0]] + SBOX[w[1]] + SBOX[w[2]] + SBOX[w[3]]


def invsubword(w):
    """
    Transformation in the Inverse Cipher that is the inverse of SubBytes().
    """
    w = w.reshape(4, 8)
    return ISBOX[w[0]] + ISBOX[w[1]] + ISBOX[w[2]] + ISBOX[w[3]]


def rotword(w):
    """
    Function used in the Key Expansion routine that takes a four-byte word and
    performs a cyclic permutation.
    """
    w = w.reshape(4, 8)
    return w[1] + w[2] + w[3] + w[0]


def multiply(a, col):
    """Multiply a matrix by one column."""
    a = a.reshape(4, 4, 4)
    col = col.reshape(4, 8)
    return fcat(
        rowxcol(a[0], col),
        rowxcol(a[1], col),
        rowxcol(a[2], col),
        rowxcol(a[3], col),
    )


def rowxcol(row, col):
    """Multiply one row and one column."""
    row = row.reshape(4, 4)
    col = col.reshape(4, 8)
    ret = uint2exprs(0, 8)
    for i in range(4):
        for j in range(4):
            if row[i, j]:
                ret ^= xtime(col[i], j)
    return ret


def xtime(b, n):
    """Repeated polynomial multiplication in GF(2^8)."""
    b = b.reshape(8)
    for _ in range(n):
        b = exprzeros(1) + b[:7] ^ uint2exprs(0x1b, 8) & b[7]*8
    return b


def add_round_key(state, rkey):
    """
    Transformation in the Cipher and Inverse Cipher in which a Round Key is
    added to the State using an XOR operation. The length of a Round Key equals
    the size of the State (i.e., for Nb = 4, the Round Key length equals 128
    bits/16 bytes).
    """
    state = state.reshape(4, 32)
    rkey = rkey.reshape(4, 32)
    return fcat(
        state[0] ^ rkey[0],
        state[1] ^ rkey[1],
        state[2] ^ rkey[2],
        state[3] ^ rkey[3],
    )


def sub_bytes(state):
    """
    Transformation in the Cipher that processes the State using a non­linear
    byte substitution table (S-box) that operates on each of the State bytes
    independently.
    """
    state = state.reshape(4, 32)
    return fcat(
        subword(state[0]),
        subword(state[1]),
        subword(state[2]),
        subword(state[3]),
    )


def inv_sub_bytes(state):
    """
    Transformation in the Inverse Cipher that is the inverse of SubBytes().
    """
    state = state.reshape(4, 32)
    return fcat(
        invsubword(state[0]),
        invsubword(state[1]),
        invsubword(state[2]),
        invsubword(state[3]),
    )


def shift_rows(state):
    """
    Transformation in the Cipher that processes the State by cyclically shifting
    the last three rows of the State by different offsets.
    """
    state = state.reshape(4, 4, 8)
    return fcat(
        state[0][0], state[1][1], state[2][2], state[3][3],
        state[1][0], state[2][1], state[3][2], state[0][3],
        state[2][0], state[3][1], state[0][2], state[1][3],
        state[3][0], state[0][1], state[1][2], state[2][3]
    )


def inv_shift_rows(state):
    """
    Transformation in the Inverse Cipher that is the inverse of ShiftRows().
    """
    state = state.reshape(4, 4, 8)
    return fcat(
        state[0][0], state[3][1], state[2][2], state[1][3],
        state[1][0], state[0][1], state[3][2], state[2][3],
        state[2][0], state[1][1], state[0][2], state[3][3],
        state[3][0], state[2][1], state[1][2], state[0][3]
    )


def mix_columns(state):
    """
    Transformation in the Cipher that takes all of the columns of the State and
    mixes their data (independently of one another) to produce new columns.
    """
    state = state.reshape(4, 4, 8)
    return fcat(
        multiply(MA, state[0]),
        multiply(MA, state[1]),
        multiply(MA, state[2]),
        multiply(MA, state[3]),
    )


def inv_mix_columns(state):
    """
    Transformation in the Inverse Cipher that is the inverse of MixColumns().
    """
    state = state.reshape(4, 4, 8)
    return fcat(
        multiply(IMA, state[0]),
        multiply(IMA, state[1]),
        multiply(IMA, state[2]),
        multiply(IMA, state[3]),
    )


def key_expand(key, Nk=4):
    """Expand the key into the round key."""
    assert Nk in {4, 6, 8}
    Nr = Nk + 6

    key = key.reshape(Nk, 32)
    rkey = exprzeros(4*(Nr+1), 32)

    for i in range(Nk):
        rkey[i] = key[i]

    for i in range(Nk, 4*(Nr+1)):
        if i % Nk == 0:
            rkey[i] = rkey[i-Nk] ^ subword(rotword(rkey[i-1])) ^ RCON[i//Nk].zext(32-8)
        elif Nk > 6 and i % Nk == 4:
            rkey[i] = rkey[i-Nk] ^ subword(rkey[i-1])
        else:
            rkey[i] = rkey[i-Nk] ^ rkey[i-1]

    return rkey


def cipher(rkey, pt, Nk=4):
    """AES encryption cipher."""
    assert Nk in {4, 6, 8}
    Nr = Nk + 6

    rkey = rkey.reshape(4*(Nr+1), 32)
    pt = pt.reshape(128)

    # first round
    state = add_round_key(pt, rkey[0:4])

    for i in range(1, Nr):
        state = sub_bytes(state)
        state = shift_rows(state)
        state = mix_columns(state)
        state = add_round_key(state, rkey[4*i:4*(i+1)])

    # final round
    state = sub_bytes(state)
    state = shift_rows(state)
    state = add_round_key(state, rkey[4*Nr:4*(Nr+1)])

    return state


def inv_cipher(rkey, ct, Nk=4):
    """AES decryption cipher."""
    assert Nk in {4, 6, 8}
    Nr = Nk + 6

    rkey = rkey.reshape(4*(Nr+1), 32)
    ct = ct.reshape(128)

    # first round
    state = add_round_key(ct, rkey[4*Nr:4*(Nr+1)])

    for i in range(Nr-1, 0, -1):
        state = inv_shift_rows(state)
        state = inv_sub_bytes(state)
        state = add_round_key(state, rkey[4*i:4*(i+1)])
        state = inv_mix_columns(state)

    # final round
    state = inv_shift_rows(state)
    state = inv_sub_bytes(state)
    state = add_round_key(state, rkey[0:4])

    return state


def encrypt(key, pt, Nk=4):
    """Encrypt a plain text block."""
    assert Nk in {4, 6, 8}
    rkey = key_expand(key, Nk)
    ct = cipher(rkey, pt, Nk)

    return ct


def decrypt(key, ct, Nk=4):
    """Decrypt a plain text block."""
    assert Nk in {4, 6, 8}
    rkey = key_expand(key, Nk)
    pt = inv_cipher(rkey, ct, Nk)

    return pt

